/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*		THOMAS R. DAVIS				*/
/*							*/
/********************************************************/

/*---------------------------------------------------------------------------
-----------------------------------------------------------------------------

                     STRUCTURED DISPLAY FILE MANAGER
                         (support routines only)

This module supplies the structured display file manager (described in
sdf.c) with lower level routines necessary to implement a
hashed lookup scheme for the named display items.

It contains all the functions necessary to insert the global names in a
hashed structure containing pointers to local display_records.  The names
may be inserted in the structure by calling Insert(item) and then
retrieved at a later time via a call to Find(item).  An unneeded name
may be deleted from the hash tables by calling DeleteHash(item).
The hashing function used is simply item MOD hashsize.

                                Rocky Rhodes
                                February 18, 1982

-----------------------------------------------------------------------------
---------------------------------------------------------------------------*/

# include "Vgts.h"
# include "sdf.h"

#define SDFHASHSIZE 128

#define Hash(global) ((global>0) ? global : -global)%SDFHASHSIZE
                                      /* Use a macro expansion in place of
                                                     a hashing function.  */

typedef struct nlist  {
    DisplayRecord *dispptr;   /* pointer to the proper display record */
    short name;                   /* host's name for this item            */
    struct nlist *link;  /* link to other names hashed to the same bucket */
} HASH_RECORD, *HASH_RECORD_PTR;

static DisplayRecord *FreeDisplayList = NIL;

HASH_RECORD_PTR GetHashRecord()
{
  return((HASH_RECORD_PTR) GetStorageChunk(sizeof(HASH_RECORD)));
}

DisplayRecord *GetDisplayRecord()
{
    register DisplayRecord *dispRec;

/*
    if (FreeDisplayList != NIL)
      {
	dispRec = FreeDisplayList;
	FreeDisplayList = dispRec->rightSib;
      }
    else
*/
	dispRec = (DisplayRecord *)
    		GetStorageChunk(sizeof(DisplayRecord));
    if (dispRec != NIL)
      {
	    dispRec->rightSib = NIL;
	    dispRec->leftChild = NIL;
      }
    return(dispRec);
}

ReturnHashRecord(hashrec)
HASH_RECORD_PTR hashrec;
{
    free(hashrec);
/*
    hashrec->link = FreeHashList;
    FreeHashList = hashrec;
 */
}

ReturnDisplayRecord(disprec)
DisplayRecord *disprec;
{
    disprec->type = SDF_FREE;
/*
    disprec->rightSib = FreeDisplayList;
    FreeDisplayList = disprec;
*/
    free( disprec );
}

GetStorageChunk(size)
	/*
	 * All calls to malloc are done through this routine,
	 * so we can experiment with tuning various parameters.
	 */
  { return malloc(size); }


typedef short HASH_INDEX;   /* data type to index into the table.  */


static HASH_RECORD *hashtab[NumberSDF][SDFHASHSIZE];  /* hash table */



/* FindHash(sdf,item) is called with the Host's name as a parameter and
  will return a pointer to a hash record containing the pointer to the display
   record associated with this item.  A failure to find the named item is
   signaled by the return of a NIL pointer.  */

HASH_RECORD_PTR FindHash(sdf,item)
short sdf;
short item;
{
 register HASH_RECORD_PTR hashrec;

 if (item == NIL)
    return(NIL);
 
 hashrec = hashtab[sdf][Hash(item)];  /* Initial hashrecord in the proper
                                           bucket's list of records. */
 if (!hashrec) return(NIL);  /* If list is empty, the wanted record must
                                not be here! */
 while (TRUE)
    {
     if (hashrec->name == item) return(hashrec);  /* Found it! */
     if (!(hashrec = hashrec->link)) return(NIL);
           /* Get next record in the list.  If none left, report failure. */
    }
}


/* Find(item) is called with the Host's name as a parameter.  The
   procedure will return a pointer to the display record for this item.
   The difference between this procedure and the FindHash procedure is
   that this one returns the display_record pointer and FindHash returns
   the hash_record for the same item.  */

DisplayRecord *Find(sdf,item)
short sdf;
short item;
{
 HASH_RECORD_PTR hashrec;
 if (sdf<0 || sdf>=NumberSDF) return(NIL);
 if (!(hashrec=FindHash(sdf,item))) return(NIL);
 return(hashrec->dispptr);
}

/* Insert is called with the Host's name as a parameter.  The
   procedure will use this name to insert a pointer to a newly created
   display record into the proper hash bucket chain.  When called with an
   already existent name, the procedure will return a pointer to the existent
   display record with the leftChild field emptied.  */

DisplayRecord *Insert(sdf,item)
  short sdf;
  short item;
{
 DisplayRecord *disprec;
 HASH_RECORD_PTR hashrec, FindHash();
 HASH_INDEX bucketNum;

 if (sdf<0 || sdf>=NumberSDF) return(NULL);
 if (item && (hashrec=FindHash(sdf,item)))
                                  /* Return pointer to existent placeholder
                                     if possible.  Should only happen for
                                     symbol calls that create the placeholder
                                     before the actual definition is given. */
    disprec = hashrec->dispptr;
 else
   {
     disprec = GetDisplayRecord(); 
     if (disprec == NIL)
	 return(NIL);
     if (item != NIL)  /* NIL item means there's no need to
                                 enter this name in the hash table. */
       {        /* link a new hash record to the proper bucket */
	 hashrec = GetHashRecord();
	 if (hashrec == NIL)
	     return(NIL);
         hashrec->link =
            hashtab[sdf][bucketNum=Hash(item)];
         hashtab[sdf][bucketNum] = hashrec;
         hashrec->dispptr = disprec;
         hashrec->name=item;
       }
    }
 disprec->leftChild = NIL;  /* Mark this item as unexpanded in case this
                                is only a symbol call and not a definition. */
 return(disprec);
}

/* DeleteHash(item) is called with the Host's name as a parameter
   and will delete the corresponding hash record from the hash table.  This
   procedure does not delete any DisplayRecords, but only removes access
   to them.  */

DeleteHash(sdf, item)
short item;
{
 HASH_RECORD_PTR hashrec, saveHashRec;
 short globalHash;
/* DEBUG printf("DHash "); */
 if (!(hashrec=hashtab[sdf][globalHash = Hash(item)]))
    return; /* if list is empty item must already be deleted. */
 if (hashrec->name == item)
   {
    hashtab[sdf][globalHash]=hashrec->link;
    ReturnHashRecord(hashrec); /* return this record to storage manager. */
    return;
   }
 else
    while(hashrec->link!=NIL)
       {
        if (hashrec->link->name==item)
          {
	   saveHashRec = hashrec->link;
           hashrec->link=saveHashRec->link;
           ReturnHashRecord(saveHashRec);
           return;
          }
        hashrec=hashrec->link;
       }
}


FreeString(sdf,item)
  {
  	/*
	 * Free up the string space used by the indicated item.
	 * We first have to check to make sure the left Child pointer
	 * is not null or used for a pointer.
	 * We then set it to NULL so we do not try to free it twice.
	 */
    register DisplayRecord *p = Find(sdf,item);
    
    if (p==NULL || p->leftChild==NULL) return;
    if (p->type == SDF_SYMBOL_CALL || p->type == SDF_SYMBOL_DEF) return;
    if (p->type == SDF_RASTER) freeRaster(p->leftChild);
    if (p->type == SDF_TEXT)
      if ( ((TextStructure *) p->leftChild)->string )
	free(((TextStructure*)p->leftChild)->string);
    free(p->leftChild);
    p->leftChild = NULL;
  }

FreeLeftChild( p )
  register DisplayRecord *p;
  {
  	/*
	 * Free up the string space used by the indicated item.
	 * We first have to check to make sure the left Child pointer
	 * is not null or used for a pointer.
	 * We then set it to NULL so we do not try to free it twice.
	 */
    
    if (p==NULL || p->leftChild==NULL) return;
    if (p->type == SDF_SYMBOL_CALL ) return;
    if (p->type == SDF_RASTER) freeRaster(p->leftChild);
    if (p->type == SDF_TEXT)
      if ( ((TextStructure *) p->leftChild)->string )
	free(((TextStructure*)p->leftChild)->string);
    free(p->leftChild);
    p->leftChild = NULL;
  }


/* FreeSdfAll: frees the entire SDF */

FreeSdfAll(sdf,andStrings)
  short sdf;
  int andStrings;
{
    short i;
    HASH_RECORD_PTR hashPtr, saveHashPtr;

    for (i = 0; i < SDFHASHSIZE; i++)
      if (hashtab[sdf][i] != NIL)
       {
	hashPtr = hashtab[sdf][i];
	hashtab[sdf][i] = NIL;
	while(hashPtr != NIL)
	  {
	    FreeSdfDef(sdf, hashPtr->dispptr,andStrings);
	    saveHashPtr = hashPtr;
	    hashPtr = hashPtr->link;
	    ReturnHashRecord(saveHashPtr);
	  }
	}
}

FreeSdfDef(sdf,sdfPtr,andStrings)
short sdf;
DisplayRecord *sdfPtr;
int andStrings;
{
    DisplayRecord *saveSdfPtr;
    BOOLEAN loopDone;

    if (sdfPtr==NULL || sdfPtr->type == SDF_FREE)
        return;
    loopDone = FALSE;
    while (!loopDone)
      {
        if ( andStrings && sdfPtr->leftChild != NULL )            
	  {
	  	/* 
		 * if requested, free up the strings in the objects
		 */
             FreeLeftChild( sdfPtr );
	  }
        saveSdfPtr = sdfPtr;
        sdfPtr = sdfPtr->rightSib;
        ReturnDisplayRecord(saveSdfPtr);
        if (sdfPtr==NULL || sdfPtr->type == SDF_FREE)
	    loopDone = TRUE;
    }
}
