/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * File: graphdisp.c
 *
 * Based on a package for the Yale layout editor by Thomas R. Davis
 *
 * This file contains the routines used to draw graphical
 * displays.  DrawSegment() is the primary drawing "engine" and is
 * the routine called to interprete vgts (i.e. the
 * structured display file).  Is is only visible internally, however.
 * There are three externally visible drawing routines.  These resp.
 * redraw everything inside a bounding box, inside a window, and
 * inside the entire sdf.
 */

#include <Vioprotocol.h>
#include <Vgts.h>
#include <Vgtp.h>
#include "sdf.h"
#include "vgt.h"
#include "splines.h"

extern short ScreenLimitX, ScreenLimitY;
extern short *Patterns[];

/* attribute variables */
extern short fill_border_style;
extern short fill_area_color;
extern short fill_area_opacity;
extern short fill_pattern;
extern short marker_type;
extern short marker_color;
extern short polyline_color;
extern short text_color;
extern short font_index;


/* sdfHeader stays constant through all recursive levels of one redraw.
 * It gives us access to the header of the active SDF, allowing us to find
 * its attribute definition tables.
 */
SdfTableType *sdfHeader;


/* For a given set of viewport and world coordinates, we need to have
 * instantly available the coefficients for the transformations from
 * world to screen coordinates.  In the first implementation, all "zoom"
 * factors will be restricted to powers of two.
 *
 *    zoom
 *   2    * (WorldXmax - WorldXmin) = ScreenXmax - ScreenXmin
 *
 *    zoom
 *   2    * (WorldYmax - WorldYmin) = ScreenYmax - ScreenYmin
 *
 */

/* Given a point (XWorld, YWorld) in world coordinates, the following
 * transformations can be used to convert them to screen coordinates:
 *
 *                           zoom
 *   XScreen = ScreenXmin + 2    *(XWorld - WorldXmin)
 *
 *                           zoom
 *   YScreen = ScreenYmax - 2    *(YWorld - WorldYmin)
 *
 * The apparent asymmetry in the two equations above is due to the fact that
 * the screen coordinates are left-handed and the world coordinates are
 * right-handed.
 *
 * These equations can be simplified to:
 *
 *                       zoom
 *   XScreen = Xconst + 2    * XWorld
 *
 *                       zoom
 *   YScreen = Yconst - 2    * YWorld
 *
 * where:
 *
 *                          zoom
 *   Xconst = ScreenXmin - 2    * WorldXmin
 *
 *                          zoom
 *   Yconst = ScreenYmax + 2    * WorldYmin
 */

extern short Debug;

/*******************************************************************/


/*
 * ClearWorld:
 * Clears the region of the screen corresponding to the
 * input coordinates (which are in world coordinates).
 * Special allows one of the dimensions to not be scaled by the
 * zoom factor:
 *	0 - none
 *	1 - xmax
 *	2 - xmin
 *	3 - ymax
 *	4 - ymin
 */

ClearWorld(xmin, xmax, ymin, ymax, wp, special)
  int xmin, xmax, ymin, ymax;
  register View_PTR wp;
  short special;
{
    register struct SubView *r;
    int sxmin, sxmax, symin, symax, h, w;
    int rxmin, rxmax, rymin, rymax;

    sxmin = XcvtToScreen(xmin, wp);
    symax = YcvtToScreen(ymin, wp);
    sxmax = XcvtToScreen(xmax, wp);
    symin = YcvtToScreen(ymax, wp);
    switch (special) 
      {
	case 1:
	    sxmax = sxmin+xmax-xmin;
	    break;
	case 2:
	    sxmin = sxmax-xmax+xmin;
	    break;
	case 3:
	    symin = symax-ymax+ymin;
	    break;
	case 4:
	    symax = symin+ymax-ymin;
	    break;
      }

    for (r=wp->rect;r;r=r->next)
      {
        if (r->bannerFlag) continue;

    	rxmin = max(sxmin, r->ScreenXmin);
    	rymin = max(symin, r->ScreenYmin);
    	rxmax = min(sxmax, r->ScreenXmax);
    	rymax = min(symax, r->ScreenYmax);

    	h = rymax - rymin + 1;
    	w = rxmax - rxmin + 1;
    	if ((h > 0) && (w > 0))
        	ClearRectangle(rxmin, rymin, h, w, wp->showGrid);
	if (Debug & DebugVgtsBasicDraw)
	     dprintf("Clearing xmin=%d, ymin=%d, h=%d, w=%d\n",
    		     rxmin, rymin, h, w);
      }
}
  

/*******************************************************************/


/*
 * drawRefPoint:
 * This is the routine which is used to draw SILT reference
 * points in an arbitrary window.  It assumes that there is a value in
 * x or y which is the world coordinate (either x or y for vertical
 * and horizontal reference points, respectively), and that the xmin,
 * ..., ymax values completely surround all parts of the icon.  This is
 * not easy to do, since the reference point is half in screen
 * coordinates, and it is not clear how long and high the text might be.
 * The delete reference point routine will have to be very careful or very
 * wasteful.  The leftChild pointer is to the textual name for the
 * reference point.
 */

static drawRefPoint(sdf,wp,r)
  register DisplayRecord *sdf;
  register View_PTR wp;
  register struct SubView *r;
  {
    int screenX, screenY, screenMax, screenMin, barMin, barMax;
    
	if (Debug & DebugVgtsBasicDraw) 
		dprintf("drawRefPt xmin=%d, xmax=%d, ymin=%d, ymax=%d\n",
          		sdf->xmin, sdf->xmax, sdf->ymin, sdf->ymax);

    if (sdf->type == SDF_HORIZONTAL_REF || sdf->type == SDF_SEL_HORIZ_REF)
      {
        screenY = YcvtToScreen((sdf->ymax+sdf->ymin)/2, wp);
        screenMin = r->ScreenXmin;
        screenMax = r->ScreenXmax;
	if (r->EdgeSet&LeftEdge)
	    screenMin += 16;
	if (r->EdgeSet&RightEdge)
	    screenMax -= 16;

        if ((r->ScreenYmin < screenY) && (screenY < r->ScreenYmax))
    	    FillRegion (screenMin, screenY,
    	      (sdf->type == SDF_HORIZONTAL_REF) ? 1 : 2, 
	      screenMax - screenMin + 1);
	if (r->EdgeSet&LeftEdge)
	  {
	    if (sdf->leftChild)
              DrawSimpleText(screenMin+3, screenY-2, sdf->leftChild, r, TRUE);
            barMin = max(screenY - 4, r->ScreenYmin);
    	    barMax = min(screenY + 4, r->ScreenYmax);
            if (barMax - barMin > 0)
                FillRegion (screenMin, barMin, barMax - barMin + 1, 1);
	  }
      }
    else	/* it's a vertical reference point */
      {
        screenX = XcvtToScreen((sdf->xmax + sdf->xmin)/2, wp);
        screenMin = r->ScreenYmin;
        screenMax = r->ScreenYmax;
	if (r->EdgeSet&TopEdge) 
	    screenMin += 20;
	if (r->EdgeSet&BottomEdge) 
	    screenMax -= 16;

        if ((r->ScreenXmin < screenX) && (screenX < r->ScreenXmax))
    	  FillRegion(screenX, screenMin,
    	     screenMax - screenMin + 1,
    	     (sdf->type == SDF_VERTICAL_REF) ? 1 : 2);
	if (r->EdgeSet&TopEdge)
	  {
	    if (sdf->leftChild)
                DrawSimpleText(screenX - 8, screenMin -2,
				 sdf->leftChild, r, TRUE);
    
            barMin = max(screenX - 4, r->ScreenXmin);
            barMax = min(screenX + 4, r->ScreenXmax);
            if (barMax - barMin > 0)
                FillRegion (barMin, screenMin, 1, barMax - barMin + 1);
	  }
      }
  }
    

/*******************************************************************/


/*
 * DrawSegment:
 * Redraws everything inside the given bounding box.
 * It assumes that the segment in question has already been cleared.
 * It also makes sure that the segment is within the window.
 *
 * The outline parameter is used as an optimized special case for SDF_OUTLINEs
 * which are hollow and thus do not require that anything completely inside
 * them be re-drawn.
 */

DrawSegment(wXmin, wXmax, wYmin, wYmax, sdf, xtrans, ytrans, depth, wp, r, outline)
short wXmin, wXmax, wYmin, wYmax, xtrans, ytrans, depth;
  BOOLEAN *outline;
  register DisplayRecord *sdf;
  register View_PTR wp;
  register struct SubView *r;
  {
    int locXtrans, locYtrans;
    short Sxmin, Sxmax, Symin, Symax;	/* World coords of object */
    register int Rxmin, Rymin, Rxmax, Rymax; /* Screen Coords after clipping */
    int lenx,leny;			/* length of drawing rect in pixels*/
    int Tx, Ty;				/* Screen Coords of (sdf->x, sdf->y) */
    DisplayRecord *sdfSub;
    short edgeMask;
    int xmin, xmax, ymin, ymax;		/* Screen Coords before cliping */
    struct SubView r2;			/* The tightest redraw box possible */


    if (sdf == NIL)   return;
    
    wXmin = max(wXmin, wp->WorldXmin);
    wYmin = max(wYmin, wp->WorldYmin);
    wXmax = min(wXmax, wp->WorldXmax);
    wYmax = min(wYmax, wp->WorldYmax);

    if (((wXmax - wXmin) <= 0) || ((wYmax - wYmin) <= 0))
        return;			/* Non-overlapping boxes. */
    
    while (sdf->type != SDF_SYMBOL_DEF)
      {
        if (sdf->type == SDF_SYMBOL_CALL)
    	  {
	    if (Debug & DebugVgtsDraw)
		dprintf("doing item %d, symbol call to %d at %d, %d\n",
		    sdf->item, sdf->leftChild->item, sdf->x, sdf->y);
    	    locXtrans = xtrans + sdf->x;
    	    locYtrans = ytrans + sdf->y;
    	    sdfSub = sdf->leftChild;	/* points to symbol definition */
	    if (sdfSub == NIL)
		return;	/* bullet-proofing -- TRD */
    	    if ((sdfSub->xmin + locXtrans) < wXmax &&
	        (sdfSub->xmax + locXtrans) > wXmin &&
	        (sdfSub->ymin + locYtrans) < wYmax &&
	        (sdfSub->ymax + locYtrans) > wYmin) /* then we gotta expand */
    	    if (!*outline ||
		(sdfSub->xmin + locXtrans) < wXmin+1 ||
	        (sdfSub->xmax + locXtrans) > wXmax-1 ||
	        (sdfSub->ymin + locYtrans) < wYmin+1 ||
	        (sdfSub->ymax + locYtrans) > wYmax-1) /* symbol is not within outline */
		{ *outline = FALSE;
	    	    if (depth < wp->expansionDepth)
		      {
			PushAttributes();
	    	        DrawSegment(wXmin, wXmax, wYmin, wYmax,
				sdfSub->rightSib, locXtrans,
				locYtrans, depth + 1, wp,r,outline);
			PopAttributes();
		      }
		else
		  {	/* Draw the outline for the symbol */
		    Rxmin = XcvtToScreen(max(
			Sxmin=sdfSub->xmin + locXtrans, wXmin), wp);
		    Rxmax = XcvtToScreen(min(
			Sxmax=sdfSub->xmax + locXtrans, wXmax), wp);
		    Rymin = YcvtToScreen(min(
			Symax=sdfSub->ymax + locYtrans, wYmax), wp);
		    Rymax = YcvtToScreen(max(
			Symin=sdfSub->ymin + locYtrans, wYmin), wp);

	    	    Rxmin = max(Rxmin,r->ScreenXmin);
	    	    Rymin = max(Rymin,r->ScreenYmin);
	    	    Rxmax = min(Rxmax,r->ScreenXmax);
	    	    Rymax = min(Rymax,r->ScreenYmax);

		    edgeMask = 0;

		    if (Rxmin > r->ScreenXmin && Sxmin > wXmin)
			edgeMask += LeftEdge;
		    if (Rxmax < r->ScreenXmax && Sxmax < wXmax)
			edgeMask += RightEdge;
		    if (Rymin > r->ScreenYmin && Symax < wYmax)
			edgeMask += TopEdge;
		    if (Rymax < r->ScreenYmax && Symin > wYmin)
			edgeMask += BottomEdge;

		    if (((Rxmax - Rxmin) > 4) && ((Rymax - Rymin) > 4))
		        DrawOutline(Rxmin, Rymin, Rxmax-Rxmin+1,
					Rymax-Rymin+1, 1, edgeMask);
		    if (((Rxmax - Rxmin) > 40) && ((Rymax - Rymin) > 20))
		   	DrawSimpleText(Rxmin+4, Rymin+20,
					 sdfSub->leftChild, r, TRUE);
		  }
		}
	}
    /* attributes: non-spatial, so no bounding box checking */
    else if (sdf->type >= SDF_FILL_PATTERN && sdf->type <= SDF_MAX_ATTRIBUTE)
      {
	if (Debug & DebugVgtsDraw)
	    dprintf("doing item %d, attribute %d value %d\n", sdf->item,
	    sdf->type, sdf->typedata);
	switch (sdf->type)
	  {
	    case SDF_FILL_PATTERN:
		fill_pattern = sdf->typedata;
		break;

	    case SDF_FILL_AREA_OPACITY:
		fill_area_opacity = sdf->typedata;
		break;

	    case SDF_FILL_BORDER_STYLE:
		fill_border_style = sdf->typedata;
		break;

	    case SDF_FILL_AREA_COLOR:
		fill_area_color = sdf->typedata;
		break;

	    case SDF_MARKER_TYPE:
		marker_type = sdf->typedata;
		break;

	    case SDF_MARKER_COLOR:
		marker_color = sdf->typedata;
		break;

	    case SDF_POLYLINE_COLOR:
		polyline_color = sdf->typedata;
		break;

	    case SDF_TEXT_COLOR:
		text_color = sdf->typedata;
		break;

	    case SDF_FONT_INDEX:
		font_index = sdf->typedata;
		break;

	    case SDF_COLOR:  /* set all of them at once */
		fill_area_color = marker_color = polyline_color =
		text_color = sdf->typedata;
		break;
	  }
      }
    /* drawing primitives */
    else if ((Sxmin = sdf->xmin + xtrans) < wXmax &&
	     (Sxmax = sdf->xmax + xtrans) > wXmin &&
	     (Symin = sdf->ymin + ytrans) < wYmax &&
	     (Symax = sdf->ymax + ytrans) > wYmin &&	
    	     (!*outline ||
	      Sxmin < wXmin+1 ||
	      Sxmax > wXmax-1 ||
	      Symin < wYmin+1 ||
	      Symax > wYmax-1))/* then we gotta expand */
	  {

	    /*
	     * Sxmxx = Object rectangle, world coordinates (y = 0 at bottom)
	     * computed above
	     */
	    
	    /* 
	     * Object rectangle, screen coordinates (y = 0 at top)
	     */
	    ymax = YcvtToScreen(Symin, wp);
	    ymin = YcvtToScreen(Symax, wp);
	    xmax = XcvtToScreen(Sxmax, wp);
	    xmin = XcvtToScreen(Sxmin, wp);

	    /*
	     * Intersection of object and redraw rectangles, screeen 
	     * coordinates (y = 0 at top)
	     */
	    Rxmin = XcvtToScreen(max(Sxmin,wXmin), wp);
	    Rymin = YcvtToScreen(min(Symax,wYmax), wp);
	    Rxmax = XcvtToScreen(min(Sxmax,wXmax), wp);
	    Rymax = YcvtToScreen(max(Symin,wYmin), wp);

	    /*
	     * Object location using a reference point instead of a bounding
	     * box, screen coordinates
	     */
	    Tx = XcvtToScreen(sdf->x + xtrans, wp);
	    Ty = YcvtToScreen(sdf->y + ytrans, wp);

#if 0
	    Rxmin = max( xmin,r->ScreenXmin);
	    Rymin = max(Rymin,r->ScreenYmin);
	    Rxmax = min(Rxmax,r->ScreenXmax);
	    Rymax = min( ymax,r->ScreenYmax);
#endif 0

	    r2 = *r;	/*Temporarily replace clip with tighter rectangle*/

	    /* 
	     * Intersection of object, redraw, and subview recangles,
	     * Screen coordinates
	     */
	    r->ScreenXmin = Rxmin = max(Rxmin,r->ScreenXmin);
	    r->ScreenYmin = Rymin = max(Rymin,r->ScreenYmin);
	    r->ScreenXmax = Rxmax = min(Rxmax,r->ScreenXmax);
	    r->ScreenYmax = Rymax = min(Rymax,r->ScreenYmax);

	    if (Debug & DebugVgtsDraw && sdf->type != SDF_InternalText) 
		dprintf("drawing item %d, type %d at %d,%d\n",
		    sdf->item, sdf->type, sdf->x, sdf->y);

	    lenx = Rxmax-Rxmin+1;
	    leny = Rymax-Rymin+1;

	    if (lenx > 0 && leny > 0 )
	     {
	      if (*outline)
	         *outline = 2;  /* routines below set *outline to TRUE if the
				   outline optimization can be salvaged */
	      switch (sdf->type)
	        {
		case SDF_FILLED_RECTANGLE:
		    DrawRect(Rxmin, Rymin, lenx, leny, sdf->typedata);
		    if (fill_border_style == 0) break;
		    /* otherwise drop down to draw the outline */

		case SDF_OUTLINE:
		    /* the Rxmin, ... above give the dimensions of the
		     * rectangle to be drawn, but if it is reduced, we
		     * don't want to draw the whole outline.  Hence,
		     * we have the following monkey business to twiddle
		     * the selected edgeset:
		     */
		    edgeMask = 0;
		    if (Rxmin <= xmin)
			edgeMask += LeftEdge;
		    if (Rxmax >= xmax)
			edgeMask += RightEdge;
		    if (Rymin <= ymin)
			edgeMask += TopEdge;
		    if (Rymax >= ymax)
			edgeMask += BottomEdge;
		    {
		        register short tt;
			if ((tt=sdf->typedata & edgeMask)!=0)
		            DrawOutline(Rxmin,Rymin,lenx,leny,2,tt);
			else if (*outline == 2)
			    *outline = TRUE;
		    }
		    break;

		case SDF_HORIZONTAL_REF:
		case SDF_VERTICAL_REF:
		case SDF_SEL_HORIZ_REF:
		case SDF_SEL_VERT_REF:
		    drawRefPoint(sdf, wp, r);
		    break;

		case SDF_HORIZONTAL_LINE:
		    FillRegion(Rxmin, Rymin, 1, lenx);
		    break;

		case SDF_VERTICAL_LINE:
		    FillRegion(Rxmin, Rymin, leny, 1);
		    break;

		case SDF_POINT:
		    FillRegion(Rxmin, Rymin, min(2,leny),min(2,lenx));
		    break;

		/*
		 * The three text routines use the big rectangle so that
		 * zoomed text works halfway decently - Future work - Allow
		 * text size to change when zoomed - then, we can replace
		 * the next thre occurrences of &r2 (big rectangle) with r
		 * (small rectangle) 	-- GAF
		 */
		case SDF_InternalText:
		    DrawSimpleText( xmin, ymax, sdf->leftChild, &r2, FALSE);
		    break;

		case SDF_SIMPLE_TEXT:
		    DrawSimpleText( xmin, ymax, sdf->leftChild, &r2, TRUE);
		    break;

		case SDF_TEXT:
		  { register TextStructure *txt =
			(TextStructure*)sdf->leftChild;
		    DrawGeneralText( xmin + txt->xskip, ymax - txt->descent,
			txt->string, txt->length, &r2, 
		    	sdf->typedata, TRUE);
		    break;		
		  }

		case SDF_UpLine:
		case SDF_DownLine:
		    Rxmax = XcvtToScreen(Sxmax,wp);
		    Rymin = YcvtToScreen(Symax,wp);
   		    if (sdf->type==SDF_DownLine)
		        DisplayLine(xmin,Rymin,Rxmax,ymax,r);
		    else
		        DisplayLine(xmin,ymax,Rxmax,Rymin,r);
		    break;

		case SDF_RASTER:
    	  	    xmin = XcvtToScreen(Sxmin+1, wp);
		    Rymin = YcvtToScreen(Symax-1,wp);
		    if (sdf->leftChild)
		        DisplayRaster(sdf->leftChild, xmin, Rymin,
		 		wp, r, sdf->typedata);
		    break;

		case SDF_SPLINE:
		    /* It turns out that the line drawing algorithm is so much
		       faster for non-clipped streight lines, that it is faster
		       to draw the entire polygon than just the part needed.
		       so use the larger rectangle for clipping... GAF */
		    if (((SPLINE *)sdf->leftChild)->filled ||
			((SPLINE *)sdf->leftChild)->order!=2) /* clip small */
		        DisplaySpline( sdf->leftChild, Sxmin, Symin,
			    wp, r, sdf->typedata );
		    else /* clip big */
		        DisplaySpline( sdf->leftChild, Sxmin, Symin,
			    wp, &r2, sdf->typedata );
		    break;

		case SDF_POLYLINE:
		    DisplayPolyline(sdf->typedata, sdf->leftChild, 
			   Tx, Ty, wp->zoom, r);
		    break;

		case SDF_CIRCLE:
		  { short radius, cx, cy;
		    radius = (sdf->xmax - sdf->xmin) / 2;
		    cx = XcvtToScreen(sdf->xmin+radius, wp);
		    cy = YcvtToScreen(sdf->ymax-radius, wp);
		    if (wp->zoom > 0)  radius <<= wp->zoom;
		    else if (wp->zoom < 0)  radius >>= -wp->zoom;
		    Circle(cx, cy, radius, r);
		    break;
		  }

		case SDF_POLYARC:
		    DisplayPolyarc(sdf->typedata, sdf->leftChild, Tx, Ty, 
			   wp->zoom, r);
		    break;

		    
		default:
		    if (Debug & (DebugVgtsDraw | DebugVgtsErrors)) 
			dprintf("VGTS ERROR*** Bad SDF type =%d\n",sdf->type);
		    break;
		} /* end switch */
		if (*outline == 2)
		    *outline = FALSE;
	    } /* end if (Rxmax - Rxmin ... */
	  else if (Debug & DebugVgtsDraw && sdf->type != SDF_InternalText)
	    dprintf("draw of %d prevented by redraw rectangle\n", sdf->item);
	  *r = r2;
      } /* end drawing primitives : else if ((sdf->xmin ... */
    else if (Debug & DebugVgtsDraw && sdf->type != SDF_InternalText)
	dprintf("draw of %d prevented by window\n", sdf->item);

    sdf = sdf->rightSib;
    if (sdf == NIL)
	return;	/* bullet-proofing -- TRD */
  }  /* end while (sdf->type != SDF_SYMBOL_DEF) */
}


/*******************************************************************/

RedrawVGT(vgt, wXmin, wXmax, wYmin, wYmax)
  short vgt;
  int wXmin, wXmax, wYmin, wYmax;
{
  BOOLEAN outline = FALSE;
  RedrawVGT2(vgt, wXmin, wXmax, wYmin, wYmax, &outline);
}

RedrawVGTOutline(vgt, wXmin, wXmax, wYmin, wYmax)
  short vgt;
  int wXmin, wXmax, wYmin, wYmax;
{
  BOOLEAN outline = TRUE;
  RedrawVGT2(vgt, wXmin, wXmax, wYmin, wYmax, &outline);
}
 
RedrawVGT2(vgt, wXmin, wXmax, wYmin, wYmax, outline)
  short vgt;
  int wXmin, wXmax, wYmin, wYmax;
  BOOLEAN *outline;
{
    short windowId;
    DisplayRecord *sdf;
    register View_PTR wp = ViewTable;
    register struct SubView *r;

    if (vgt<0) return;

    wXmin = max(wXmin, -32760);
    wXmax = min(wXmax,  32760);
    wYmin = max(wYmin, -32760);
    wYmax = min(wYmax,  32760);
    
    for (windowId = 0; windowId < MAX_Views; windowId++, wp++)
        if (wp->vgt == vgt && wp->usage!=UNUSED)
          {
	    VGT *vgtPtr = &VGTtable[wp->vgt];
	    sdf = vgtPtr->topSymbol;

    	    if (sdf == NIL)
    	        continue;
	    if (Debug & DebugVgtsBasicDraw)
	        dprintf("Redrawing wXmin=%d, wXmax=%d, wYmin=%d, wYmax=%d\n",
			wXmin, wXmax, wYmin, wYmax);
  	    if (EraseCursor( max( XcvtToScreen(wXmin-2,wp), wp->ScreenXmin),
	    		 min( XcvtToScreen(wXmax+2,wp), wp->ScreenXmax),
	    		 max( YcvtToScreen(wYmax+2,wp), wp->ScreenYmin),
			 min( YcvtToScreen(wYmin-2,wp), wp->ScreenYmax) ))
		continue;
	    if (wp->usage & GRAPHICS)
		if (*outline) {
		    ClearWorld(wXmin,   wXmin+1, wYmin,   wYmax,   wp, 1);
		    ClearWorld(wXmax-1, wXmax,   wYmin,   wYmax,   wp, 2);
		    ClearWorld(wXmin,   wXmax,   wYmin,   wYmin+1, wp, 3);
		    ClearWorld(wXmin,   wXmax,   wYmax-1, wYmax,   wp, 4);
		}
		else
    	            ClearWorld(wXmin, wXmax, wYmin, wYmax, wp, 0);
	    for (r=wp->rect;r;r=r->next)
	      if (r->bannerFlag==FALSE)
		{
		  InitAttributes();
		  DrawSegment(wXmin-1, wXmax+1, wYmin-1, wYmax+1, 
		      sdf->rightSib, 0, 0, 0, wp, r, outline);
		}
            Cursor();
          }
}


RedrawPartialVGT(vgt, wXmin, wXmax, wYmin, wYmax, sdf)
  short vgt;
  int wXmin, wXmax, wYmin, wYmax;
  DisplayRecord *sdf;
{
	/*
	 * draw only added items in the given VGT
	 */
    short windowId;
    register View_PTR wp = ViewTable;
    register struct SubView *r;
    BOOLEAN outline = FALSE;

    if (vgt<0 || sdf==NULL) return;

    wXmin = max(wXmin, -32760);
    wXmax = min(wXmax,  32760);
    wYmin = max(wYmin, -32760);
    wYmax = min(wYmax,  32760);
    
    for (windowId = 0; windowId < MAX_Views; windowId++, wp++)
        if (wp->vgt == vgt && wp->usage!=UNUSED)
          {
	    if (Debug & DebugVgtsBasicDraw)
	        dprintf( "Redrawing wXmin=%d, wXmax=%d, wYmin=%d, wYmax=%d\n",
	            wXmin, wXmax, wYmin, wYmax);
  	    if (EraseCursor( max( XcvtToScreen(wXmin-2,wp), wp->ScreenXmin),
	    		 min( XcvtToScreen(wXmax+2,wp), wp->ScreenXmax),
	    		 max( YcvtToScreen(wYmax+2,wp), wp->ScreenYmin),
			 min( YcvtToScreen(wYmin-2,wp), wp->ScreenYmax) ))
		continue;

	    for (r=wp->rect;r;r=r->next)
	      if (r->bannerFlag==FALSE)
		{
		  InitAttributes();
    	          DrawSegment(wXmin-1, wXmax+1, wYmin-1, wYmax+1, sdf, 0, 0, 
			0, wp, r, &outline);
		}
            Cursor();
          }
}/*******************************************************************/


/*
 * RedrawEntireView:
 * Redraws all features of the window, including
 * the outline.
 */

RedrawEntireView(windowId)
short windowId;
{
    register DisplayRecord *sdf;
    register View_PTR wp = ViewTable + windowId;
    register struct SubView *r;
    VGT *vgtPtr;
    BOOLEAN outline = FALSE;

    if (wp->usage==UNUSED) return;
    if (EraseCursor(wp->ScreenXmin, wp->ScreenXmax, wp->ScreenYmin, wp->ScreenYmax))
        return;
    for (r=wp->rect;r;r=r->next)
      {
	if (r->bannerFlag)
	  {
	    if ( *(wp->banner) & 0200)
                DrawRect(r->ScreenXmin, r->ScreenYmin, 
    		  r->ScreenXmax - r->ScreenXmin + 1, 
		  r->ScreenYmax - r->ScreenYmin + 1,
		  BLACK );
	    else
                ClearRectangle(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1, 0);
	  }
	else 
            ClearRectangle(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1, 
		wp->showGrid);
        DrawOutline( r->ScreenXmin-2, r->ScreenYmin-2,
    	  r->ScreenXmax - r->ScreenXmin + 5, 
	  r->ScreenYmax - r->ScreenYmin + 5, 2, r->EdgeSet );

	vgtPtr = &VGTtable[wp->vgt];
	sdf = vgtPtr->topSymbol;

	if (r->bannerFlag)
	      DisplayBanner( wp, r);
         else
           if (sdf != NIL)
          {
              if (Debug & DebugVgtsBasicDraw)
              	  dprintf("Redoing vgt %d\n",wp->vgt);
	      InitAttributes();
              DrawSegment(wp->WorldXmin, wp->WorldXmax, wp->WorldYmin,
    		  	  wp->WorldYmax, sdf->rightSib, 0, 0, 0, wp, r,&outline);
          }
     }
    Cursor();
}


InvertVGT(vgt)
  short vgt;
{
    register View_PTR wp = ViewTable;
    register struct SubView *r;
    short windowId;

    for (windowId = 0; windowId < MAX_Views; windowId++, wp++)
        if (wp->vgt == vgt && wp->usage!=UNUSED)
         {
           if (EraseCursor(wp->ScreenXmin, wp->ScreenXmax, 
	     		     wp->ScreenYmin, wp->ScreenYmax))
        	return;
           for (r=wp->rect;r;r=r->next)
           {
            InvertRegion(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1);
           }
         Cursor();
       }
}

/*
 * RedrawBanner:
 * redraw ONLY the banner parts of a view
 */

RedrawBanner(wp)
    register View_PTR wp;
{
    register struct SubView *r;

    if (wp->usage==UNUSED) return;
    if (EraseCursor(wp->ScreenXmin, wp->ScreenXmax, wp->ScreenYmin, wp->ScreenYmax))
        return;
    for (r=wp->rect;r;r=r->next)
     if (r->bannerFlag)
      {
         if ( *(wp->banner) & 0200)
                DrawRect(r->ScreenXmin, r->ScreenYmin, 
    		  r->ScreenXmax - r->ScreenXmin + 1, 
		  r->ScreenYmax - r->ScreenYmin + 1, BLACK );
	    else
                ClearRectangle(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1, FALSE );
        DisplayBanner( wp, r);
     }
    Cursor();
}


/*******************************************************************/


/*
 * SdfExpand:
 * Redraws the entire sdf.  It assumes that all of the
 * global window parameters have been set.
 */

SdfExpand()
{
    short windowId;

    EraseCursor(0, ScreenLimitX, 0, ScreenLimitY);
    DrawBackground(0, 0, ScreenLimitX, ScreenLimitY);
    for (windowId =0; windowId < MAX_Views; windowId++)
    	RedrawEntireView(windowId);
}


/*******************************************************************/


/*
 * XcvtToScreen:
 * Converts world x coord. to screen x coord.
 */

XcvtToScreen(x,w)
  int x;
  register View_PTR w;  
{
    return(w->Xconst+((w->zoom>=0)?((x)<<w->zoom):((x)>>(-w->zoom))));
}


/*
 * YcvtToScreen:
 * Converts world y coord. to screen y coord.
 */

YcvtToScreen(y,w)
  int y;
  register View_PTR w;  
{
    return(w->Yconst-((w->zoom>=0)?((y)<<w->zoom):((y)>>(-w->zoom))));
}



