#include <Vioprotocol.h>
#include <rasterops.h>
#include "Vgts.h"
#include "sdf.h"
#include "vgt.h"

/* The "mixed" type, used to represent slopes, is a signed, fixed-point
 * mixed number.  The upper 16 bits represent the integer part; the lower
 * 16 bits represent the fraction.  The MixedQuotient macro produces mixed
 * values.
 */
typedef int mixed;

/* int / int -> mixed */
#define MixedQuotient(a, b)  (((a) << 16) / (b))


typedef struct edge {
	struct edge *next;
	short ymin, ymax;
	short xmin;
	mixed slope;
	short x;  unsigned short xlo; /* together they are one "mixed" value */
	}  Edge;

Edge *edgetable[1024];  /* one per scan line, for bucket sorting */
char leftstate[1024];	/* whether we are inside the polygon at left clip */

Edge *edgefree;		/* freelist */
Edge *edgetrash;	/* edges to be freed at end of this polygon */
Edge *horizedges;	/* special list of horizontal edges */
Edge *edgelist;		/* list of edges before bucket sorting */
Edge *MakeEdge();

/* The trashlist is used because of edge shortening.  We must be sure that
 * an edge record, even if deleted by clipping, does not get reused during
 * this polygon, because it might get shortened based on its pointer stored
 * in oldedge or firstedge.
 */
#define FreeEdge(e) \
  { \
    e->next = edgetrash; \
    edgetrash = e; \
  }

extern SdfPoint *TransformPoints();
static FreeWholeList();


/* aim values for remembering which way we are proceeding */
#define UP 1
#define DOWN 2

Edge BogusEdge;			/* for initializing edge linkage */
Edge *firstedge;
short firstaim;


/* The FillArea drawing routine.  Algorithm comes straight from
 * Foley & Van Dam, pp. 457-460.  Clipping is done in an opportunistic
 * way: we have to clip lines that extend above the rectangle so we can
 * bucket sort them, but we do nothing about lines that extend below; we
 * simply quit scanning at the bottom of the rectangle.  Similarly for
 * clipping on the right.
 *
 * Input to this routine is an array of npoints SdfPoint elements, of
 * which the last must be the same as the first (else this routine will
 * probably crash!)  All x,y coordinates stored in that array are relative to
 * the (xoffset, yoffset) passed to the function.
 */
FillArea(npoints, indata, pattern, xoffset, yoffset, zoom, clip)
  short xoffset, yoffset, npoints, zoom;
  SdfPoint indata[];
  unsigned pattern[];  /* array of 16 shorts, the fill pattern */
  register struct SubView *clip;
  {
    SdfPoint *data;
    register SdfPoint *p;
    register Edge *edge, *oldedge, *temp;
    short aim, oldaim, xbot;
    register int i;

    /* for zoom, and eventually rotate, etc. we make a transformed copy	
     * of the input data array.
     */
    if (zoom == 0) data = indata;
    else data = TransformPoints(npoints, indata, zoom);
    if (!data) return;

    horizedges = edgelist = edgetrash = NULL;
    for (i=clip->ScreenYmin; i <= clip->ScreenYmax; i++)
	edgetable[i] = NULL;

    oldaim = 0;  /* neither up nor down, prevents edge shortening 1st time */
    firstaim = 0;
    oldedge = &BogusEdge;

    /* Convert point data into edge records, computing absolute coordinates */
    for (p = data; p < data + npoints - 1; p++)
      {
	if (p->y > (p+1)->y)
	  {
	    edge = MakeEdge(p, p+1, xoffset, yoffset);
	    aim = DOWN;
	    xbot = (p+1)->x;
	  }
	else if (p->y < (p+1)->y)
	  {
	    edge = MakeEdge(p+1, p, xoffset, yoffset);
	    aim = UP;
	    xbot = p->x;
	  }
	else  /* horizontal, special treatment */
	  {
	    HorizontalEdge(MakeEdge(p, p+1, xoffset, yoffset), clip);
	    continue;
	  }

	if (!edge) return;  /* malloc failed, can't display */
	if (firstaim == 0)  /* remember the first nonhorizontal edge */
	  {
	    firstedge = edge;
	    firstaim = aim;
	  }

	/* edge shortening to avoid double-counting at vertices */
	if (oldaim == DOWN && aim == DOWN)
	    oldedge->ymax -= 1;
	else if (oldaim == UP && aim == UP)
	    edge->ymax -= 1;
	oldaim = aim;
	oldedge = edge;

	/* edgelist forms in reverse order */
	edge->next = edgelist;
	edgelist = edge;
      } /* end loop on points */

    /* edge shortening: last edge to first edge */
    if (aim == DOWN && firstaim == DOWN)
	edge->ymax -= 1;
    else if (aim == UP && firstaim == UP)
	firstedge->ymax -= 1;

    /* clip and sort the edges into edgetable, with each bucket sorted
     * on xmin
     */
    for (edge = edgelist; edge; edge = temp)
      {
	temp = edge->next;

	/* clip if entirely above; ymax < ymin is possible due to shortening */
	if (edge->ymax < clip->ScreenYmin) {FreeEdge(edge);}
	/* fine clipping at top edge */
	else if (edge->ymin < clip->ScreenYmin)
	  { register mixed t;

	    t = edge->slope * (clip->ScreenYmin - edge->ymin);
	    edge->xmin += (t >> 16);
	    edge->x = edge->xmin;
	    edge->xlo = t & 0xffff;
	    edge->ymin = clip->ScreenYmin;
	    StoreEdge(edge);
	  }
	/* clip if entirely below */
	else if (edge->ymin > clip->ScreenYmax)  {FreeEdge(edge);}
	/* edges that are partially below will take care of themselves */
	else  StoreEdge(edge);
      } /* end loop on edges */

    /* Use data stored in edgetable to draw the image */
    ScanFillPolygon(pattern, clip);

    FreeWholeList(edgetrash);
    if (zoom) free(data);
  }


Edge *MakeEdge(p1, p2, xoffset, yoffset)
  register SdfPoint *p1, *p2;
  register short xoffset, yoffset;  /* must declare exactly 2 data registers */
  {
    register Edge *edge;

    if (!edgefree)
      {
	edge = (Edge *) malloc(sizeof(Edge));
	if (!edge)  return(NULL);
      }
    else
      {
	edge = edgefree;
	edgefree = edge->next;
      }
    
    edge->ymin = yoffset - p1->y;  /* - handles reversed y coordinate system */
    if (p1->y == p2->y)  /* infinite slope, horizontal */
      { /* special usage: ymax field is used to store xmax */
	if (p1->x > p2->x)
	  { 
	    edge->xmin = xoffset + p2->x;
	    edge->ymax = xoffset + p1->x;
	  }
	else
	  {
	    edge->xmin = xoffset + p1->x;
	    edge->ymax = xoffset + p2->x;
	  }
      }
    else
      {
	edge->ymax = yoffset - p2->y;
	edge->x = edge->xmin = xoffset + p1->x;
	edge->xlo = 0x8000; /* fractional part = 1/2 to counteract truncation */
	edge->slope = MixedQuotient(p2->x - p1->x, p1->y - p2->y);
      }
    return(edge);
  }


/* Bucket sort on ymin; with each bucket, insertion sort on xmin */
StoreEdge(edge)
  register Edge *edge;
  {
    Edge **bucket;
    register Edge *p;

    bucket = &edgetable[edge->ymin];
    if (*bucket == NULL || edge->xmin < (*bucket)->xmin)
      {
	edge->next = *bucket;
	*bucket = edge;
      }
    else 
      {
	for (p = *bucket; p->next && edge->xmin > p->next->x; p = p->next) ;
	edge->next = p->next;
	p->next = edge;
      }
  }


/* Clipping and storing of horizontal edges */
HorizontalEdge(edge, clip)
  register Edge *edge;
  register struct SubView *clip;
  {
    if (edge->ymin < clip->ScreenYmin ||
	edge->ymin > clip->ScreenYmax ||
	edge->ymax < clip->ScreenXmin ||
	edge->xmin > clip->ScreenXmax)
      {
	FreeEdge(edge);
      }
    else
      {
	if (edge->xmin < clip->ScreenXmin) edge->xmin = clip->ScreenXmin;
	if (edge->ymax > clip->ScreenXmax) edge->ymax = clip->ScreenXmax;
	edge->next = horizedges;
	horizedges = edge;
      }
  }



/* macro to update the current x value of an edge */
#define newx(e) \
  { \
    t = e->xlo + (e->x << 16) + e->slope; \
    e->xlo = t & 0xffff; \
    e->x = t >> 16; \
  }

extern short fill_area_color, fill_area_opacity;

Edge *activedges;  /* Head of the active edge table */
extern Edge BogusLeftEdge;  /* in fillpolyarc.c */

ScanFillPolygon(pattern, clip)
  unsigned pattern[];
  register struct SubView *clip;
  {
    register Edge *edge, *p, *q;
/* newedge and temp are logically separate variables from p and q, but I
 * need them to occupy the same registers with p and q! */
#define temp p
#define newedge q
    register short y, x1, x2;
    register int t;
    short on;

    /* set the framebuffer function once and for all */
    SetFillFunction(fill_area_color, fill_area_opacity);

    activedges = &BogusLeftEdge;  /* simplifies insertion */
    activedges->next = NULL;
    for (y = clip->ScreenYmin; y <= clip->ScreenYmax; y++)
      {
	/* remove outdated edges */
	for (edge = activedges; edge->next; )
	  {
	    if (edge->next->ymax < y)
	      {
		temp = edge->next;
		edge->next = edge->next->next;
		temp->next = edgefree;
		edgefree = temp;
	      }
	    else  edge = edge->next;
	  }

	/* update x values and make sure active edge list stays sorted in x */
	{
	    edge = activedges;
	    p = edge->next;
	    if (!p)  goto newedges;  /* see if there are new edges to insert */
	    newx(p);
	    q = p->next;
	    while(q)
	      {
		newx(q);
		if (p->x > q->x)  /* time for a swap */
		  {
		    edge->next = q;
		    p->next = q->next;
		    q->next = p;
		    edge = q;
		  }
		else
		  {
		    edge = p;
		    p = q;
		  }
		q = p->next;
	      }
	  }

	/* insert new edges that we may have come to */
newedges:
	edge = activedges;
	newedge = edgetable[y];  /* most common case: empty */
	while (newedge)
	  {
	    /* insert it into the list at the right place */
	    while (edge->next && edge->next->x < newedge->x)  edge = edge->next;
	    temp = newedge->next;
	    newedge->next = edge->next;
	    edge->next = newedge;
	    newedge = temp;
	  }
	
	/* The drawing itself */
	on = 0;
	for (edge = activedges->next; edge && edge->x < clip->ScreenXmin; 
		edge = edge->next)
	    on ^= 1;
	if (on)
	  {
	    x1 = clip->ScreenXmin;
	    x2 = edge->x;
	    if (x2 > clip->ScreenXmax)  x2 = clip->ScreenXmax;
	    FillRow(y, x1, x2, pattern);
	    edge = edge->next;
	  }
	while (edge && edge->x <= clip->ScreenXmax)
	  {
	    x1 = edge->x;
	    edge = edge->next;
	    x2 = edge->x;
	    if (x2 > clip->ScreenXmax)  x2 = clip->ScreenXmax;
	    FillRow(y, x1, x2, pattern);
	    edge = edge->next;
	  }
      }  /* end loop on scan lines */
    FreeWholeList(activedges->next);

    /* now we draw in all the horizontals */
    for (edge = horizedges; edge; edge = edge->next)
	FillRow(edge->ymin, edge->xmin, edge->ymax, pattern);
    FreeWholeList(horizedges);
  } /* end ScanFillPolygon */


static FreeWholeList(head)
  Edge *head;
  {
    register Edge *e;
    if (!head) return;
    for (e = head; e->next; e = e->next) ;
    e->next = edgefree;
    edgefree = head;
  }
