/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* execs.c - VGTS executive control routines
 *
 * This file contains the routines to handle the Exec Control
 * option of the View manager in the vgtsexec program.
 *
 * Bill Nowicki April 1983
 *
 */

#include <Venviron.h>
#include <Vioprotocol.h>
#include <Vtermagent.h>
#include "Vgts.h"
#include <Vexec.h>
#include <Vprocess.h>
#include "sdf.h"
#include "vgt.h"
#include "interp.h"
#include "pad.h"

/* for debugging ease */
#define static

extern short MouseX, MouseY, MouseButtons, InputVGT, TtySDF, Unique;
extern short Debug;		/* imported from master mux */

extern struct Buffer *CreateBuffer();

ProcessId execserver;

#define C_CreateExec 1
#define C_DeleteExec 2
#define C_KillProgram 3
#define C_BigCreateExec 4
#define C_SetSize 5
#define C_TogglePagedOutput 6
#define C_InterruptProgram 7
#define C_ResetState 8

#define BiggestExecSize 1000

char bigMessage[128] = "Create 48 Line Executive";
short bigExecSize = 48;

static PopUpEntry execMenu[] =
    {
      "Toggle Paged Output Mode", C_TogglePagedOutput,
      bigMessage, C_BigCreateExec,
      "Set Alternate Exec Size", C_SetSize,
      "Create 28 Line Executive", C_CreateExec,
      "Reset State", C_ResetState,
      "Interrupt Program", C_InterruptProgram,
      "Kill Program", C_KillProgram,
      "Delete Executive", C_DeleteExec,
      0, 0
    };


short ExecControl(pad)
{
  short newExecSize;

  execserver = GetPid(EXEC_SERVER, LOCAL_PID);
  switch (popup(execMenu))
    {
      case C_CreateExec:
          return(DoCreateExec(28));
	  break;

      case C_SetSize:
          newExecSize = TtyGetNumber("Enter number of lines for executives:");
	  if (newExecSize > BiggestExecSize || newExecSize == 0)
	    {
	      TtyMessage("Bad length for executive");
	    }
	  else /* ok */
	    {
	      bigExecSize = newExecSize;
	      sprintf( bigMessage, "Create %d line Executive", bigExecSize );
	    }
	  break;

      case C_BigCreateExec:
          return(DoCreateExec(bigExecSize));
	  break;

      case C_DeleteExec:
          return(DoDeleteExec());
	  break;

      case C_KillProgram:
          DoKillProgram(FALSE);
	  break;

      case C_InterruptProgram:
          DoKillProgram(TRUE);
	  break;

      case C_TogglePagedOutput:
          DoTogglePagedOutput();
	  break;

     case C_ResetState:
     	  DoResetState();
	  break;

     default:
          break;
    }
   return(pad);
}


static DoKillProgram(interrupt)
  {
    short view, vgtindex;
    register Term1 *term1;
    register Term2 *term2;
    SystemCode r; 

    TtyPutString(interrupt ? "Interrupt Program B: " : "Kill Program B: ");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
     if ((view = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError("Viewport not found");
        return;
      }
    vgtindex = FindTopVGT(MouseX, MouseY);
    if (vgtindex<0) return;
    term1 = Term1Table + vgtindex;
    term2 = term2Ptr(term1);
    if (term2 && term2->isexec)
      {
        if (interrupt)
	    r = (term2->termPid ? ForceException(term2->termPid) :
	    			   NONEXISTENT_PROCESS);
	else
	  {
	    r = KillProgram(execserver, term2->execid);
	    term2->termPid = 0;	/* Unregister the break process */
	  }
	if (r == OK) TtyPutString("\r\n");
	else TtyBlinkError("No program");
      }
    else TtyBlinkError("No Executive in that view");
  }


static DoTogglePagedOutput()
  {
    short view, vgtindex;
    register Term1 *term1;
    register Term2 *term2;
    SystemCode r; 

    TtyPutString("Click in desired Pad.\r\n");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
     if ((view = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError("Viewport not found");
        return;
      }
    vgtindex = FindTopVGT(MouseX, MouseY);
    if (vgtindex<=0) return;
    term1 = Term1Table + vgtindex;
    term2 = term2Ptr(term1);

    term1->mode ^= PageOutputEnable;
    
    if (term1->mode & PageOutputEnable)
      {
	term1->mode |= PageOutput;
	padPtr(term1)->newlinesLeft = padPtr(term1)->pageLength;
        TtyPutString("Paged Output Mode ENABLED.\r\n");
      }
    else
      {
	if (term2->writerPid)
	  {
	    term2->outputFlag |= PageOutputShutdown;
	    WriteUnblock(term1, ' ');
	  }
	else term1->mode &= ~PageOutput;
        TtyPutString("Paged Output Mode DISABLED.\r\n");
      }
  }


static DoResetState()
  {
    short view, vgtindex;
    register Term1 *term1;
    register Term2 *term2;
    register struct TtyPadRec *pad;
    SystemCode r; 

    TtyPutString("Reset state of pad B:");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
     if ((view = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError(" Viewport not found");
        return;
      }
    vgtindex = ViewTable[view].vgt;
    if (vgtindex==0) 
      {
	TtyPutString("\r\n");
        return;
      }
    term1 = Term1Table + vgtindex;
    pad = padPtr(term1);

    pad->regionTop = 0;
    pad->regionBot = pad->length;
    pad->standout = 0;
    pad->state = Normal;
    pad->newlinesLeft = pad->pageLength = pad->length;
    
    TtyPutString(" Reset\r\n");
    
  }


static DoDeleteExec()
  {
    register Term1 *term1;
    register Term2 *term2; 
    short view, vgtindex;

    TtyPutString("Delete Exec B: ");
    if (GetMouseInput() == 7)
      {
	TtyPutString(" Aborted\r\n");
        return(InputVGT);
      }
     if ((view = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError("Viewport not found");
        return(InputVGT);
      }
    vgtindex = FindTopVGT(MouseX, MouseY);
    if (vgtindex<0) return;
    term1 = Term1Table + vgtindex;
    term2 = term2Ptr(term1);

    if (term2 && term2->isexec)
      {
	DeleteExec(execserver, term2->execid);
	term2->isexec = 0;
	if ( term2->lineEditBuf )
	    free( term2->lineEditBuf );
	term2->lineEditBuf = NULL;
        TtyPutString("\r\n");
      }
    else TtyBlinkError("No Executive in that view");
    return(vgtindex);
  }


extern InitExec();

static DoCreateExec(size)
    short size;		/* lines in desired pad */
  {
    short xCorner, yCorner, height, width;
    short vgtindex;

    height = ViewHeight(size);
    width = ViewWidth(PadWidth);

    TtyPutString("Creating an Exec, upper left corner at B: ");
    SetExecCursor();
    WaitForDownButton();
    xCorner = MouseX;
    yCorner = MouseY;
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    while (MouseButtons!=0)
      {
        ReadMouseInput();

	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
	if (MouseButtons==7)
          {
            SetMainCursor();
	    TtyPutString(" Aborted\r\n");
    	    return(InputVGT);
      	 }
        xCorner = MouseX;
        yCorner = MouseY;
	CursorUpdate();
	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    }
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    SetMainCursor();

    vgtindex = NewTextVGT(TtySDF, "Executive", size, PadWidth);
    if (vgtindex<0)
      {
        TtyBlinkError("Can't create exec");
        return(InputVGT);
      }

    CreateView( vgtindex, xCorner, yCorner, 
        	xCorner+width, yCorner+height, 0, 0, 0, 0);

    /* we've made the view, now start an exec in it */
    TtyPutString("\r\n");

    SpawnExec(vgtindex);
    return(vgtindex);
  }

SpawnExec(vgtindex)
short vgtindex;
  {
    ProcessId initpid;
    register Term1 *term1;
    register Term2 *term2;
    
    term1 = Term1Table + vgtindex;
    term2 = term2Ptr(term1);
    term1->mode = CR_Input+LF_Output+PageOutputEnable;
    				         /* must be done before starting exec */
    term1->vgt = vgtindex;
    term1->master = -1;
    term2->isexec = 1;
    term2->lineEditBuf = CreateBuffer();
    term2->bytesWritten = 0;
    term2->outputFlag = 0;
    term1->reader = 0;
    term2->writerPid = 0;

    /* outrageous hack warning: InitExec is called as a separate process
     * because the exec server must be able to QueryInstance back to the
     * VGTS to open the exec's i/o files.  So DoCreateExec must return
     * while InitExec is still waiting on the exec server.
     */
    initpid = Create(10, InitExec, 2048);
    term1->owner = initpid;
    Ready(initpid, 1, vgtindex);
  }

static char errbuf[120];

/*
 * InitExec: start up the exec, directing its standard input, output, and
 * error files to our new pad, and its context to [home].
 */
InitExec(vgtindex)
 short vgtindex;
  {
    register Term1 *term1 = Term1Table + vgtindex;
    register Term2 *term2 = term2Ptr(term1);
    ProcessId execpid;
    SystemCode error;
    short execflags;
    struct InterpUser *i;
    
    extern struct InterpUser *CreateInterp();

    if (term1->interp==0)
      {
        i = CreateInterp(vgtindex);
        if (i==NULL) return(0);
        term1->interp = i;
      }

    execflags = RELEASE_INPUT + RELEASE_OUTPUT + RELEASE_ERR;

    term2->execid = CreateExec(execserver, VgtsPid, vgtindex, VgtsPid, vgtindex,
				VgtsPid, vgtindex, 0, 0,
				execflags, &execpid, &error);
    if (error)
      {
	sprintf(errbuf, "can't create exec: %s", ErrorString(error));
	TtyBlinkError(errbuf);
	term2->isexec = 0;
        DeleteTextVGT(vgtindex,TRUE);
	term1->owner = 0;
	if (term1->interp)
	    free(term1->interp);
	term1->interp = NULL;
	if ( term1->lineEditBuf )
	    free ( term1->lineEditBuf );
	term1->lineEditBuf = 0;
	return;
      }
    term1->owner = execpid;

    SelectForInput(vgtindex);
    return;
}

