/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* event.c - Handles VGTS event queues
 *
 *
 * Bill Nowicki April 1983 
 *
 */

#include <Venviron.h>
#include <Vioprotocol.h>
#include <Vtermagent.h>
#include "Vgts.h"
#include "sdf.h"
#include "vgt.h"
#include "pad.h"
#include "interp.h"
#include <chars.h>

struct Event *FreeEvents = NULL;

BOOLEAN EmulateGraphicsInput = TRUE; /* User control of graphics-input	*/
				     /* emulation			*/

enum
  {
    NoPrefix,
    EscPrefix,
    ArrowPrefix,
    PFPrefix
  } KbdEscSeqState = NoPrefix;	/* state of escape sequence	*/
				/* interpreter for converting	*/
				/* keystrokes to graphics input */
				/* device events		*/ 

short KbdEscSeqAge;		/* Number of timer messages	*/
				/* since current KbdEscSeqState	*/
				/* was established		*/
	
unsigned ClicksSinceLastArrow = 0;	/* during graphics-input emulation, */
					/* counts # clicks since last arrow */
					/* key hit (mouse motion)	    */

extern short InputVGT;			/* the current input pad */
extern short MouseX, MouseY, MouseButtons;
extern short Debug;			/* for extra debugging info */

/*
 * Routines that handle character events
 */

UserPut(term1,c)
  register Term1 *term1;
  char c;
    {
	/*
	 * Check if the last event is a Keystroke event that has not
	 * been filled up.  If not, tack another one onto the end
	 * of the queue.  Stuff the character into the last event.
	 */
      IoReply msg;
      register struct Event *e = term1->eventTail;

      if (e==NULL || e->code != ReportCharacter
      		  || e->bytecount >= IO_MSG_BUFFER)
        {
		/*
		 * try to get a free Event and tack it onto the end
		 */
	  e = FreeEvents;
	  if (e==NULL)
	    {
	      if (Debug & DebugVgtsEvents) dprintf("No free Events!\n");
	      return;
	    }
	    
	  FreeEvents = e->next;

	  if (term1->eventTail)  
	    {
	      term1->eventTail->next = e;
	    }
	  if (term1->eventQ == NULL) term1->eventQ = e;
	  term1->eventTail = e;
	  e->next = NULL;
	  e->code = ReportCharacter;
	  e->bytecount = 0;
	}
      
      e->shortbuffer[e->bytecount++] = c;
    }


UserPush(term1,c)
  register Term1 *term1;
  {
  	/*
	 * This is the last character we are giving back to the user,
	 * so without further ado, send it to the eagerly awaiting
	 * client process.  But throw it away if a pure mouse request
	 * is outstanding.
	 */

    UserPut(term1,c);
    EventToUser(term1);
  }


BOOLEAN ConvertKeyToGraphics(msg)
    register IoRequest *msg;

  {
  	/*
	 * If graphics-input emulation is turned on, convert arrow keys to
	 * graphics device motion and PF1 to PF3 to graphics device button
	 * clicks.  PF4 toggles graphics-input emulation.
	 * 
	 * If the message is a Keyboard message and the keystroke completes
	 * an escape sequence from one of the above function keys, the
	 * message is converted to an appropriate GraphicsHelperMsg and TRUE is
	 * returned.  Otherwise, the message is left unchanged and FALSE is
	 * returned.
	 *
	 * If the message is a Keyboard message and the keystroke does not
	 * start an escape sequence, it is sent to UserCook for interpretation.
	 * If the keystroke continues an escape sequence, the escape-sequence
	 * interpreter advances to the next state.  If the keystroke causes
	 * an in-progress escape sequence to fail, the previously swallowed
	 * characters and the current keystroke are sent to UserCook.
	 * 
	 */

#define FineMouseIncrement 5	/* motion per individual keystroke */

#define CoarseMouseIncrement 15	/* motion per keycode received when key is */
				/* held down				   */

/* NOTE -- ticks are 4/second timer ticks, NOT clicks */
#define MaxTicksBetweenKeyRepeats 1	/* max #ticks between keycodes when */
					/* key is held down		     */
#define MaxEscAge 3		/* max #ticks for escape sequence age */

    register EventReq *ereq = (EventReq *) msg;
    register char c = (char) msg->shortbuffer[0];
    register Term1 *term1 = Term1Table + InputVGT;
    short mouseIncrement;

    KbdEscSeqAge = 0;

    switch (KbdEscSeqState)
    
      {

	case NoPrefix:

	    if (c == ESC) KbdEscSeqState = EscPrefix;
	    else UserCook(term1,c); 
	    return(FALSE);
	    break;


	case EscPrefix:

	    switch(c)

	      {

		case 'O':  KbdEscSeqState = PFPrefix; break;

		case '[':
		    if (EmulateGraphicsInput)
		      {
			KbdEscSeqState = ArrowPrefix;
			break;
		      }

		default:
		    UserCook(term1, ESC);
		    UserCook(term1,c);
		    KbdEscSeqState = NoPrefix;

	      }

	    return(FALSE);
	    break;


	case ArrowPrefix:

	    if (ClicksSinceLastArrow > MaxTicksBetweenKeyRepeats)
		mouseIncrement = FineMouseIncrement;
	    else mouseIncrement = CoarseMouseIncrement;
	    ClicksSinceLastArrow = 0;

	    switch(c)

	      {

		case 'A':  MouseY -= mouseIncrement; break;

		case 'B':  MouseY += mouseIncrement; break;

		case 'C':  MouseX += mouseIncrement; break;

		case 'D':  MouseX -= mouseIncrement; break;

		default:
		    UserCook(term1, ESC);
		    UserCook(term1, '[');
		    UserCook(term1, c);
		    KbdEscSeqState = NoPrefix;
		    return(FALSE);

	      }

	    ConstrainCursorXY(&MouseX, &MouseY);
	    break;


	case PFPrefix:

	    if (!EmulateGraphicsInput && c != 'S') return(FALSE);

	    switch(c)

	      {

		case 'P':
		    MouseButtons ^= LeftButton;
		    TtyPutString("Left button ");
		    if (MouseButtons & LeftButton) TtyPutString("down\r\n");
		    else TtyPutString("up\r\n");
		    break;

		case 'Q':
		    MouseButtons ^= MiddleButton;
		    TtyPutString("Middle button ");
		    if (MouseButtons & MiddleButton) TtyPutString("down\r\n");
		    else TtyPutString("up\r\n");
		    break;

		case 'R':
		    MouseButtons ^= RightButton;
		    TtyPutString("Right button ");
		    if (MouseButtons & RightButton) TtyPutString("down\r\n");
		    else TtyPutString("up\r\n");
		    break;

		case 'S':
		    EmulateGraphicsInput = !EmulateGraphicsInput;
		    TtyPutString("Mouse emulation ");
		    if (EmulateGraphicsInput) TtyPutString("on\r\n");
		    else TtyPutString("off\r\n");
		    break;

		default:
		    UserCook(term1, ESC);
		    UserCook(term1, 'O');
		    UserCook(term1, c);
		    KbdEscSeqState = NoPrefix;
		    return(FALSE);

	      }

      }

    msg->requestcode = (SystemCode) GraphicsHelperMsg;
    ereq->x = MouseX;
    ereq->y = MouseY;
    ereq->buttons = MouseButtons; 
    KbdEscSeqState = NoPrefix;
    return(TRUE);

  }


AgeKbdEscSequence()

  {
  	/*
	 * Called upon receipt of timer message when using keyboard to emulate
	 * graphics input device.  We want to recognize arrow and PF function
	 * keys, which send rapid-fire escape sequences.  If more than one
	 * timer message is received in the middle of an escape sequence, that
	 * sequence cannot have been transmitted by these keys, so we don't
	 * intercept it.
	 *
	 * Also, increments the count of timer ticks seen since the last
	 * time an arrow key was hit (mouse motion), so we can offer coarse
	 * mouse motion when arrow keys are held down and fine mouse motion
	 * for individual keystrokes.
	 */
    register Term1 *term1 = Term1Table + InputVGT;

    if (EmulateGraphicsInput && KbdEscSeqState != NoPrefix && ++KbdEscSeqAge>MaxEscAge )

      {

	UserCook(term1, ESC);

        switch(KbdEscSeqState)

	  {

	      case ArrowPrefix:
		  UserCook(term1, '[');
		  break;

	      case PFPrefix: 
		  UserCook(term1, 'O');

	  }

	KbdEscSeqState = NoPrefix;

      }

    ClicksSinceLastArrow++;

  }


UserCook(term1,c)
    register Term1 *term1;
    register char	c;
  {
  	/*
	 * Possibly cook an input character.
	 * Drop the character if a pure mouse request is outstanding
	 */
    register Term2 *term2 = term2Ptr(term1);

    if ( term2->owner == 0 ) return;
    if ( term2->writerPid && WriteUnblock(term1, c) ) return;
 
    if (term1->requestFlag == Mousereq) return;

    if (Debug & DebugVgtsLineEdit) dprintf("before Line Edit\n");
    if (term1->mode & LineBuffer && term2->lineEditBuf)
      return( LineEdit(term1,c) );

    if (c=='\r' && (term1->mode & CR_Input) )
      c = '\n';
    if (term1->mode & Echo)
      {
        if (c=='\n' && (term1->mode&LF_Output))
	      	 PadPutChar('\r', padPtr(term1));
        PadPutChar(c, padPtr(term1));
      }
    UserPush(term1,c);
  }


HandleRead(term1, pid)
    register Term1 *term1;
  {
    IoReply msg;
	/*
	 * The user is trying to read.  See if any characters are
	 * on the event queue.  If not, set flags to wait for some.
	 */
    if (term1->requestFlag&BlockBits)
      {
        if( ! ValidPid(term1->reader))
	  {
	    msg.replycode = DISCARD_REPLY;
	    Reply(&msg, term1->reader);
	  }
	else
	  {
	    /* Another client is queued; ask this one to try again.
      	     * General queueing would be better, but this feature is
      	     * rarely used, so we get by without it. */
      	    msg.replycode = RETRY;
      	    msg.bytecount = 0;
      	    Reply(&msg, pid);
      	    return;
	  }
      }
    term1->requestFlag |= Keyreq;
    term1->reader = pid;
    EventToUser(term1);
  }



HandleWrite(msg, pid)
  IoRequest *msg;
  ProcessId pid;
  {
  	/*
	 * Do a MoveFrom to get written data
	 * if in a different team.
	 */
    unsigned vgtindex, total;
    static char buf[BlockSize];
    static char string[ 80 ];
    char *from;
    register count, actual;
    register Term1 *term1;
    register Term2 *term2;

    vgtindex = msg->fileid;
    term1 = Term1Table + vgtindex;
    term2 = term2Ptr(term1);

#ifdef DUPLICATE_SUPPRESSION 
    if (( msg->blocknumber - 1 ) == term1->BlockNumber )
      {

	/* no duplicate, so "write" it */
#endif DUPLICATE_SUPPRESSION 

        from = msg->bufferptr;
        total = msg->bytecount;


        if ( PadBusy(vgtindex, pid) )
          {
	    msg->requestcode = BUSY;
	    Reply(msg, pid);
	    return;
          }

        while (total>0)
          {
            count = total;
	    if (count>BlockSize) count=BlockSize;
	    total -= count;
	    MoveFrom( pid, buf, from, count);
	    from += count;
     	    if (term1->mode & DiscardOutput) actual = count;
            else actual = InterpPutString(term1->interp,
					  buf, count, term1->mode);
	    term2->lastBytesWritten = actual;
	    term2->bytesWritten += actual;
	    if (actual < count) break;
          }

        if (actual == count)
          {
            term1->TotalBytesWritten += term2->bytesWritten;
	    ++(term1->BlockNumber);
	    term2->writerPid = 0;
	    term2->bytesWritten = 0;
            msg->requestcode = OK;
            Reply(msg, pid);
          }
        else
          {    
	    term1->mode |= NoCursor;
            term2->writerPid = pid;
	    /* Make a copy of the request message (unless already did). */
	    if (msg != (IoRequest *)&(term2->reqMessage))
	        Copy(&(term2->reqMessage), msg, sizeof(MsgStruct));
	    PadCursorOff(vgtindex);
	    PadRedraw(vgtindex);
          }
#ifdef DUPLICATE_SUPPRESSION 
      }
    else if ( msg->blocknumber == term1->BlockNumber )
      {

	/* this is a duplicate, so drop it & reply "ok" */

	msg->requestcode = OK;
	Reply(msg, pid);

        sprintf( string,
	 "Suppressed duplicate stream write on block %d, expected block %d",
	 msg->blocknumber, term1->BlockNumber+1 );
	TtyMessage( string );

      }
    else
      {

	/* this is some strange block, so reply "bad block number" */

	msg->requestcode = BAD_BLOCK_NO;
	Reply(msg, pid);

        sprintf( string,
	 "Dropped bad stream write on block %d, expected block %d",
	 msg->blocknumber, term1->BlockNumber+1 );
	TtyMessage( string );

      }
#endif DUPLICATE_SUPPRESSION 
   }



HandleShortWrite(msg, pid)
  IoRequest *msg;
  ProcessId pid;
  {
    static char string[ 80 ];
    unsigned int vgtindex;
    register actual;
    register Term1 *term1;
    register Term2 *term2;

    vgtindex = msg->fileid;
    term1 = Term1Table + vgtindex;
    term2 = term2Ptr(term1);

#ifdef DUPLICATE_SUPPRESSION 
    if (( msg->blocknumber - 1 ) == term1->BlockNumber )
      {

	/* no duplicate, so "write" it */
#endif DUPLICATE_SUPPRESSION 

	if (DifferentByteOrder(pid))
	    ByteSwapLongInPlace(msg->shortbuffer, (msg->bytecount+3)&~3);


        if ( PadBusy(vgtindex, pid) )
          {
	    msg->requestcode = BUSY;
	    Reply(msg, pid);
	    return;
          }

        if (term1->mode & DiscardOutput) actual = msg->bytecount;
        else actual = InterpPutString(term1->interp,
    			              msg->shortbuffer + term2->bytesWritten, 
    			              msg->bytecount, term1->mode);
        term2->lastBytesWritten = actual;
        term2->bytesWritten += actual;

        if (actual == msg->bytecount)
          {
            term1->TotalBytesWritten += term2->bytesWritten;
	    ++(term1->BlockNumber);
	    term2->writerPid = 0;
	    term2->bytesWritten = 0;
            msg->requestcode = OK;
            Reply(msg, pid);
          }
        else
          {    
	    term1->mode |= NoCursor;
	    PadCursorOff(vgtindex);
	    PadRedraw(vgtindex);
            term2->writerPid = pid;
	    /* Make a copy of the request message (unless already did). */
	    if (msg != (IoRequest *)&(term2->reqMessage))
	        Copy(&(term2->reqMessage), msg, sizeof(MsgStruct));
          }
#ifdef DUPLICATE_SUPPRESSION 
      }
    else if ( msg->blocknumber == term1->BlockNumber )
      {

	/* this is a duplicate block, so drop it, but still reply "ok" */

	msg->requestcode = OK;
	Reply(msg, pid);

        sprintf( string,
	 "Suppressed duplicate stream write on block %d, expected block %d",
	 msg->blocknumber, term1->BlockNumber+1 );
	TtyMessage( string );

      }
    else
      {

	/* this is some strange block, so reply "bad block number" */

	msg->requestcode = BAD_BLOCK_NO;
	Reply(msg, pid);

        sprintf( string,
	 "Dropped bad stream write on block %d, expected block %d",
	 msg->blocknumber, term1->BlockNumber+1 );
	TtyMessage( string );

      }
#endif DUPLICATE_SUPPRESSION 
  }



int PadBusy(vgtindex, pid)
  register unsigned vgtindex;
  register ProcessId pid;
  {
	/*
	 * If there is an existing writer (pad is blocked on output), and the
	 * pid of the new writer differs from that of the existing writer,
	 * return 1.  Otherwise, return 0.
	 *
	 * If the existing writer died unexpectedly, clean up after him and
	 * return 0.
	 *
	 */
    register Term1 *term1 = &Term1Table[vgtindex];
    register Term2 *term2 = term2Ptr(term1);
    register TtyPadType *pad = padPtr(term1);


    if (term2->writerPid)
      {
        if ( ValidPid(term2->writerPid) )
	  {
	    if (pid == term2->writerPid) return(0);
	    else return(1);
	  }
        else
          {
	    MsgStruct m;
	    m.sysCode = DISCARD_REPLY;
	    Reply(&m, term2->writerPid);
	    term2->writerPid = 0;
	    term2->bytesWritten = 0;
	    pad->newlinesLeft = pad->length;
	    pad->pageLength = pad->length;
	    term1->mode &= ~NoCursor;
   	    SetBannerState(vgtindex, DisplayString);
	    PadCursorOn(vgtindex);
	    PadRedraw(vgtindex);
          }
      }
    return(0);
  }



int WriteUnblock(term1, c)
  register Term1 *term1;
  register char c;
  {
	/*
	 * Client is blocked on a pad-filling write and we just received a
	 * character. If the character is meaningful, return 1, else 0.
	 * 
	 * The following list describes the available commands.  Most commands
	 * are optionally preceded by an integer argument k.  Defaults are
	 * in brackets.  Star (*) indicates argument becomes new default.
	 *
	 * <space>	Display next k lines [current page size]
	 * 
	 * z		Display next k lines [current page size]*
	 * Z
	 * 
	 * <CR>		Display next k lines [1]
	 * <LF>
	 * 
	 * 'q'		Throw away all output until next input read
	 * 'Q'
	 *
	 * 's'		Skip forward k lines [1]
	 *
	 * 'S'		Skip forward to last line
	 *
	 * 'f'		Skip forward k pages [1]
	 *
	 * 'F'		Skip forward to last page
	 *
	 * <backspace>	Erase last character of argument
	 * DEL
	 *
	 * '.'		Repeat previous command
	 *
	 */

    static int	Arg = 0;
    static int	PrevArg = 0;
    static char	PrevCommand = ' ';

    IoRequest *msg;
    register Term2 *term2 = term2Ptr(term1);
    register TtyPadType *pad = padPtr(term1);
    int vgtindex;

    msg = (IoRequest *)&(term2->reqMessage);

    if ( (c >= '0') && (c <= '9') )
      {
	Arg = 10 * Arg + c - '0';	
	return(1);
      }
    else if (c == '.')
      {
	Arg = PrevArg;
        c = PrevCommand;
      }
    else
      {
	PrevArg = Arg;
        PrevCommand = c;
      }

    switch(c)
      {
 	case ' ':
	case 'z':
	case 'Z':
	    pad->newlinesLeft = Arg ? Arg : pad->pageLength;
	    if (c != ' ') pad->pageLength = pad->newlinesLeft;
	    break;

	case '\r':
	case '\n':
	    pad->newlinesLeft = Arg ? Arg : 1;
	    if (Arg) pad->pageLength = Arg;
	    break;
	
	case 'q':
	case 'Q':
	    PadPutString("\r\n", pad);
	    term1->mode |= DiscardOutput;
	    break;
	
	case 's':
	    pad->newlinesLeft = pad->pageLength + (Arg?Arg:1);
	    break;
	    
	case 'f':
	    pad->newlinesLeft = pad->pageLength +
	    				pad->pageLength * (Arg?Arg:1);
	    break;

	case 'S':
	case 'F':
	    pad->newlinesLeft = 32767;
	    break;

	case '\b':
	case DEL:
	    Arg = Arg / 10;
	    return(1);
	    break;

	default:
	    Arg = 0;
	    return(0);
	    break;
      }
  
    Arg = 0;

    if (term2->outputFlag & PageOutputShutdown)
      {
	term1->mode &= ~PageOutput;
	term2->outputFlag &= ~PageOutputShutdown;
      }

    vgtindex = term1 - Term1Table;
    SetBannerState(vgtindex, DisplayString);
    term1->mode &= ~NoCursor;
    PadCursorOn(vgtindex);
    PadRedraw(vgtindex);
    msg->bytecount -= term2->lastBytesWritten;

    switch(msg->requestcode)
      {
	case WRITE_INSTANCE:
	    msg->bufferptr += term2->lastBytesWritten;
	    HandleWrite(msg, term2->writerPid);
	    break;
	
	case WRITESHORT_INSTANCE:
	    HandleShortWrite(msg, term2->writerPid);
	    break;
      }

    return(1);
  }



PutEndEvent(term1)
    register Term1 *term1;
  {
	/*
	 * Put an end-of-file event on the queue. These are returned
	 * immediately if the user was waiting for anything.
	 */
    register struct Event *e;
    register int mousemode = term1->mode & (ReportTransition | ReportClick);

    e = FreeEvents;
    if (e == NULL)
      {
        if (Debug & DebugVgtsEvents) dprintf("No free Events!\n");
	return;
      }
      
    FreeEvents = e->next;

    if (term1->eventTail)  term1->eventTail->next = e;
    if (term1->eventQ == NULL) term1->eventQ = e;
    term1->eventTail = e;
    e->next = NULL;
    e->code = EOFcode + ReportCharacter + ReportClick;
    e->bytecount = 0;
    EventToUser(term1);
  }



/*
 * Routines that are common to character and graphical events
 */

EventToUser(term1)
    register Term1 *term1;
  {
	/*
	 * If the user was requesting input, Reply with the event
	 * descriptor.
	 */
    int requestmode;
    register Term2 *term2 = term2Ptr(term1);
    register struct Event *e;
    register TtyPadType *pad = padPtr(term1);
    short vgtindex = term1 - Term1Table;

    if (term2->writerPid != 0)
      {
        pad->newlinesLeft = pad->pageLength;
	term1->mode &= ~DiscardOutput;
      }

    requestmode = 0;
    if (term1->requestFlag&Mousereq) 
    	requestmode |= ReportClick|ReportTransition;
    if (term1->requestFlag&Keyreq) 
    	requestmode |= ReportCharacter;

    if ( (term1->requestFlag&BlockBits)==0) return;

    e = term1->eventQ;
    while (e && (!(e->code & requestmode)))
      {
	if ( (term1->requestFlag & SlaveReq) &&
	    (e->code & ReportClick) )
	  {
	  	/*
		 * Here we convert the graphics event to character events.
		 * We just return from this, since ReturnToMaster eventually
		 * calls UserPush, which calls this routine recursively.
		 * For this reason, it is crucial that the SlaveReq bit
		 * be turned off before you call ReturnToMaster.
		 */

	    DropEvent(term1,e);
	    term1->requestFlag &= ~SlaveReq;
	    ReturnToMaster(vgtindex,e->x,e->y,e->buttons,e->fileid);
	    return;
	  }
	e = e->next;
      }

    if (e) 
      {
	  	/*
		 * We got one!! yank it off the event queue,
		 * put it on the free queue, and Reply to the client.
		 */
	register EventReq *rep;  /* variant record: IoReply */

	/* hack! EventReq and Event structs should overlap like this! */
	rep = (EventReq *) (&e->next + 2);
	if (e->code & ReportCharacter)
	  {
	    rep = (EventReq *) ((char*)rep + 2*sizeof(short));
	    rep->eventcode = ReportCharacter;
	  }

	DropEvent(term1,e); 	/* free the one we are using */
	
	if (e->code & EOFcode)
	    rep->requestcode = END_OF_FILE;
	else
	    rep->requestcode = OK;
	rep->fileid = vgtindex;
	if (DifferentByteOrder(term1->reader))
	    ByteSwapLongInPlace(e->shortbuffer, (e->bytecount+3)&~3);
	Reply(rep, term1->reader);
        term1->requestFlag &= ~BlockBits;
      }
  }


static DropEvent(term1,victim)
    register Term1 *term1;
    struct Event *victim;
  {
	/*
	 * Remove the given event from the the Event Queue
	 * for this vgt
	 */
    register struct Event *e, *last;
    
    last = NULL;
    e = term1->eventQ;

    while (e && victim != e)
      {
        last = e;
        e = e->next;
      }
    if (e==NULL) return;

    if (last)
      last->next = e->next;
    else
      term1->eventQ = e->next;

    if (term1->eventTail==e) term1->eventTail = last;
    e->next = FreeEvents;
    FreeEvents = e;
  }



/*
 * Routines that handle graphical events
 */

PutGraphicsEvent(vgtindex, x, y, buttons, uptrans)
    short vgtindex, x, y, buttons;
    register int uptrans;
  {
	/*
	 * Put a mouse event on the queue.  Should be invoked only when
	 * this VGT is open to mouse input in some form.
	 */
    register Term1 *term1 = Term1Table + vgtindex;
    register struct Event *e;
    register int mousemode = term1->mode & (ReportTransition | ReportClick);
    register eventcode;

    if (term1->master>0)
      {
        term1 = Term1Table + term1->master;
	mousemode = ReportClick;
      }

    e = FreeEvents;
    if (e == NULL)
      {
        if (Debug & DebugVgtsEvents) dprintf("No free Events!\n");
	return;
      }
      
    FreeEvents = e->next;

    if (term1->eventTail)  term1->eventTail->next = e;
    if (term1->eventQ == NULL) term1->eventQ = e;
    term1->eventTail = e;
    e->next = NULL;

    eventcode = ReportTransition;
    if (uptrans) eventcode |= ReportClick;

    e->code = eventcode;
    e->x = x;  e->y = y;
    e->buttons = buttons;
    e->fileid = vgtindex;

    EventToUser(term1);
  }

int MouseUpKludge = 0;

HandleGraphics(msg)
    struct EventMsg *msg;
  {
	/*
	 * The graphics input device has changed state, 
	 * and the view manager was
	 * not waiting for it.  In a requesting graphics view, or a selected
	 * requesting pad, we queue and possibly send an event.  Elsewhere,
	 * the right button invokes the view manager and the other buttons
	 * cause this VGT to be selected, if it is a pad.  Graphics VGTs
	 * cannot be selected, but the middle button always pulls a graphics
	 * view to the top.
	 */
    short vgtindex, user, view;
    short xw, yw, buttons = msg->buttons;
    static lastButtons = 0, gatherButtons = 0;
    int padmode;

    CursorUpdate();

    if (MouseUpKludge)
      {
        lastButtons = gatherButtons = 0;
	MouseUpKludge = 0;
      }

    if (buttons==lastButtons) return;

    view = FindMouseClick(msg->x, msg->y, &xw, &yw);
    vgtindex = view < 0 ? -1 : ViewTable[view].vgt;

    /* short circuit for dynamic manager */
    if (view >= 0 &&
        lastButtons == 0 &&
	msg->y < ViewTable[view].ScreenYmin + BannerHeight)
      {
    	vgtindex = quickmanager(InputVGT, buttons);
	gatherButtons = 0;
	return;
      }
    

    lastButtons = buttons;
    gatherButtons |= buttons;

    if (vgtindex<0 || Term1Table[vgtindex].master <= 0)
      user = vgtindex;
    else
      user = Term1Table[vgtindex].master;
    padmode = user<0 ? 0 : Term1Table[user].mode;

    if (
#ifndef PASSTHRU
        user==InputVGT &&
#endif PASSTHRU
    (((padmode & ReportClick) && buttons==0) || padmode & ReportTransition))
      {
#ifdef PASSTHRU
        if (user != InputVGT)
	    SelectForInput(user);
#endif PASSTHRU
	PutGraphicsEvent(vgtindex, xw, yw, gatherButtons, buttons==0);
	gatherButtons = 0;
	return;
      }

    if (
#ifndef PASSTHRU
        vgtindex==InputVGT &&
#endif PASSTHRU
        (padmode & ReportEscSeq))
	  {
	    if (buttons == 0 && gatherButtons != 0) 
		{
#ifdef PASSTHRU
	          if (user != InputVGT)
		      SelectForInput(user);
#endif PASSTHRU
		  InterpMouseEscape(vgtindex, xw, yw, gatherButtons);
		  gatherButtons = 0;
		}
	    return;
	  }

    if (buttons == 0 /* && user != InputVGT */) 
     {
     		/*
		 * This is the default case, if the click was 
		 * not in a VGT associated with the current intput master.
		 */
         switch (gatherButtons)
          { 
            case RightButton:  
	        manager(InputVGT);
	        break;

            case RightButton | LeftButton:
	        graphics();
		break;

	    case LeftButton | MiddleButton:
	        ZoomMode(InputVGT);
		break;
		
	    case RightButton | MiddleButton:
		ExecControl(InputVGT);
		break;
		
	    case MiddleButton:
	    case LeftButton:
	        if (user>0)
	            SelectForInput(user);
	        if (user != vgtindex || vgtindex == 0) MakePadTop(vgtindex);
	        break;
          }
     }
   if (buttons == 0)
       gatherButtons = 0;
  }




HandleEventRequest(term1, msg, pid)
    register Term1 *term1;
    register struct EventMsg *msg;
    ProcessId pid;
  {
	/*
	 * The user has called for an event of either type.  Reporting of
	 * mouse events is controlled by the mode of the pad.  Whichever
	 * comes first, a mouse click (or transition), or a keystroke, that
	 * will be returned.
	 */
    if (msg->eventcode & ReportMotion)
	{
	  HandleGetGraphicsStatus(msg, pid, 0);
	  return;
	}
    if ((term1->requestFlag&BlockBits) && 
    	  ValidPid(term1->reader) )
    	return;			/* should reply with error code!! */
    term1->requestFlag |= Mousereq;
    if (msg->eventcode & ReportCharacter)
        term1->requestFlag |= Keyreq;
    term1->reader = pid;
    EventToUser(term1);
  }


HandleGetGraphicsStatus(msg, pid, mouseFlag)
    EventReq *msg;
    int pid;
    int mouseFlag;	/* View manager is in control */
  {
    short xw, yw, topvgt;
    short vgtindex = msg->fileid;
    register Term1 *term1 = Term1Table + vgtindex;

	/*
	 * a direct graphics status request flushes
	 *  ALL queued events in this pad, and the controlling pad.
	 */

    FlushEvents(term1);

    if (term1->master>0)
        FlushEvents(Term1Table+term1->master);
    topvgt = FindTopVGT(MouseX, MouseY);

    if (mouseFlag==0 && topvgt>0 && (vgtindex == topvgt || 
                         	     Term1Table[topvgt].master == vgtindex) )
      {
	FindMouseClick(MouseX, MouseY, &xw, &yw);
	msg->x = xw;  msg->y = yw;
	msg->buttons = MouseButtons;
	msg->fileid = topvgt;
	msg->requestcode = OK;
      }
    else msg->requestcode = END_OF_FILE;  /* mouse not in requesting VGT */
    Reply(msg, pid);
  }

 
FlushEvents(term1)
    register Term1 *term1;
  {
   if (term1->eventQ != NULL)
      {
	term1->eventTail->next = FreeEvents;
	FreeEvents = term1->eventQ;
	term1->eventQ = term1->eventTail = NULL;
      }
}


InterpMouseEscape(vgtindex, xw, yw, buttons)
  short vgtindex, xw, yw, buttons;
  {
  	/*
	 * Send back the appropriate escape codes for an Emacs hack.
	 */
      switch (buttons)
	  {
	    case LeftButton:
	        SendEmacsHack(vgtindex,xw,yw,-1);	/* move cursor */
		break;
		
	    case LeftButton+MiddleButton:
	        SendEmacsHack(vgtindex,xw,yw,0);	/* null - set mark */
		break;
		
	    case LeftButton+RightButton:
	        SendEmacsHack(vgtindex,xw,yw, 'W' & 31); /* control W -  kill */
		break;
		
	    case MiddleButton+RightButton:
	        SendEmacsHack(vgtindex,xw,yw,'Y' & 31); /* control Y - yank */
		break;
		
	    case MiddleButton:
		SelectForInput(vgtindex);
	        break;

	    case RightButton:
		SelectForInput(manager(InputVGT));
		break;		
	   }
  }


SendEmacsHack(vgtindex,x,y,c)
  {
  	/*
	 * send the escape sequence for Emacs to position the
	 * cursor at a given character.
	 * If c is greater than zero, it is another command character
	 * to send.
	 */
    short line, col;
    register Term1 *term1 = Term1Table + vgtindex;
    
    if (vgtindex<1) return;
    PadFindPoint(vgtindex, TtyPadList[vgtindex].length, x, y, &line, &col);
    UserPut(term1,ESC);
    UserPut(term1,'M');
    UserPut(term1,col+1);
    if (c<0)
        UserPush(term1,line+1);
    else
      {
        UserPut(term1,line+1);
        UserPush(term1,c); 
      }
  }


