/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1984			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: draw2.c
 *
 * This file (and draw1.c) contains the low-level routines that interface with
 * the graphics libraries.
 */

#include <Vioprotocol.h>
#include <Vgts.h>
#include <rasterops.h>
#include "sdf.h"
#include "splines.h"
#include <text.h>
#include "vgt.h"
#include <fonttable.h>
#include <Vfont.h>

#define PadFontLoaded Fonts.first16[CMR_Font].font

/* Include the appropriate header file, depending on the framebuffer type. */

#ifdef SUN100FB

#include "gl_sun100.h"
#include <bitmaps.h>
int FontOptions = FontSun100;

#else 
#ifdef  SUN120FB

#include "gl_sun120.h"
int FontOptions = 0;

#else
#ifdef  QVSS

#include "gl_mvax.h"
int FontOptions = 0;
extern VRaster *WholeScreen;

#else

Somebody blew it again.

#endif  QVSS
#endif SUN120FB
#endif SUN100FB

#define FontHeight 16
#define FontDescent 3

extern short ScreenLimitX, ScreenLimitY;	/* framebuffer limits */


DrawLine(x,y,dx,dy)
 int x, y, dx, dy;
  {
    /*
     * Draw a vector 
     */
    gl_DrawLine(x,y,dx,dy);
  }


/*******************************************************************/
/*
 * DrawSimpleText:
 * Draws specified string at designated location on the screen.
 * Checks for violation of screen boundaries - clip if necessary.
 * If paintFlag is true, the text is painted instead of written.
 */

DrawSimpleText(xLeft, yBottom, string, r, paintFlag)
  int xLeft, yBottom;
  char *string;
  register struct SubView *r;
  int paintFlag;
  {
    int length;
    int xRight;
    char *ch;
#ifndef MC68000
    Rectangle clip;
    Vector start;
#endif MC68000

    if (string==NULL) return;
    length = strlen(string);

#ifdef SUN100FB
    xRight = xLeft + length*FontWidth;

    if (yBottom < r->ScreenYmin || 
        (yBottom - FontHeight)>r->ScreenYmax ||
	 xLeft  > r->ScreenXmax ||
	 xRight < r->ScreenXmin	 )
		/*
		 * Don't bother drawing it at all if it is
		 * going to be completely out of the view
		 */
        return;	

    if (yBottom > r->ScreenYmax || yBottom-FontHeight+1 < r->ScreenYmin)
      {
        if (xLeft >= ScreenLimitX) return;
        if (PadFontLoaded)
	  DrawGeneralText(xLeft, yBottom - FontDescent, string, length,
		r, CMR_Font, paintFlag );
	return;
      }

    if (xLeft >= r->ScreenXmin && xRight <= r->ScreenXmax)
	    DrawText(xLeft, yBottom, string, length, paintFlag);
    else
      {
        if (xLeft >= r->ScreenXmin)
	  {
	    int fullChars = (r->ScreenXmax - xLeft)/FontWidth;
	    
		DrawText(xLeft, yBottom, string, fullChars, paintFlag);
	    string += fullChars;
	    xLeft += fullChars*FontWidth;
	    if (xLeft >= r->ScreenXmax || xLeft >= ScreenLimitX) return;
	  }
        if (PadFontLoaded)
            DrawGeneralText(xLeft, yBottom - FontDescent, string, length,
		r, CMR_Font, paintFlag );
      }

#endif SUN100FB

#ifdef SUN120FB

    /* The Sun-2 graphics library routine does clipping for us: */
    Sun2_DrawSimpleText(xLeft, yBottom, string, length, paintFlag,
			r->ScreenXmin, r->ScreenXmax,
			r->ScreenYmin, r->ScreenYmax);
#endif SUN120FB

#ifdef  QVSS

    start.h = xLeft; start.v = yBottom - (FontHeight - 1);
    clip.start.h = r->ScreenXmin; clip.size.h = r->ScreenXmax-r->ScreenXmin+1;
    clip.start.v = r->ScreenYmin; clip.size.v = r->ScreenYmax-r->ScreenYmin+1;
    SimpleText(string, length, &start, &clip);

#endif  QVSS
  }


DrawGeneralText(xLeft, yBottom, text, length, r, fontNumber, paintFlag)
  short xLeft, yBottom; /* Relative to top left of screen */
  char *text;
  register struct SubView *r;
  unsigned char fontNumber;
  {
  	/*
	 * Draw some generalized text, using Per's routine.
	 */

#ifdef  SUN100FB

    static struct fb_raster temp;
    int code =  (paintFlag ? TextStdPaint : (TextStdCopy+InverseIfHigh))
	    + fontNumber;

#endif  SUN100FB

#ifdef  SUN120FB

    static struct fb_raster temp;
    int code = fontNumber + (paintFlag ? ((GXpaint & 0xFF)<<16)
    				       : ((GXcopy & 0xFF)<<16) + InverseIfHigh);

#endif  SUN120FB

#ifdef  QVSS

    Rectangle temp;
    Vector start;
    FontEntry *fEntry;
    int code = fontNumber + (paintFlag ? ((GXpaintInverted & 0xFF)<<16)
    		: ((GXcopyInverted & 0xFF)<<16) + InverseIfHigh);

#endif  QVSS

    if (fontNumber==255) return;

#ifdef MC68000
    temp.x = r->ScreenXmin;
    temp.y = r->ScreenYmin;
    temp.width = r->ScreenXmax - r->ScreenXmin + 1;
    temp.height= r->ScreenYmax - r->ScreenYmin + 1;

    if (text)
	gl_WriteGeneralText(text, length,
	    xLeft, yBottom, code, &temp);
#else
    temp.start.h = r->ScreenXmin;
    temp.start.v = r->ScreenYmin;
    temp.size.h = r->ScreenXmax - r->ScreenXmin + 1;
    temp.size.v = r->ScreenYmax - r->ScreenYmin + 1;
    start.h = xLeft;
    start.v = yBottom;
    fEntry = (FontEntry*)LookupFont(code & 0xFF);
    if (fEntry) WriteText(text,length,&start,code,fEntry->font,&temp);
#endif
  }


/*******************************************************************/
/* Write a raster (in Per's general format) onto the screen. */

DisplayRaster(item, x, y, view, r, data)
#ifdef SUN100FB
	MemRaster *item;
#else SUN120FB or VAX
	VRaster *item; /* note: actual address of prefix, withou adding 1 */
#endif
	int x, y;
	View *view;
	register struct SubView *r;
	unsigned char data;
  {
    int scale = view->zoom;
#ifdef SUN100FB
    register struct fb_raster temp;
    int adjust;     /* so that pixels \centers/ are on grid lines */

    temp.x = r->ScreenXmin;
    temp.y = r->ScreenYmin;
    temp.width = r->ScreenXmax - r->ScreenXmin + 1;
    temp.height= r->ScreenYmax - r->ScreenYmin + 1;
    GXfunction = item->where & InverseRaster ? GXcopy : GXcopyInverted;
    if (scale < 0) scale = 0; if (scale > 4) scale = 4;
    if ((scale == 3 || scale == 4) && view->showGrid) scale |= ScaleFramed;
    adjust = (1 << (scale & 15)) >> 1;
    ShowRaster(item, x - adjust, y - adjust, scale, &temp);

#else /* SUN120FB or VAX */

#ifdef MC68000
#define RasterOp(dst, src, op) RasterOpS(dst, src, op)
#define RestrictRaster SubRasterS
#define RopColor(bool) (bool ? GXcopyInverted : GXcopy)
#else
#define RopColor(bool) (bool ? GXcopy : GXcopyInverted)
#endif

    VRaster src, dst;
    int dx = r->ScreenXmin - x;	/* how much to chop off on left */
    int dy = r->ScreenYmin - y;	/* how much to chop off on top */

    if (dx < 0) dx = 0;
    if (dy < 0) dy = 0;
    if (scale == 0)
      {
	x += dx; y += dy;
#ifndef MC68000
	RestrictRaster(&src+1, item+1, dy, dx, item->bBox.v-dy, item->bBox.h-dx);
	RestrictRaster(&dst+1, WholeScreen, y, x,
	    r->ScreenYmax + 1 - y, r->ScreenXmax + 1 - x);
	RasterOp(&dst+1, &src+1,
	    RopColor(Options(*item) & RasterInverse));
#else MC68000
	SubRasterS(&src+1, item+1, dy, dx, item->bBox.v-dy, item->bBox.h-dx);
	SubRasterS(&dst+1, &Sun120Fb+1, y, x,
	    r->ScreenYmax + 1 - y, r->ScreenXmax + 1 - x);
	RasterOpS(&dst+1, &src+1,
	    item->options & RasterInverse ? GXcopyInverted : GXcopy);
#endif MC68000
      }
    else
      { /* NOT IMPLEMENTED YET FOR THE SUN-2 FRAMEBUFFER */ }
#endif
  }

freeRaster(r)
    /* Invokes the "FreeRaster" routine in the graphics library.  In fact, a
     * lot of Per's raster code in the graphics library is really device
     * independent, so it should eventually be pulled back into the VGTS (gag).
     */
#ifdef SUN100FB
    MemRaster *r;
  {
    FreeRaster(r);
#else /* SUN120FB or VAX */
    VRaster *r;
  {
    if (r->start != NULL) /* for paranoia's sake */
	free(r->start);
#endif
  }


/*
 * Stripped down version of Per's generalized raster put mechanism.
 *							DRK
 */
/* RSF: another reinvented wheel - bletch! */
    
/* show a MemRaster on the screen, optionally magnified.
 * Per Bothner. March 1983.
 */

DisplayNib(in,x,y,clip)
  struct mem_raster *in;	/* raster (must be in memory) to display */
  int x,y;			/* top, left corner of FB rectangle. */
  struct SubView *clip;		/* bounding box of raster */
  /* GXfunction must be set before calling this routine */
  {
#ifdef SUN100FB
   short moreWidth = in->width;
    short curOffset = 0;
    int height = in->height;
    short curWidth;
    register shift, i;
    extern int GXBase;
    register char *RegGXBase = (char *)GXBase;
#define GXBase RegGXBase
    register short *yFB = y + (short *)((GXupdate|GXselectY|GXsource)+GXBase);
    short *colTop = in->start;
    register short *memPtr;
    short maxColWidth = 16;

    GXfunction = GXpaintInverted;

    if (x < clip->ScreenXmin)
      { /* left clipping */
	i = (clip->ScreenXmin - x);
	x += i;
	moreWidth -= i;
	if (moreWidth <= 0) return;
	curOffset += i;
	colTop += in->height * (curOffset >> 4);
	curOffset &= 0xF;
      }
    if (y < clip->ScreenYmin)
      { /* top clipping */
	i = (clip->ScreenYmin - y);
	height -= i;
	if (height <= 0) return;
	colTop += i;
	y += i;
	yFB += i;
      }
    i = x - (clip->ScreenXmax + 1) + moreWidth;
    if (i > 0)
      { /* right clipping */
	moreWidth -= i;
	if (moreWidth <= 0) return;
      }
    i = y - (clip->ScreenYmax + 1) + height;
    if (i > 0)
      { /* bottom clipping */
	height -= i;
	if (height <= 0) return;
      }
    while (moreWidth > 0)
      {
        curWidth = 16 - curOffset;
	if (curWidth > moreWidth) curWidth = moreWidth;
	if (curWidth > maxColWidth) curWidth = maxColWidth;
	memPtr = colTop;
	shift = 16 - curOffset - curWidth;
	GXwidth = curWidth;
	GXBase[x<<1] = shift;
	  {
	    register j; register short word;
		for (i = height; i > 0; i--)
		  { 
		    word = (*memPtr++ >> shift) & ~(-1 << curWidth);
		    *yFB++ = (word <<= 16 - curWidth);
		  }
          }
	x += curWidth;
	moreWidth -= curWidth;
	curOffset += curWidth;
	yFB -= height;
	if (curOffset >= 16) {curOffset = 0; colTop += in->height;}
      }
#undef GXBase
#endif  SUN100FB

#ifdef  SUN120FB

  /* Implemented by Gus Fernandez, March 15, 1985. This code assumes
     that there is a BitLong which is twice the size of a BitWord */

#define BitLong unsigned int
#define BitLongLen (2*BitWordLen)
#define BitLongLog (1+BitWordLog)

  static BitWord LeftMasks[] = 
    {
      0xFFFF,0x7FFF,0x3FFF,0x1FFF,0x0FFF,0x07FF,0x03FF,0x01FF,
      0x00FF,0x007F,0x003F,0x001F,0x000F,0x0007,0x0003,0x0001,0x0000
    };

  static BitWord RightMasks[] =
    {
      0x0000,0x8000,0xC000,0xE000,0xF000,0xF800,0xFC00,0xFE00,
      0xFF00,0xFF80,0xFFC0,0xFFE0,0xFFF0,0xFFF8,0xFFFC,0xFFFE,0xFFFF      
    };
    register BitWord mask, *source;
    register BitUnit *faddr;
    register int stridebytes,offset,numrows,y1,t,t2,t3,closest;

    closest=(x >> BitWordLog) << BitWordLog;
    offset=BitWordLen - x + closest;
    if (x < clip->ScreenXmin) 
      {
	t = clip->ScreenXmin-x;
        if (t > in->width) return;
      }
    else t=0;
    t2=in->width;
    if (x+t2 > clip->ScreenXmax)
      {
        t3 = clip->ScreenXmax - x + 1;
        if (t3 <= 0) return;
        if (t3 < t2) t2=t3;
      }
    mask=LeftMasks[t] & RightMasks[t2];
    y1=y;
    source=(BitWord *)in->start;
    numrows=in->height;
    if (y < clip->ScreenYmin)
      {
        t=clip->ScreenYmin-y;
        if (t > numrows) return;
        y1+=t;
        source+=t;
        numrows-=t;
      }
    if (y1 + numrows > clip->ScreenYmax)
      {
        t3 = clip->ScreenYmax - y1 + 1;
        if (t3 <=0) return;
        if (t3 < numrows) numrows=t3;
      }
    stridebytes=Sun120Fb.stride;
    faddr=Sun120Fb.start+(y1*stridebytes)+(closest >> BitUnitLog);
    numrows--;
    do
      {
	*(BitLong *)faddr |= ((BitLong)(*source++ & mask) << offset);
	(BitUnit *)faddr += stridebytes;
      }
    while (--numrows >=0);

#endif  SUN120FB

#ifdef  QVSS
	/* Taken from sun120 code above */
#define BitLong unsigned int
#define BitLongLen (2*BitWordLen)
#define BitLongLog (1+BitWordLog)

    /*
     * On the QVSS, the left-most pixel is the low-order bit, so
     * most things are reversed from the way they are done on the Sun.
     *
     * The nib cannot exceed one BitWord (16 bits) in width.
     *
     * Note:  This routine does NOT do bit-reversal on the nibs.
     * This works only because they are left/right symmetric.
     * IF ASYMMETRIC NIBS ARE USED, THIS WON'T WORK.  Either this
     * routine can be changed to do bit-reversal, or the nibs can
     * be conditionally compiled according to machine type, since
     * they are all defined at compile time.
     */


    /*
     * LeftMasks[n] is used to mask off the n leftmost bits of the
     * nib.  It is used for clipping.
     *
     * RightMasks[n] is used to mask off everything except the n
     * leftmost bits on the nib, and is used both for clipping and for nibs
     * less than the length of a BitWord (16 bits).
     */

  static BitWord LeftMasks[] = 
    {
      0xFFFF,0xFFFE,0xFFFC,0xFFF8,0xFFF0,0xFFE0,0xFFC0,0xFF80,
      0xFF00,0xFE00,0xFC00,0xF800,0xF000,0xE000,0xC000,0x8000,0x0000
    };

  static BitWord RightMasks[] =
    {
      0x0000,0x0001,0x0003,0x0007,0x000F,0x001F,0x003F,0x007F,
      0x00FF,0x01FF,0x03FF,0x07FF,0x0FFF,0x1FFF,0x3FFF,0x7FFF,0xFFFF
    };

    register BitWord mask, *source;
    register BitUnit *faddr;
    register int stridebytes,offset,numrows,y1,t,width,t3,closest;


    /* Find what word this is in */
    closest=(x >> BitWordLog) << BitWordLog;

    if (x < clip->ScreenXmin) 
      {
	t = clip->ScreenXmin-x;
        if (t > in->width) return;
      }
    else t=0;   /* Number of bits to mask off on the left */
    width=in->width;   /* Width of nib */
    if (width > BitWordLen)
	width = BitWordLen; /* Nib must fit into one BitWord (short word) */
    if (x+width > clip->ScreenXmax)
      {
        t3 = clip->ScreenXmax - x + 1;
        if (t3 <= 0) return;
	if (t3 < width) width=t3;
      }

    /*
     * This is the amount by which each word will be left-shifted before
     * being stored in the frame buffer.  (Remember that the << left-shift
     * operator actually shifts the image to the right, since it moves
     * it into higher-order bits.)
     *
     * The term x - closest is the bit offset within the word where the
     * image is to go.
     *
     * The term BitWordLen - width is to get the nib left-justified, since
     * the definitions assume the high-order bit is at the left, whereas
     * here it's at the right.  (See note above about symmetry.)
     *
     * Logically these two are independent -- first we would do
     * << (x-closest) and then >> (BitWordLen-width).  The shifts are
     * combined for efficiency.
     */
    offset = (x - closest) - (BitWordLen - width);

    /*
     * LeftMasks[i] and RightMasks[i] have to be shifted to line up
     * the high-order bits in the nib.
     */
    mask = (LeftMasks[t] & RightMasks[width]) << (BitWordLen - width);
    if (mask == 0)      /* This can be 0 when clipping occurs */
	return;

    y1=y;
    source=(BitWord *)in->start;    /* Bitmap of nib */
    numrows=in->height;
    if (y < clip->ScreenYmin)
      {
        t=clip->ScreenYmin-y;
        if (t > numrows) return;
        y1+=t;
        source+=t;
        numrows-=t;
      }
    if (y1 + numrows > clip->ScreenYmax)
      {
        t3 = clip->ScreenYmax - y1 + 1;
        if (t3 <=0) return;
        if (t3 < numrows) numrows=t3;
      }
    stridebytes=WholeScreen[-1].stride;
    faddr=WholeScreen[-1].start+(y1*stridebytes)+(closest >> BitUnitLog);
    numrows--;
    do
      {
#ifdef  VAX
	/* The VAX treats a negative left shift as a right shift */
	*(BitLong *)faddr &= ~((BitLong)(*source++ & mask) << offset);
#else
	if (offset > 0)
	    *(BitLong *)faddr &= ~((BitLong)(*source++ & mask) << offset);
	else
	    *(BitLong *)faddr &= ~((BitLong)(*source++ & mask) >> -offset);
#endif  VAX
	(BitUnit *)faddr += stridebytes;
      }
    while (--numrows >=0);
#endif  QVSS
  }


/*  Draw one marker, for the GKS polymarker command.  A marker is defined
 *  by an 8x8 bitmap.  The (3,3) pixel, counted from (0,0) at the upper left
 *  corner, is considered to be the center, and falls on the specified point
 */
#ifndef MC68000
Rectangle fbraster;
VRaster memraster;
BitUnit markerbuf[16];
#else MC68000
struct fb_raster fbraster;
struct mem_raster memraster;
short markerbuf[8];
#endif MC68000
static ShiftRaster();

DrawMarker(x, y, marker, r)
  short x, y;
#ifndef MC68000
  BitUnit *marker;
#else MC68000
  short *marker;
#endif MC68000
  register struct SubView *r;
  {
#ifndef MC68000
    register Rectangle *f = &fbraster;
    register VRaster *m = &memraster+1;
#else MC68000
    register struct fb_raster *f = &fbraster;
    register struct mem_raster *m = &memraster;
#endif MC68000
    register short xleft = x - 3, xright = x + 4, ytop = y - 3, ybot = y + 4;
    register short dif;

#ifndef MC68000
    f->size.h = 8;
    f->size.v = 8;
    m[-1].start = marker;
#else MC68000
    f->width = 8;
    f->height = 8;
    m->start = marker;
#endif MC68000

    if (xright < r->ScreenXmin || xleft > r->ScreenXmax ||
	ybot < r->ScreenYmin || ytop > r->ScreenYmax)	   return;

    if (ytop < r->ScreenYmin) 		/* top clipping */
      {
	dif = r->ScreenYmin - ytop;
	ytop = r->ScreenYmin;
#ifndef MC68000
	f->size.v -= dif;
	m[-1].start += dif;
#else MC68000
	f->height -= dif;
	m->start += dif;
#endif MC68000
      }
    if (ybot > r->ScreenYmax)		/* bottom clipping */
#ifndef MC68000
	f->size.v -= ybot - r->ScreenYmax;
#else MC68000
	f->height -= ybot - r->ScreenYmax;
#endif MC68000
    if (xleft < r->ScreenXmin)  	/* left clipping */
      {
	dif = r->ScreenXmin - xleft;
	xleft = r->ScreenXmin;
#ifndef MC68000
	f->size.v -= dif;
	ShiftRaster(m->start, markerbuf, f->size.v, dif);
	m[-1].start = markerbuf;
#else MC68000
	f->width -= dif;
	ShiftRaster(m->start, markerbuf, f->height, dif);
	m->start = markerbuf;
#endif MC68000
      }
    if (xright > r->ScreenXmax)		/* right clipping */
#ifndef MC68000
	f->size.h -= xright - r->ScreenXmax;
#else MC68000
	f->width -= xright - r->ScreenXmax;
#endif MC68000

#ifndef MC68000
    f->start.h = xleft;
    f->start.v = ytop;
#else MC68000
    f->x = xleft;
    f->y = ytop;
#endif MC68000

    gl_PaintRaster(f, m);
  }


static ShiftRaster(src, dst, length, shift)
  short *src, *dst;
  int length, shift;
  {
    register int i;

    for (i=0; i<length; i++)  *dst++ = *src++ << shift;
  }


/* Low-level routines for implementing Kenneth's new SDF item types: */

/* NOTE: Plot() should someday be put in the graphics libraries. */

Plot(x, y)
  int x, y;
  {
#ifdef SUN100FB
    register short *Xaddress = 
	(short *) (GXBase | GXupdate | GXselectX | (x << 1));
    register short *Yaddress =
	(short *) (GXBase | GXmask | GXselectY | (y << 1));

    GXfunction = GXclear;  /* should not be here; just for debugging */
    GXwidth = 1;
    *Yaddress = 1;  /* set y */
    *Xaddress = 1;  /* set x and plot the point */
#else SUN120FB
    return;  /* NOT IMPLEMENTED YET FOR THE SUN-2 FRAMEBUFFER */
#endif
  }

FillRow(y, x1, x2, pattern)
  register short y, x1, x2;
  register short *pattern;
  {
#ifdef SUN100FB
    register short *Xaddress = 
	(short *) (GXBase | GXupdate | GXselectX | (x1 << 1));
    register short *Yaddress =
	(short *) (GXBase | GXmask | GXselectY | (y << 1));
    register int width = x2 - x1 + 1;
    register int i;

    GXwidth = 16;
/* debug only */ GXfunction = GXandInvPattern;
    *Yaddress = pattern[y & 15];  /* set y and pattern */
    for (i = width >> 4; i > 0; i--)
      {
	*Xaddress = 1;		/* draw 16 pixels: no source */
	Xaddress += 16;
      }
    /* the tag end, less than 16 pixels */
    width &= 15;
    if (width != 0)
      {
	GXwidth = width;
	*Xaddress = 1;
      }
#endif  SUN100FB

#ifdef  SUN120FB

#define ROWLENGTH (ScreenLimitX>>3)  /* number of bytes in a framebuffer row */
    register short *fbaddr;
    register int width = x2 - x1 + 1;
    register short patternbits, mask;
    register int i, tagend;
    int wholewords;

    patternbits = pattern[y & 15];
    fbaddr = (short *) ((GXBase + y*ROWLENGTH + (x1>>3)) & ~1);

    /* special case: a short strip all in the same word */
    if ((x1 ^ x2) < 0x10)  /* i.e. they differ only in the lowest 4 bits */
      { register short mask2;
	tagend = x1 & 0xf;
	mask = 0xffff >> tagend;
	tagend = x2 & 0xf;
	mask &= ~(0xffff >> tagend);
	*fbaddr = (*fbaddr & ~mask) | (patternbits & mask);
	return;
      }

    tagend = x1 & 0xf;
    if (tagend)
      {
	mask = 0xffff >> tagend;
	*fbaddr = (*fbaddr & ~mask) | (patternbits & mask);
	fbaddr++;
      }
	
    wholewords = (x2 >> 4) - ((x1+0xf) >> 4);
    for (i = 0; i < wholewords; i++)
	*fbaddr++ = patternbits;

    /* the tag end, less than 16 pixels */
    tagend = x2 & 0xf;
    if (tagend)
      {
	mask = 0xffff >> tagend;
	*fbaddr = (*fbaddr & mask) | (patternbits & ~mask);
      }

#endif  SUN120FB

#ifdef  QVSS

#define ROWLENGTH (ScreenLimitX>>3)  /* number of bytes in a framebuffer row */
    register short *fbaddr;
    register int width = x2 - x1 + 1;
    register short patternbits, mask;
    register int i, tagend;
    int wholewords;

    patternbits = pattern[y & 15];
#define GXBase ((int)(WholeScreen[-1].start))
    fbaddr = (short *) ((GXBase + y*ROWLENGTH + (x1>>3)) & ~1);

    /* special case: a short strip all in the same word */
    if ((x1 ^ x2) < 0x10)  /* i.e. they differ only in the lowest 4 bits */
      { register short mask2;
	tagend = x1 & 0xf;
	mask = 0xffff >> tagend;
	tagend = x2 & 0xf;
	mask &= ~(0xffff >> tagend);
	*fbaddr = (*fbaddr & ~mask) | (patternbits & mask);
	return;
      }

    tagend = x1 & 0xf;
    if (tagend)
      {
	mask = 0xffff >> tagend;
	*fbaddr = (*fbaddr & ~mask) | (patternbits & mask);
	fbaddr++;
      }
	
    wholewords = (x2 >> 4) - ((x1+0xf) >> 4);
    for (i = 0; i < wholewords; i++)
	*fbaddr++ = patternbits;

    /* the tag end, less than 16 pixels */
    tagend = x2 & 0xf;
    if (tagend)
      {
	mask = 0xffff >> tagend;
	*fbaddr = (*fbaddr & mask) | (patternbits & ~mask);
      }
#endif  QVSS
  }


#ifdef SUN100FB
/* Given color and opacity attributes, set GXfunction */
static short fillfunctions[2][3] = { 
	{GXandInvPattern, GXpaintPattern, GXinvertPattern},
	{GXcopyInvPattern, GXcopyPattern, GXinvertPattern}
	};
#endif SUN100FB

SetFillFunction(color, opacity)
  short color, opacity;
  {
#ifdef SUN100FB
    GXfunction = fillfunctions[opacity][color];
#else
    return;  /* eventually, set function pointers */
#endif
  }


#ifdef SUN100FB
/* set GXfunction for line drawing, where no pattern is involved */
static short colorfunctions[3] = {
	{GXclear, GXset, GXinvert}
	};
#endif

SetColorFunction(color)
  short color;
  {
#ifdef SUN100FB
    GXfunction = colorfunctions[color];
#else
    return;  /* eventually, set function pointers */
#endif
  }
