/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: display.c
 *
 * Bill Nowicki August 1982
 *
 * This file contains various display support
 * routines for the virtual graphics terminal service.
 * popup menu, display (clipped) text in a graphics window, and
 * display a cursor.
 * These are the routines that have to know about VGTS private
 * data structures; the raster-op and draw.c routines are lower level.
 */



#include "Vgts.h"
#include <chars.h>
#include "sdf.h"
#include "vgt.h"

extern short	ScreenLimitX,
		ScreenLimitY;

extern short Debug;

#include <fonttable.h>
#define UseVarFont Fonts.first16[Menu_Font].font
short	 MouseX, MouseY, MouseButtons;

		/* Variables defining the state of the mouse device. */


/*******************************************************************/

/*
 * popup:
 * The routine popup() takes a menu and puts up (at current mouse position)
 * the appropriate pup-up menu on the screen.  It returns the number of the
 * selected menu item, or -1 if none were selected.
 * To create a menu, a region of suitable size of the screen is copied to a
 * newly allocated portion of memory, and a menu is drawn over it.  A selection
 * is taken and is returned.
 *
 * The menu is passed as a 0-terminated array of strings.
 */

extern int MouseUpKludge;

int popup(menu)
  PopUpEntry menu[];
{
    short i;
    short menu_count;			/* number of strings in the menu */
    short screenX, screenY;		/* Upper left screen corner */
    short BltHeight;			/* Height of menu */
# define BltWidth 208			/* Width of menu */
# define MAX_MENU_ITEMS 40		/* (see below) */
    
    short saveXmin, saveXmax, saveYmin, saveYmax;
    int popup_value;
    short buttons;
    static struct SubView fakeSubView;
    
/* First, count items in menu -- we stop if the number of entries exceeds
 * MAX_MENU_ITEMS, not because of space considerations, but to protect against
 * some loser providing a menu with no terminating NULL string.
 */

    menu_count = 0;
    while (menu[++menu_count].string)
	if (menu_count == MAX_MENU_ITEMS) break;

/* Now figure out where the upper-left corner goes, and the height of the
 * menu.  The menu is always located so that
 * it doesn't go off the edge of the screen.  
 * It is placed slightly farther from
 * the bottom of the screen so that it is easy to get the cursor tip into the
 * bottom selection in all cases.
 */

    BltHeight = 16*menu_count + 1;		/* +1 for bottom line */
    screenX = (MouseX - 2*BltWidth/3);
    screenY = (MouseY - BltHeight/2);
    if (screenX < 0) screenX = 0;
    if (screenY < 0) screenY = 0;
    if (screenX > ScreenLimitX - BltWidth)  screenX = ScreenLimitX - BltWidth;
    if (screenY > ScreenLimitY - BltHeight) screenY = ScreenLimitY - BltHeight;

    EraseCursor(screenX, screenX+BltWidth,
    		screenY, screenY+BltHeight);
    
    SaveBeforeMenu(screenX, screenY, BltWidth, BltHeight);

/* Now clear the Frame: */

    ClearRegion( screenX, screenY, BltWidth, BltHeight);

/* Now draw in the various lines */

    FillRegion(screenX, screenY, BltHeight, 1);
    FillRegion(screenX + BltWidth - 1, screenY, BltHeight, 1);
    for (i = 0; i <= menu_count; i++)
        FillRegion(screenX, screenY + i * 16, 1, BltWidth);

/*
 * The fake window is so the Display Text function does the
 * character clipping correctly.
 */
    fakeSubView.ScreenXmin = screenX;
    fakeSubView.ScreenXmax = screenX+BltWidth;
    fakeSubView.ScreenYmin = screenY;
    fakeSubView.ScreenYmax = screenY+BltHeight;

    for (i = 0; i < menu_count; i++)
# ifdef OneFont
        DrawSimpleText(screenX+8, screenY + 16*i + (16 + 1),
			menu[i].string, &fakeSubView, TRUE );
# else OneFont
	if (UseVarFont)
            DrawGeneralText(screenX+8, screenY + 16*i + (16 - 3), 
			menu[i].string, strlen(menu[i].string),
			&fakeSubView, Menu_Font, TRUE );
	else
            DrawSimpleText(screenX+8, screenY + 16*i+ (16 + 1), 
			menu[i].string, &fakeSubView, TRUE );
# endif OneFont

/* OK.  The menu is up -- all we need to do is to interpret it. */
    
    MenuCursor(screenX,screenX+BltWidth,screenY,screenY+BltHeight);
    buttons = GetMouseInput();

    if (buttons == 7 ||		/* universal abort */
        MouseX < screenX || MouseX > screenX + BltWidth ||
        MouseY < screenY || MouseY > screenY + BltHeight - 1)
    	popup_value = -1;
    else
        popup_value = menu[(MouseY - screenY - 1)/FontHeight].menuNumber;

    if (!DisableCursor())
      {
      	/*
	 * If we did not have to redraw the world,
	 * we can just blit back what was there before.
	 */
        EraseCursor(screenX, screenX+BltWidth,
		    screenY, screenY+BltHeight);
	RestoreAfterMenu();
      }

    FreeMenuSaveSpace();

    Cursor();
    MouseUpKludge = 1;
    return (popup_value);
}


DisplayBackground(w)
  View *w;
  {
  	/*
	 * draw in the background pattern for the indicated subViewPorts.
	 * this is trickier than it sounds, because we have to take into
	 * account any possible edges.
	 */
    register struct SubView *r;

    if (EraseCursor(w->ScreenXmin, w->ScreenXmax, w->ScreenYmin, w->ScreenYmax))
        return;
    for (r=w->rect;r;r=r->next)
      {
        int left   = (r->EdgeSet&LeftEdge)   ? 2 : 0;
        int right  = (r->EdgeSet&RightEdge)  ? 2 : 0;
        int top    = (r->EdgeSet&TopEdge)    ? 2 : 0;
        int bottom = (r->EdgeSet&BottomEdge) ? 2 : 0;

         DrawBackground( r->ScreenXmin - left, r->ScreenYmin - top, 
	     r->ScreenXmax - r->ScreenXmin+left+right+1, 
	     r->ScreenYmax - r->ScreenYmin+top+bottom+1 );
       }	
    Cursor();
  }



# define Code(x,y) (x<r->ScreenXmin ? LeftEdge : (x>r->ScreenXmax ? RightEdge : 0)) + \
   (y<r->ScreenYmin ? TopEdge : (y>r->ScreenYmax ? BottomEdge : 0)) 


# define Swap(x,y) {t=y; y=x; x=t;}

/*******************************************************************/

DisplayLine( x1, y1, x2, y2, r )
  register short x1, y1, x2, y2;
  register struct SubView *r;
  {
  	/*
	 * clips a line to a window and draws it.
	 */
    int edgeset1 = Code(x1,y1);
    int edgeset2 = Code(x2,y2);
    int t;
    
    if (x1==x2 && y1==y2) return;
    
    while (edgeset1 || edgeset2)
      {
          /*
	   * If neither part of the line is visible, return.
	   * otherwise, clip to a side determined by an edgeset,
	   * and try again.
	   */
        if (edgeset1 & edgeset2) return;
	if (edgeset1==0) 
	  {
	    Swap(edgeset1,edgeset2);
	    Swap(x1,x2);
	    Swap(y1,y2);
	  }
	if (edgeset1 & LeftEdge)
	  {
	    y1 = y1+(y2-y1)*(r->ScreenXmin-x1)/(x2-x1);
	    x1 = r->ScreenXmin;
	  }
	else if (edgeset1 & RightEdge)
	  {
	    y1 = y1+(y2-y1)*(r->ScreenXmax-x1)/(x2-x1);
	    x1 = r->ScreenXmax;
	  }
	else if (edgeset1 & TopEdge)
	  {
	    x1 = x1+(x2-x1)*(r->ScreenYmin-y1)/(y2-y1);
	    y1 = r->ScreenYmin;
	  }
	else if (edgeset1 & BottomEdge)
	  {
	    x1 = x1+(x2-x1)*(r->ScreenYmax-y1)/(y2-y1);
	    y1 = r->ScreenYmax;
	  }
	edgeset1 = Code(x1,y1);
      }
    DrawLine(x1,y1, x2-x1,y2-y1);
    
  }


DisplayBanner(w,r)
    register View_PTR w;
    register struct SubView *r;
  {
  	/*
	 * Display the banner for the given viewport with the
	 * given subviewport.
	 */

    if (UseVarFont)
      DrawGeneralText( w->ScreenXmin + 16, w->ScreenYmin+BannerHeight-1,
    		 	w->banner, strlen(w->banner), r, Menu_Font, FALSE );
    else
      DrawSimpleText( w->ScreenXmin + 16, w->ScreenYmin+BannerHeight+1, 
    		 	w->banner, r, FALSE );

    if (w->bannerState == DisplayPageMessage)

      {

	register short xMin;

	xMin = Max(w->ScreenXmin + 16, w->ScreenXmax - PageMessageWidth);
        DrawSimpleText( xMin, w->ScreenYmin+BannerHeight+1,
			 PageMessage, r, FALSE);

      }

  }



/*********************  "poly" drawing routines  **********************/
/* Transform the point array to account for a zoom factor.  This same
 * routine should someday support general scaling and rotation.
 */
SdfPoint *TransformPoints(npoints, indata, zoom)
  short npoints, zoom;
  register SdfPoint *indata;
  {
    register SdfPoint *outdata, *outarray;
    register int i;
#define RoundShift(a, sh) 	( ((a) >> sh) + (((a) >> (sh -1)) & 1) )

    outarray = outdata = (SdfPoint *) malloc((npoints+1) * sizeof(SdfPoint));
    if (!outdata) return(NULL);

    if (zoom > 0)
      {
	for (i=0; i < npoints; i++)
	  {
	    outdata->x = indata->x << zoom;
	    outdata->y = indata->y << zoom;
	    indata++;
	    outdata++;
	  }
      }
    else
      {
	zoom = -zoom;
	for (i=0; i <= npoints; i++)  /* yes, I really meant <= */
	  {
	    outdata->x = RoundShift(indata->x, zoom);
	    outdata->y = RoundShift(indata->y, zoom);
	    indata++;
	    outdata++;
	  }
      }
    return(outarray);
  }


DisplayPolyline(npoints, indata, x, y, zoom, clip)
  short x, y, npoints, zoom;
  SdfPoint indata[];
  register struct SubView *clip;
  {
    SdfPoint *data;
    register SdfPoint *p;

    if (npoints < 2) return;

    /* for zoom, and eventually rotate, etc. we make a transformed copy	
     * of the input data array.
     */
    if (zoom == 0) data = indata;
    else data = TransformPoints(npoints, indata, zoom);
    if (!data) return;

    for (p = data; p < data+npoints-1; p++)
      { /* y - p->y is because screen y axis points down, not up */
	DisplayLine(x + p->x, y - p->y, x + (p+1)->x, y - (p+1)->y, clip);
      }
    if (zoom) free(data);
  }


DisplayPolymarker(npoints, indata, bitmap, x, y, zoom, clip)
  short x, y, npoints, zoom;
  short *bitmap;
  SdfPoint indata[];
  register struct SubView *clip;
  {
    SdfPoint *data;
    register SdfPoint *p;

    if (npoints < 1) return;

    /* for zoom, and eventually rotate, etc. we make a transformed copy	
     * of the input data array.
     */
    if (zoom == 0) data = indata;
    else data = TransformPoints(npoints, indata, zoom);
    if (!data) return;

    for (p = data; p < data+npoints; p++)
      { /* y - p->y is because screen y axis points down, not up */
	DrawMarker(x + p->x, y - p->y, bitmap, clip);
      }
    if (zoom) free(data);
  }


#include "arc.h"
extern Arcdata *TransformArcData();
extern Arc *MakeSeg();
extern Seg *segfree;	/* freelist of Seg (or Arc) records */

#define FreeArc(e) 	( e->next = segfree, segfree = e )

DisplayPolyarc(npoints, indata, x, y, zoom, clip)
  short x, y, npoints, zoom;
  Arcdata indata[];
  register struct SubView *clip;
  {
    Arcdata *data;
    register Arcdata *p;
    register Seg *arc;
    extern Seg *arclist;

    if (npoints < 2) return;

    /* for zoom, and eventually rotate, etc. we make a transformed copy	
     * of the input data array.
     */
    if (zoom == 0) data = indata;
    else data = TransformArcData(npoints, indata, zoom);
    if (!data) return;

    arclist = NULL;
    for (p = data; p < data+npoints-1; p++)
      {
	if (p->type == LINE)  
	    DisplayLine(x + p->x, y - p->y, x + (p+1)->x, y - (p+1)->y, clip);
	else 
	  {
	    arc = MakeSeg(p, p+1, x, y); /* see fillpolyarc.c */
	    if (arc->type == VESTIGE) /* a very small arc, too small to draw */
	      {
		FreeArc(arc);
	      }
	    else
	      {
		arc->next = arclist;
		arclist = arc;
	      }
	  }
      }
    DrawArcs(arclist);

    FreeWholeList(arclist);
    if (zoom) free(data);
  }


