/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * General support routines for the V team server.
 */

#include "team.h"
#include <Vgroupids.h>
#include <Vquerykernel.h>
#include <Vteams.h>
#include <Vgroupids.h>

unsigned char OurProcFamily; /* What type of processor this is */

TeamsEntry *FindNextTeam();
TeamsEntry *AdvanceCurrentTeamPtr();
SystemCode CheckIfValidIO();
SystemCode MapCid(), LookupName();
short ExternalPriorityType();

extern SystemCode SetTeamPriority();
extern ProcessId LoadTeamFromNamedFile();
extern SystemCode SetUpEnvironment();
extern char *malloc();

/* Align pointer p to the next b-byte boundary, assuming b is a power of 2. */
#define align(p, b) (  (p) + ( -(int)(p) & ((b)-1) )  )

/* Query the kernel to find our processor family type. */
unsigned char getProcFamily()
  {
    Message msg;
    MachineConfigurationReply *reply = (MachineConfigurationReply *)msg;
    unsigned char procFamily;

    QueryKernel(0, MACHINE_CONFIG, reply);
    if (reply->replycode != OK)
	return(-1);
    procFamily = (reply->processor)&PROC_FAMILY;
    if ((procFamily < 0) ||
	((procFamily>>PROC_FAMILY_SHIFT) >= NUM_PROC_FAMILIES))
	return(-1);
    return (procFamily);
  }


/*
 * InitServerDataStructures:
 * Initialize the various data structures of the team server.
 */

InitServerDataStructures()
  {
    int i;
    SystemCode r;

    /* Initialize free list of team entry records. */
    TeamsFreeList = InitStack();
    for (i = 0; i < MaxTeams; i++)
      {
        Teams[i].inUse = 0;
	PushStack(TeamsFreeList, &Teams[i]);
      }

    /* Initialize list of active team entry records to null. */
    TeamsInUseList = InitCList();

    /* Initialize round-robin scheduling pointers to "null". */
    CurrentRunningTeam = NULL;
    CurrentFgTeam = NULL;
    CurrentBgTeam = NULL;
    CurrentGuestTeam = NULL;

    /* Initialize host status data structures. */
    InitHostStatus();

    /* Set our processor family type */
    OurProcFamily = getProcFamily();
  }


/*
 * TerminateSingleTeam:
 * Destroy the specified team if a TERMINATE_TEAM requestcode is supplied.
 * Then terminate the associated postmortem debugger (if any) for the
 * designated Teams table entry and then reclaim the table entry.
 * If the requestcode is DeadTeamNotice then don't try to kill the team's
 * rootPid.  (If it was on a different logical host then the DestroyProcess
 * will timeout non-locally.)
 */

TerminateSingleTeam(teamEntryPtr, requestcode)
    TeamsEntry *teamEntryPtr;	/* Ptr to a team record. */
    SystemCode requestcode;	/* Determines whether we have to destroy the
				   the root pid of the team or not. */
  {
    TeamsEntry *p;
    ProcessId deadPid;

    Lock(LockId);
    teamEntryPtr->inUse = 0;

    deadPid = teamEntryPtr->rootPid;

    /* Destroy the "regular" team specified in the table entry. */
    if (requestcode == TERMINATE_TEAM)
      {
	DestroyProcess(deadPid);
      }

    /* Release the team's program image file instance */
    ReleaseInstance(teamEntryPtr->loadServer, teamEntryPtr->loadFile, OK);

    /* Return the space for the name to free space. */
    free(teamEntryPtr->name);

    /* Check if we must readjust the round-robin scheduling ptrs. */
    ReadjustCurrentPtrs(teamEntryPtr);

    /* Remove the table entry. */
    RemoveCList(TeamsInUseList, teamEntryPtr);
    Unlock(LockId);
    PushStack(TeamsFreeList, teamEntryPtr);
    teamEntryPtr->ownerPid = 0;

    /* Run through the list of teams and see if this team was the owner of
       any of them.  Terminate any teams whose owner was on this team
       unless the requestcode specifies otherwise. */
    if (requestcode != DeleteTeamEntryOnly)
      {
	requestcode = TERMINATE_TEAM;
      }
    teamEntryPtr = FirstCList(TeamsInUseList);
    while (teamEntryPtr != NULL)
      {
        if (SameTeam(deadPid, teamEntryPtr->ownerPid))
	  {
	    TerminateSingleTeam(teamEntryPtr, requestcode);
	    teamEntryPtr = FirstCList(TeamsInUseList);
				/* Reset because we don't know what else has
				   been deleted from the list. */
	  }
	else
	  {
	    teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr);
	  }
      }
  }


/*
 * InvokeDebugger:
 * Invokes the postmortem debugger to handle a team that has incurred an
 * exception.
 */

SystemCode InvokeDebugger(reqMsg, reqPid)
    LoadTeamRequest *reqMsg;
    ProcessId reqPid;
  {
    TeamsEntry *teamEntryPtr, *debugTeamEntryPtr;
    ProcessId debuggerPid, invokerPid;
    SystemCode error;
    char lineBuf[MaxArgsLength];
    char *args[10];
    char **argv;
    char *teamend;
    char *name;
    int dontPrintErrFlag = 0;

    /* Find the Teams table entry for the team incurring the exception. */
    for (teamEntryPtr = FirstCList(TeamsInUseList);
		teamEntryPtr != NULL;
		teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
	if (SameTeam(teamEntryPtr->rootPid, reqPid))
	    break;
    if (teamEntryPtr == NULL)	/* Unregistered team! */	
      {
	PrintStdExceptInfo(stdout->fileserver, stdout->fileid,
	    reqMsg, reqPid, 
	    "ERROR - Got an exception from a nonregistered team!");
	DestroyProcess(reqPid);
			    /* Waste it! */
	return(NO_REPLY);
      }
    if (Equal(teamEntryPtr->name, PostDebugger))
				/* Exception occurred in the postmortem
				   debugger! */
      {
	PrintStdExceptInfo(stdout->fileserver, stdout->fileid,
		reqMsg, reqPid, "Exception in postmortem debugger");
	   
	/* Destroy both the Postmortem debugger and the team it was invoked
	   on. */
	invokerPid = teamEntryPtr->ownerPid;
	TerminateSingleTeam(teamEntryPtr, TERMINATE_TEAM);
	for (teamEntryPtr = FirstCList(TeamsInUseList);
		    teamEntryPtr != NULL;
		    teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
	    if (teamEntryPtr->rootPid == invokerPid)
		break;
	TerminateSingleTeam(teamEntryPtr, TERMINATE_TEAM);

	return(NO_REPLY);
      }

    /* Make sure the debugRtMsg for this team is valid. */
    error = CheckIfValidIO(&teamEntryPtr->debugRtMsg);
    if (error != OK)
      {
        dontPrintErrFlag = 1;
	goto abortPM;
      }

    /* Make sure we have enough resources left. */
    if (EmptyStack(TeamsFreeList))
      {
        error = NO_SERVER_RESOURCES;
	goto abortPM;
      }
    name = malloc(strlen(PostDebugger) + 1);
    if (name == NULL)
      {
	goto abortPM;
      }
    strcpy(name, PostDebugger);

    /* Load the postmortem debugger from a prespecified file. */
    debuggerPid = LoadTeamFromNamedFile(reqPid, PostDebugger, &error, &teamend);
    if (debuggerPid == NULL)
      {
        free(name);
        goto abortPM;
      }
      
    /* Set up the arguments for the debugger and set its team priority. */
    strcpy(lineBuf, PostDebugger);
    strcat(lineBuf, " -p -d");
    strcat(lineBuf, " ");
    strcat(lineBuf, teamEntryPtr->name);
    ParseLine(lineBuf, args, 10);
    argv = args;
    teamEntryPtr->debugRtMsg.envblock = align(teamend + INIT_STACK_SIZE, 4);
    error = SetUpEnvironment(debuggerPid, argv, *PerProcess->env,
	    *PerProcess->namecache, teamEntryPtr->debugRtMsg.envblock);
    if (error != OK)
      {
	DestroyProcess(debuggerPid);
	free(name);
	goto abortPM;
      }
    
    /* Enter debugger into the Teams table and set its owner to be the
       team which incurred the exception so that killing that will force
       the debugger to vanish too. */
    /* Find an unused entry in the Teams table. */
    debugTeamEntryPtr = PopStack(TeamsFreeList);
    /* Set user number to same as requestor. */
    SetUserNumber( debuggerPid, User(reqPid) );    
     /* Initialize Teams table entry. */
    Lock(LockId);
    AddCList(TeamsInUseList, TeamsInUseList, debugTeamEntryPtr);
    debugTeamEntryPtr->ownerPid = teamEntryPtr->rootPid;
    debugTeamEntryPtr->runPriority = NONRUNNING_FG_TEAM_PRIORITY;
    debugTeamEntryPtr->inUse = 1;
    Unlock(LockId);
    SetTeamPriority(debuggerPid, debugTeamEntryPtr->runPriority);
    debugTeamEntryPtr->rootPid = debuggerPid;
    debugTeamEntryPtr->name = name;
    /* We don't care about the loadServer, loadFile, and rtMsg fields for a
       postmortem debugger because we NEVER invoke a postmortem debugger on
       a postmortem debugger. */
    debugTeamEntryPtr->loadServer = 0;
    debugTeamEntryPtr->loadFile = 0;
   
    /* Set the debugger running and forward the exception message to it. */
    Reply(&(teamEntryPtr->debugRtMsg), debuggerPid);
    Forward(reqMsg, reqPid, debuggerPid);
    
    return(NO_REPLY);

    /*
     * Postmortem debugger couldn't be invoked.
     * Print standard exception information and destroy team that caused
     * the exception.
     */
abortPM:
    if (dontPrintErrFlag == 0)
      {
	strcpy(lineBuf,"Error loading postmortem debugger: ");
	strcat(lineBuf,ErrorString(error));
	PrintStdExceptInfo(stdout->fileserver, stdout->fileid, reqMsg, 
				    reqPid, lineBuf);
      }
       
    /* Terminate the team causing the exception. */
    TerminateSingleTeam(teamEntryPtr, TERMINATE_TEAM);
    
    return(NO_REPLY);
  }
    

/*
 * BuildDescriptor:
 * Creates a descriptor record that can be handed back to a client.
 * Returns length of record in bytes.
 * If teamEntryPtr is non-null, it identifies a team.  If it is NULL,
 *  the cid identifies a context.
 */

unsigned BuildDescriptor(desc, teamEntryPtr, cid, pid)
    ArbitraryDescriptor *desc;
    TeamsEntry *teamEntryPtr;
    ContextId cid;
    ProcessId pid;	/* client pid */
  {
    if (teamEntryPtr != NULL && teamEntryPtr->ownerPid != 0)
      {
	/* Some team */
	desc->t.descriptortype = TEAM_DESCRIPTOR;
	strncpy(desc->t.fileName, teamEntryPtr->name, MAX_NAME_LENGTH-1);
	desc->t.fileName[MAX_NAME_LENGTH-1] = '\0';
			/* Needed because strncpy doesn't NULL-term.
			   if more than n bytes are encountered. */
	desc->t.ownerPid = teamEntryPtr->ownerPid;
	desc->t.rootPid = teamEntryPtr->rootPid;
	desc->t.fileserver = teamEntryPtr->loadServer;
	desc->t.fileid = teamEntryPtr->loadFile;
	desc->t.rtMsg = teamEntryPtr->debugRtMsg;
	desc->t.runPriority = 
			ExternalPriorityType(teamEntryPtr->runPriority);
	if (DifferentByteOrder(pid))
	    ByteSwapTeamDescriptor(&desc->t);
	return (sizeof(TeamDescriptor));
      }

    /* Not a team */
    switch (cid)
      {
	case GLOBAL_ROOT_CONTEXT:
	  desc->c.descriptortype = CONTEXT_DESCRIPTOR;
	  strncpy(desc->c.name, "[", sizeof(desc->c.name));
	  desc->c.ctxpair.pid = VCSNH_SERVER_GROUP;
	  desc->c.ctxpair.cid = cid;
	  if (DifferentByteOrder(pid))
	      ByteSwapContextDescriptor(&desc->c);
	  return (sizeof(ContextDescriptor));

	case TEAM_SERVER_CONTEXT:
	  desc->c.descriptortype = CONTEXT_DESCRIPTOR;
	  strncpy(desc->c.name, "team", sizeof(desc->c.name));
	  desc->c.ctxpair.pid = VTEAM_SERVER_GROUP;
	  desc->c.ctxpair.cid = cid;
	  if (DifferentByteOrder(pid))
	      ByteSwapContextDescriptor(&desc->c);
	  return (sizeof(ContextDescriptor));

	case DEFAULT_CONTEXT:
	  /* This uses the old descriptor type defined for the service
	   *  server.  Yucch.
	   */
	  desc->h.descriptortype = HOST_DESCRIPTOR;
	  strncpy(desc->h.serverType, "host", sizeof(desc->h.serverType));
	  strncpy(desc->h.hostName, hostname(), sizeof(desc->h.hostName));
	  strncpy(desc->h.userName, hostUsername(), sizeof(desc->h.userName));
	  desc->h.regPid = TeamServerPid;
	  desc->h.availability = hostAvailability();
	  if (DifferentByteOrder(pid))
	      ByteSwapHostDescriptor(&desc->h);
	  return (sizeof(HostDescriptor));
      }

    /* Something weird */
    desc->e.descriptortype = EMPTY_DESCRIPTOR;
    if (DifferentByteOrder(pid))
	ByteSwapEmptyDescriptor(&desc->e);
    return (sizeof(EmptyDescriptor));
  }


/*
 * PrintStdExceptInfo:
 * Prints an error message and then calls the standard exception handler
 * routine.
 */

PrintStdExceptInfo(fileServer, fileid, reqMsg, reqPid, errMsg)
    ProcessId fileServer;
    InstanceId fileid;
    ExceptionRequest *reqMsg;
    ProcessId reqPid;
    char *errMsg;
  {
    File *outFile;
    SystemCode error;

    /* Open the output file instance. */
    outFile = OpenFile(fileServer, fileid, FAPPEND, &error);
    if (outFile == NULL)
      {
	outFile = stdout;
      }

    /* Print the error message if specified. */
    if (errMsg)
      {
	fprintf(outFile, errMsg);
	Flush(outFile);
      }

    /* Call the standard exception handler routine to print out the
       exception information. */
    StandardExceptionHandler(reqMsg, reqPid, outFile);

    /* Close the output file instance unless it is stdout . */
    if (outFile != stdout)
	Close(outFile);
  }


    extern int UpdateIdleTimeCount() , UpdateIdleCounterTime ;  /*ms*/


/*
 * TeamTimerProcess:
 * Periodically wakes up and sends a msg to the main team server process
 * so that it can perform various maintenance functions.
 */

TeamTimerProcess()
  {
    int counter2 = 0;						/*ms*/
    TeamsEntry *pFg, *pBg, *pGuest;
    ProcessId stpNonRunning, stpRunning;
    unsigned short prioNonRunning;

    while (1)
      {
	Delay(0, TimerInterval);
				/* Put the Delay below the Send so that
				   maintenance gets performed immediately
				   after startup. */
	counter2 += TimerInterval;				/*ms*/


	if (counter2 >= UpdateIdleCounterTime)			/*ms*/
	  {				     			/*ms*/
	    counter2 = 0 ;		     			/*ms*/
          UpdateIdleTimeCount() ;	     			/*ms*/
        }				     			/*ms*/

	/* Perform round-robin scheduling. */
	stpNonRunning = 0;
	stpRunning = 0;
	Lock(LockId);
	if (EmptyCList(TeamsInUseList))
	  {
	    Unlock(LockId);
	    continue;
	  }
	if (CurrentRunningTeam != NULL)
	  {
	    stpNonRunning = CurrentRunningTeam->rootPid;
	    prioNonRunning = CurrentRunningTeam->runPriority;
	  }
	pFg = FindNextTeam(CurrentFgTeam, NONRUNNING_FG_TEAM_PRIORITY);
	if (pFg != NULL)
	  {
	    CurrentRunningTeam = pFg;
	    CurrentFgTeam = pFg;
	    stpRunning = pFg->rootPid;
	  }
	else			/* There are no local foreground teams to 
				   run. */
	  {
	    pBg = FindNextTeam(CurrentBgTeam, NONRUNNING_BG_TEAM_PRIORITY);
	    if (pBg != NULL)
	      {
	        CurrentRunningTeam = pBg;
		CurrentBgTeam = pBg;
		stpRunning = pBg->rootPid;
	      }
	    else		/* There are no local background teams to 
				   run. */
	      {
	        pGuest = FindNextTeam(CurrentGuestTeam, 
				NONRUNNING_GUEST_TEAM_PRIORITY);
		if (pGuest != NULL)
		  {
		    CurrentRunningTeam = pGuest;
		    CurrentGuestTeam = pGuest;
		    stpRunning = pGuest->rootPid;
		  }
	      }
	  }
	Unlock(LockId);
	/* We set team priorities outside the critical section because
	   the operation may blocked on a frozen logical host. */
	if (stpNonRunning)
	  {
	    SetTeamPriority(stpNonRunning, prioNonRunning);
	  }
	if (stpRunning)
	  {
	    SetTeamPriority(stpRunning, RUNNING_TEAM_PRIORITY);
	  }
      }
  }


/*
 * FindNextTeam:
 * Find the next team with the specified priority starting at  curTeam.
 */

TeamsEntry *FindNextTeam(curTeam, priority)
    TeamsEntry *curTeam;
    int priority;
  {
    TeamsEntry *p;

    if (curTeam == NULL)
      {
	curTeam = LastCList(TeamsInUseList);
	if (curTeam == NULL)
	  {
	    return(NULL);
	  }
      }
    for (p = curTeam->next; p != curTeam; p = p->next)
      {
	if (p->runPriority == priority)
	  {
	    break;
	  }
      }
    if (p->runPriority == priority)
      {
	return(p);
      }
    else
      {
	return(NULL);
      }
  }


/*
 * AdvanceCurrentTeamPtr:
 * Advance the specified ptr to the next team descr. of the specified
 * priority.
 */

TeamsEntry *AdvanceCurrentTeamPtr(curPtr, priority)
    TeamsEntry *curPtr;
    int priority;
  {
    TeamsEntry *p, *newCurPtr;

    p = FindNextTeam(curPtr, priority);
    if (p == curPtr)
      {
        newCurPtr = NULL;
      }
    else
      {
        newCurPtr = p;
      }
    return(newCurPtr);
  }


/*
 * ReadjustCurrentPtrs:
 * Readjust the Current running, back, and foreground ptrs to take into
 * account the absence of the team specified by teamEntryPtr.
 */

ReadjustCurrentPtrs(teamEntryPtr)
    TeamsEntry *teamEntryPtr;
  {
    if (CurrentRunningTeam == teamEntryPtr)
      {
	CurrentRunningTeam = NULL;
				/* No team now has the current time slice. */
      }
    if (CurrentFgTeam == teamEntryPtr)
      {
        CurrentFgTeam = AdvanceCurrentTeamPtr(CurrentFgTeam, 
					NONRUNNING_FG_TEAM_PRIORITY);
      }
    else if (CurrentBgTeam == teamEntryPtr)
      {
        CurrentBgTeam = AdvanceCurrentTeamPtr(CurrentBgTeam, 
					NONRUNNING_BG_TEAM_PRIORITY);
      }
    else if (CurrentGuestTeam == teamEntryPtr)
      {
        CurrentGuestTeam = AdvanceCurrentTeamPtr(CurrentGuestTeam, 
					NONRUNNING_GUEST_TEAM_PRIORITY);
      }
  }



/*
 * SetTeamPriorityReq:
 * Sets a team's priority wrt. scheduling.
 *
 * The pid specified may be any from the team whose priority is being set.
 *
 * Users may request one of the real-time priority classes or round-robin
 * scheduling as either foreground, background, or guest jobs.
 * Scheduling class is specified by the manifest constants:
 * FOREGROUND, BACKGROUND, GUEST, REALTIME1 - REAL_TIMEn, where REAL_TIME1 is 
 * the best priority class (i.e. scheduled first).  STOP_TEAM_PRIORITY
 * designates a priority that will cause the team to not run.
 *
 * %%%Note: could be modified to allow specifying the team by name. --TPM
 */

SystemCode SetTeamPriorityReq(reqMsg, reqPid)
    KernelRequest *reqMsg;
				/* We use the same format as the equivalent
				   kernel request. */
    ProcessId reqPid;
  {
    int priority;
    ProcessId pid;
    TeamsEntry *teamEntryPtr;
    SystemCode retCode;

    priority = reqMsg->unspecified[0];
    pid = reqMsg->pid;
    if (pid == 0) pid = reqPid;

    /* Find the Teams table entry for the specified team. */
    for (teamEntryPtr = FirstCList(TeamsInUseList);
	    teamEntryPtr != NULL;
	    teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
      {
	if (SameTeam(pid, teamEntryPtr->rootPid))
	    break;
      }
    if (teamEntryPtr == NULL)
      {
	return(NOT_FOUND);
      }

    /* Check whether the request is permissible. */
    if (!RequestAllowed(reqMsg, reqPid, teamEntryPtr))
      {
	return(NO_PERMISSION);
      }

    /* Make sure we have a reasonable priority value. */
    switch (priority)
      {
	case FOREGROUND:
	    priority = NONRUNNING_FG_TEAM_PRIORITY;
	    break;
	case BACKGROUND:
	    priority = NONRUNNING_BG_TEAM_PRIORITY;
	    break;
	case GUEST:
	    priority = NONRUNNING_GUEST_TEAM_PRIORITY;
	    break;

	case REAL_TIME1:
	case REAL_TIME2:
	case REAL_TIME3:
	case REAL_TIME4:
	    switch (priority)
	      {
		case REAL_TIME1:
		    priority = REAL_TIME_PRIORITY1;
		    break;
		case REAL_TIME2:
		    priority = REAL_TIME_PRIORITY2;
		    break;
		case REAL_TIME3:
		    priority = REAL_TIME_PRIORITY3;
		    break;
		case REAL_TIME4:
		    priority = REAL_TIME_PRIORITY4;
		    break;
	      }
	    break;

	case STOP_TEAM_PRIORITY:
	    priority = STOPPED_TEAM_PRIORITY;
	    break;

	default:
	    return(BAD_ARGS);
      }
    retCode = SetTeamPriority(teamEntryPtr->rootPid, priority);
    if (retCode != OK )
      {
	return(retCode);
      }
    Lock(LockId);
    teamEntryPtr->runPriority = priority;
    ReadjustCurrentPtrs(teamEntryPtr);
    Unlock(LockId);

    return(OK);
  }


/*
 * ExternalPriorityType:
 */

short ExternalPriorityType(runPriority)
    unsigned short runPriority;
  {
    short priority;

    switch(runPriority)
      {
	case NONRUNNING_FG_TEAM_PRIORITY:
	    priority = FOREGROUND;
	    break;
	case NONRUNNING_BG_TEAM_PRIORITY:
	    priority = BACKGROUND;
	    break;
	case NONRUNNING_GUEST_TEAM_PRIORITY:
	    priority = GUEST;
	    break;
	case STOPPED_TEAM_PRIORITY:
	    priority = STOP_TEAM_PRIORITY;
	    break;
	case REAL_TIME_PRIORITY1:
	    priority = REAL_TIME1;
	    break;
	case REAL_TIME_PRIORITY2:
	    priority = REAL_TIME2;
	    break;
	case REAL_TIME_PRIORITY3:
	    priority = REAL_TIME3;
	    break;
	case REAL_TIME_PRIORITY4:
	    priority = REAL_TIME4;
	    break;
	default:
	    priority = 0;
	    printf("Bad runPriority encountered: %x!\n", runPriority);
	    break;
      }
    return(priority);
  }



/*
 * GetTeamPriorityReq:
 * Returns a team's priority class.
 *
 * %%%Note: could be modified to allow specifying the team by name. --TPM
 */

SystemCode GetTeamPriorityReq(reqMsg, reqPid)
    KernelRequest *reqMsg;
    ProcessId reqPid;
  {
    ProcessId pid;
    TeamsEntry *teamEntryPtr;

    pid = reqMsg->pid;

    /* Find the Teams table entry for the specified team. */
    for (teamEntryPtr = FirstCList(TeamsInUseList);
	    teamEntryPtr != NULL;
	    teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
      {
	if (SameTeam(pid, teamEntryPtr->rootPid))
	    break;
      }
    if (teamEntryPtr == NULL)
      {
	return(NOT_FOUND);
      }

    reqMsg->unspecified[0] = 
		(unsigned) ExternalPriorityType(teamEntryPtr->runPriority);
    return(OK);
  }



/*
 * ValidatorProcess:
 * Periodically checks to see that all teams and their owners are still
 * alive.  Notifies the team server of those that aren't.
 */

ValidatorProcess()
  {
    Message msg;
    ExitTeamRequest *req = (ExitTeamRequest *) msg;
    TeamsEntry *t;
    register int i;


    while (1)
      {
	Delay(0, ValidationInterval);

	for (i = 0, t = Teams; i < MaxTeams; i++, t++)
	  {
	    if (!t->inUse)
	      {
		continue;
	      }
	    if (!ValidPid(t->rootPid))
	      {
		req->requestcode = DeadTeamNotice;
		req->rootPid = t->rootPid;
		req->requestType = SpecificTeam;
		req->status = -1;		/* ??? */
		Send(msg, TeamServerPid);
	      }
	    else if (!ValidPid(t->ownerPid))
	      {
		req->requestcode = TERMINATE_TEAM;
		req->rootPid = t->rootPid;
		req->requestType = SpecificTeam;
		req->status = -1;		/* ??? */
		Send(msg, TeamServerPid);
	      }
	  }
      }
  }


/*
 * CheckIfValidIO:
 * Queries the servers of the I/O instances specified in rtMsg to see if they
 * are valid.
 */

SystemCode CheckIfValidIO(rtMsg)
    RootMessage *rtMsg;
  {
    MsgStruct msg;
    QueryInstanceRequest *req = (QueryInstanceRequest *) &msg;

    req->requestcode = QUERY_INSTANCE;
    req->fileid = rtMsg->stdinfile;
    Send(req, rtMsg->stdinserver);
    if (req->requestcode != OK)
      {
	return(req->requestcode);
      }
    req->requestcode = QUERY_INSTANCE;
    req->fileid = rtMsg->stdoutfile;
    Send(req, rtMsg->stdoutserver);
    if (req->requestcode != OK)
      {
	return(req->requestcode);
      }
    req->requestcode = QUERY_INSTANCE;
    req->fileid = rtMsg->stderrfile;
    Send(req, rtMsg->stderrserver);
    return(req->requestcode);
  }


/*
 * Remote:
 * Returns true if pid is from another physical host.
 */

int Remote(pid)
    ProcessId pid;
  {
    TeamsEntry *teamEntryPtr;

    for (teamEntryPtr = FirstCList(TeamsInUseList);
	    teamEntryPtr != NULL;
	    teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
      {
	if (SameTeam(pid, teamEntryPtr->rootPid))
	  {
	    return(FALSE);
	  }
      }
    if (SameTeam(pid, TeamServerPid))
      {
	return(FALSE);
      }
    return(TRUE);
  }


/*
 * RequestAllowed:
 * Returns TRUE if the requestor represented by reqPid is allowed to 
 * perform the operation specified in reqMsg.
 */

int RequestAllowed(reqMsg, reqPid, teamEntryPtr)
    KernelRequest *reqMsg;
    ProcessId reqPid;
    TeamsEntry *teamEntryPtr;
  {
    int priority;

    /* A user should be able to reach his executing programs from a remote 
       host. */
    if (User(reqPid) == hostUserNumber())
      {
	return(TRUE);
      }

    /* Allow all locally originating requests. */
    if (!Remote(reqPid))
      {
	return(TRUE);
      }

    /* At this point we know that the request is from a remote requestor who 
       is not the same user as the one logged in, if anyone is logged in. */
    switch (reqMsg->opcode)
      {
	case TeamLoadRequest:
	    if (hostAvailability() == -1)
	      {
		return(FALSE);
	      }
	    else
	      {
		return(TRUE);
	      }

	case CHANGE_TEAM_PRIORITY:
	    /* Only let the user of a program change its priority. */
	    if (User(reqPid) != User(teamEntryPtr->rootPid))
	      {
		return(FALSE);
	      }

	    /* Guest programs start at GUEST priority.  Let them lower their
	       priority as they wish.  Ony let them increase their priority 
	       if noone is logged in. */
	    priority = reqMsg->unspecified[0];
	    switch (priority)
	      {
		case FOREGROUND:
		case BACKGROUND:
		case REAL_TIME1:
		case REAL_TIME2:
		case REAL_TIME3:
		case REAL_TIME4:
		    if (hostAvailability() == 1)
		      {
			return(TRUE);
		      }
		    else
		      {
			return(FALSE);
		      }

		case GUEST:
		case STOP_TEAM_PRIORITY:
		default:
		    return(TRUE);
	      }

	default:
	    printf(
	         "Error in RequestAllowed: Should never have gotten here!\n");
	    return(FALSE);
      }
  }

/*** Character-string naming support ***/

char NameBuffer[MAX_PATH_LENGTH];

#define PrefixMatch(prefix) \
( (strncmp(path, prefix, l = strlen(prefix)) == 0 && IsDelimiter(path[l])) ? \
    (path += l + (path[l] != '\0'), 1) : 0 )

SystemCode LookupName(req, pid, segbuf, segsize, suffix, ctx, team)
  NameRequest *req;
  ProcessId pid;
  char *segbuf;
  unsigned segsize;
  char **suffix;
  ContextPair *ctx;
  TeamsEntry **team;
  {
    /*
     * Given a naming-protocol request from process "pid", find the
     *  named object.  "segbuf" and "segsize" give the appended data segment
     *  received with the request message, if any.
     * Returns a ContextPair for the object, plus a pointer into the teams
     *  table if it is a team, else NULL.  If the object is not found,
     *  on return "suffix" points to the first character of the undefined
     *  part of the name, while "ctx" identifies the context to which the
     *  defined part maps.
     * Exception: if req->requestcode is QUERY_NAME, the lookup stops with
     *  the first single-manager context found.
     * This is all pretty heavy-duty code considering how little character
     *  string naming the team server really needs to do.
     */
    register char *path;
    register int len, longflag, l;
    register SystemCode status;
    TeamsEntry *dummy;

    /* Initialize */
    longflag = 0;
    status = OK;
    if (team == NULL) team = &dummy;

    /* First move pathname into NameBuffer */
    path = NameBuffer;
    len = req->namelength - req->nameindex;
    if (len > MAX_PATH_LENGTH)
      {
	/* Indicated length is too great.  Name may still be legal
	 *  if null-terminated short of this length. */
        longflag++;
	len = MAX_PATH_LENGTH;
      }

    if (req->namelength <= segsize)
      {
	/* Entire segment is in segment buffer, including all of name,
	 *  so we can get it from there. */
	Copy(path, segbuf + req->nameindex, len);
      }
    else
      {
        /* Some of the name may not be there, so we do a MoveFrom to be
	 *  sure we have it. */
	status = MoveFrom(pid, path, req->nameptr+req->nameindex, len);
        if (status != OK) goto exit;
      }

    /* Be sure name is null-terminated */
    path[len] = '\0';

    /* Check for length overflow, which is deemed to have occurred
     *  if namelength indicated the name could be longer than
     *  MAX_PATH_LENGTH, and it was not null-terminated soon enough 
     *  to be shorter than MAX_PATH_LENGTH.  The case where it was
     *  null-terminated to be exactly MAX_PATH_LENGTH is unfortunately
     *  treated as an overflow.  Seems not worth the effort to fix this.
     */
    if (longflag && strlen(path) == MAX_PATH_LENGTH)
      {
	status = ILLEGAL_NAME;
	goto exit;
      }


    /*
     * Find identifier for starting context and map to a teams
     *  table entry if it is a team.
     */
    ctx->cid = req->namecontextid;
    status = MapCid(ctx, team);
    if (status != OK) goto exit;

    /* Now parse the character string */
    while (*path)
      {
	/* If QueryName, we are done as soon as we reach
	 *  a single-manager context */
	if (req->requestcode == QUERY_NAME &&
	    ctx->pid == TeamServerPid) goto exit;

	/* What we do next depends on the context */
	switch (ctx->cid)
	  {
	    case GLOBAL_ROOT_CONTEXT:
	      if (PrefixMatch("team"))
	        {
		  ctx->pid = VTEAM_SERVER_GROUP;
		  ctx->cid = TEAM_SERVER_CONTEXT;
		  break;
		}
	      else if (PrefixMatch("."))
	        {
		  continue;
		}
	      else
	        {
		  status = NOT_HERE;
		  goto exit;
		}

	    case TEAM_SERVER_CONTEXT:
	      if ( PrefixMatch(hostname()) ||
	           PrefixMatch("any") ||
		   PrefixMatch(MACHINE) )
		{
		  ctx->pid = TeamServerPid;
		  ctx->cid = DEFAULT_CONTEXT;
		  break;
		}
	      else if ( PrefixMatch("local") && !Remote(pid) )
		{
		  ctx->pid = LTEAM_SERVER_GROUP;
		  ctx->cid = DEFAULT_CONTEXT;
		  break;
		}
	      else if (PrefixMatch("."))
	        {
		  continue;
		}
	      else if ( PrefixMatch("..") )
	        {
		  /* Upward reference to larger context */
		  req->nameindex += path - NameBuffer;
		  req->namecontextid = GLOBAL_ROOT_CONTEXT;
		  Forward(req, VCSNH_SERVER_GROUP);
		  status = NO_REPLY;
		  goto exit;
		}
	      else
	        {
		  status = NOT_HERE;
		  goto exit;
		}

	    case DEFAULT_CONTEXT:
	      /* Team specified by string name.
	       *  We don't implement "." or ".." here because
	       *  team names can legitimately start with "./" or "../".
	       */
	      for ((*team) = FirstCList(TeamsInUseList);
		   (*team) != NULL;
		   (*team) = IterCList(TeamsInUseList, (*team)))
	        {
	          if (Equal((*team)->name, path))
		    {
		      path += strlen(path);
		      break;
		    }
		}
	      if ((*team) == NULL)
	          status = NOT_FOUND;
	      else
	          ctx->cid = (*team)->rootPid;
	      goto exit;
   
	    default:
	      /* The current context is a team, but
	       *  team names do not have any substructure,
	       *  and we don't implement ".." from them either.
	       */
#ifdef STRICT
	      status = NOT_FOUND;
#else
	      /* %%% For now, we ignore the remaining piece of name,
	       *  because some clients currently leave garbage there.
	       */
	      *path = '\0';
#endif STRICT
	      goto exit;
	  }
    }

      
exit:
    if (suffix != NULL) *suffix = path;
    if ( status != OK && status != NOT_FOUND && IsGroupId(Forwarder(pid)) )
        return DISCARD_REPLY;
    else
        return status;
  }


/*
 * Find object corresponding to a context id.  Fills in the pid
 *  field of the context pair and returns a pointer to a teams table
 *  entry if object is a team.
 */
SystemCode MapCid(ctx, team)
  ContextPair *ctx;
  TeamsEntry **team;
  {
    *team = NULL;
    switch (ctx->cid)
      {
	case GLOBAL_ROOT_CONTEXT:
	  ctx->pid = VCSNH_SERVER_GROUP;
	  break;
	case TEAM_SERVER_CONTEXT:
	  ctx->pid = VTEAM_SERVER_GROUP;
	  break;
	case DEFAULT_CONTEXT:
	  ctx->pid = TeamServerPid;
	  break;
	default:
	  /* Context-id must be the root pid of one of our teams;
	   *  otherwise it's invalid.
	   */
	  for (*team = FirstCList(TeamsInUseList);
	       *team != NULL;
	       *team = IterCList(TeamsInUseList, (*team)))
	    {
	      if ((*team)->rootPid == ctx->cid) break;
	    }
	  if (*team == NULL)
	      return INVALID_CONTEXT;

	  ctx->pid = TeamServerPid;
	  break;
      }
    return OK;
  }
