/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Migration support routines for the V team server.
 */


#include "team.h"
#include <Vgroupids.h>
#include <Vquerykernel.h>
#include <Vteams.h>
#include <Vdirectory.h>

#define DEFAULT_PRIORITY 4	/* For dummy team root processes. */
#define DEFAULT_ENTRY 0		/* Dummy entry point for dummy team root 
				   processes. */

#define DummyTeamName "Dummy Team"

SystemCode AcceptMigrationRequest();
SystemCode RegisterDummyTeam();
SystemCode QueryRemoteTeamServer();

extern char *GetTeamSize(), *SetTeamSize();
extern char *malloc();


#define SameHost(a, b) (((a) & 0xffff0000) == ((b) & 0xffff0000))


SystemCode MigrationRequest(reqMsg, reqPid)
    MsgStruct *reqMsg;
    ProcessId reqPid;
  {
    int redoFlag;
    char *descBuffer;
    ProcessId *processes, *dummyProcesses;
    unsigned *memSizes;
    TeamsEntry *teamEntryPtr;
    SystemCode status;
    int nT, nP;
    unsigned totalMemSize;
    register int i;
    ProcessId pid, lhost;
    char *loadptr, *size;
    MsgStruct msg;
    MsgStruct *fReq = &msg;

    if (reqMsg->sysCode == MIGRATION_REQUEST)
      {
	redoFlag = 0;
      }
    else
      {
	redoFlag = 1;
	lhost = (ProcessId) reqMsg->unspecified[3];
      }

    /* Get the information on resource requirements. */
    nT = (int) reqMsg->unspecified[0];
    nP = (int) reqMsg->unspecified[1];
    totalMemSize = reqMsg->unspecified[2];
    /* Are we willing to accept this request? */
    if (!redoFlag)
      {
	status = AcceptMigrationRequest(nT, nP, totalMemSize, reqPid);
	if (status != OK)
	  {
	    return(status);
	  }
      }

    /* Get the information on individual team pids and memory requirements. */
    descBuffer = malloc(reqMsg->segmentSize);
    if (descBuffer == NULL)
      {
	return(NO_SERVER_RESOURCES);
      }
    status = MoveFrom(reqPid, descBuffer,
    				reqMsg->segmentPtr, reqMsg->segmentSize);
    if (status != OK)
      {
	free(descBuffer);
	return(status);
      }
    processes = (ProcessId *) descBuffer;
    memSizes = (unsigned *) (descBuffer + nT * sizeof(ProcessId));
    dummyProcesses = (ProcessId *) (descBuffer + nT * sizeof(ProcessId) +
					         nT * sizeof(unsigned));

    /* Create a new dummy team for each team which is to migrate and set its
       team size to that specified by memSizes.  Forward the team back to the
       requestor along with an indication of which teams correspond to each
       other.
       We create the dummy teams in REVERSE order of their placement in the 
       descriptor.  This will result in an order of creation of teams that 
       corresponds to that on the original host.  The TransferHost operation 
       relies on this ordering.
       If this is a redo migration request then check first if the dummy
       team already exists.  Do nothing if so. */
    for (i = nT - 1; i >= 0; i--)
      {
        if (redoFlag)
	  {
	    for (teamEntryPtr = FirstCList(TeamsInUseList);
		 teamEntryPtr != NULL;
		 teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
	      {
		if (teamEntryPtr->migrationPid == processes[i])
		    break;
	      }
	    if (teamEntryPtr != NULL)
	      {
		continue;
	      }
	  }
        if ((i == (nT - 1)) && !redoFlag)
	  {
	    pid = CreateHost(DEFAULT_PRIORITY, DEFAULT_ENTRY, 0);
	    lhost = pid;
	  }
	else
	  {
	    pid = CreateTeam(DEFAULT_PRIORITY, DEFAULT_ENTRY, 0, lhost);
	  }
	if (pid == NULL)
	  {
	    free(descBuffer);
	    return(NO_TDS);
	  }
	loadptr = GetTeamSize(pid);
	size = SetTeamSize(pid, loadptr + memSizes[i]);
	if (size != (loadptr + memSizes[i]))
	  {
	    DestroyProcess(pid);
	    free(descBuffer);
	    return(NO_MEMORY);
	  }
	SetUserNumber( pid, User(reqPid) );    
	status = RegisterDummyTeam(pid, reqPid, processes[i]);
	if (status != OK)
	  {
	    DestroyProcess(pid);
	    free(descBuffer);
	    return(status);
	  }
	dummyProcesses[i] = pid;
	fReq->sysCode = READ_AND_FORWARD;
	fReq->segmentPtr = loadptr;
	fReq->segmentSize = memSizes[i];
	Forward(fReq, pid, reqPid);
      }
    status = MoveTo(reqPid, reqMsg->segmentPtr, descBuffer,
    				reqMsg->segmentSize);
    free(descBuffer);
    reqMsg->unspecified[0] = lhost;
    return(status);
  }


SystemCode TransferHostRequest(reqMsg, reqPid)
    MsgStruct *reqMsg;
    ProcessId reqPid;
  {
    ProcessId lhost, newLHost;
    ProcessId teamServer;
    char *desc;
    SystemCode status;
    TeamsEntry *teamEntryPtr;
    ProcessId groupId;

    /* Get the logical host id of the host to transfer into. */
    lhost = reqMsg->unspecified[0];
    teamServer = reqMsg->unspecified[1];
    /* Make sure the host exists. */
    for (teamEntryPtr = FirstCList(TeamsInUseList);
         teamEntryPtr != NULL;
	 teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
      {
        if (SameHost(teamEntryPtr->rootPid, lhost))
	    break;
      }
    if (teamEntryPtr == NULL)
      {
	return(BAD_ARGS);
      }
    newLHost = teamEntryPtr->migrationPid;
				/* Remember the migrating logical host's id.*/

    /* Get hold of the descriptor. */
    desc = malloc(reqMsg->segmentSize);
    if (desc == NULL)
      {
	return(NO_SERVER_RESOURCES);
      }
    status = MoveFrom(reqPid, desc, 
    				reqMsg->segmentPtr, reqMsg->segmentSize);
    if (status != OK)
      {
        free(desc);
	return(status);
      }

    /* TransferHost the kernel descriptor information into the dummy logical 
       host that has been created previously. */
    status = TransferHost(lhost, desc, reqMsg->segmentSize);
    free(desc);
    if (status != OK)
      {
	return(status);
      }

    /* Query the requesting machine's team server for its information about
       each team on the migrating logical host. */
    for (;	/* We already have a ptr. to the first team. */
         teamEntryPtr != NULL;
	 teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
      {
        if (!SameHost(teamEntryPtr->rootPid, lhost))
	  {
	    continue;
	  }
	status = QueryRemoteTeamServer(teamEntryPtr, teamServer);
	if (status != OK)
	  {
	    return(status);
	  }
      }

    /* Join the local team server group for this logical host. */
    groupId = (newLHost & LOGICAL_HOST_PART) | LTEAM_SERVER_GROUP;
    status = JoinGroup(groupId, TeamServerPid);
    if (status != OK)
      {
	printf("ERROR: couldn't join local team server group: %s.\n",
				ErrorString(status));
      }

    /* Unfreeze the newly migrated logical host to set it running. */
    status = UnfreezeHost(newLHost);

    return(status);
  }


/*
 * AcceptMigrationRequest:
 * Determine whether we are willing to accept this migration request.
 * Depends on whether we have the necessary resources and whether the
 * policy database allows us to accept the request from this (or any) user.
 */

SystemCode AcceptMigrationRequest(nT, nP, memSize, reqPid)
    int nT;
    int nP;
    unsigned memSize;
    ProcessId reqPid;
  {
    Message msg;
    MemoryStatisticsReply *msRep = (MemoryStatisticsReply *)msg;
    KernelStatisticsReply *ksRep = (KernelStatisticsReply *)msg;

    /* Check whether we have enough resources to handle this host. */
    QueryKernel(0, KERNEL_STATS, ksRep);
    if (ksRep->freeTds < nT)
      {
	return(NO_TDS);
      }
    if (ksRep->freePds < nP)
      {
	return(NO_PDS);
      }
    QueryKernel(0, MEMORY_STATS, msRep);
    if ((msRep->unusedFastMemory + msRep->unusedSlowMemory) < memSize)
      {
	return(NO_MEMORY);
      }
    return(OK);
  }


SystemCode RegisterDummyTeam(pid, reqPid, migrationPid)
    ProcessId pid;
    ProcessId reqPid;
    ProcessId migrationPid;
  {
    TeamsEntry *teamEntryPtr;
    char *name;

    /* Find an unused entry in the Teams table. */
    teamEntryPtr = PopStack(TeamsFreeList);
    if (teamEntryPtr == NULL)
      {
	return(NO_SERVER_RESOURCES);
      }

    name = malloc(strlen(DummyTeamName) + 1);
    if (name == NULL)
      {
        PushStack(TeamsFreeList, teamEntryPtr);
	return(NO_MEMORY);
      }
    strcpy(name, DummyTeamName);

    /* Initialize Teams table entry. */
    Lock(LockId);
    AddCList(TeamsInUseList, TeamsInUseList, teamEntryPtr);
    teamEntryPtr->ownerPid = reqPid;
    teamEntryPtr->runPriority = NONRUNNING_GUEST_TEAM_PRIORITY;
    teamEntryPtr->inUse = 1;
    Unlock(LockId);
    SetTeamPriority(pid, teamEntryPtr->runPriority);
    teamEntryPtr->rootPid = pid;
    teamEntryPtr->name = name;
    /* We don't care about the loadServer, loadFile, and rtMsg fields for a
       postmortem debugger because we NEVER invoke a postmortem debugger on
       a dummy team. */
    teamEntryPtr->loadServer = 0;
    teamEntryPtr->loadFile = 0;
    teamEntryPtr->migrationPid = migrationPid;

    return(OK);
  }


SystemCode QueryRemoteTeamServer(teamEntryPtr, teamServer)
    TeamsEntry *teamEntryPtr;
    ProcessId teamServer;
  {
    Message msg;
    DescriptorRequest *reqMsg = (DescriptorRequest *) msg;
    DescriptorReply *repMsg = (DescriptorReply *) msg;
    TeamDescriptor teamDesc;
    char *name;

    /* Query the designated team server for its info. on the migrated team. */
    teamDesc.fileName[0] = '\0';/* Team server is queried by team root, 
				   not by name. */
    reqMsg->requestcode = NREAD_DESCRIPTOR;
    reqMsg->nameindex = ((int) &(teamDesc.fileName[0])) - (int)(&teamDesc);
    reqMsg->dataindex = 0;
    reqMsg->fileid = 0;
    reqMsg->namecontextid = (ContextId) teamEntryPtr->migrationPid;
    reqMsg->segmentptr = (char *) (&teamDesc);
    reqMsg->segmentlen = sizeof(teamDesc);
    Send(msg, teamServer);
    if (repMsg->replycode != OK)
      {
	return(repMsg->replycode);
      }

    /* Change the record for the dummy team to be the record for the migrated 
       team. */
    teamEntryPtr->ownerPid = teamDesc.ownerPid;
    teamEntryPtr->rootPid = teamDesc.rootPid;
    free(teamEntryPtr->name);
    name = malloc(strlen(teamDesc.fileName) + 1);
    if (name == NULL)
      {
        TerminateSingleTeam(teamEntryPtr, TERMINATE_TEAM);
	return(NO_MEMORY);
      }
    strcpy(name, teamDesc.fileName);
    teamEntryPtr->name = name;
    teamEntryPtr->loadServer = teamDesc.fileserver;
    teamEntryPtr->loadFile = teamDesc.fileid;
    teamEntryPtr->debugRtMsg = teamDesc.rtMsg;
    /* We DON'T set the runPriority since the program is now a guest 
       program. */

    return(OK);
  }


/*
 * DestroyHostRequest:
 * Destroy a logical host that has either migrated or is a dummy host to be
 * migrated to.  In the latter case we don't want to use DestroyHost because
 * that tries to forward messages, etc.  Instead, we unfreeze the host 
 * (in case it was already frozen) and apply the normal team destruction 
 * routines.
 */

SystemCode DestroyHostRequest(reqMsg, reqPid)
    MsgStruct *reqMsg;
    ProcessId reqPid;
  {
    unsigned opType;
    int requestType;
    ProcessId lhost;
    TeamsEntry *teamEntryPtr;
    SystemCode status;

    opType = reqMsg->unspecified[0];
    lhost = reqMsg->unspecified[1];
    if (opType == MIGRATED_HOST)
      {
	requestType = DeleteTeamEntryOnly;
	
      }
    else			/* KILL_HOST */
      {
	requestType = TERMINATE_TEAM;
	UnfreezeHost(lhost);
      }
    teamEntryPtr = FirstCList(TeamsInUseList);
    while (teamEntryPtr != NULL)
      {
        if (SameHost(lhost, teamEntryPtr->rootPid))
	  {
	    TerminateSingleTeam(teamEntryPtr, requestType);
	    teamEntryPtr = FirstCList(TeamsInUseList);
				/* Reset because we don't know what else has
				   been deleted from the list. */
	  }
	else
	  {
	    teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr);
	  }
      }
    if (opType == MIGRATED_HOST)
      {
	status = DestroyHost(lhost);
				/* This goes last so that the SameTeam 
				   operations in TerminateSingleTeam stay 
				   local. */
      }
    return(status);
  }
