/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/


/*
 * Team loading routines.
 */


#include "team.h"
#include <Vgroupids.h>
#include <b.out.h>

/* important hack */
#ifndef VAX
#undef N_BADMAG
#undef N_TXTOFF

#define	N_BADMAG(x) \
    (((x).fmagic)!=FMAGIC)

#define	N_TXTOFF(x) \
	(sizeof (struct bhdr))
#endif

extern File *Open();
extern char *GetTeamSize(), *SetTeamSize();

#define TRUE 1
#define FALSE 0
#define NUL '\0'
#define CHUNK 256
#define BUFFERSIZE 1024
#define DEFAULT_PRIORITY 4

/*
 * LoadTeamFromFileInstance:
 * Team file loading function which loads from an already opened file.
 */
ProcessId LoadTeamFromFileInstance(lhost, fileServer, fileid, errcode, teamend)
    ProcessId lhost;
    ProcessId fileServer;
    InstanceId fileid;
    SystemCode *errcode;
    char **teamend;
  /*
   * Create a new team and load the program contained in the specified 
   *   file instance into it.
   * If lhost is 0 then create the team in a new logical host.  Otherwise
   *   put it on the logical host specified by lhost.
   *
   * The file must be in b.out format. The b.out header is checked for a
   *   valid "magic number."
   * The file (text+data segments) is loaded into the new team.  The bss
   *   segment is NOT zeroed here, nor is memory allocated for it.
   * If errors occur during loading, NULL is returned, and a system reply
   *   code describing the error is placed in errcode if it is not null.
   *
   * Team layout:
   *  |kernel|text|data|uninitialized data(bss)|root process stack|heap|
   */

  {
    File *teamfile;
    int  rsize, bytes, blocksize;
    char *readend, *loadptr;
    static short *databuffer = NULL;
    register char *buffer;
    ProcessId teampid;
    struct bhdr header;
    SystemCode r;
    unsigned int blocknumber, blockoffset;
    SystemCode status;
    ProcessId groupId;

    if (errcode == NULL) errcode = &r;

    /* Allocate a buffer if not already done */
    if ( databuffer == NULL &&
	(databuffer = (short *) malloc(BUFFERSIZE)) == NULL )
      {
	*errcode = NO_MEMORY;
	return (NULL);
      }
    buffer = (char *) databuffer;

    /* Open team code file */
    teamfile = OpenFile(fileServer, fileid, FREAD | FBLOCK_MODE, errcode);
    if( *errcode || (teamfile == NULL) )
      {
	return (NULL);
      }
    blocksize = teamfile->blocksize;

    /* Read first block of file to get b.out header */
    rsize = Read(teamfile, buffer, blocksize);
    if( rsize < sizeof(struct bhdr) )
      { 
	Close(teamfile);
	*errcode = teamfile->lastexception;
	return (NULL);
      }

    /* Copy b.out header to header buffer */
    header = *(struct bhdr *) buffer;
    
    /* Check the header has the legal magic number */
    if( N_BADMAG(header))
      {
	Close(teamfile);
	*errcode = BAD_STATE;
	return (NULL);
      }

    /* Create a team */
    if (lhost == 0)
      {
	teampid = CreateHost(DEFAULT_PRIORITY, header.entry, 0);
	groupId = (teampid & LOGICAL_HOST_PART) | LTEAM_SERVER_GROUP;
	status = JoinGroup(groupId, TeamServerPid);
	if (status != OK)
	  {
	    printf("ERROR: couldn't join local team server group: %s.\n",
				ErrorString(status));
	  }
      }
    else
      {
        teampid = CreateTeam(DEFAULT_PRIORITY, header.entry, 0, lhost);
      }
    if (teampid == NULL )
      {
	Close(teamfile);
	*errcode = NO_TDS;	/* could also be illegal priority, but ... */
	return (NULL);
      }

    /* Calculate the required sizes and pointers */
    loadptr = GetTeamSize( teampid ); /* Point to start loading */
    /* End of data segment in team (just beyond last byte to read) */
    readend = loadptr + header.tsize + header.dsize;
    /* Size of text+data+bss */
    *teamend = readend + header.bsize;

    /* Set size of destination team */
    if( SetTeamSize(teampid, readend) != readend )
      {
	Close(teamfile);
	DestroyProcess(teampid);
	*errcode = NO_MEMORY;
	return (NULL);
      }

    /* get first block that has the program in it, dealing with N_TXTOFF */
    blocknumber = N_TXTOFF(header) / blocksize;
    blockoffset = N_TXTOFF(header) % blocksize;
    if ( blocknumber != 0 )
      {
        /* read the proper block */
	teamfile->block = blocknumber;
	rsize = Read(teamfile, buffer, blocksize);
      }
    buffer += blockoffset;
    rsize -= blockoffset;

    /* Move remainder of first data block to team code segment */

    /* In case amount to move is less than that in one block */
    if( loadptr+rsize > readend ) rsize = (int) (readend - loadptr);

    if( (*errcode = MoveTo(teampid,loadptr,buffer,rsize)) != OK )
      {
	Close(teamfile);
	DestroyProcess(teampid);
	return(NULL);
      }
    buffer = (char *) databuffer;
    loadptr += rsize;
    teamfile->block++; /* Set to next block number */

    /* Load rest of file */
#ifdef SLOWLOAD
    while( loadptr < readend )
      {
	bytes = (int) (readend - loadptr);
	bytes = (bytes < blocksize) ? bytes : blocksize;

	if( (rsize = Read( teamfile, buffer, bytes)) < bytes )
	  {
#ifdef DEBUG
printf("Read failed on (%d)\n", teamfile->block);
#endif DEBUG
	    Close(teamfile);
	    *errcode = teamfile->lastexception;
	    DestroyProcess(teampid);
	    return (NULL);
	  }
	if( (*errcode = MoveTo(teampid,loadptr,buffer,rsize)) != OK )
	  {
#ifdef DEBUG
printf("MoveTo failed on (%d)\n", teamfile->block);
#endif DEBUG
	    Close(teamfile);
	    DestroyProcess(teampid);
	    return(NULL);
	  }
	loadptr += rsize;
	teamfile->block++; /* move to next file block */
      }
#else
    /* Fast team load using the forwarding trick */
     {
	Message msg;
	IoRequest *req = (IoRequest *) msg;

	req->requestcode = READ_AND_FORWARD;
	req->fileid = teamfile->fileid;
	req->blocknumber = teamfile->block;
	req->bufferptr = loadptr;
	req->bytecount = (unsigned) (readend - loadptr);
	Forward( req, teampid, teamfile->fileserver );
	ReceiveSpecific( req, teampid );
	/* A kludge:  If either our forward or the storage server's
	 *  return forward fails, the team will start running (!)
	 *  and probably get an exception very early in the game.  We
	 *  only check for exceptions here, but it is possible the
	 *  process might send some other message (perhaps to some other
	 *  process!) first.  A complete fix for this problem would be
	 *  somewhat involved.  TPM
	 */
	if (req->requestcode == EXCEPTION_REQUEST)
	  {
	    *errcode = BAD_FORWARD;
	    PrintStdExceptInfo(stdout->fileserver, stdout->fileid, req,
	        teampid, "ERROR - Forward failure on load!");
	  }
	else
	  {
	    *errcode = (SystemCode) req->blocknumber; 
				/* Special place for replycode */
	  }

	if( *errcode != OK )
	  {
	    Close(teamfile);
	    DestroyProcess(teampid);
	    return( NULL );
	  }
	loadptr = readend;
      }
#endif  SLOWLOAD
    Close(teamfile);

    return(teampid);
  }


#define BUFLEN 128

/*
 * LoadTeamFromNamedFile:
 * Load a team from a file specified by name
 */
ProcessId LoadTeamFromNamedFile(lhost, filename, errcode, teamend)
    ProcessId lhost;		/* Which logical host to place on. */
    char *filename;
    SystemCode *errcode;
    char **teamend;
  {
    extern unsigned char OurProcFamily;
    extern File *OpenProgFile();

    File *teamfile;
    ProcessId teampid;
    SystemCode r;
    static char nameBuf[BUFLEN];

    if (errcode == NULL) errcode = &r;

    /* Try to open the named program file (and check for an appropriate 
     * machine-specific suffix if necessary).
     */
    strncpy(nameBuf, filename, BUFLEN - 1); /* allow space for any suffix */
    teamfile = OpenProgFile(nameBuf, OurProcFamily, errcode);

    if (teamfile == NULL) return(0);

    teampid = LoadTeamFromFileInstance(lhost, teamfile->fileserver,
				       teamfile->fileid, errcode, teamend);

    Close(teamfile);

    return (teampid);
  }

