/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Host status package.
 */

#include <Vioprotocol.h>
#include "Vdirectory.h"
#include "Vteams.h"
#include "team.h"


#define ByteSwapHostStatus(hs) do { \
    ByteSwapLongInPlace(&((hs)->availability), 2*sizeof(long)); \
  } while (0)


HostStatusRecord HostStatus;


extern char *index();
extern File *Open();
extern InstanceId RegisterObject();



/*
 * InitHostStatus:
 * Initialize host status data structures of the team server.
 */

InitHostStatus()
  {
    extern SystemCode QueryWorkstationConfig();

    if (QueryWorkstationConfig("name", HostStatus.hostName, 
	    MaxObjectNameLength) != OK)
        sprintf(HostStatus.hostName, "%08x", TeamServerPid);

    HostStatus.userName[0] = '\0';
    HostStatus.availability = 1;
    HostStatus.userNumber = User(TeamServerPid);

    UpdateServerStatus(0);	/* Update the service server's status. */
  }


/*
 * UpdateHostStatusReq:
 * Update the host status kept by the team server or return the current
 * host status, depending on which requesttype is specified.
 */

SystemCode UpdateHostStatusReq(reqMsg, reqPid)
    UpdateHostStatusRequest *reqMsg;
    ProcessId reqPid;
  {
    SystemCode status;
    int len;

    if (reqMsg->requesttype == UpdateInformation)
      {
        if (reqMsg->bytecount > MaxObjectNameLength)
	  {
	    return(BAD_BYTE_COUNT);
	  }
        HostStatus.availability = reqMsg->availability;
        status = MoveFrom(reqPid, HostStatus.userName, 
		reqMsg->bufferptr, reqMsg->bytecount);
	HostStatus.userNumber = User(reqPid);
	UpdateServerStatus(0);
      }
    else if (reqMsg->requesttype == GetHostStatus )
      {
	HostStatusRecord h;		/* copy for byte-swapping */
	h = HostStatus;

	len = sizeof ( HostStatusRecord ) ;
	if (reqMsg->bytecount < len)
	  {
	    return(BAD_BYTE_COUNT);
	  }
	/* Since we're transferring the whole descriptor in this case
	 * (i.e. not just the string "userName"), we may have to byte-swap it.
	 */
	if (DifferentByteOrder(reqPid))
	    ByteSwapHostStatus(&h);
	status = MoveTo(reqPid, reqMsg->bufferptr,
		&h, len);
      }	
    else if (reqMsg->requesttype == ReturnInformation)
      {
        if (HostStatus.availability == -1)
	    reqMsg->availability = 0;
	else
	    reqMsg->availability = HostStatus.availability;
	len = strlen(HostStatus.userName) + 1;
	if (reqMsg->bytecount < len)
	  {
	    return(BAD_BYTE_COUNT);
	  }
	status = MoveTo(reqPid, reqMsg->bufferptr,
		HostStatus.userName, len);
      }
    else if (reqMsg->requesttype == GetLongHostStatus)
      {
	UpdateServerStatus(reqPid);
	status = OK;
      }
    else
      {
	status = REQUEST_NOT_SUPPORTED;
      }
    return(status);
  }


/*
 * CheckOnUpdateStatus:
 * Checks whether an update should be posted to the service server.
 */

CheckOnUpdateStatus(reqMsg, reqPid)
    ServiceServerRequest *reqMsg;
    ProcessId reqPid;
  {
    if (reqMsg->pleaseUpdate)
      {
	UpdateServerStatus(reqPid);
      }
    return(DISCARD_REPLY);
  }


/*
 * UpdaterProcess:
 * Updates the host status to the service server.
 * Runs at higher priority than the main team server process, so that 
 * there is no conflict of access to shared data structures.  Note that
 * all shared data structures are accessed before the blocking send is
 * performed.
 */

UpdaterProcess(serverPid)
    ProcessId serverPid;	/* 0 => service server. */
  {
    static HostDescriptor h;
    int nameIndex, typeIndex;
    SystemCode error;
    InstanceId serverId;
    Message msg;
    RegisterObjectRequest *reqMsg = (RegisterObjectRequest *) msg;

    nameIndex = h.hostName - ((char *) &h);
    typeIndex = h.serverType - ((char *) &h);
    h.descriptortype = HOST_DESCRIPTOR;
    strcpy(h.hostName, HostStatus.hostName);
    strcpy(h.serverType, HOST_TYPE);
    h.regPid = TeamServerPid;
    strcpy(h.userName, HostStatus.userName);
    if (HostStatus.availability == -1)
	h.availability = 0;
    else
	h.availability = HostStatus.availability;

    if (serverPid == 0)		/* Service server by default. */
      {
	serverId = RegisterObject(STRING_NAME, nameIndex, typeIndex,
		TeamServerPid, &h, sizeof(HostDescriptor), &error);
      }
    else
      {
	reqMsg->requestcode = REGISTER_OBJECT;
	reqMsg->nameType = STRING_NAME;
	reqMsg->nameIndex = nameIndex;
	reqMsg->typeIndex = typeIndex;
	reqMsg->ownerPid = TeamServerPid;
	reqMsg->bufferptr = (char *) &h;
	reqMsg->bytecount = sizeof(HostDescriptor);
	Send(msg, serverPid);
      }

    Suicide();
  }


/*
 * UpdateServerStatus:
 * Invoked when a change in host status should be sent to the service
 * server or the server designated by serverPid.
 * Creates a new process to do the actual send to the 
 * server so that the main team server process won't block on this action.
 */

UpdateServerStatus(serverPid)
    ProcessId serverPid;	/* 0 => service server. */
  {
    ProcessId updaterPid;

    /* Create the process which will run the actual update procedure. */
    updaterPid = Create(2, UpdaterProcess, 2000);
    if (updaterPid == 0)
      {
	printf("ERROR - couldn't create updater process.\n");
	return;
      }
    if (!Ready(updaterPid, 1, serverPid))
      {
	printf("ERROR - couldn't ready updater process.\n");
	return;
      }
  }

/* addons to let other procedures of the teamsserver access this info */

char *
hostname()
  {
    return (HostStatus.hostName) ;
  }
 
char *
hostUsername()
  {
    return (HostStatus.userName) ;
  }

hostAvailability()
  {
    return(HostStatus.availability);
  }

hostUserNumber()
  {
    return(HostStatus.userNumber);
  }
