/********************************************************/
/*							*/
/*	       Simple Terminal Server			*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* mux.c - Simple Terminal Server Master Multiplexor
 *
 * Kenneth Brooks, November 1983
 *
 * The Simple Terminal Server is a vastly downgraded descendant of the Virtual
 * Graphics Terminal Server.  Its purpose is to provide line editing and
 * other "cooked i/o" features in an environment where a plain terminal or
 * console device is being used for i/o.  Certain features have been left
 * in place in view of a future development which ought to take place, the
 * Virtual Terminal Server.  That server implements multiple pads, but talks
 * to a plain terminal; switching between pads consists of redrawing the screen
 * with new contents and directing keyboard input to a different client.
 */

# include <Venviron.h>
# include <Vioprotocol.h>
# include <Vtermagent.h>
# include <Vgts.h>
# include "client.h"
# include <Vquerykernel.h>
# include "interp.h"
# include <Vgroupids.h>
# include <Vteams.h>

#include <stsio.h>

short Debug;		/* extra debugging information */

struct Event *FreeEvents;

#undef MaxClients
#define MaxClients 1
struct Client ClientTable[MaxClients];

/*
 * Number of clock ticks before declaring screen to be idle
 */
# define TicksPerMinute 240
static IdleLimit = 10*TicksPerMinute;
static IdleTime = 0;

/*
 * Process stack sizes.
 */
# define HelperSize 500

ProcessId StsPid;

short IsCadlinc = FALSE;

/*******************************************************************/


/*
 * InitHelperProcesses:
 * Starts up keyboard helper process.
 */

InitHelperProcesses()
  {
    Message msg;
    ProcessId pid;
    extern KeyboardProcess();

    pid = Create( 5, KeyboardProcess, HelperSize );
    Ready(pid, 0);
  }


/*******************************************************************/

InitSts(xmin,ymin,xmax,ymax)
  {
/*
 * InitSts:
 * Initializes the terminal server multiplexor.
 * This process must eventually be the one who calls GetMessage()
 */
    int i;
    register struct Client *client = ClientTable; 
    char queryResult[4];
    SystemCode queryCode;
    extern SystemCode QueryWorkstationConfig();
    EditBuffer *CreateBuffer();

    unsigned char *periphEntry;
    Message configMsg;
    register PeripheralConfigurationReply *configReply =
				(PeripheralConfigurationReply *) configMsg; 

    StsPid = GetPid(0, LOCAL_PID);
    InitHelperProcesses();

    for (i = 0; i < MaxClients; i++, client++)
      {
	client->writerPid = 0;
	client->owner = 0;
	client->termPid = 0;
	client->master = -1;
	client->isexec = FALSE;
	client->readQ = NULL;
	client->eventQ = NULL;
	client->eventTail = NULL;
	client->lineEditBuf = CreateBuffer();
	client->bytesWritten = 0;
	client->mode = CR_Input+LF_Output+Echo+LineBuffer
		       +PageOutput+PageOutputEnable;
	client->outputFlag = 0;
      }

    InitEvents();

#ifdef  USE_STDIO
    ModifyPad(stdin, 0); /* allows this program to be tested under VGTS */
#endif  USE_STDIO

#ifndef RTS
    JoinGroup(LTERMINAL_SERVER_GROUP, StsPid);
    ChangeTeamPriority(StsPid, REAL_TIME4);

    QueryKernel(0, PERIPHERAL_CONFIG, configMsg);
    periphEntry = configReply->peripheral;
    for (i = 0 ; i < MAX_PERIPHERALS ; i++, periphEntry++)
	if (*periphEntry == PRF_CADLINC_KEYBOARD) {IsCadlinc = TRUE; break;}

    queryCode = QueryWorkstationConfig("terminal-type",queryResult,4);
    if ((queryCode == (SystemCode) OK) && (strcmp("h19", queryResult) == 0))
       IsCadlinc = TRUE;
#else   RTS
    IsCadlinc = TRUE;
#endif  RTS
  }


/*******************************************************************/


GetMessage(mouseFlag, keyboardFlag, msg)
  int mouseFlag;	/* true if we return on mouse input */
  int keyboardFlag;	/* true if we retun on keyboard input */
  register IoRequest *msg;	/* message that we return */
   {
	/*
	 * handle a request by dispatching to the correct handler
	 */
    ProcessId pid;
    int index;
    char c;
    /* we have only one "client", so this variable never changes */
    register struct Client *client = ClientTable;

    while (1)
      {
	pid = Receive(msg);
	index = msg->fileid;
	if (msg->requestcode != CREATE_INSTANCE)
	  {
	  	/*
		 * Do all the file id Validity checking here.
		 * after this point, we can index the Client table
		 * with impunity, since we know that the index is in range
		 */
	    if (index >= MaxClients) 
	      {
		msg->requestcode = INVALID_FILE_ID;
		Reply( msg, pid);
	  	return;
	      }
	  }
	switch (msg->requestcode)
	  {
	    case QUERY_INSTANCE:
		HandleQuery(msg,pid);
	        Reply(msg, pid);
	        break;

	    case CREATE_INSTANCE:
		HandleCreate(msg,pid);
	        Reply(msg, pid);
	        break;

	    case SET_BREAK_PROCESS:
		HandleSetBreak(msg);
		Reply(msg,pid);
		break;

	    case SET_INSTANCE_OWNER:
		HandleSetOwner(msg);
		Reply(msg,pid);
		break;

	    case SetBannerRequest:
		msg->requestcode = REQUEST_NOT_SUPPORTED;
		Reply(msg,pid);
		break;

	    case RELEASE_INSTANCE:
		HandleRelease(msg);
	        Reply(msg, pid);
	        break;

	    case WRITE_INSTANCE:
	        HandleWrite(msg, pid);
	        break;

	    case WRITESHORT_INSTANCE:
	        HandleShortWrite(msg, pid);
	        break;

	    case READ_INSTANCE:
		HandleRead(ClientTable+index,pid);
		break;

	    case QueryPadRequest:
		HandleQueryFile(msg,pid);
	        Reply(msg, pid);
	        break;

	    case ModifyPadRequest:
		HandleModify(msg);
	        Reply(msg, pid);
	        break;

	    case EventRequest:
		HandleEventRequest(ClientTable+index, msg, pid);
		break;

	    case LineEditRequest:	 
		HandleShoveInput(ClientTable+index, msg, pid);
		break;

	    case RedrawRequest:
		msg->requestcode = OK;
		Reply( msg, pid);
		break;

	    case GetRawIO:
		HandleGetRawIO(msg);
		Reply(msg, pid);
		break; 

	    case Keyboard:
		Reply(msg, pid);
		if (keyboardFlag) return(pid);
		c = (char) msg->shortbuffer[0];
		UserCook(client,c);
		break;

	    case Die: /* a request to commit suicide */
		msg->requestcode = OK;
		Reply(msg, pid);
		DestroyProcess(0);
		break;

	    default:
		if (Debug) printf("Bad request: 0x%x, pid 0x%x\n", 
			msg->requestcode, pid);
		msg->requestcode = ILLEGAL_REQUEST;
		Reply(msg, pid);
		break;
	  }
      }
  }



HandleCreate(msg,pid)
  register CreateInstanceReply *msg;
  ProcessId pid;
  {
  	/*
	 * Return apropriate stuff to create an instance.
	 * msg becomes the reply message.
	 * This creates a new pad, unless the Directory bit is set,
	 * in which case it just returns the Directory instance.
	 */
    register CreateInstanceRequest *req = (CreateInstanceRequest *)msg;
    int readMode = (req->filemode & FBASIC_MASK) == FREAD;
    register struct Client *client;
    int interp;
    EditBuffer *editBuffer, *CreateBuffer();
    
    if (req->filemode & FDIRECTORY)
      {
	msg->replycode = REQUEST_NOT_SUPPORTED;
	return;
      }

    if ( ( interp = CreateInterp(msg->fileid) ) == 0 ) 
      {
        msg->replycode = NO_SERVER_RESOURCES;
        return;
      }

    if ( !(editBuffer = CreateBuffer(msg->fileid)) )
      {
	free(interp);
	msg->replycode = NO_SERVER_RESOURCES;
	return;
      }

    msg->fileid = 0;

    client = ClientTable+msg->fileid;
    client->interp = interp;
    client->vgt = msg->fileid;
    client->writerPid = 0;
    client->termPid = 0;
    client->owner = pid;
    client->master = -1;
    client->isexec = FALSE;
    client->readQ = NULL;
    client->block = 0;
    client->bytesWritten = 0;
    client->mode = CR_Input+LF_Output+Echo+LineBuffer
    		   +PageOutput+PageOutputEnable;
    client->outputFlag = 0;
    client->lineEditBuf = editBuffer;
       
    msg->fileserver = StsPid;
    if (readMode)
      {
        msg->blocksize = IO_MSG_BUFFER;
	msg->filetype = READABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = MAXUNSIGNED;
	msg->filelastbytes = MAXUNSIGNED;
      }
    else 
      {
        msg->blocksize = BlockSize;
        msg->filetype = WRITEABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = 0;
	msg->filelastbytes = 0;
      }
    msg->filenextblock = 0;
  }


HandleRelease(msg)
  register CreateInstanceReply *msg;
  {
  	/*
	 * Return apropriate stuff to release an instance.
	 * msg becomes the reply message.
	 *
	 * If we are deleting the current input pad, we need
	 * to find another one.
	 */
    
    msg->replycode = OK;

    if (ClientTable[msg->fileid].lineEditBuf)
      {
        DestroyBuffer(ClientTable[msg->fileid].lineEditBuf);
	ClientTable[msg->fileid].lineEditBuf = NULL;
      }

    if (ClientTable[msg->fileid].interp)
      {
        free(ClientTable[msg->fileid].interp);
        ClientTable[msg->fileid].interp = 0;
      }

  }


HandleQuery(msg,pid)
  register CreateInstanceReply *msg;
  {
  	/*
	 * Return apropriate stuff to a query instance.
	 * msg becomes the reply message.
	 */
    QueryInstanceRequest *req = (QueryInstanceRequest *)msg;
    int readMode = (req->filemode & 0x3F)
    			== FREAD;
    register struct Client *client = ClientTable + msg->fileid;
    
    if (req->filemode & 0xFF00) readMode = 0;

    if (Debug) 
        printf("Query on id %d, mode=0x%x, readMode=%d\n", msg->fileid,
		 ((QueryInstanceRequest *)msg) ->filemode,
		 readMode);
    msg->replycode = OK;
    msg->fileserver = StsPid;
    
    if (client->owner==0)
	client->owner = pid;      
    if (client->vgt <= 0)
	client->vgt = msg->fileid;
    if (client->interp==0)
        client->interp = CreateInterp(msg->fileid);

    if (readMode)
      {
        msg->blocksize = IO_MSG_BUFFER;
	msg->filetype = READABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = MAXUNSIGNED;
	msg->filelastbytes = MAXUNSIGNED;
      }
    else 
      {
        msg->blocksize = BlockSize;
        msg->filetype = WRITEABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = client->block;
	msg->filelastbytes = 0;
      }
    msg->filenextblock = 0;
  }


HandleQueryFile(msg)
  register struct ModifyMsg *msg;
  {
  	/*
	 * Return apropriate stuff to a query file.
	 * Right now this is just the degree of cooking.
	 * msg becomes the reply message.
	 */
    int pad = msg->fileid;
    
    if (pad<MaxClients)
      {
        msg->mode = ClientTable[pad].mode;
	msg->requestcode = OK;
      }
  }


HandleModify(msg)
  register struct ModifyMsg *msg;
  {
   	/*
	 * Set stuff for a Modify file request.
	 * Right now this is just the degree of cooking.
	 * msg becomes the reply message.
	 */

    int pad = msg->fileid;
    register struct Client *client = ClientTable + msg->fileid;
    register struct InterpUser *interp = (struct InterpUser *)client->interp;
    EditBuffer *CreateBuffer();
    
    if (pad<MaxClients)

      {

	if (msg->mode & PageOutputEnable)
	
	  {
	    if ( !(msg->mode & PageOutput) ) msg->mode &= ~PageOutputEnable;
	  }

	else if (client->mode & PageOutputEnable) msg->mode |= PageOutputEnable;

	else msg->mode &= ~PageOutput;

        client->mode = msg->mode;

	if (client->mode & PageOutput)
	  {
	    short length, width;
	
	    PadGetSize(0, &length, &width);
	    interp->newlinesLeft = interp->pageLength = length - 1;
	  }
	else if (client->writerPid)
	    client->outputFlag |= PageOutputShutdown;

	if ( (client->mode & LineBuffer) && !(client->lineEditBuf) )
	  {
	    if ( !(client->lineEditBuf = CreateBuffer(pad)) )
	      {
		msg->mode &= ~LineBuffer;
		client->mode = msg->mode;
		msg->requestcode = NO_SERVER_RESOURCES;
		return;
	      }
          }
	else if ( !(client->mode & LineBuffer) && (client->lineEditBuf) )
	  {
	    DestroyBuffer(client->lineEditBuf);
	    client->lineEditBuf = NULL;
	  }

	msg->requestcode = OK;
      }
  }


HandleSetBreak(msg)
    SetBreakRequest *msg;
  {
  	/*
	 * The break process is destroyed when the Kill Program
	 * command is selected, or the user types the break character.
	 */
    if (msg->fileid >= MaxClients)
      {
	msg->requestcode = INVALID_FILE_ID;
	return;
      }
    ClientTable[msg->fileid].termPid = msg->breakprocess;
    msg->requestcode = OK;
  }

HandleSetOwner(msg)
    register IoRequest *msg;
  {
  	/*
	 * Change the "owner" of an instance
	 */
    if (msg->fileid >= MaxClients)
      {
	msg->requestcode = INVALID_FILE_ID;
	return;
      }
    ClientTable[msg->fileid].owner = msg->instanceowner;
    msg->requestcode = OK;
  }


HandleGetRawIO(msg)
  register RawIOReply *msg;
  {
#ifdef USE_STDIO
    msg->inserver = stdin->fileserver;
    msg->infile = stdin->fileid;
    msg->outserver = stdout->fileserver;
    msg->outfile = stdout->fileid;
    msg->replycode = OK;
#else  USE_STDIO
    msg->replycode = REQUEST_NOT_SUPPORTED;
#endif USE_STDIO
  }
