/********************************************************/
/*							*/
/*	       Simple Terminal Server			*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* event.c - Handles event queues
 *
 *
 * Bill Nowicki April 1983 
 *
 * Simple Terminal Server version by Kenneth Brooks, November 1983
 *
 * Craig Dunwoody December 1983
 * 	Added calls to LineEdit in UserCook and HandleGraphics; moved character 
 *	echoing from UserCook to EventToUser; added call to DisplayInput in
 *	EventToUser.
 *
 */

# include <Venviron.h>
# include <Vioprotocol.h>
# include "Vtermagent.h"
# include "Vgts.h"
# include "client.h"
# include "pad.h"
# include "interp.h"
# include "chars.h"

#include <stsio.h>

extern short Debug;			/* for extra debugging info */
extern short	Nrows, Ncols;
extern ProcessId StsPid;
extern struct ReadRec *GetReadRec();

struct Event *FreeEvents = NULL;
struct ReadRec *FreeReadRecs = NULL;

#define MaxRead 8
struct ReadRec ReadTable[MaxRead];

/* initialize event and read request descriptors */
InitEvents()
  {
    register struct Event *e = EventTable;
    register struct ReadRec *r = ReadTable;
    register int i;

    /* Put all of the event descriptors onto the free event list */
    for (i = 0; i < MaxEvents; i++, e++)
      {
        e->next = FreeEvents;
	FreeEvents = e;
      }

    for (i = 0; i < MaxRead; i++, r++)
      {
	r->next = FreeReadRecs;
	FreeReadRecs = r;
      }
  }


/*
 * Routines that handle character events
 */

UserPut(client,c)
  register struct Client *client;
  char c;
    {
	/*
	 * Check if the last event is a Keystroke event that has not
	 * been filled up.  If not, tack another one onto the end
	 * of the queue.  Stuff the character into the last event.
	 */
      IoReply msg;
      register struct Event *e = client->eventTail;

      if (e==NULL || e->code != ReportCharacter
      		  || e->bytecount >= IO_MSG_BUFFER)
        {
		/*
		 * try to get a free Event and tack it onto the end
		 */
	  e = FreeEvents;
	  if (e==NULL) return;
	  FreeEvents = e->next;

	  if (client->eventTail)  
	    {
	      client->eventTail->next = e;
	    }
	  if (client->eventQ == NULL) client->eventQ = e;
	  client->eventTail = e;
	  e->next = NULL;
	  e->code = ReportCharacter;
	  e->bytecount = 0;
	}
      
      e->shortbuffer[e->bytecount++] = c;
    }


UserPush(client,c)
  register struct Client *client;
  {
  	/*
	 * This is the last character we are giving back to the user,
	 * so without further ado, send it to the eagerly awaiting
	 * client process.  But throw it away if a pure mouse request
	 * is outstanding.
	 */

    UserPut(client,c);
    EventToUser(client);
  }


UserCook(client,c)
    register struct Client *client;
  {
  	/*
	 * Possibly cook an input character.
	 * Drop the character if a pure mouse request is outstanding
	 * Send character to line editor if pad's LineBuffer bit is set.
	 */

    if ( client->writerPid && WriteUnblock(client, c) ) return;
 
    if (client->readQ && client->readQ->requestFlag == Mousereq) return;

    if (client->mode & LineBuffer)
      return( LineEdit(client,c) );

    if (c=='\r' && (client->mode & CR_Input) )
      c = '\n';

    UserPush(client,c);
  }



/*
 * The user is trying to read.  See if any characters are
 * on the event queue.  If not, set flags to wait for some.
 */
HandleRead(client, pid)
  register struct Client *client;
  {
    register struct ReadRec *read;

    read = GetReadRec();
    if (!read) return;  /* out of descriptors, requestor hangs forever */

    read->requestFlag = Keyreq;
    read->pid = pid;
    read->next = NULL;
    EnqueueRead(client, read);
    EventToUser(client);
  }



HandleWrite(msg, pid)
  IoRequest *msg;
  ProcessId pid;
  {
  	/*
	 * Do a MoveFrom to get clients' written data
	 * if in a different team.
	 */
    unsigned user, total;
    static char buf[BlockSize];
    char *from;
    register count, actual;
    register struct Client *client;

    user = msg->fileid;

    client = ClientTable + user;

    if ( PadBusy(client, pid) )

      {
	msg->requestcode = BUSY;
	Reply(msg, pid);
	return;
      }

    from = msg->bufferptr;
    total = msg->bytecount;

    while (total>0)
      {
        count = total;
	if (count>BlockSize) count=BlockSize;
	total -= count;
	MoveFrom( pid, buf, from, count);
	from += count;
 	if (client->mode & DiscardOutput)
	    actual = count;
        else
	    actual = InterpPutString(client->interp,buf, count, client->mode);
	client->lastBytesWritten = actual;
	client->bytesWritten += actual;
	if (actual < count) break;
      }

    if (actual == count)
      {
	client->writerPid = 0;
        client->block += client->bytesWritten;
	client->bytesWritten = 0;
        msg->requestcode = OK;
        Reply(msg, pid);
      }
    else
      {
	client->writerPid = pid;
	/* Make a copy of the request message (unless we've already done so). */
	if (msg != (IoRequest *)&(client->reqMessage))
	    Copy(&(client->reqMessage), msg, sizeof(MsgStruct));
      }
  }



HandleShortWrite(msg, pid)
  IoRequest *msg;
  ProcessId pid;
  {
    register actual;
    register struct Client *client;
    unsigned int user;

    user = msg->fileid;
    client = ClientTable+user;

    if (DifferentByteOrder(pid))
	ByteSwapLongInPlace(msg->shortbuffer, (msg->bytecount+3)&~3);

    if (PadBusy(client, pid))
      {
	msg->requestcode = BUSY;
	Reply(msg, pid);
	return;
      }

    if (client->mode & DiscardOutput) actual = msg->bytecount;
    else actual = InterpPutString(client->interp,
    			          msg->shortbuffer + client->bytesWritten, 
    			          msg->bytecount, client->mode);
    client->lastBytesWritten = actual;
    client->bytesWritten += actual;

    if (actual == msg->bytecount)
      {
	client->writerPid = 0;
        client->block += client->bytesWritten;
	client->bytesWritten = 0;
        msg->requestcode = OK;
        Reply(msg, pid);
      }
    else client->writerPid = pid;
  }



int PadBusy(client, pid)

  register struct Client	*client;
  register ProcessId		pid;
  
  {

	/*
	 * If there is an existing writer (pad is blocked on output), and the
	 * pid of the new writer differs from that of the existing writer,
	 * return 1.  Otherwise, return 0.
	 *
	 * If the existing writer died unexpectedly, clean up after him and
	 * return 0.
	 *
	 */

    if (client->writerPid)

      {

        if ( AwaitingReply(StsPid, client->writerPid) )
	
	  {
	    if (pid == client->writerPid) return(0);
	    else return(1);
	  }

        else

          {
    	    register struct InterpUser *interp;
	    short length, width;

    	    interp = (struct InterpUser *) client->interp;
	    PadGetSize(0, &length, &width);

	    client->writerPid = 0;
	    client->bytesWritten = 0;
	    interp->newlinesLeft = length - 1;
	    interp->pageLength = length - 1;
	    PutCharacter('\r');
	    PadClearToEOL(0);
	    FlushOutput();
          }

      }

    return(0);

  }



int WriteUnblock(client, c)

  register struct Client	*client;
  register char			c;
  
  {

	/*
	 * Client is blocked on a pad-filling write and we just received a
	 * character. If the character is meaningful, return 1, else 0.
	 * 
	 * The following list describes the available commands.  Most commands
	 * are optionally preceded by an integer argument k.  Defaults are
	 * in brackets.  Star (*) indicates argument becomes new default.
	 *
	 * <space>	Display next k lines [current page size]
	 * 
	 * z		Display next k lines [current page size]*
	 * Z
	 * 
	 * <CR>		Display next k lines [1]
	 * <LF>
	 * 
	 * 'q'		Throw away all output until next input read
	 * 'Q'
	 *
	 * 's'		Skip forward k lines [1]
	 *
	 * 'S'		Skip forward to last line
	 *
	 * 'f'		Skip forward k pages [1]
	 *
	 * 'F'		Skip forward to last page
	 *
	 * <backspace>	Erase last character of argument
	 * DEL
	 *
	 * '.'		Repeat previous command
	 *
	 */

    static int	Arg = 0;
    static int	PrevArg = 0;
    static char	PrevCommand = ' ';

    IoRequest *msg;
    register struct InterpUser *interp = (struct InterpUser *) client->interp;

    msg = (IoRequest *)&(client->reqMessage);

    if ( (c >= '0') && (c <= '9') )

      {

	Arg = 10 * Arg + c - '0';	
	return(1);
      
      }

    else if (c == '.')
    
      {
	Arg = PrevArg;
        c = PrevCommand;
      }

    else
    
      {
	PrevArg = Arg;
        PrevCommand = c;
      }

    switch(c)
    
      {
      
	case ' ':
	case 'z':
	case 'Z':
	    interp->newlinesLeft = Arg ? Arg : interp->pageLength;
	    if (c != ' ') interp->pageLength = interp->newlinesLeft;
	    break;

	case '\r':
	case '\n':
	    interp->newlinesLeft = Arg ? Arg : 1;
	    if (Arg) interp->pageLength = Arg;
	    break;
	
	case 'q':
	case 'Q':
	    PadPutString("\r\n", interp);
	    client->mode |= DiscardOutput;
	    break;
	
	case 's':
	    interp->newlinesLeft = interp->pageLength + (Arg?Arg:1);
	    break;
	    
	case 'f':
	    interp->newlinesLeft = interp->pageLength +
	    			   interp->pageLength * (Arg?Arg:1);
	    break;

	case 'S':
	case 'F':
	    interp->newlinesLeft = 32767;
	    break;

	case '\b':
	case DEL:
	    Arg = Arg / 10;
	    return(1);
	    break;

	default:
	    Arg = 0;
	    return(0);
	    break;
      
      }
  
    Arg = 0;

    if (client->outputFlag & PageOutputShutdown)
    
      {
	client->mode &= ~PageOutput;
	client->outputFlag &= ~PageOutputShutdown;
      }

    PutCharacter('\r');
    PadClearToEOL(0);
    FlushOutput();
    msg->bytecount -= client->lastBytesWritten;

    switch(msg->requestcode)
    
      {
      
	case WRITE_INSTANCE:
	    msg->bufferptr += client->lastBytesWritten;
	    HandleWrite(msg, client->writerPid);
	    break;
	
	case WRITESHORT_INSTANCE:
	    HandleShortWrite(msg, client->writerPid);
	    break;

      }

    return(1);
  
  }


PutEndOfFile(client)
    register struct Client *client;
  {
	/*
	 * Put an end-of-file event on the queue. These are returned
	 * immediately if the user was waiting for anything.
	 */
    register struct Event *e;
    register int mousemode = client->mode & (ReportTransition | ReportClick);

    e = FreeEvents;
    if (e == NULL) return;
    FreeEvents = e->next;

    if (client->eventTail)  client->eventTail->next = e;
    if (client->eventQ == NULL) client->eventQ = e;
    client->eventTail = e;
    e->next = NULL;
    e->code = EOFcode + ReportCharacter + ReportClick;
    e->bytecount = 0;
    EventToUser(client);
  }

PutEndEvent(client)
    register struct Client *client;
  {
	/*
	 * Put an end-of-file event on the queue. These are returned
	 * immediately if the user was waiting for anything.
	 */
    register struct Event *e;
    register int mousemode = client->mode & (ReportTransition | ReportClick);

    e = FreeEvents;
    if (e == NULL) return;
    FreeEvents = e->next;

    if (client->eventTail)  client->eventTail->next = e;
    if (client->eventQ == NULL) client->eventQ = e;
    client->eventTail = e;
    e->next = NULL;
    e->code = EOFcode + ReportCharacter + ReportClick;
    e->bytecount = 0;
    EventToUser(client);
  }




/*
 * Routines that are common to character and graphical events
 */

EventToUser(client)
    register struct Client *client;
  {
	/*
	 * If the user was requesting input, Reply with the event
	 * descriptor.
	 */
    int requestmode;
    register struct Event *e;
    register struct InterpUser *interp = (struct InterpUser *) client->interp;
    register struct ReadRec *read = client->readQ;
    int replyResult;

    if (!client->writerPid)  /* blocked on output: page mode, page is full */
      {
	short length, width;
	
	PadGetSize(0, &length, &width);

        interp->newlinesLeft = interp->pageLength = length - 1;
	client->mode &= ~DiscardOutput;
      }

    if (read == NULL) return;  /* no read request outstanding, no action */

    requestmode = 0;
    if (read->requestFlag&Keyreq) 
    	requestmode |= ReportCharacter|ReportEditedText;

    e = client->eventQ;
    if (e) 
      {
	  	/*
		 * We got one!! yank it off the event queue,
		 * put it on the free queue, and Reply to the client.
		 */
	register EventReq *rep;  /* variant record: IoReply */

	if ( e->code & (ReportCharacter|ReportEditedText) ) 
	  {

	    rep = (EventReq *) &(e->code);
	    rep->eventcode = ReportCharacter;

	    if ( (client->mode & Echo) && (e->code & ReportCharacter) )

	      {
		/*
		 * Perform the input character echoing now that the
		 * client is ready to accept the input.
		 */

		register char *c = e->shortbuffer;
		short i = 0;

		for ( ; ++i <= e->bytecount ; c++ )
		  {
		    if ( *c == '\n' && (client->mode & LF_Output) )
		        PadPutChar( '\r', TtyPadList+client->vgt );
		    PadPutChar( *c, TtyPadList+client->vgt );
		  }
	      }
	  }
    	else rep = (EventReq *) &(e->requestcode );

	DropEvent(client,e); 	/* free the one we are using */
	
	do
	  {
	    if (e->code & EOFcode)
		rep->requestcode = END_OF_FILE;
	    else
		rep->requestcode = OK;
	    rep->fileid = client-> vgt;
	    if (DifferentByteOrder(read->pid))
		ByteSwapLongInPlace(e->shortbuffer, (e->bytecount+3)&~3);
	    replyResult = Reply(rep, read->pid);
	    read->requestFlag &= ~BlockBits;

	    client->readQ = read->next; /* read is satisfied, free its record */
	    FreeReadRec(read);
	    read = client->readQ;
	  }
	    /* Note: if the requesting pid turns out to no longer exist, then 
		     reply to the next requester instead. */
	while ((replyResult == 0) && (read != NULL));
      }

    else if ( (requestmode & ReportCharacter) && (client->mode & LineBuffer) &&
	      (client->mode & Echo) )

	/*
	 * The client wants to get line-buffered input and a complete
	 * line is not yet available.  Start the user line editing by
	 * displaying what has been typed so far.
	 */

	DisplayInput(client);
  }


DropEvent(client,victim)
    register struct Client *client;
    struct Event *victim;
  {
	/*
	 * Remove the given event from the the Event Queue
	 * for this client
	 */
    register struct Event *e, *last;
    
    last = NULL;
    e = client->eventQ;

    while (e && victim != e)
      {
        last = e;
        e = e->next;
      }
    if (e==NULL) return;

    if (last)
      last->next = e->next;
    else
      client->eventQ = e->next;

    if (client->eventTail==e) client->eventTail = last;
    e->next = FreeEvents;
    FreeEvents = e;
  }


/*
 * The user has called for an event of either type.  Reporting of
 * mouse events is controlled by the mode of the pad.  Whichever
 * comes first, a mouse click (or transition), or a keystroke, that
 * will be returned.  No mouse clicks currently supported in STS.
 */
HandleEventRequest(client, msg, pid)
  register struct Client *client;
  register struct EventMsg *msg;
  ProcessId pid;
  {
    register struct ReadRec *read;

    read = GetReadRec();
    if (!read) return;  /* out of descriptors, requestor hangs forever */

/* Not usable in Terminal Server --
    if (msg->eventcode & ReportMotion)
	{
	  HandleGetGraphicsStatus(msg, pid, 0);
	  return;
	}
 */
    read->requestFlag = Mousereq;
    if (msg->eventcode & ReportCharacter)
        read->requestFlag |= Keyreq;
    read->pid = pid;
    read->next = NULL;
    EnqueueRead(client, read);
    EventToUser(client);
  }


struct ReadRec *GetReadRec()
  {
    register struct ReadRec *r;

    r = FreeReadRecs;
    if (!r) return(NULL);
    FreeReadRecs = r->next;
    r->requestFlag = 0;
    return(r);
  }


FreeReadRec(r)
  register struct ReadRec *r;
  {
    r->next = FreeReadRecs;
    FreeReadRecs = r;
  }


/* add a read record to the end of client's queue */
EnqueueRead(client, read)
  register struct Client *client;
  register struct ReadRec *read;
  {
    register struct ReadRec *r = client->readQ;

    if (!r)  client->readQ = read;
    else
      {
	while (r->next)  r = r->next;
	r->next = read;
      }
  }
