/*
 * SMI "Network Disk" boot server
 *
 * A quick (?) and dirty program to act as a boot server using
 *   SMI's remote disk protocol.
 *
 * based on "smiboot" by Tim Mann 11/05/83
 *
 * Rewritten for Unix by Bill Nowicki December 1983
 *  Updated January 1984 to read file & take args (WIN)
 *  and March 1984 to read device and IP address from command line
 * Kludged up further by TPM (6/84) to take account of the fact
 *  that the newer Suns use SMI ethernet addresses instead of 3Com.
 * Hacked on some more by Lance Berc (2/85) to support multiple
 *  boot files and on-the-fly modification of boot & config files.
 */
 
/* Things to know:
 *  Hosttable and bootfile descriptors cannot be removed from
 *   the tables. The buffers are not flushed if the files are
 *   removed from disk.
 *  The ndserver doesn't crash if it runs out of bootfile
 *   descriptors; it just refuses to load more. Maybe it
 *   should vfork, execl(argv), and die...
 *  File modification times are checked only when processing
 *   a block zero read request. Woe be the workstation that
 *   is in the middle of rebooting when a bootfile is recached.
 *  Address aritmetic is used to determine the offset of the
 *   block to be written, a Dangerous Practice.
 *  Only the first line of the fancy awk output is used when
 *   searching alt-ether-addrs.
 *  For now the -t option is disabled by initializing the
 *   CheckTimes switch to one; modification times are always
 *   checked.
 */

/* Configuration parameters */

# define DefaultDevice "/dev/enet"
# define DefaultIpAddress {36, 8, 0, 2}

# define ConfigDir "/usr/sun/Vboot/config"
# define BootProperty "boot:"
# define BootPropertySize 5
# ifndef NO_GRANDFATHER
/* Should be ripped out once the config files all have "boot:" instead */
/*   of "ndboot:"						       */
# define OldBootProperty "ndboot:"
# define OldBootPropertySize 7
# endif  NO_GRANDFATHER

# define DefaultBootfile "/usr/sun/Vboot/Vload10.d"
# define Bootfile "bootfile:"
# define BootfileSize 9

# define BLOCK_SIZE 512
# define BOOT_BUFFER_SIZE 32
# define PACKET_BUFFER_SIZE 2048
# define MaxHosts 256
# define MaxBootfiles 32

/* Constants */
# define YES 1
# define NO 0

# include <sys/types.h>
# include <sys/socket.h>
# include <netinet/in.h>
# include <netdb.h>
# include <sys/enet.h>
# include <sys/uio.h>
# include <stdio.h>
# include <errno.h>
# include <sys/stat.h>

extern int errno;

/* How ugly can you get department:
 * The ndprotocol asks for the first sixteen blocks off the
 * disk, but it expects the first two to be garbage directory
 * blocks. So we allocate the boot buffer to be 10k instead
 * of 8k and leave the first two blocks empty when reading
 * in the boot file.
 */
typedef char BootBuffer[BOOT_BUFFER_SIZE+2][BLOCK_SIZE];

char EtherHeader[14];
struct iovec EtherVec[] = 
    {
      {(caddr_t)EtherHeader,	sizeof(EtherHeader) },
      {0,	0}
    };

struct BootfileDescriptor /* Structure for storing a bootfile buffer */
  {
    char *path;		/* Path to the file. Should be absolute */
    char *BufferPtr;		/* Pointer to a sizeof(BootBuffer) buffer */
    time_t mtime;		/* Modification time of the file */
  } BFD[MaxBootfiles];

struct HostTableEntry
  {
  	/*
	 * Table to recall whom we are allowed to boot.
	 * Note that addresses are stored in big-endian byte order!
	 */
    char *path;			/* Path to the config file */
    time_t mtime;		/* Config file modification time */
    unsigned char ether[6];	/* Ether address */
    struct BootfileDescriptor *bfd;	/* Ptr to the bootfile or NULL */
  } HostTable[MaxHosts];

/*
 * Set at run-time from command line arguments
 */
char DefaultHost[] = DefaultIpAddress;
char *HostIP = DefaultHost;
char *Device = DefaultDevice;
char tmpBootfile[1024];

int LastHost = 0;
int LastBFD = 0;

/* Forward */
struct BootfileDescriptor *LoadBootfile(), *FindBootfile();
struct HostTableEntry *MapToHostTable(), *TryToReadConfigFile();
extern char *malloc();

#define ND_READ 1
#define ND_WRITE 2
#define ND_ERROR 3
#define ND_OP_MASK 7
#define ND_WAIT_FLAG 010
#define ND_DONE_FLAG 020
#define ND_MAXDATA 1024

char PacketBuffer[PACKET_BUFFER_SIZE];

int Debug = 0;		/* Nonzero causes messages */
int CheckTimes = 1;	/* Nonzero causes modification times of config
			 * files and boot files to be compared against
			 * the cached versions.
			 */

char name[1024];
struct stat statbuf;

struct HostTableEntry *ReadConfigFileByAddress(h)
unsigned char *h;
  {
   /*
    * Read the host configuration file to determine whether we should boot
    *   the host with ethernet address h, and update our host table if
    *   read successfully.  Return the new entry, else NULL if none made.
    */
     
    sprintf(name,"%s/C.%02x%02x%02x%02x%02x%02x", ConfigDir,
		 h[0], h[1], h[2], h[3], h[4], h[5] );
    if (stat(name, &statbuf) == -1)
      {
	if (Debug) printf("Couldn't stat %s\n", name);
	return(NULL);
      }
    else
        return(TryToReadConfigFile(name, h));
  }

struct HostTableEntry *ReadConfigFileWithAwk(h)
unsigned char *h;
  {
    register struct HostTableEntry *ht;
    FILE *gf, *popen();
    char name[1024];

    /* Get out your airsickness bag... 
     *   We couldn't find a config file for this ethernet address, so
     *   we search for it in the other config files, where it may
     *   be listed as an alternate address.  We use awk just in case
     *   the address we want is on a continuation line (overkill).
     */
    sprintf(name, "						\
		awk -F:     					\
		 '{						\
		    if ( $1 != \"\" )				\
			keyword = $1 ;				\
		    if ( $1 == \"alt-ether-addr\" && 		\
			 $2 == \"%02x%02x.%02x%02x.%02x%02x\" ) \
			print FILENAME ;			\
		  }'						\
		%s/C.*[0-9a-f]",
	h[0], h[1], h[2], h[3], h[4], h[5], ConfigDir);

    if (Debug) printf("Issuing command: %s\n", name);
    gf = popen(name, "r");
    if (gf == NULL)
      {
        perror("popen failed");
        return(NULL);
      }
    /* Try to read the name of the real config file as the output of awk */
    ht = NULL;
    while (fscanf(gf, "%s", name) == 1 && ht == NULL)
      {
	if (stat(name, &statbuf) == -1)
          {
            if (Debug) printf("Cannot open config file %s\n", name);
	  }
	else
	  ht = TryToReadConfigFile(name, h);
      }
    pclose(gf);
    return(ht);
  }

struct HostTableEntry *TryToReadConfigFile(name, h)
char *name;
char *h; /* address */
  { 
    struct HostTableEntry *ht;   

    if (Debug) printf("Stat of %s succeeded.\n", name);

    /* Cache this guy so we don't have to keep reading the config files.
     * We only cache him if both the config file and bootfile exist, so
     * we can add a new config file or fix a bogus bootfile at anytime.
     * (The bootfile check covers the case of a bootfile: field that
     * points to a non-existant bootifile.
     */
    ht = &HostTable[LastHost++];
    if (!ReadConfigFile(name, ht))
      {
	/* Bootfile doesn't exist */
        LastHost--;
	return(NULL);
      }
    ht->ether[0]= h[0];
    ht->ether[1]= h[1];
    ht->ether[2]= h[2];
    ht->ether[3]= h[3];
    ht->ether[4]= h[4];
    ht->ether[5]= h[5];
    ht->mtime = statbuf.st_mtime;
    ht->path = malloc(strlen(name));
    strcpy(ht->path, name);
    
    return(ht);
  }

int ReadConfigFile(path, ht)
struct HostTableEntry *ht;
  {
    char line[1024];
    FILE *fopen(), *hf;

    strcpy(tmpBootfile, DefaultBootfile);
    if ((hf = fopen(path, "r")) == NULL) return(NULL);
    while (fgets(line,sizeof(line),hf))
      {
        if (strncmp(line,BootProperty,BootPropertySize)==0)
	  {
	    if (strncmp(line+BootPropertySize,"no",2) == 0)
		*tmpBootfile = NULL;
	    if (Debug) printf("Want to boot=%d, string=%s.\n",
	    	(*tmpBootfile == NULL)?0:1,line+BootPropertySize);
	    break;
	  }
#ifndef NO_GRANDFATHER
	/* This should go away once the config files all use "boot:" */
	/*   instead of "ndboot:"				     */
        if (strncmp(line,OldBootProperty,OldBootPropertySize)==0)
	  {
	    if (Debug) printf(
     "-- Config file uses \"ndboot:\" - should be updated to \"boot:\" --\n");
	    if (strncmp(line+OldBootPropertySize,"no",2) == 0)
		*tmpBootfile = NULL;
	    if (Debug) printf("Want to boot=%d, string=%s.\n",
	    	(*tmpBootfile == NULL)?0:1,line+OldBootPropertySize);
	    break;
	  }
#endif  NO_GRANDFATHER
        if (strncmp(line,Bootfile,BootfileSize)==0)
	  {
	    strcpy(tmpBootfile, line+BootfileSize);
	    tmpBootfile[strlen(tmpBootfile)-1] = NULL;
	    if (Debug) printf("Bootfile=%s.\n", tmpBootfile);
	  }
      }
    fclose(hf);

    /* See if the boot file has been cached yet. */
    if (*tmpBootfile == NULL)
	ht->bfd = NULL;
    else
        if (((ht->bfd = FindBootfile(tmpBootfile)) == NULL) &&
	    ((ht->bfd = LoadBootfile(tmpBootfile)) == NULL))
	    return(NO); /* bootfile doesn't exist ! */
    return(YES);
  }

/* Search the Bootfile 'path' in the Bootfile cache */ 
struct BootfileDescriptor *FindBootfile(path)
char *path;
  {
    register struct BootfileDescriptor *bfd;
    for (bfd = &BFD[0]; bfd < &BFD[LastBFD]; bfd++)
        if (strcmp(path, bfd->path)==0) return(bfd);
    if (Debug) printf("Couldn't find BFD for %s.\n", path);
    return(NULL);
  }

/* Load the Bootfile 'path' into a new Bootfile descriptor */
struct BootfileDescriptor *LoadBootfile(path)
char *path;
  {
    struct BootfileDescriptor *bfd;
    struct stat statbuf;
    if (LastBFD == MaxBootfiles)
      {
	printf("ndserver: Ran out of boot file descriptors.\n");
	return(NULL);
      }
    if (stat(path, &statbuf) == -1)
      {
	if (Debug) { printf("Couldn't stat %s:", path); perror("Crud"); }
	return(NULL);
      }
    bfd = &BFD[LastBFD++];
    bfd->BufferPtr = malloc(sizeof(BootBuffer));
    if (ReadBootfile(path, bfd->BufferPtr+(2*BLOCK_SIZE)) == 1)
      {
	if (Debug) printf("Couldn't load bootfile %s.\n", path);
	free(bfd->BufferPtr);
	LastBFD--;
	return(NULL);
      }
    if (Debug) printf("Loaded %s into BFD %d.\n", path, LastBFD - 1);
    bfd->path = malloc(strlen(path));
    strcpy(bfd->path, path);
    bfd->mtime = statbuf.st_mtime;
    return(bfd);
  }

CheckBootfileTime(bfd)
struct BootfileDescriptor *bfd;
  {
    struct stat statbuf;
    
    /* If we can't stat the file don't touch the BFD */
    if ((stat(bfd->path, &statbuf) == -1) ||
	(statbuf.st_mtime == bfd->mtime)) return;
    if (Debug) printf("Rereading %s.\n", bfd->path);
    ReadBootfile(bfd->path, bfd->BufferPtr+(2*BLOCK_SIZE));
    bfd->mtime = statbuf.st_mtime;
  }

CheckConfigFileTime(ht)
struct HostTableEntry *ht;
  {
    struct stat statbuf;
    
    /* If we can't touch the file don't touch the hosttable */
    if ((stat(ht->path, &statbuf) == -1) ||
	(statbuf.st_mtime == ht->mtime)) return;
    if (Debug) printf("Recaching %s.\n", ht->path);
    ReadConfigFile(ht->path, ht);
    ht->mtime = statbuf.st_mtime;
  }

struct HostTableEntry *HostLookup(h)
    unsigned char *h;
  {
    /* Find host in our table. */
     register struct HostTableEntry *ht = HostTable;
     int i;
     
     for (i=0;i<LastHost;i++, ht++)
        if (bcmp(h,ht->ether,6)==0) return(ht);
     return(NULL);
  }

main(argc, argv)
    int argc;
    char *argv[];
  {
    char *bootfile = DefaultBootfile;
    int packetsize;

    argv++;
    while (argc-- > 1 && **argv == '-')
      {
       while (**argv)
         {
           switch (**argv)
	     {
	       case '-': break;
	       case 'D': case 'd': Debug++; break;
	       case 'T': case 't': CheckTimes++; break;
	       default:
	         printf("Unrecognized option: %c\n",**argv);
		 printf("Usage: ndserver [-d] /dev/enet inter.net.add.ress [file]\n");
	     }
	   (*argv)++;
	 }
       argv++;
      }
    if (Debug) printf("SMI boot protocol server\n");

    /* 
     * Arguments are: device, IP address, boot file.
     * All are optional and defaulted.
     */

    if (argc > 0) 
      {
        Device = *argv;
	if (Debug) printf("Device name is %s\n", Device);
        argc--; argv++;
      }

    if (argc > 0) 
      {
        static char host[4];
	struct hostent *h;

	h = gethostbyname(*argv);
	if (h==NULL)
	  {
	    long int i = inet_addr(*argv);
	    
	    if (i == -1)
	      printf(
	"Illegal host specified: %s, using default: [%d.%d.%d.%d]\n",
		    *argv, HostIP[0], HostIP[1], HostIP[2], HostIP[3]);
	    else bcopy(&i,host,4);
	  }
	else
	  {
	    bcopy(h->h_addr,host,4);
            HostIP = host;
	  }
	if (Debug) 
		printf("Host IP address is %s, [%d.%d.%d.%d]\n",
		    *argv, HostIP[0], HostIP[1], HostIP[2], HostIP[3]);
        argc--; argv++;
      }

    if (argc > 0) printf("Extra arguments are being tossed.\n");

    LastBFD = 0;
    LastHost = 0;
    InitNet();

    if (Debug) printf("Starting operation\r\n");
    while (1)
      {
        struct sockaddr_in src;

	ReadPacket(PacketBuffer, &packetsize, &src);  
	ProcessNDRequest(PacketBuffer, packetsize, &src);
      }
  }


/* Process a Network Disk protocol request */
ProcessNDRequest(buf, size, src)
   char *buf;
    int size;
    struct sockaddr_in *src;
  {
    register char *p=buf;
    unsigned char op, minordev, err, ver;
    char *opp, *ccp;
    unsigned int seqnum;
    unsigned int blocknum; 
    unsigned int bytecount;
    unsigned int residual;
    unsigned int offset;
    unsigned int ccount;
    char ipAddress[4];
    struct HostTableEntry *ht;

    if (Debug) printf("host %08x: size %d   ", src->sin_addr.s_addr, size);
    
	/*
	 * Skip the IP header, which is not supposed to
	 * be given to the user according to documentation,
	 * but is anyhow.
	 */
    p += 4;
    p += 4;
    p += 4;
    p += 4;
    p += 4;

    opp = p;
    op = *p++;
    minordev = *p++;
    err = *p++;
    ver = *p++;

    if (Debug) 
	printf("op=0%o minor=0%o err=0%o ver=%d\r\n", op, minordev, err, ver );

    seqnum = GetInt(p);
    p += 4;
    if (Debug) printf("seq# %d, ", seqnum);

    blocknum = GetInt(p); p += 4;
    if (Debug) printf("block %d, ", blocknum);

    bytecount = GetInt(p);
    p += 4;
    if (Debug) printf("bytes %d, ", bytecount);

    residual = GetInt(p); 
    p += 4;
    if (Debug) printf("Residual %d ", residual);

    offset = GetInt(p); 
    p += 4;
    if (Debug) printf("offset %d ", offset);

    ccp = p;
    ccount = GetInt(p); 
    p += 4;
    if (Debug) printf("in packet %d\r\n", ccount);

    switch (op & ND_OP_MASK)
      {
      case ND_READ: /* Read */
        *opp = ND_READ+ND_WAIT_FLAG+ND_DONE_FLAG;
        ccp[2] = ND_MAXDATA>>8;
	ccp[3] = ND_MAXDATA & 0xFF;

	if ((ht = MapToHostTable(src)) == NULL) break;
	if (blocknum == 0 && CheckTimes)
	  {
	    if (Debug) printf("Using config file %s.\n", ht->path);
	    CheckConfigFileTime(ht);
	    if (ht->bfd == NULL) break;
	    CheckBootfileTime(ht->bfd);
	  }
	bcopy( ht->bfd->BufferPtr + (blocknum*BLOCK_SIZE), p, ND_MAXDATA);
        WritePacket(buf, size + ND_MAXDATA,src);
	break;

      case ND_WRITE: /* Write */
	if (Debug) printf("Attempted ND Write!\r\n");
	break;

      case ND_ERROR: /* Error */
	if (Debug) printf("Received ND Error Packet!\r\n");
	break;
      }
  }

GetInt(buf)
    register unsigned char *buf;
  {
    register int i;
    
    i = *buf++;
    i <<= 8;
    i |= *buf++;
    i <<= 8;
    i |= *buf++;
    i <<= 8;
    i |= *buf++;
    return(i);
  }

/*
 * Note: due to the strange way that Berkeley did their networing
 * code, we have to do input via the "socket" mechanism (otherwise
 * they will throw the packets away), but have to do output by
 * the CMU packet-filtering mechanism since raw IP output seems
 * to always come back with routing errors.
 */

int S;	/* Socket */
int E;	/* enet file */
static struct enfilter MainFilter =   /* Filter for no packets */
  {
    1,	/* Priority */
    1,	/* Length */
    ENF_PUSHZERO,
  };

InitNet()
  {
    struct sockaddr_in adr;
    struct eniocb ecb;
    int i, maxwaiting;
    char fn[16];

    S = socket(AF_INET, SOCK_RAW, IPPROTO_ND);
    if (S<0) 
      {
        perror("Unable to open socket");
	exit(0);
       }
	  
    adr.sin_family = AF_INET;
    adr.sin_port = 0;
    adr.sin_addr.s_addr = INADDR_ANY;

/*    if (bind(S,&adr,sizeof(adr),0)<0)
      {
        perror("Unable to bind socket");
	exit(0);
      } */
    
    /* try all ethernet minors from 0 on up.
     *
     * Algorithm: assumption is that minors start at /dev/ether1
     * and run up as decimal numbers without gaps.  We search
     * until we get an error that is not EBUSY (i.e., probably
     * either ENXIO or ENOENT), or until we sucessfully open one.
     */
    for( i=0;; i++)
      {
	sprintf (fn, "%s%d", Device, i);
	if ( (E = open(fn, 1)) >= 0 ) 	
	  {
	    break;
	  }
	/* if we get past the break, we got an error */
	if (errno != EBUSY) 
	  {
       	    perror("Unable to open enet");
	    exit(0);
	  }
      }
    if (Debug) printf( "Opened %s\n", fn );    

    /* Get the parameters for this file */
    ioctl( E, EIOCGETP, &ecb );

    /* Ask for minimum incoming packet buffering */
    maxwaiting = 1;
    ioctl( E, EIOCSETW, &maxwaiting );

    /* Flush packets in the input queue */
    ioctl( E, EIOCFLUSH, 0 );
    ioctl( E, EIOCSETF, &MainFilter );

   }

ReadPacket(buf, size, fromp)
    int *size;
    struct sockaddr_in *fromp;
  {
    /*
     * Read an ND packet
     */
     int fromlen;

     fromlen = sizeof(struct sockaddr_in);
     *size = recvfrom(S,buf,PACKET_BUFFER_SIZE, 0, fromp, &fromlen);
     if (fromlen != sizeof(struct sockaddr_in)) 
         if (Debug) printf("Fromlength=%d\n",fromlen);
  }

int ReadBootfile(bootfile, buf)
    char *bootfile, *buf;
  {
    FILE *bf;

    if (Debug) printf("Reading bootfile %s\n", bootfile);
    if ((bf = fopen(bootfile,"r")) == NULL)
      {
	if (Debug)
	  {
	    printf("Opening '%s' failed: ", bootfile);
	    perror("Tough luck");
	  }
	return(1);
      }
    fread(buf,1,sizeof(BootBuffer),bf);
    fclose(bf);
    return(0);
  }


struct HostTableEntry *MapToHostTable(to)
struct sockaddr_in *to;
  {
    /* We didn't really get the client's ethernet address because we
     *  read the packet at the IP level.  So we use the following kludge.
     *  We know the SMI boot PROMs construct a bogus IP address with
     *  network number 0 and host number equal to the low-order 24 bits
     *  of the ethernet address.  We construct an ethernet address by
     *  taking the high order 24 bits to be either 3Com's or SMI's
     *  assigned address block number, and use it both to look up the
     *  requestor in our client table (to find whether we should really
     *  reply or not) and to route the reply packet.
     */
    struct HostTableEntry *ht;
     
    /* Ether Dest: 3Com or SMI, then low order IP addr */
    EtherHeader[3] = to->sin_addr.S_un.S_un_b.s_b2;
    EtherHeader[4] = to->sin_addr.S_un.S_un_b.s_b3;
    EtherHeader[5] = to->sin_addr.S_un.S_un_b.s_b4;

    /* First look for 3Com address in our table */
    EtherHeader[0] = 0x02;
    EtherHeader[1] = 0x60;
    EtherHeader[2] = 0x8c;
    if ((ht = HostLookup(EtherHeader)) != NULL) return(ht);

    /* No good.  Look for SMI address in our table */
    EtherHeader[0] = 0x08;
    EtherHeader[1] = 0x00;
    EtherHeader[2] = 0x20;
    if ((ht = HostLookup(EtherHeader)) != NULL) return(ht);

    /* Look for config file under 3Com address */
    EtherHeader[0] = 0x02;
    EtherHeader[1] = 0x60;
    EtherHeader[2] = 0x8c;
    if ((ht = ReadConfigFileByAddress(EtherHeader)) != NULL) return(ht);

    /* Look for config file under SMI address */
    EtherHeader[0] = 0x08;
    EtherHeader[1] = 0x00;
    EtherHeader[2] = 0x20;
    if ((ht = ReadConfigFileByAddress(EtherHeader)) != NULL) return(ht);

    /* Last chance.  Look for config file with alt-ether-addr SMI address */
    EtherHeader[0] = 0x08;
    EtherHeader[1] = 0x00;
    EtherHeader[2] = 0x20;
    if ((ht = ReadConfigFileWithAwk(EtherHeader)) != NULL) return(ht);

    /* Last chance.  Look for config file with alt-ether-addr 3Com address */
    EtherHeader[0] = 0x02;
    EtherHeader[1] = 0x60;
    EtherHeader[2] = 0x8c;
    if ((ht = ReadConfigFileWithAwk(EtherHeader)) != NULL) return(ht);

    return(NULL);
  }

WritePacket(buf,size,to)
    register char *buf;
    int size;
    struct sockaddr_in *to;
  {
    register struct HostTableEntry *ht;
    int cc;
    
    /* Ether Source: Filled in by driver or hardware */
    /* Ether Type: IP */
    EtherHeader[12] = 0x08;
    EtherHeader[13] = 0x00;
    
    /* IP Length */
    buf[2] = size>>8;
    buf[3] = size & 0xFF; 

    /* IP Source */
    buf[12] = HostIP[0];
    buf[13] = HostIP[1];
    buf[14] = HostIP[2];
    buf[15] = HostIP[3];

    /* IP dest */
    buf[16] = to->sin_addr.S_un.S_un_b.s_b1;
    buf[17] = to->sin_addr.S_un.S_un_b.s_b2;
    buf[18] = to->sin_addr.S_un.S_un_b.s_b3;
    buf[19] = to->sin_addr.S_un.S_un_b.s_b4;

    EtherVec[1].iov_base = buf;
    EtherVec[1].iov_len = size;
    cc = writev(E,EtherVec,2);
    if (cc<0) perror("Ether write");
    if (Debug) printf("Wrote %d bytes to host %02x%02x.%02x%02x.%02x%02x\n",cc,
		(unsigned char)EtherHeader[0],(unsigned char)EtherHeader[1],
		(unsigned char)EtherHeader[2],(unsigned char)EtherHeader[3],
		(unsigned char)EtherHeader[4],(unsigned char)EtherHeader[5]);
/*
 * The following does not seem to work.
 *
    to->sin_addr.S_un.S_un_b.s_b1 = 36;
    to->sin_addr.S_un.S_un_b.s_b2 = 8;
    cc = sendto(S,buf,size,to,sizeof(struct sockaddr_in));
    
    if (cc<0)
      {
        if (Debug) printf("To host %08x ", to->sin_addr.s_addr);      
	perror("Sendto");
      }
    else
      if (Debug) printf("Wrote %d bytes to host %08x\n",cc, to->sin_addr.s_addr);
   */
  }
