/*
 * Routines to search config files for:
 *   - name of bootstrap file to load
 *   - whether we should actually service load requests
 * Implements a cache for this information.
 *
 * These are hacked from similar routines in the ND server (for Suns).  The ND
 * server also caches the bootstrap files themselves; we don't do this because
 * our bootstrap files may be arbitrarily large, whereas ND boot files are 
 * limited to about 8K (?).
 *
 * It might be a considerable win to do some sort of file caching.
 */

#include <stdio.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>

#include "configboot.h"

#ifndef DEFAULT_CONFIGDIR
>>>>>>> DEFAULT_CONFIGDIR must be defined to be the pathname of a default
>>>>>>> directory to search for config files
#endif  DEFAULT_CONFIGDIR

int ccDebug = 0, ccVerbose = 0, ccCheckTimes = 1;
char *ccDefaultProgram = 0;
char *ccConfigDir = DEFAULT_CONFIGDIR;

#define MaxHosts 256
struct HostTableEntry HostTable[MaxHosts];

int LastHost = 0;

/* Forward */
struct HostTableEntry *MapToHostTable(), *TryToReadConfigFile();

char name[1024];
struct stat statbuf;

struct HostTableEntry *ReadConfigFileByAddress(h)
    struct ether_addr *h;
  {
   /*
    * Read the host configuration file to determine whether we should boot
    *   the host with ethernet address h, and update our host table if
    *   read successfully.  Return the new entry, else NULL if none made.
    */
     
    sprintf(name,"%s/C.%02x%02x%02x%02x%02x%02x", ccConfigDir,
		 h->ether_addr_octet[0], h->ether_addr_octet[1],
		 h->ether_addr_octet[2], h->ether_addr_octet[3],
		 h->ether_addr_octet[4], h->ether_addr_octet[5] );
    if (stat(name, &statbuf) == -1)
      {
	if (ccVerbose) printf("Couldn't stat %s\n", name);
	return(NULL);
      }
    else
        return(TryToReadConfigFile(name, h));
  }

/*
 * This made sense for Suns with multiple Ethernet interfaces.  Does it make
 *   any sense for other machines?
 */
struct HostTableEntry *ReadConfigFileWithAwk(h)
    struct ether_addr *h;
  {
    register struct HostTableEntry *ht;
    FILE *gf, *popen();
    char name[1024];

    /* Get out your airsickness bag... 
     *   We couldn't find a config file for this ethernet address, so
     *   we search for it in the other config files, where it may
     *   be listed as an alternate address.  We use awk just in case
     *   the address we want is on a continuation line (overkill).
     */
    sprintf(name, "						\
		awk -F:     					\
		 '{						\
		    if ( $1 != \"\" )				\
			keyword = $1 ;				\
		    if ( $1 == \"alt-ether-addr\" && 		\
			 $2 == \"%02x%02x.%02x%02x.%02x%02x\" ) \
			print FILENAME ;			\
		  }'						\
		%s/C.*[0-9a-f]",
	h->ether_addr_octet[0], h->ether_addr_octet[1],
	h->ether_addr_octet[2], h->ether_addr_octet[3],
	h->ether_addr_octet[4], h->ether_addr_octet[5], ccConfigDir);

    if (ccDebug) printf("Issuing command: %s\n", name);
    gf = popen(name, "r");
    if (gf == NULL)
      {
        perror("popen failed");
        return(NULL);
      }
    /* Try to read the name of the real config file as the output of awk */
    ht = NULL;
    while (fscanf(gf, "%s", name) == 1 && ht == NULL)
      {
	if (*name == '\0' || stat(name, &statbuf) == -1)
          {
            if (ccVerbose) printf("Cannot open config file \"%s\"\n", name);
	  }
	else
	  ht = TryToReadConfigFile(name, h);
      }
    pclose(gf);
    return(ht);
  }

struct HostTableEntry *TryToReadConfigFile(name, h)
    char *name;
    struct ether_addr *h; /* address */
  { 
    struct HostTableEntry *ht;   

    if (ccDebug) printf("Stat of %s succeeded.\n", name);
    ht = &HostTable[LastHost++ % MaxHosts];
	/*
	 * In the unlikely event that there are more than MaxHosts machines out
	 *   there, we recycle entries in FIFO fashion because I couldn't be
	 *   bothered doing anything more fancy.  We should free() the strings
	 *   in the entries we're recycling, but that would be just far too
	 *   thorough.
	 */
    if (!ReadConfigFile(name, ht))
      {
	/* Either config file or its boot file doesn't exist, so we */
	/*   don't cache even the config file			    */
        LastHost--;
	return(NULL);
      }
    ht->ether = *h;
    ht->mtime = statbuf.st_mtime;
    ht->path = (char *)malloc(strlen(name)+1);
    strcpy(ht->path, name);
    
    return(ht);
  }

/*
 * Reads the config file, looks for "don't boot" or "bootfile name =" entries,
 *   and - if we are meant to boot - checks whether the boot file exists.
 *   Returns 1 if we should keep this HostTableEntry, 0 if not.
 *   Also fills in the bootpath field of the HostTableEntry.
 */
int ReadConfigFile(path, ht)
char *path;
struct HostTableEntry *ht;
  {
    char line[1024], bootpath[1024];
    FILE *fopen(), *hf;
    int  loadfile = 1;
    int BootPropertySize = strlen(BootProperty);
    int BootfileSize     = strlen(Bootfile    );

    strcpy(bootpath, ccDefaultProgram);
    if ((hf = fopen(path, "r")) == NULL) return(0);
    while (fgets(line,sizeof(line),hf))
      {
        if (strncmp(line,BootProperty,BootPropertySize)==0)
	  {
	    if (strncmp(line+BootPropertySize,"no",2) == 0)
		loadfile = 0;
	    if (ccVerbose) printf("Want to boot=%d, string=%s.\n",
		loadfile, line+BootPropertySize);
	  }
        if (strncmp(line,Bootfile,BootfileSize)==0)
	  {
	    strcpy(bootpath, line+BootfileSize);
	    bootpath[strlen(bootpath)-1] = 0;
	    if (ccVerbose) printf("Bootfile=%s.\n", bootpath);
	  }
      }
    fclose(hf);

    if (!loadfile)
      {
	ht->bootpath = NULL;	/* Config file said don't boot, so we build */
	return(1); 		/*   a HostTableEntry with bootpath = NULL  */
      }
    else if (FindBootFile(bootpath))
      {
	ht->bootpath = (char *)malloc(strlen(bootpath)+1);
	strcpy(ht->bootpath, bootpath);
	return(1);
      }
    else
	return(0);
  }

FindBootFile(path)
char *path;
  {
    struct stat statbuf;
    if (path == NULL || *path == '\0' || stat(path, &statbuf) == -1)
      {
	if (ccVerbose) { printf("Couldn't stat \"%s\":", path); perror("Crud"); }
	return(0);
      }
    return(1);
  }

CheckConfigFileTime(ht)
struct HostTableEntry *ht;
  {
    struct stat statbuf;
    
    /* If we can't touch the file don't touch the hosttable */
    if ((ht->path == NULL) || (*(ht->path) == '\0') ||
	(stat(ht->path, &statbuf) == -1) ||
	(statbuf.st_mtime == ht->mtime)) return;
    if (ccVerbose) printf("Recaching %s.\n", ht->path);
    ReadConfigFile(ht->path, ht);
    ht->mtime = statbuf.st_mtime;
  }

struct HostTableEntry *HostLookup(h)
    struct ether_addr *h;
  {
    /* Find host in our table. */
     register struct HostTableEntry *ht = HostTable;
     int i;
     
     for (i=0;i<( (LastHost<MaxHosts)?LastHost:MaxHosts );i++, ht++)
        if (bcmp(h, &(ht->ether), 6) == 0)
	  {
	    if (ccCheckTimes)		/* If the file no longer exists, we */
		CheckConfigFileTime(ht);/*   will retain the cache info.    */
	    return(ht);			/*   Is that the semantics we want? */
	  }
     return(NULL);
  }


struct HostTableEntry *UseDefaults(h)
    struct ether_addr *h;
  {
    /*
     * It's a bit of a crock to use just one entry like this when there may
     * be multiple machines out there that use it, but we seem to get away
     * with it.  We don't insert the default info in the host cache, mainly
     * because I haven't stopped to think about the implications of that.
     */

    static struct HostTableEntry DefaultEntry;

    if ( ! FindBootFile(ccDefaultProgram) )
	return(0);
    DefaultEntry.path	  = 0;
    DefaultEntry.bootpath = ccDefaultProgram;
    DefaultEntry.mtime	  = (time_t) 0;
    DefaultEntry.ether	  = *h;
    return(&DefaultEntry);
  }
