/*
 * Routines relating to naming.
 */
 
#include <Vnaming.h>
#include <Vgroupids.h>
#include "memserv.h"

#define MAX_PATH_LENGTH (MAX_NAME_LENGTH*16)

/* Imports */
Filedesc *WellKnown[];

/* Globals */
char NameBuffer[MAX_PATH_LENGTH+1];


Filedesc *ContextToFd(cid)
    ContextId cid;
  {
    /* Given a context id, find the corresponding file descriptor.
     *  Return NULL if none exists.
     */
    register int index;
    register Filedesc *fd;
    
    if (cid < MAX_WELL_KNOWN_CONTEXTS) return WellKnown[cid];

    index = cid & 0x7fff;
    if (index > nfiles) return NULL;
    fd = &fdtable[index];
    if (fd->ctxpair.cid != cid) return NULL;

    return fd;
  }


SystemCode LookupFile(req, pid, fdpointer, suffix, segbuf, segsize)
    NameRequest *req;
    ProcessId pid;
    Filedesc **fdpointer;
    char **suffix;
    char *segbuf;
    unsigned segsize;
  {
    /* Look up the pathname specified by req->filename+req->filenameindex, 
     *   the node corresponding to req->contextid, unless the first
     *   character of the pathname is a '/'; in the latter case, start 
     *   with DEFAULT_CONTEXT.  Continue searching until either (1) the 
     *   end of the pathname is reached, (2) a path component is not found
     *   in the current search context, or (3) req->requestcode was 
     *   QUERY_NAME and a single-manager context was reached.
     * Upon return *suffix points to the first component of the pathname 
     *   that was not successfully mapped (or to the terminating null byte 
     *   if the entire name was mapped).  A pointer to the last node reached 
     *   is returned in *fdpointer (unless fdpointer == NULL); this pointer
     *   may be NULL if none of the name could be mapped.
     */
    register char *path, *comp;
    register int len, longflag = 0;
    register Filedesc *fd = NULL, *son;
    register char term;
    register SystemCode status = OK;

    /* First move pathname into NameBuffer */
    path = NameBuffer;
    len = req->namelength - req->nameindex;
    if (len > MAX_PATH_LENGTH)
      {
	/* Indicated length is too great.  Name may still be legal
	 *  if null-terminated short of this length. */
        longflag++;
	len = MAX_PATH_LENGTH;
      }

    if (req->namelength <= segsize)
      {
	/* Entire segment is in segment buffer, including all of name,
	 *  so we can get it from there. */
	Copy(path, segbuf + req->nameindex, len);
      }
    else
      {
        /* Some of the name may not be there, so we do a MoveFrom to be
	 *  sure we have it. */
	status = MoveFrom(pid, path, req->nameptr+req->nameindex, len);
        if (status != OK) return status;
      }

    /* Be sure name is null-terminated */
    path[len] = '\0';

    /* Check for length overflow, which is deemed to have occurred
     *  if namelength indicated the name could be longer than
     *  MAX_PATH_LENGTH, and it was not null-terminated soon enough 
     *  to be shorter than MAX_PATH_LENGTH.  The case where it was
     *  null-terminated to be exactly MAX_PATH_LENGTH is unfortunately
     *  treated as an overflow.  Seems not worth the effort to fix this.
     */
    if (longflag && strlen(path) == MAX_PATH_LENGTH)
      {
	status = ILLEGAL_NAME;
	goto exit;
      }

    /* Now ready to parse the name.
     * First check for '/' escape to local root or '['
     *   escape to global root.
     */
    switch (*path)
      {
        case '/':
	  fd = WellKnown[DEFAULT_CONTEXT];
	  path++;
	  break;

	case '[':
	  fd = WellKnown[GLOBAL_ROOT_CONTEXT];
	  path++;
	  break;

	default:
	  fd = ContextToFd(req->namecontextid);
	  if (fd == NULL)
	    {
	      status = INVALID_CONTEXT_ID;
	      goto exit;
	    }
	  break;
      }
      
    /* Scan the pathname component by component */
    do
      {
	/* If QueryName, we are done as soon as we reach
	 *  a single-manager context */
	if (req->requestcode == QUERY_NAME &&
	    fd->ctxpair.pid == mypid) goto exit;

	/* Find the terminator for the next component and
	 *  replace it with a null, saving the old value in "term".
	 */
	comp = path;
	while ((term = *path) != '\0' && term != '/' && term != ']')
	    path++;
	if (path - comp > MAX_NAME_LENGTH) 
	  {
	    status = ILLEGAL_NAME;
	    goto exit;
	  }
	if (term != '\0') *path++ = '\0';

	/* Check for special values */
	if (comp[0] == '\0') continue;
	if (comp[0] == '.')
	  {
	    if (comp[1] == '\0') continue;
	    if (comp[1] == '.' && comp[2] == '\0')
	      {
		if (fd->father->ctxpair.pid != fd->ctxpair.pid &&
		    req->requestcode != 0)
		  {
		    /* Must strip off the prefix parsed so far,
		     *  and forward the remainder of the name
		     *  to the group implementing the father context.
		     * requestcode == 0 flags this as the second
		     *  name in a RenameFile request, which is not
		     *  forwardable.
		     */
		    req->nameindex += (path - NameBuffer);
		    req->namecontextid = fd->father->ctxpair.cid;
		    Forward(req, pid, fd->father->ctxpair.pid);
		    return NO_REPLY;
		  }
	        fd = fd->father;
		continue;
	      }
	  }

	/* Check for sons of this node */
	for (son = fd->son; son != NULL; son = son->brother)
	  {	
	    if (strncmp(comp, son->name, MAX_NAME_LENGTH) == 0)
	      {
		if (son->size == MS_ALIAS)
		  {
		    /* Aliases are only recognized
		     * for local requestors */
		    if (!IsLocal(pid)) continue;
		    fd = son->son;
		  }
		else
	            fd = son;
		break;
	      }
	  }

	if (son == NULL)
	  {
	    /* No son with the specified name */
	    if (fd->ctxpair.pid == mypid)
	        status = NOT_FOUND;
    	    else
	        status = NOT_HERE;
	    goto exit;
	  }
      }        
    while (term != '\0');
      
exit:
    if (fdpointer != NULL) *fdpointer = fd;
    if (status == OK)
      {
	if (suffix != NULL) *suffix = path;
	return OK;
      }
    else
      {
	if (suffix != NULL) *suffix = comp;
	if (term != '\0') *--path = term;
        if ( (status != NOT_FOUND) && IsGroupId(Forwarder(pid)) )
            return DISCARD_REPLY;
        else
            return status;
      }
 }

SystemCode HandleGetContextId(req, pid, segbuf, segsize)
  ContextRequest *req;
  ProcessId pid;
  char *segbuf;
  unsigned segsize;
  {
    register ContextReply  *reply = (ContextReply *) req;
    Filedesc *fd;
    SystemCode r;
    
    r = LookupFile(req, pid, &fd, NULL, segbuf, segsize);
    if (r != OK) return (r);

    reply->context.pid = fd->ctxpair.pid;
    reply->context.cid = fd->ctxpair.cid;
    return(OK);
  }


SystemCode HandleGetName(req, pid, segbuf, segsize)
  register ContextRequest *req;
  ProcessId      pid;
  char *segbuf;
  unsigned segsize;
  {
    Filedesc *fd;
    char *suffix;
    SystemCode r;
    char tempbuffer[MAX_PATH_LENGTH+1];
    register Instance  *instance;
    register int index, diff;
#define reply ((ContextReply *) req)

    /* Initialize buffer for filename.  Grows from right to left. */
    tempbuffer[MAX_PATH_LENGTH] = '\0';
    index = MAX_PATH_LENGTH;

    /* Find node whose name we need */
    switch (req->requestcode)
      {
	case GET_FILE_NAME:
	  instance = GetInstance(req->instanceid);
      	  if (instance == NULL) return( INVALID_FILE_ID );
	  reply->context = instance->fd->ctxpair;
	  fd = instance->fd;
	  break;

	case GET_ABSOLUTE_NAME:
	  r = LookupFile(req, pid, &fd, &suffix, segbuf, segsize);
	  if (r == OK)
	    {
	      reply->context = fd->ctxpair;
	    }
	  else if (r == NOT_FOUND)
	    {
	      /* Special code to get absolute names for files that
	       *  don't exist yet.  We append the tail of the path
	       *  name (which didn't exist) to the absolute name
	       *  for the last context in the path that did exist.
	       */
	      reply->context.pid = 0;	/* indicate nonexistent as yet */
	      reply->context.cid = 0;
	      diff = strlen(suffix);
	      index -= diff;
	      strncpy(tempbuffer + index, suffix, diff);
	      tempbuffer[--index] = '/';
	    }
	  else
	      return r;
	  break;

	case GET_CONTEXT_NAME:
	  fd = ContextToFd(req->context.cid);
	  if (fd == NULL) return INVALID_CONTEXT_ID;
	  reply->context = fd->ctxpair;
	  break;
      }

    
    /* Walk up the tree, prepending name components as we go */
    for (;;)
      {
	diff = strlen(fd->name);
	index -= diff;
	strncpy(tempbuffer+index, fd->name, diff);
	fd = fd->father;
	if (fd->ctxpair.cid == GLOBAL_ROOT_CONTEXT)
	  {
	    tempbuffer[--index] = '[';
	    break;
	  }
	else
	    tempbuffer[--index] = '/';
      }

    reply->replycode = OK;
    reply->namelength = MAX_PATH_LENGTH + 1 - index;
    ReplyWithSegment( reply, pid, tempbuffer+index, req->nameptr, 
                         reply->namelength );

    return NO_REPLY;

#undef reply
  }


SystemCode HandleQueryName(req, pid, segbuf, segsize)
  register ContextRequest *req;
  ProcessId pid;
  char *segbuf;
  unsigned segsize;
  {
    SystemCode r;
    char *suffix;
    Filedesc *fd;
#define reply ((ContextReply *) req)
    
    r = LookupFile(req, pid, &fd, &suffix, segbuf, segsize);
    if (r != OK) return (r|IMMEDIATE_REPLY_BIT);
    reply->nameindex = req->nameindex + (suffix - NameBuffer);
    reply->context = fd->ctxpair;
    reply->flags = 0;
    return (OK|IMMEDIATE_REPLY_BIT);

#undef reply
  }
