/*			Memory File Server
 *
 * Main server module.  Waits for requests and handles them.
 *
 * This program is a fake file server.  It maintains files in contiguous
 * blocks of main memory.  Its interface to the world is modelled after
 * the V storage server, with one difference.  There is no ownership of
 * files, and no protection.  A file descriptor always shows maximal
 * permission to owner, group, and others, and that is an accurate portrayal
 * of this server's behavior.
 *
 * Files are stored in contiguous sections of memory.  The server uses
 * SetTeamSize rather than malloc, so that it will shrink when its memory
 * needs shrink.  It moves files around in memory to keep a compact use of
 * its team space.  The only wastage is some internal fragmentation - memory
 * is allocated in blocks of 1024 bytes, in anticipation of a page management
 * facility.  The memory hogged by this server can be limited by command
 * line options:
 *	-m <number>	Maximum storage size will be <number> bytes,
 *			interpreted in decimal, rounded down as needed.
 *	-k <number>	Keep <number> bytes free.  When the server starts,
 *			it will set its maximum size <number> bytes short
 *			of the available memory.  It does not check again
 *			later.
 */

#include <Vioprotocol.h>
#include <Vtermagent.h>
#include <Vnaming.h>
#include <Vgroupids.h>
#include "memserv.h"

/* Global variables */
ProcessId	mypid;
unsigned	memtop,		/* current top of team */
		memlimit;	/* maximum team size ever allowed */


Filedesc	*fdtable,	/* variable array of file descriptors */
		*fdfile;	/* fd for pseudo-file of file descriptors */

unsigned	nfiles;		/* number of file descriptors in fdtable */

unsigned	debug = 0,
		maxmem = 0,	/* -m flag value */
		keepmem = 0;	/* -k flag value */

Filedesc	*WellKnown[MAX_WELL_KNOWN_CONTEXTS];
char		HostName[MAX_NAME_LENGTH+1];

extern SystemCode HandleCreateInstance(), HandleReleaseInstance(), 
		HandleQueryInstance(), 	HandleReadInstance(), 
		HandleWriteInstance(), HandleReadAndForward(),
		HandleSetInstanceOwner(), HandleRemoveFile(), 
		HandleGetContextId(), HandleGetContextName(), 
		HandleGetFileName(), HandleReadDescriptor(),
		HandleNreadDescriptor(), HandleCreateFile(),
		HandleRenameFile(), HandleQueryName(), 
		HandleGetAbsoluteName();
extern unsigned MemFree() ;

char SegmentBuffer[MAX_APPENDED_SEGMENT];


/* Main program.  Initializes the server, then handles requests in a loop
 * forever.
 */
main(argc, argv)
  int  argc;
  char **argv;
  {
    Message msg;
    register IoRequest  *req = (IoRequest *) msg;
    register IoReply *replymsg = (IoReply *) msg;

    register SystemCode replycode;
    register ProcessId id;
    unsigned segsize;

    ReadOptions(argc, argv);
    InitMemserver();
   
    while( 1 )
      {
	if (debug) Flush(stdout);
	segsize = MAX_APPENDED_SEGMENT;

	id = ReceiveWithSegment( req, SegmentBuffer, &segsize );

        switch( req->requestcode )
          {
            case CREATE_INSTANCE: 
            case CREATE_INSTANCE_RETRY: 
              replycode = 
	      	  HandleCreateInstance(req, id, SegmentBuffer, segsize);
              break;

            case RELEASE_INSTANCE:
              replycode = HandleReleaseInstance(req);
              break;
            
            case QUERY_INSTANCE: 
                replycode = HandleQueryInstance(req);
                break;

            case READ_INSTANCE:
                replycode = HandleReadInstance(req, id);
                break;

            case WRITESHORT_INSTANCE:
            case WRITE_INSTANCE: 
                replycode = 
		    HandleWriteInstance(req, id, SegmentBuffer, segsize);
                break;

            case READ_AND_FORWARD:
                replycode = HandleReadAndForward(req, id);
                break;

            case SET_INSTANCE_OWNER:
                replycode = HandleSetInstanceOwner(req, id);
                break;

            case CREATE_FILE: 
              replycode = HandleCreateFile(req, id, SegmentBuffer, segsize);
              break;

            case RENAME_FILE: 
              replycode = HandleRenameFile(req, id, SegmentBuffer, segsize);
              break;

            case REMOVE_FILE:
              replycode = HandleRemoveFile(req, id, SegmentBuffer, segsize);
              break;

            case GET_CONTEXT_ID:
              replycode = HandleGetContextId(req, id, SegmentBuffer, segsize);
              break;

	    case QUERY_NAME:
	      replycode = HandleQueryName(req, id, SegmentBuffer, segsize);
	      break;

            case GET_CONTEXT_NAME:
            case GET_FILE_NAME:
	    case GET_ABSOLUTE_NAME:
              replycode = HandleGetName(req, id, SegmentBuffer, segsize);
              break;

            case READ_DESCRIPTOR:
              replycode = HandleReadDescriptor(req, id);
              break;

            case NREAD_DESCRIPTOR:
              replycode = 
	      	  HandleNreadDescriptor(req, id, SegmentBuffer, segsize);
              break;
           
	    case SET_DEBUG_MODE:	/* for debugging use only */
	      replycode = HandleDebugAction(req, id);
	      break;

            default:
	      if (IsGroupId(Forwarder(id)))
		replycode = DISCARD_REPLY;
	      else
                replycode = ILLEGAL_REQUEST;
          }
        if (replycode == NO_REPLY)
          continue;

        replymsg->replycode = replycode;
        Reply(replymsg, id);
      }
  }


/* ReadOptions: interpret the command line flags */
ReadOptions(argc, argv)
  register int argc;
  register char *argv[];
  {
    register int nArg;
    register char *flagp;

    nArg = 1;
    while ( nArg < argc )
      {
	/* parse a string of flags */
	if (argv [nArg][0] == '-')
	  {
	    flagp = &argv[nArg++][1];
	    while (*flagp != 0)
	      {
		switch (*flagp)
		  {
		    case 'd': debug = 1;
			    printf("debug on\n");
			    break;

		    case 'k':
			    if (nArg >= argc) 
				Fatal("-k flag with no number");
			    keepmem = getnumber(argv[nArg++]);
			    goto nextarg;  /* break 2 levels */

		    case 'm':
			    if (nArg >= argc) 
				Fatal("-m flag with no number");
			    maxmem = getnumber(argv[nArg++]);
			    goto nextarg;  /* break 2 levels */

		  default:  printf("unknown flag %c\n", *flagp); exit(1);
		  }  /* end switch */

		flagp++;
	      }  /* end while loop on flags */
nextarg:    ;
	  } /* end if */
	else  nArg++;	/* we have no use for non-flag arguments */
      }  /* end while loop on args */
  }  /* end ReadOptions */


InitMemserver()
  {
    unsigned	oldteamsize, roundteamsize, memfree, memallowed;
    unsigned	inittime;
    register Filedesc *fd;
    extern SystemCode GetContextId();
    SystemCode	err;
    char namebuf[128];
    ContextPair ctx;

    mypid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    ChangeTeamPriority(mypid, FOREGROUND);
    inittime = GetTime(NULL);
    err = QueryWorkstationConfig("name", HostName, sizeof(HostName));
    if (err != OK) Fatal("Can't get host name");

    /* Check for name conflicts and exit if present.
     * Another alternative would be to pick a different name.
     */
    strcpy(namebuf, "[storage/");
    strcat(namebuf, HostName);
    if (GetContextId(namebuf, &ctx) == OK)
        Fatal("A memserver or storage server is already running!\n");

    /* This server does not use malloc.  It uses SetTeamSize itself, always
     * in aligned 1K blocks.  Here we align to start with.
     */
    oldteamsize = GetTeamSize(mypid);
    if (oldteamsize & (BLOCKSIZE-1))
      {
	roundteamsize = 1 + (oldteamsize | (BLOCKSIZE-1));
	memtop = SetTeamSize(mypid, roundteamsize);
	if (memtop != roundteamsize)  Fatal("Not enough memory to get started");
      }
    
    /* The -m and -k options allow the user to put limits on the growth
     * of this team.  They are implemented here.
     */
    memfree = MemFree();
    if (maxmem)
	memlimit = memtop + maxmem;  /* user specifies size of file storage */
    else if (keepmem)
      {
	maxmem = memfree - keepmem;  /* user specifies space to keep free */
	memlimit = memtop + maxmem;
      }
    else memlimit = 1<<30;  /* we'll never see this much memory! */

    if (memlimit < 1<<30)
      {
	memallowed = memlimit - memtop;
	printf("Memory server started, %dK allowed, %dK reserved\n",
		memallowed/1024,
		(memfree > memallowed) ? (memfree-memallowed)/1024 : 0 );
      }
    else  printf("Memory server started, unlimited memory allowed, %dK available\n", memfree/1024);
    Flush(stdout);

    /* Create the file descriptor table.  It is a variable-length array
     * of file descriptors, stored in a pseudo-file described by the first
     * descriptor.  Its size is stored in nfiles; it never decreases.
     * The advantage of a pseudo-file is that I can expand it using the
     * same mechanisms used to expand files in general.
     */
    fdtable = (Filedesc *) RequireBlocks(1);

    /* the file descriptor pseudo-file (local root) */
    fd = fdfile = fdtable;
    strncpy(fd->name, HostName, sizeof(fd->name));
    fd->start = (mem *) fdtable;
    fd->size = 2 * sizeof(Filedesc);
    fd->nblocks = 1;
    fd->timestamp = inittime;
    fd->ctxpair.pid = mypid;
    fd->ctxpair.cid = DEFAULT_CONTEXT;
    WellKnown[DEFAULT_CONTEXT] = fd;    

    /* the global root node */
    fd++;
    strncpy(fd->name, "[", sizeof(fd->name));
    fd->start = (mem *) fdtable;
    fd->size = 0;
    fd->nblocks = 0;
    fd->timestamp = inittime;
    fd->father = fd;
    fd->brother = 0;	/* son set later */
    fd->ctxpair.pid = VCSNH_SERVER_GROUP;
    fd->ctxpair.cid = GLOBAL_ROOT_CONTEXT;
    WellKnown[GLOBAL_ROOT_CONTEXT] = fd;

    /* a node for the context of all storage servers */
    fd++;
    strncpy(fd->name, "storage", sizeof(fd->name));
    fd->start = (mem *) fdtable;
    fd->size = 0;
    fd->nblocks = 0;
    fd->timestamp = inittime;
    fd->father = WellKnown[GLOBAL_ROOT_CONTEXT];
    WellKnown[GLOBAL_ROOT_CONTEXT]->son = fd;
    fd->brother = 0;
    fd->son = WellKnown[DEFAULT_CONTEXT];
    fdfile->father = fd;    
    fd->ctxpair.pid = VSTORAGE_SERVER_GROUP;
    fd->ctxpair.cid = STORAGE_SERVER_CONTEXT;
    WellKnown[STORAGE_SERVER_CONTEXT] = fd;

    /* an alias for the local root */
    fd++;
    strncpy(fd->name, "local", sizeof(fd->name));
    fd->start = (mem *) fdtable;
    fd->size = MS_ALIAS;
    fd->father = WellKnown[STORAGE_SERVER_CONTEXT];
    fd->brother = 0;
    fd->son = fdfile;
    fdfile->brother = fd;

    /* /tmp directory */
    fd++;
    strncpy(fd->name, "tmp", sizeof(fd->name));
    fd->start = (mem *) fdtable;
    fd->size = 0;
    fd->nblocks = 0;
    fd->timestamp = inittime;
    fd->father = fdfile;
    fdfile->son = fd;
    fd->brother = 0;
    fd->son = 0;
    fd->ctxpair.pid = mypid;
    fd->ctxpair.cid = 0x8000 + (fd - fdtable);

    nfiles = 5;

    /* the instance table is statically initialized */

    /* Join the groups involved in naming */
    if ( (err = JoinGroup(VCSNH_SERVER_GROUP, mypid)) != OK ||
         (err = JoinGroup(VSTORAGE_SERVER_GROUP, mypid)) != OK ||
         (err = JoinGroup(LSTORAGE_SERVER_GROUP, mypid)) != OK )
      {
	PrintError(err, "can't join naming groups");
	exit(1);
      }
  }


#include <Vquerykernel.h>

/* Determine how much memory is left in the workstation */
unsigned MemFree()
  {
    Message msg;
    register MemoryStatisticsReply *reply = (MemoryStatisticsReply *)msg;
    unsigned memfree;

    QueryKernel(0, MEMORY_STATS, reply);
    memfree = reply->unusedFastMemory + reply->unusedSlowMemory;
    memfree &= ~1023;	/* fractional K units are not interesting */
    return(memfree);
  }


/* Decimal input conversion, with the <number>K and <number>M abbreviations
 * supported.
 */
int getnumber(s)
  register char *s;
  {
    register int n = 0;

    while (*s >= '0' && *s <= '9')  {n = n*10 + (*s - '0');  s++;}
    if (*s == 'k' || *s == 'K') n <<= 10;
    else if (*s == 'm' || *s == 'M') n <<= 20;
    return(n);
  }
