/*			Memory File Server
 *
 * Requests that deal with file instances
 */

#include <Vioprotocol.h>
#include <Vdirectory.h>
#include <Vgroupids.h>
#include "memserv.h"

extern SystemCode CreateDirectoryInstance(), HandleQueryInstance();


SystemCode HandleCreateInstance( req, id, segbuf, segsize )
  register CreateInstanceRequest *req;
  ProcessId id;
  char *segbuf;
  unsigned segsize;
  {
    register             Instance *instance;
    IoReply              *replymsg = (IoReply *) req;
    Filedesc             *fd;
    register             SystemCode  r;
    unsigned             mode, type;
    char		 *suffix;

    if ( req->requestcode == CREATE_INSTANCE_RETRY &&
	 IgnoreRetry(req, id, segbuf, &segsize, mypid) ) return DISCARD_REPLY;

    mode = req->filemode;

    switch( mode )
      {
        case FREAD:
          type = READABLE|FIXED_LENGTH|MULTI_BLOCK;
          break;

        case FREAD|FDIRECTORY:
	  return(CreateDirectoryInstance(req, id, segbuf, segsize));
          break;

        case FCREATE:
          type = READABLE|WRITEABLE|MULTI_BLOCK;
          break;

        case FAPPEND:
          type = READABLE|WRITEABLE|APPEND_ONLY|MULTI_BLOCK;
          break;

        case FMODIFY: 
          type = READABLE|WRITEABLE|MULTI_BLOCK;
          break;

        case FEXECUTE|FCREATE:
          return(MODE_NOT_SUPPORTED);
          break;

        default: 
          return(INVALID_MODE);
          break;
      }

    if( (r = LookupFile(req, id, &fd, &suffix, segbuf, segsize)) != OK )
      {
        if (r != NOT_FOUND)  return(r);
	if (mode == FCREATE || mode == FAPPEND)
	  {  /* create the file and allocate it some blocks */
	    if ((r = NewFile(fd, suffix, &fd)) != OK)  return(r);
	    if (!AddBlocksToFile(fd, CLEARING)) 
	      {
		if (debug) printf("File %s: cannot allocate a clearing\n",
					fd->name);
		return(NO_SERVER_RESOURCES);
	      }
	  }
	else return(NOT_FOUND);
      }

    if (IsGroupId(fd->ctxpair.pid)) return MULTI_MANAGER;

    if( (instance=AllocInstance(id)) == NULL )
        return( NO_SERVER_RESOURCES );

    if (mode == FCREATE) fd->size = 0;	/* truncate */

    instance->owner = id;
    instance->fd = fd;
    instance->type = type;
    instance->kind = FILE_INSTANCE;
    
    replymsg->fileid = instance - insttable;  /* id = index in instance table */
    return(HandleQueryInstance(replymsg) );
  }


/*
 * ReleaseInstance: release the file instance.  If the file was open for
 * writing, close the file's "clearing" to prevent waste of memory.
 * Apart from that, has no effect on the file.
 */
SystemCode HandleReleaseInstance(req)
    register IoRequest  *req;
  {
    register Instance	*instance;
    register Filedesc	*fd;

    /* Map to the desired instance */
    if((instance = GetInstance(req->fileid)) == NULL ) return( OK );

    fd = instance->fd;
    if (instance->type & WRITEABLE)  CloseClearing(fd);

    instance->fd = NULL;
    instance->owner = 0;  /* sign of a free instance descriptor */

    return( OK );
  }


/* Return information on the specified file instance. */
SystemCode HandleQueryInstance(req)
  CreateInstanceReply *req;
  {
    Instance        *instance;
    Filedesc            *fd;

    /* Map to the desired instance */
    if( (instance = GetInstance(req->fileid)) == NULL )
        return( INVALID_FILE_ID );
    fd = instance->fd;

    req->fileserver = mypid;
    req->filetype = instance->type;

    if (instance->kind == DIRECTORY_INSTANCE)
      {
	req->blocksize = sizeof(FileDescriptor);
	req->filelastblock = MAXBLOCKS; /* size unknown at this time */
	req->filelastbytes = req->blocksize;
      }
    else
      {
	req->blocksize = BLOCKSIZE;
	req->filelastblock = fd->size / BLOCKSIZE;
	req->filelastbytes = fd->size % BLOCKSIZE;
	if (req->filelastbytes == 0)  req->filelastbytes = BLOCKSIZE;
      }
    return(OK);
  }


SystemCode HandleSetInstanceOwner(req, id)
   IoRequest  *req;
   ProcessId id;
  {
    Instance        *instance;

    /* Map to the desired instance */
    if ((instance = GetInstance(req->fileid)) == NULL)
        return(INVALID_FILE_ID);
    if (instance->owner != id) return (NO_PERMISSION);

    instance->owner = req->instanceowner;
    return( OK );
  }


