/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Lower-level support routines for the TCP protocol.
 */


#include "net.h"
#include "internet.h"


/*
 * Imported routines
 */

extern PktBuf AllocBuf();
extern SystemCode QueryInstance();
extern PktBuf DeQueue();
extern SpinLockType *InstanceLock;
extern int InstanceLockHolder;
extern int SendTcpStats();
extern Boolean StatsEnabled;


/*
 * HandleOther:
 * Check fin and urg bits of incoming segment as a fcn. of sendAckedFlag.
 * Handle any options present.
 * Queue any data in the segment for the receiver.
 * Reply to CreateInstance request for this connection.
 */

HandleOther(pTcb, curPkt, sendAckedFlag)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    Boolean sendAckedFlag;
  {
    Message msg;
    CreateInstanceReply *repMsg = (CreateInstanceReply *) msg;

    if (TcpPkt(curPkt->segAdr)->hdr.fin)
	if (sendAckedFlag)
	    pTcb->state = CloseWait;
	else
	  {
	    DiscardSegment(pTcb, curPkt->segAdr, 
	    	    "fin set and our syn not yet acked.");
	    return;
	  }
    if (TcpPkt(curPkt->segAdr)->hdr.urg)
      {
	pTcb->rcvUrgentFlag = True;
	pTcb->rcvUrgentPointer = TcpPkt(curPkt->segAdr)->hdr.sequenceNumber + 
		TcpPkt(curPkt->segAdr)->hdr.urgentPointer;
      }
    if (curPkt->segDataOffset > StandardByteDataOffset)
	HandleOptions(pTcb, curPkt);
    if (curPkt->segLen > 0)
      {
	QSegment(pTcb, curPkt, sendAckedFlag);
				/* sendAckedFlag here indicates if we have
				   already reached a state where Receive
				   calls can be honored. */
      }
    else
	DiscardSegment(pTcb, curPkt->segAdr, NULL);
    repMsg->replycode = OK;
    repMsg->fileid = pTcb->netId << 1;
    QueryInstance(repMsg);
    Reply(msg, pTcb->rcvId);
  }




/*
 * HandleOptions:
 * Processes options present in incoming Tcp packet.
 * Assumes that options are present.
 */

HandleOptions(pTcb, curPkt)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
  {
    int i;
    TcpPktPtr p;
    Bit16 *v;
    Boolean done;

    i = 0;
    p = TcpPkt(curPkt->segAdr);
    done = False;
    while (! done)
      {
	switch (p->data[i])
	  {
	    case EndOptionList:
		done = True;
		break;
	    case NopOption:
		i++;
		break;
	    case MaxSegSizeOption:
		i += 2;
		v = (Bit16 *) &(p->data[i]);
#ifdef LITTLE_ENDIAN
		ByteSwapShortInPlace(v);
#endif LITTLE_ENDIAN
		i += 2;
		pTcb->maxPacketDataLength = Min((int) (*v), 
			MaxPacketDataLength);
		NetInstTable[pTcb->netId].wblocksize =
			pTcb->maxPacketDataLength;
		break;
	    default:
		done = True;
		break;
	  }
      }
  }




/*
 * FindWhichConnection:
 * Determines which connection is being referred to by the current event.
 * Returns id of Netinst record for that connection.
 * If no connection exists then -1 is returned.
 */

int FindWhichConnection(
		createInstance, localPort, foreignPort, foreignHost)
    Boolean createInstance;	/* Signals whether this represents a
				   CreateInstance situation or not. */
    Bit16 localPort, foreignPort;
    Bit32 foreignHost;
  {
    int i, indx = -1, indx1 = -1;
    TcpTcbPtr p;

    if (createInstance)
      {
	if (foreignHost == 0)
	    return(-1);
	LOCKID(InstanceLock,InstanceLockHolder,21);
	for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	    if (NetInstTable[i].prot == TcpProt)
	      {
		p = (TcpTcbPtr) NetInstTable[i].tcbId;
		if ((localPort == p->localPort) &&
			(foreignPort == p->foreignSocket.port) &&
			(foreignHost == p->foreignSocket.host))
		  {
		    indx = i;
		    break;
		  }
	      }
	UNLOCK(InstanceLock,InstanceLockHolder);
	return(indx);
      }
    else			/* packet from the network case */
      {
	LOCKID(InstanceLock,InstanceLockHolder,22);
	for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	  {
	    if (NetInstTable[i].prot == TcpProt)
	      {
		p = (TcpTcbPtr) NetInstTable[i].tcbId;
		if (p->instState & TCP_CONN_CLOSED)
			/* Connection is already closed. */
		    continue;
		if ((localPort == p->localPort) &&
			(foreignPort == p->foreignSocket.port) &&
			(foreignHost == p->foreignSocket.host))
		  {
		    indx = i;
		    break;
		  }
		else if ((localPort == p->localPort) &&
			(p->foreignSocket.host == 0))
		  {
		    indx1 = i;
		  }
	      }
	  }

	UNLOCK(InstanceLock,InstanceLockHolder);
	if (indx != -1)
	    return(indx);
	else
	    return(indx1);
      }
  }




/*
 * NoAccess:
 * Checks if user has access to specified connection.
 *
 * Currently this is a dummy routine.
 */

Boolean     NoAccess()
  {
            return(False);
  }




/*
 * VerifySecurityAndPrecedence:
 * Returns OK if security and precedence specified in event parms.
 * are ok, otherwise returns status indicating which is in error.
 *
 * Currently a dummy routine.
 */

SystemCode VerifySecurityAndPrecedence(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to current connection. */
  {
            return(OK);
  }




/*
 * CloseConnection:
 * Close connection by discarding the incoming segment and then
 * deallocating the connection.
 */

CloseConnection(pTcb, curPkt, errCode, reason)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    SystemCode     errCode;
    char *reason;		/* reason for closing */
  {
    if (InternetDebug > 2)
	printf("[%d]Closing connection: %s\n",pTcb->netId,reason);
    DiscardSegment(pTcb, curPkt->segAdr, ErrorString(errCode));
    DeallocateTcb(pTcb, errCode);
  }




/*
 * CheckSum:
 * Computes the checksum for designated Tcp packet.
 * Optionally also copies the packet to a second one.
 *
 * The checksum consists of the one's complement of the one's 
 * complement sum of the 16-bit words in the tcp packet with
 * the checksum field set to zero.  This sum also includes the
 * pseudo header of the tcp packet which consists of the packet's
 * source and destination address, its length, and its type (TCP).
 */

Bit16 CheckSum(ptr, tcpLength, srcAdr, dstAdr, tcpType)
    TcpPktPtr ptr;		/* Ptr to Tcp packet to calc. checksum
				   for. */
    Bit16 tcpLength;		/* Length of tcp packet. */
    Bit32 srcAdr, dstAdr;	/* Source and destination address of
				   the packet. */
    Bit16 tcpType;		/* Type of the packet.  (Should be
				   TCPprot.) */
  {
    int len, len2;
    register int i;
    register unsigned long sum;
    unsigned short *w1, *w2, *w3, *w4;
    register unsigned short *w;
    unsigned short tmp;
    Bit16 ckSum;

    ckSum = ptr->hdr.checkSum;          /* Save old value */
    ptr->hdr.checkSum = 0;
    len = (int) tcpLength / 2;
    w = (unsigned short *) &(ptr->hdr);

    sum = 0;
    len2 = len / 2;
    for (i = 0; i < len2; i++)
      {
#ifdef  LITTLE_ENDIAN
	tmp = *w++;
	ByteSwapShortInPlace(&tmp);
	sum += tmp;
	tmp = *w++;
	ByteSwapShortInPlace(&tmp);
	sum += tmp;
#else
        sum += *w++ + *w++;
#endif  LITTLE_ENDIAN
      }

    if ((len2 << 1) != len)
      {
#ifdef  LITTLE_ENDIAN
	tmp = *w++;
	ByteSwapShortInPlace(&tmp);
	sum += tmp;
#else
	sum += *w++;		/* Add in the last (odd) word. */
#endif  LITTLE_ENDIAN
      }

    if ((tcpLength % 2) != 0)
      {
#ifdef  LITTLE_ENDIAN
	tmp = ((*w) & 0x00ff);
	ByteSwapShortInPlace(&tmp);
	sum += tmp;
#else
	sum += ((*w) & 0xff00);
#endif  LITTLE_ENDIAN
      }                         /* Add in the last (odd) byte. */

    /* Add in the pseudo header. */
    sum += ((srcAdr >> 16) & 0x0000ffff) + (srcAdr & 0x0000ffff);
    sum += ((dstAdr >> 16) & 0x0000ffff) + (dstAdr & 0x0000ffff);
    sum += TCPprot + tcpLength;

    /* Add in the one's complement carry. */
    sum = (sum & 0xffff) + (sum >> 16);
    sum = (sum + (sum >> 16));

    ptr->hdr.checkSum = ckSum;
    return((Bit16) ~sum & 0xffff);
  }




/*
 * InitTcpConnection:
 * Initialize the data structures for a Tcp connection and return a 
 * preallocated packet buffer for use with ReceiveWithSegment.
 */

PktBuf InitTcpConnection(pTcb)
    TcpTcbPtr pTcb;
  {
    int     i;
    PktBuf pkt;

    /*
     * Initialize to "blank" template state.
     */
    InitSafeQueue(&(pTcb->readerQueue), pTcb->ringBufs);
    pTcb->receiveBlocked = False;
    pTcb->netId = -1;
    pTcb->instState = 0;
    pTcb->state = Closed;
    pTcb->rcvId = 0;
    pTcb->wtSignalId = 0;
    pTcb->localPort = 0;
    pTcb->foreignSocket.port = 0;
    pTcb->foreignSocket.host = 0;
    pTcb->originalForeignSocket = pTcb->foreignSocket;
    pTcb->passiveOpen = False;
    pTcb->active = False;
    pTcb->prc = 0;
    pTcb->higherPrcAllowed = True;
    pTcb->security = 0;
    pTcb->waitSignal = False;
    pTcb->maxPacketDataLength = MaxPacketDataLength;
    for (i = SendQueue; i <= SaveQueue; i++)
	InitQueue(&(pTcb->q[i]));
    for (i = 0; i < NumTcpTimeouts; i++)
      {
	StopTimer(pTcb, i);
      }
    pTcb->rcvByteCnt = 0;
    pTcb->rcvQ = Null;
    pTcb->lastReadBuf = Null;
    pTcb->segQBytesAvail = 0;
    pTcb->sndUna = 0;
    pTcb->sndNxt = 0;
    pTcb->sndWnd = 0;
    pTcb->sndUp = 0;
    pTcb->sndWl1 = 0;
    pTcb->sndWl2 = 0;
    pTcb->sndUrgBit = False;
    pTcb->iss = 0;
    pTcb->rcvNxt = 0;
    pTcb->rcvWnd = RcvWindowSize;
    pTcb->delRcvWnd = 0;
    pTcb->rcvUrgentPointer  = Null;
    pTcb->rcvUrgentFlag     = False;
    pTcb->rcvUrgentReported = False;
    pTcb->irs = 0;
    for (i = 0; i < NumRetransTimes; i++)
      {
	pTcb->retransTimes[i] = LenRetransTimeout;
      }
    pTcb->lenRetransTimeout = LenRetransTimeout;
    pTcb->nextRetransTime = 0;

    pTcb->user = 0;
    pTcb->creationTime = GetTime(0);

    pTcb->segmentsSent = 0;
    pTcb->bytesSent = 0;
    pTcb->totAckTime = 0;
    pTcb->numRetransTimeouts = 0;

    pTcb->segmentsRcvd = 0;
    pTcb->bytesRcvd = 0;
    pTcb->numOutOfOrderPkts = 0;

    /* Preallocate a buffer for doing ReceiveWithSegment on WriteInstances. */
    pkt = AllocBuf();
    while (pkt == NULL)
      {
        Delay(0,10);
	pkt = AllocBuf();	/* Keep trying forever */
      }
    pkt->dataptr = 
    		&(pkt->data[(MAXPBUFSIZE - MaxPacketDataLength)]);
				/* Initialize the dataptr so that the Tcp
				   data gets put in the right part of the
				   packet buffer. */
    return(pkt);
  }




/*
 * DeallocateTcb:
 * Resets all fields of the Tcb, returns all queued buffers, replies
 * to all outstanding user calls.
 */

DeallocateTcb(pTcb, statusCode)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    SystemCode     statusCode;	/* Status to return to any outstanding
				   calls. */
  {
    Message msg;
    int i;

    if (InternetDebug > 0)
      {
        printf("***** DeallocateTcb %d: %s *****\n", pTcb->netId,
		ErrorString(statusCode));
      }
    /* 
     * Clean up loose ends.
     */
    if (pTcb->state == SynSent)
      {				/* Respond to outstanding createInstance. */
        ((CreateInstanceReply *)msg)->replycode = statusCode;
	Reply(msg, pTcb->rcvId);
	/* Since CreateInstance failed, set up NetInst to be deallocated */
	NetInstTable[pTcb->netId].inUse = 0;
      }
    pTcb->state = Closed;
    if (pTcb->waitSignal)
      {
	if (InternetDebug > 5) printf("DeallocTcb: Notifying user\n");
	NotifyUser(pTcb, statusCode);
      }
    if (pTcb->rcvQ != NULL)
      {
	if (InternetDebug > 5) printf("DeallocTcb: Replying to read\n");
	ReplyToRead(statusCode, pTcb->rcvId, NULL, NULL, 0);
      }
    if (pTcb->lastReadBuf != NULL)
      {
	if (InternetDebug > 5) printf("DeallocTcb: FreeBuf(lastReadBuf)\n");
	FreeBuf(pTcb->lastReadBuf);
      }
    if ((InternetDebug>0) && !Empty(pTcb->q[SaveQueue]))
      {
	printf("[%d]----- Still have packets on the save queue when ",
	pTcb->netId);
	printf("deallocating a Tcp connection! -----\n");
      }
    for (i = SendQueue; i <= SaveQueue; i++)
      {
	if (InternetDebug > 5) printf("DeallocTcb: Flushing queue %d\n",i);
	FlushQueue(pTcb, i);
      }
    for (i = 0; i < NumTcpTimeouts; i++)
      {
	if (InternetDebug > 5) printf("DeallocTcb: stopping timer %d\n",i);
	StopTimer(pTcb, i);
      }
    if (StatsEnabled) SendTcpStats(pTcb);
    /*
     * Free up the network instance.
     */
    pTcb->instState |= TCP_CONN_CLOSED;
  }





/*
 * SendIp:
 * Sends designated packet to Ip layer.
 * Returns WriteIp status code adjusted for Tcp status values.
 */

SystemCode SendIp(pTcb, packet, fHost)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. 
				   Null if no connection is associated with
				   this send. */
    PktBuf     packet;		/* Ptr to packet buffer to send
				   to Ip. */
    Bit32 fHost;		/* Foreign host to send to. */
  {
    TcpPktPtr p;		/* Ptr to actual Tcp packet. */
    SystemCode rc;
    char *saveDataptr;		/* Used to save value of 
				   packet->dataptr during WriteIp. */
    int saveLength;		/* Used to save value of
				   packet->length during WriteIp. */

    p = TcpPkt(packet);


    saveDataptr = packet->dataptr;
    saveLength = packet->length;
    rc = WriteIp(LocalIpHost, fHost,
		TCPprot, 0, 0, 30, packet, 0, 0);
				/* Note: WriteIp changes the values
				   of dataptr and length since it adds
				   an IP header to the packet. */
    packet->dataptr = saveDataptr;
    packet->length = saveLength;

    if (rc != OK)
      {
	if (InternetDebug)
	  {
	    printf("TCP: error in WriteIp (%s)\n", ErrorString(rc));
	  }
      }
    else if (pTcb != NULL)
	StopTimer(pTcb, AckTimeout);
    return(rc);
  }




/*
 * InitPacketBuffer:
 * Initialize packet buffer.
 * Intended for use with outgoing Tcp packets.
 *
 * Assumes that dOffset is a multiple of 4.
 * Note: the urg and urgentPointer fields of the packet are set at
 * actual send time.
 */

InitPacketBuffer(lPort, fPort, tcpLen, dOffset, packet, seqNum, ackNum,
	    ack, psh, rst, syn, fin, rcvWnd)
    Bit16 lPort, fPort;		/* Local and foreign ports. */
    int tcpLen;			/* Length of Tcp packet. */
    int dOffset;		/* Offset in bytes of data 
				   field. */
    PktBuf packet;		/* Packet buffer to init. */
    Bit32 seqNum, ackNum;	/* Sequence number and ack number resp. */
    Boolean ack, psh, rst, syn, fin;
				/* Equivalent packet bit fields. */
    Bit16 rcvWnd;		/* Receive window size. */
  {
    TcpPktPtr ptr;		/* Ptr to Tcp packet to init. */

    packet->next = Null;
    packet->dataptr = 
		&(packet->data[(MAXPBUFSIZE - TcpPkt_Size)]);
    packet->length = tcpLen;

    ptr = TcpPkt(packet);
    ptr -> hdr.sourcePort = lPort;
    ptr -> hdr.destinationPort = fPort;
    ptr -> hdr.sequenceNumber = seqNum;
    ptr -> hdr.acknowledgementNumber = ackNum;
    ptr -> hdr.dataOffset = dOffset / 4;
    ptr -> hdr.reserved = 0;
    ptr -> hdr.urg = 0;
    ptr -> hdr.ack = ack;
    ptr -> hdr.psh = psh;
    ptr -> hdr.rst = rst;
    ptr -> hdr.syn = syn;
    ptr -> hdr.fin = fin;
    ptr -> hdr.window = rcvWnd;
    ptr -> hdr.checkSum = Null16;
    ptr -> hdr.urgentPointer = Null16;
  }




/*
 * FlushQueue:
 * Flush designated queue.
 */

FlushQueue(pTcb, queue)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    int     queue;		/* Which queue to flush. */
  {
    PktBuf pkt;

    while (!Empty(pTcb->q[queue]))
      {
	pkt = DeQueue(&(pTcb->q[queue]));
	FreeBuf(pkt);
      }
  }




/*
 * StartTimer:
 * (Re)start specified timer.
 * NOTE: This routine assumes that there will always be a timer 
 * queue record available on the free list.
 */

StartTimer(pTcb, timer, len)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    int    timer;		/* Index of timer to start. */
    int     len;		/* Length of timeout interval. */
  {
    int secs, clicks;

    secs = GetTime(&clicks) - Time0;
    pTcb->timers[timer] = CTime(secs, clicks) + len;
    if (pTcb->timers[timer] < NextTimeout)
      {
        Wakeup(TimerPid);
      }
  }




/*
 * StopTimer:
 * Stop specified timer.
 */

StopTimer(pTcb, timer)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    int    timer;		/* Timer to stop. */
  {
    pTcb->timers[timer] = MAX_SIGNED_INT;
  }




/*
 * DiscardSegment:
 * Discard the specified segment.
 */

DiscardSegment(pTcb, packet, errStr)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    PktBuf packet;
    char *errStr;
  {
    if ((InternetDebug>2) && errStr)
      {
	printf("[%d]Discarding Tcp segment from %s: %s\n",
	    pTcb->netId, inet_ntoa(packet->unspecified[0]), errStr);
      }
    FreeBuf(packet);
  }
