/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Higher-level support routines for the TCP protocol.
 */


#include "net.h"
#include "internet.h"


/*
 * Imported routines
 */

extern PktBuf AllocBuf();
extern PktBuf DeQueue();
extern SystemCode SendIp();

/*
 * Forward function declarations
 */

SystemCode MoveTcpSegment();
SystemCode TryToSendToIp();



/*
 * CheckRcvWnd - report change in rcvWnd size if outside of range
 */

CheckRcvWnd(pTcb,change)
    TcpTcbPtr pTcb;
    int change;    
  {
    Bit16 rw;

    rw = pTcb->rcvWnd;
    if (rw + change > RcvWindowSize ||
	rw + change < 0)
      {
	if (InternetDebug > 0)
	  {
	    printf("[%d]***** rcvWnd ABOUT TO BE MADE OUT OF RANGE!\n",
		pTcb->netId);
	    printf("--current value %d to change by %d\n",rw,change);
	    PauseInternetServer("rcvWnd error");
	  }
      }
  }
/*
 * SelectIss:
 * Selects an initial sequence number for a new connection.
 */

Bit32 SelectIss()
  {
    int     clicks;		/* Number of time interrupts since last
				   second. */
    Bit32 iss;			/* Initial sequence number. */

    iss = (Bit32) GetTime(&clicks);
    return (iss + (Bit32) clicks);
  }




/*
 * SendSynSegment:
 * Issue a Syn segment to IP.
 *
 * Currently no data is sent with this segment.
 */

SystemCode SendSynSegment(pTcb, iss, ackFlag, ack)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    Bit32 iss;			/* Initial sequence number. */
    Boolean ackFlag;		/* Inicates if should send an Ack. */
    Bit32 ack;			/* Ack number to send. */
  {
    SystemCode     statusCode;
    PktBuf packet;
    TcpPktPtr p;
    Bit16 *v;
    int t, clicks;

    packet = AllocBuf();
    while (packet == Null)	/* In this case we want to wait until a
				   packet buffer becomes available rather
				   than returning a bad return code since
				   this routine gets called from code that
				   responds to incoming network packets (thus
				   the returncode isn't much good). */
      {
	Delay(0,1);
	AllocBusyWait++;
	packet = AllocBuf();
      }
    InitPacketBuffer(pTcb->localPort, pTcb->foreignSocket.port,
    	    StandardByteDataOffset + 4,
	    StandardByteDataOffset + 4, packet, iss, ack,
	    ackFlag, False, False, True, False, 
	    pTcb->rcvWnd - pTcb->delRcvWnd);

    p = TcpPkt(packet);
    p->data[0] = MaxSegSizeOption;
    p->data[1] = (Bit8) 4;
    v = (Bit16 *) &(p->data[2]);
    *v = (Bit16) MaxPacketDataLength;
/* the following option is optional and is hereby removed!
    p->data[4] = EndOptionList;
    p->data[5] = (Bit8) 0;
    p->data[6] = (Bit8) 0;
    p->data[7] = (Bit8) 0;
 */

    statusCode = SendIp(pTcb, packet, pTcb->foreignSocket.host);
    if (statusCode == OK)
      {
	pTcb->iss = iss;
	pTcb->sndUna = iss;
	pTcb->sndNxt = iss + 1;
	pTcb->state = SynSent;
	EnQueue(&(pTcb->q[RetransQueue]), packet);
	packet->unspecified[0] = 0;
				/* Init. retransmission number. */
	t = GetTime(&clicks) - Time0;
	packet->unspecified[1] = CTime(t, clicks);
				/* Time the packet was sent. */
	StartTimer(pTcb, RetransTimeout, pTcb->lenRetransTimeout);
#ifdef TIMERDEBUG
	if (TimerDebug)
	   printf("[%d]Start RTimer #1\n",pTcb->netId);
#endif TIMERDEBUG
      }
    else
      {
	FreeBuf(packet);
      }
    return(statusCode);
  }




/*
 * QSend:
 * Either queues a segment in the send queue of the connection or queues
 * a null segment with the fin bit set.  Then tries to send packets
 * to the Ip level if possible.
 * QSend is either presented with a preallocated buffer, in which case it 
 * "preallocates" another buffer to pass back; or it isn't presented with a 
 * buffer, in which case it just allocates its own.
 * NOTE: The current implementation always sets the psh bit due to the nature
 * of the V I/O protocol.
 */

SystemCode QSend(pTcb, movePid, fin, rqMsg, pkt, pktSize)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    ProcessId movePid;		/* Process id of client whose data is
				   to be sent. */
    Boolean fin;		/* Indicates if fin bit should be set in
				   last packet. */
    IoRequest *rqMsg;		/* Client's write request msg. */
    PktBuf *pkt;		/* Points to a preallocated packet buffer.
				   NULL if none provided. */
    int pktSize;		/* Number of bytes of data already read into
				   pkt by ReceiveWithSegment. */
  {
    SystemCode statusCode;
    PktBuf packet, tmpPktPtr;
    TcpPktPtr p;
    char *t, *f;
    int i;

    statusCode = OK;

    if ((FreeBufferCount <= 1) && 
	    (TotalBufferCount >= MaxBufAllocNum - 1))
      {
	return(RETRY);
      }
    packet = AllocBuf();
    if (packet == NULL)
      {
	return(RETRY);
      }
    if (pkt != NULL)		/* Preallocated buffer avail. Make packet 
				   point  to it and make pkt point to the new 
				   one. */
      {
	packet->dataptr = 
    		&(packet->data[MAXPBUFSIZE - MaxPacketDataLength]);
				/* Initialize the dataptr so that the Tcp
				   data gets put in the right part of the
				   packet buffer. */
	tmpPktPtr = packet;
	packet = *pkt;
	*pkt = tmpPktPtr;
      }
    InitPacketBuffer(pTcb->localPort, pTcb->foreignSocket.port, 
    		StandardByteDataOffset,	StandardByteDataOffset, packet,
		pTcb->sndNxt, pTcb->rcvNxt,
		True, True, False, False, False, 
		pTcb->rcvWnd - pTcb->delRcvWnd);
    EnQueue(&(pTcb->q[SendQueue]), packet);

    if (fin)
      {
	p = TcpPkt(packet);
	p->hdr.fin = 1;
	pTcb->sndNxt += 1;
      }
    else
      {
	if (rqMsg->bytecount <= IO_MSG_BUFFER)
	  { 			/* NOTE: We are assuming that IO_MSG_BUFFER
				   is always smaller than
				   pTcb->maxPacketDataLength. */
	    if (DifferentByteOrder(movePid))
	      {
		ByteSwapLongCopy((char *)rqMsg->shortbuffer,
		    (char *)(packet->dataptr + packet->length),
		    rqMsg->bytecount);
	      }
	    else
	      {
		t = (char *) (packet->dataptr + packet->length);
		f = (char *) rqMsg->shortbuffer;
		for (i = 0; i < rqMsg->bytecount; i++)
		    *t++ = *f++;
	      }
            packet->length += rqMsg->bytecount;
	  }
	else if (rqMsg->bytecount <= pTcb->maxPacketDataLength)
	  {
	    if (pktSize < rqMsg->bytecount)
				/* Haven't got all the bytes yet. */
	      {
		statusCode = MoveFrom(
			movePid, (packet->dataptr + packet->length + pktSize),
			(rqMsg->bufferptr + pktSize), 
			(rqMsg->bytecount - pktSize));
		if (statusCode != OK)
		  {
		    return(statusCode);
		  }
	      }
	    packet->length += rqMsg->bytecount;
	  }
	else			/* User sent too much data! */
	    return(BAD_BYTE_COUNT );
        pTcb->sndNxt += rqMsg->bytecount;
      }
    if (pTcb->state >= Estab)
        statusCode = TryToSendToIp(pTcb);
    return(statusCode);
  }




/*
 * TryToSendToIp:
 * Tries to send packets from the specified queue to IP.
 *
 * NOTE: Assumes that the psh bit is always set.  If this assumption should
 * ever change, then Clark's Silly Window Syndrome avoidance scheme should
 * be incorporated here; namely only send a packet if the usable window
 * size is at least 25% of the offered window size.
 */

SystemCode TryToSendToIp(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    SystemCode statusCode;
    PktBuf  packet, pkt;
    TcpPktPtr p;
    int t, clicks;

    StopTimer(pTcb, SndWndTimeout);
				/* Stop the timer on the assumption that we
				   will have a large enough window to send a 
				   packet. */
    statusCode = OK;
    while (!Empty(pTcb->q[SendQueue]))
      {
        packet = pTcb->q[SendQueue].head;
        p = TcpPkt(packet);
        if (!IN(pTcb->sndUna, p -> hdr.sequenceNumber +
    		DLength(packet), pTcb->sndUna + pTcb->sndWnd))
	  {			/* Send window is too small to send
				           packet. */
	    if (Empty(pTcb->q[RetransQueue]))
		StartTimer(pTcb, SndWndTimeout, LenSndWndTimeout);
				/* Start a timer to periodically query the
				   remote host to see if it's rcv window has
				   grown. */
	    break;
	  }
        if (pTcb->sndUp != NULL)
          {
	    p -> hdr.urgentPointer = pTcb->sndUp - p -> hdr.sequenceNumber;
	    p -> hdr.urg = True;
	    if (IN(p -> hdr.sequenceNumber, pTcb->sndUp,
			p->hdr.sequenceNumber + DLength(packet)))
	      {
		pTcb->sndUp = NULL;
	      }
          }
        else
	    p -> hdr.urg = False;
        p -> hdr.acknowledgementNumber = pTcb->rcvNxt;
        statusCode = SendIp(pTcb, packet, pTcb->foreignSocket.host);
        if (statusCode == OK)
          {
	    pTcb->segmentsSent++;
	    pTcb->bytesSent += DLength(packet);
	    pkt = DeQueue(&(pTcb->q[SendQueue]));
	    EnQueue(&(pTcb->q[RetransQueue]), pkt);
	    pkt->unspecified[0] = 0;
				/* Init. retransmission number. */
	    t = GetTime(&clicks) - Time0;
	    pkt->unspecified[1] = CTime(t, clicks);
				/* Time the packet was sent. */
	    StartTimer(pTcb, RetransTimeout, pTcb->lenRetransTimeout);
#ifdef TIMERDEBUG
	    if (TimerDebug)
   	        printf("[%d]Start RTimer #2\n",pTcb->netId);
#endif TIMERDEBUG
          }
        else
	    break;
      }
    return(statusCode);
  }




/*
 * ServiceRcvRequest:
 * Reassemble queued incoming segments into user's receive buffer.
 * Set notification for urgent data if this is the case.
 * Remove processed incoming segments from Segments Received queue.
 * Reply to user's Read call.
 *
 * Note: packet->dataptr and packet->length now refer only to the
 * actual Tcp data to be sent to the user.  Adjustments for overlapping
 * sequence numbers, etc. have already been made.
 * Assumes that a Read call is outstanding and that bytes are
 * available for receiving.
 */

ServiceRcvRequest(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    register int len;
    Message msg;
    SystemCode sCode;

    len = pTcb->q[SegQueue].head->length;

    /* check if URG flag has appeared on an incoming segment */
    if( pTcb->rcvUrgentFlag )
      {
	if( !pTcb->rcvUrgentReported )
	  {
	    /* report presence of urgent condition to receiver */
	    ReplyToRead( BEGIN_URGENT_DATA, pTcb->rcvId, NULL, NULL, 0 );
	    pTcb->rcvQ = NULL;
	    pTcb->rcvUrgentReported = True;
	    return;
	  }
	else if( pTcb->rcvUrgentPointer ==
		(pTcb->rcvNxt - (Bit32)pTcb->segQBytesAvail) )
	  {
	    /* urgent pointer has been reached -- report to receiver */
	    ReplyToRead( END_URGENT_DATA, pTcb->rcvId, NULL, NULL, 0 );
	    pTcb->rcvQ = NULL;
	    pTcb->rcvUrgentFlag = False;
	    pTcb->rcvUrgentReported = False;
	    return;
	  }
	else if( pTcb->rcvUrgentPointer < (pTcb->rcvNxt -
		(Bit32)pTcb->segQBytesAvail + pTcb->q[SegQueue].head->length) )
	  {
	    /* urgent pointer points into the middle of the head segment */
	    /* NOTE: above test ignores wraparound of sequence numbers!  */

	    if (InternetDebug > 0) 
		printf("Note: rcvUrgentPointer ptr not at head of pkt!\n");
	    len = (Bit32)pTcb->segQBytesAvail -
			(pTcb->rcvNxt - pTcb->rcvUrgentPointer);
	    if (len <= 0) 
		PauseInternetServer("urgent ptr => negative length");
	  }
	/* else urgent pointer points beyond the data in the head buffer */
      }

    if( len > pTcb->rcvByteCnt )  len = pTcb->rcvByteCnt;

    switch (pTcb->state)
      {
	case Closed:
	case LastAck:
	case Closing:
	case TimeWait:
	    sCode = END_OF_FILE;
	    break;
	default:
	    sCode = OK;
	    break;
      }

    ReplyToRead(sCode, pTcb->rcvId, pTcb->q[SegQueue].head, 
	    pTcb->rcvQ, len);
    pTcb->rcvQ = NULL;

    /* Keep around the last block read for handling duplicate reads. */

    if( len < pTcb->q[SegQueue].head->length )
      {
	/* not all bytes from the head segment were read -- split buffers */
	if( pTcb->lastReadBuf == NULL )
	    while( (pTcb->lastReadBuf = AllocBuf()) == NULL )  Delay( 0, 1 );

	pTcb->lastReadBuf->dataptr = (char *) pTcb->lastReadBuf->data;
	pTcb->lastReadBuf->length  = len;
	Copy( pTcb->lastReadBuf->dataptr, pTcb->q[SegQueue].head->dataptr, len);
	pTcb->q[SegQueue].head->dataptr += len;
	pTcb->q[SegQueue].head->length  -= len;
      }
    else
      {
	if (pTcb->lastReadBuf != NULL)
	    FreeBuf(pTcb->lastReadBuf);

	pTcb->lastReadBuf = DeQueue(&(pTcb->q[SegQueue]));
      }

    NetInstTable[pTcb->netId].filenextblock++;
    pTcb->segQBytesAvail -= len;

#ifdef TIMERDEBUG
    CheckRcvWnd(pTcb,len);
#endif TIMERDEBUG
    pTcb->rcvWnd += len;
    pTcb->delRcvWnd += len;
    if (pTcb->delRcvWnd > RcvWindowCutoff)
      {
	pTcb->delRcvWnd = 0;
	SendAckSegment(pTcb);
			    /* Notify remote side of connection of the
			       new window size. */
      }
  }




/*
 * QSegment:
 * Case 1: Queue a segment to be delivered to the host if it is on or
 *    overlapping the beginning of the receive window (SegQueue).
 * Case 2: Queue a segment for later delivery if it's sequence number 
 *    is beyond the next expected octet (SaveQueue).
 *
 * Assumes that segment length > 0.
 * Assumes that at least one octet of segment is in rcvWnd.
 */

QSegment(pTcb, curPkt, flag)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    Boolean flag;		/* Indicates if should honor Receive
				   requests. */
  {
    int     diff;
    PktBuf     packet, pkt;
    Bit32 seqNum, rcvNxt;
    extern PrintTcpPktInfo();

    if (curPkt->segLen == 0)
      {
	if (InternetDebug > 3)
	  {
	    printf("QSegment: INEFFICIENCY - tried to queue 0 length segment\n");
	    if (InternetDebug > 5) PrintTcpPktInfo(curPkt->segAdr);
	  }
	DiscardSegment(pTcb, curPkt->segAdr, NULL); 
	return;			/* Don't queue an empty segment. */
      }

    packet = curPkt->segAdr;
    seqNum = TcpPkt(packet)->hdr.sequenceNumber;
    rcvNxt = pTcb->rcvNxt;

    if (seqNum <= rcvNxt)
      {
	/* case 1: segment on or overlapping rcvNxt */
	/* Make pointers refer to data to be queued to host */
	packet->dataptr += curPkt->segDataOffset;
	packet->length = curPkt->segLen;
	if (seqNum < rcvNxt)
	  {
	    /* Trim duplicate octets from front of segment */
	    diff = rcvNxt - seqNum;
	    if (diff <= 0 || diff > packet->length)
	      {
		printf("internetserver last gasp: diff = %d\n",diff);
	        PauseInternetServer("trim error");	/* Internal error */
	      }
	    if (InternetDebug > 5)
		printf("QSegment: trimming %d bytes from front of pkt\n",
		    diff);
	    packet->dataptr += diff;
	    packet->length  -= diff;
	  }
	if (packet->length > pTcb->rcvWnd)
	  {
	    /* Trim excess octets from end of segment */
	    if (InternetDebug > 0)
	      {
		printf("[%d]*** JUMBO segment from %s. Trim to rcvWnd = %u\n",
		    inet_ntoa(curPkt->foreignHost),pTcb->rcvWnd);
		printf("\tsndLen: %8x\tsndSeq: %8x\n",
		    packet->length, seqNum);
		printf("\trcvWnd: %8x\trcvNxt: %8x\n",
		    pTcb->rcvWnd, pTcb->rcvNxt);
	      }
	    packet->length = pTcb->rcvWnd;
	    TcpPkt(packet)->hdr.fin = 0;
	  }
	pTcb->rcvWnd -= packet->length;
	pTcb->rcvNxt += packet->length;
	pTcb->segQBytesAvail += packet->length;
	EnQueue(&(pTcb->q[SegQueue]), packet);
	if (!ActiveTimer(pTcb, AckTimeout))
	  {
	    StartTimer(pTcb, AckTimeout, LenAckTimeout);
				/* Clark's Silly Window Syndrome algorithm
				   would check the psh bit here to avoid
				   processing an extra timeout interrupt.
				   For the current implementation this
				   doesn't yield any benefit. */
	  }
      }
    
    else    /* seqNum > rcvNxt */
      {
        pTcb->numOutOfOrderPkts++;
        if (InternetDebug>4)
	  {
	    printf("[%d]arriving Tcp segment starts beyond rcvNxt\n",
		pTcb->netId);
	  }

	if ((FreeBufferCount < 1) && 
		(TotalBufferCount >= (MaxBufAllocNum - 2)))
	  {
	    DiscardSegment(pTcb, curPkt->segAdr, "Not enough buffers.");	
				/* Can't afford the packet buffer. */
	    return;
	  }

        /* The segment starts beyond rcvNxt.  Store it for later processing
	   on the save queue.  Store in sorted order of sequenceNumber. */
	packet->unspecified[3] = curPkt->segLen;
	packet->unspecified[4] = curPkt->segDataOffset;
	seqNum = TcpPkt(packet)->hdr.sequenceNumber;
	if ((Empty(pTcb->q[SaveQueue])) || 
		(IN(TcpPkt(pTcb->q[SaveQueue].tail)->hdr.sequenceNumber,
			seqNum, (pTcb->rcvNxt + pTcb->rcvWnd))))
	  {
	    EnQueue(&(pTcb->q[SaveQueue]), packet);
	  }
	else if (IN(pTcb->rcvNxt, seqNum,
		TcpPkt(pTcb->q[SaveQueue].head)->hdr.sequenceNumber))
	  {
	    /* Insert packet at head of queue. */
	    packet->next = pTcb->q[SaveQueue].head;
	    pTcb->q[SaveQueue].head = packet;
	  }
	else
	  {
	    /* Find where in the queue the packet belongs.  We already 
	       know that it isn't at the head or tail of it - so no special
	       cases must be handled. */
	    for (pkt = pTcb->q[SaveQueue].head; pkt != NULL; pkt = pkt->next)
	      {
	        if (IN(TcpPkt(pkt)->hdr.sequenceNumber, seqNum,
			(pTcb->rcvNxt + pTcb->rcvWnd)))
		  {
		    packet->next = pkt->next;
		    pkt->next = packet;
		    break;
		  }
	      }
	  }
      }
  }




/*
 * RemoveAckedSendSegments:
 * Removes acked send segments from the retransmission queue.
 *
 * NOTE: only entire segments are removed from the queue.
 */

RemoveAckedSendSegments(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    PktBuf packet, pkt;
    SystemCode statusCode;
    int del;
    register int sum, i;
    int curTime, clicks;
    int pktSendTime = 0;

    statusCode = OK;

    for (packet = pTcb->q[RetransQueue].head; packet != Null;
	    packet = pTcb->q[RetransQueue].head)
      {
        if ((TcpPkt(packet)->hdr.syn) || (TcpPkt(packet)->hdr.fin))
	    del = 1;
	else
	    del = 0;
	if (IN(pTcb->iss, (TcpPkt(packet))->hdr.sequenceNumber + 
		DLength(packet) + del, pTcb->sndUna))
	  {
	    pkt = DeQueue(&(pTcb->q[RetransQueue]));
	    if (pktSendTime == 0)
	      {
		pktSendTime = pkt->unspecified[1];
	      }
	    FreeBuf(pkt);
	  }
	else
	    break;
      }

    if (pktSendTime != 0)	/* Acked at least one pkt on queue */
      {
	curTime = GetTime(&clicks) - Time0;
	curTime = CTime(curTime, clicks) - pktSendTime;
	pTcb->totAckTime += curTime;
	pTcb->retransTimes[pTcb->nextRetransTime++] = curTime;
				/* Time required to ack the packet.*/
        
	if (pTcb->nextRetransTime == NumRetransTimes)
	  {
	    pTcb->nextRetransTime = 0;
	  }
	/* Calculate the average retrans timeout value to use. */
	for (i = 0, sum = 0; i < NumRetransTimes; i++)
	  {
	    sum += pTcb->retransTimes[i];
	  }
	pTcb->lenRetransTimeout =
	    Min((sum >> RetransShift) + 20, MaxLenRetransTimeout);
				/* Add 20 to make timeout just larger than
				   the time required to ack. */
#ifdef TIMERDEBUG
	if (TimerDebug)
	  {
	    printf("[%d]RemoveAckedSendSegments: lenRetransTimeout=%d\n",
		pTcb->netId,
	        pTcb->lenRetransTimeout);
	  }
#endif TIMERDEBUG
      }

    if (pTcb->state >= Estab)
	statusCode = TryToSendToIp(pTcb);

    if (Empty(pTcb->q[RetransQueue]))
      {
	StopTimer(pTcb, RetransTimeout);
      }
    else
      {
	StartTimer(pTcb, RetransTimeout, pTcb->lenRetransTimeout);
#ifdef TIMERDEBUG
	if (TimerDebug)
            printf("[%d]Start RTimer #4\n",pTcb->netId);
#endif TIMERDEBUG
      }

    if ((statusCode != OK) && pTcb->waitSignal)
	NotifyUser(pTcb, statusCode);
  }




/*
 * NotifyUser:
 * if WaitSignal is outstanding then reply to it with statusCode.
 * Assumes that a WaitSignal is outstanding.
 */

NotifyUser(pTcb, statusCode)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    SystemCode statusCode;		/* Status code to return to user. */
  {
    Message msg;

    ((IoReply *)msg)->replycode = statusCode;
    pTcb->wtSignalId = Reply(msg, pTcb->wtSignalId);
    pTcb->waitSignal = False;
  }




/*
 * SendControlSegment:
 * Sends a Tcp control packet to IP.
 * Control packet here means no data is sent.
 * Uses spare buffer set to ensure that the packet can be sent.
 */

SendControlSegment(pTcb, sPort, fPort, fHost, seq, ackFlag, ack, rst, fin)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    Bit16 sPort;		/* source port. */
    Bit16 fPort;		/* destination port. */
    Bit32 fHost;		/* destination host. */
    Bit32 seq, ack;		/* Sequence and ack number to use. */
    Boolean ackFlag;		/* Indicates whether to send an ack. */
    Boolean rst;		/* Value of rst bit field in packet. */
    Boolean fin;		/* Value of fin bit field in packet. */
  {
    struct pbuf Spare;		/* Spare packet buffer used to guarantee that
				   a control segment can be sent. */
    PktBuf SparePacketBuffer = &Spare;
    SystemCode statusCode;

    InitPacketBuffer(sPort, fPort, 
    	    StandardByteDataOffset, StandardByteDataOffset,
	    SparePacketBuffer, seq, ack,
	    ackFlag, False, rst, False, fin, 
	    (pTcb == NULL) ? 0 : pTcb->rcvWnd - pTcb->delRcvWnd);

    statusCode = SendIp(pTcb, SparePacketBuffer, fHost);
				/* Status returned doesn't matter. */
  }




/*
 * SendAckSegment:
 * Send an Ack segment to IP.
 */

SendAckSegment(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    SendControlSegment(pTcb, pTcb->localPort, pTcb->foreignSocket.port,
		pTcb->foreignSocket.host,
		pTcb->sndNxt, True, pTcb->rcvNxt, False, False);
  }
