/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Routines which handle an incoming TCP segment.
 * The implementation used follows that suggested in the
 * Internet Handbook.
 */


#include "net.h"
#include "internet.h"


/*
 * Imported routines
 */

extern Bit32 SelectIss();
extern SystemCode SendSynSegment();
extern PktBuf DeQueue();



/*
 * SegmentArrives:
 * Processes segment arrival event.
 *
 * Must reply to outstanding Receive request if there is one and 
 * state == Estab or further.
 * Must send whatever Send Queue segments can be sent due to change
 * in state of flow control windows.
 *
 * NOTE:  Assumes that bad checksums have already been taken care of.
 */

Boolean SegmentArrives(pTcb, curPkt)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
  {
    TcpPktPtr p;		/* Ptr to start of Tcp packet in buffer. */
    PktBuf packet;

    p = TcpPkt(curPkt->segAdr);
    switch (pTcb->state)
      {
	case Closed: 
	      {
		printf("[%d]**** ERROR **** ",pTcb->netId);
		printf("Allocated Tcp connection in a Closed state!\n");
		DiscardSegment(pTcb, curPkt->segAdr, "Closed state.");
		return(False);
	      }
	case Listen: 
	      {
		DoSegListen(pTcb, curPkt, p);
		break;
	      }
	case SynSent: 
	      {
		DoSegSynSent(pTcb, curPkt, p);
		break;
	      }
	default: 
	      {
		DoSegDefault(pTcb, curPkt, p);
		break;
	      }
      }

    RemoveAckedSendSegments(pTcb);

    /* Check for saved segments that may now be able to arrive */
    if (Empty(pTcb->q[SaveQueue]))
	return(False);
    else
      {
	int seqNum;
	packet = pTcb->q[SaveQueue].head;
	seqNum = TcpPkt(packet)->hdr.sequenceNumber;
	if (seqNum > pTcb->rcvNxt)
	    return(False);
	else
	  {
	    /* Set up pkt for arrival, passing it back in curPkt */
	    packet = DeQueue(&(pTcb->q[SaveQueue]));
	    curPkt->segAdr = packet;
	    curPkt->segLen = packet->unspecified[3];
	    curPkt->segDataOffset = packet->unspecified[4];
	    return(True);
	  }
      }
  }




/*
 * ReplyToConnectionlessPacket:
 * Replies to a packet sent to a non-existent Tcp connection (i.e. a
 * Closed connection).
 */

ReplyToConnectionlessPacket(packet, fHost)
    PktBuf packet;
    Bit32 fHost;
  {
    TcpPktPtr p = TcpPkt(packet);

    if (!p->hdr.rst)
      {
	if (p->hdr.ack)
	  {
	    InitPacketBuffer(p->hdr.destinationPort, p->hdr.sourcePort,
	    	    StandardByteDataOffset, StandardByteDataOffset, packet, 
		    p->hdr.acknowledgementNumber, Null32, False,
		    False, True, False, False, 0);
	  }
	else
	  {
	    InitPacketBuffer(p->hdr.destinationPort, p->hdr.sourcePort,
		    StandardByteDataOffset, StandardByteDataOffset, packet,
		    Null32, p->hdr.sequenceNumber, True,
		    False, True, False, False, 0);
	  }
	SendIp(Null, packet, fHost);
	FreeBuf(packet);
      }
  }




/*
 * DoSegListen:
 */

DoSegListen(pTcb, curPkt, p)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    TcpPktPtr p;		/* Ptr to start of Tcp packet in buffer. */
  {
    SystemCode     statusCode;

 /* 
  * Check for an Rst.
  */

    if(p->hdr.rst)
      {
	DiscardSegment(pTcb, curPkt->segAdr, "Rst set");
	return;
      }

 /* 
  * Check for an Ack.
  */

    if (p->hdr.ack)
      {
	SendControlSegment(pTcb, p -> hdr.destinationPort,
		p -> hdr.sourcePort, curPkt->foreignHost,
		p->hdr.acknowledgementNumber, False, Null32, 
			True, False);
	DiscardSegment(pTcb, curPkt->segAdr,
		"Ack to a listening connection.");
	return;
      }

 /* 
  * Check for a Syn. 
  */

    if (p->hdr.syn)
      {
	if (pTcb->security != curPkt->segSecurity)
	  {
	    SendControlSegment(pTcb, p -> hdr.destinationPort,
		    p -> hdr.sourcePort, curPkt->foreignHost,
		    p->hdr.acknowledgementNumber, False, Null32, 
				True, False);
	    DiscardSegment(pTcb, curPkt->segAdr, "Bad security");
	    return;
	  }
	if (curPkt->segPrc > pTcb->prc)
	    if (pTcb->higherPrcAllowed)
		pTcb->prc = curPkt->segPrc;
	    else
	      {
		SendControlSegment(pTcb, p -> hdr.destinationPort,
			p -> hdr.sourcePort,
			curPkt->foreignHost,
			p->hdr.acknowledgementNumber, False, Null32, 
				True, False);
		DiscardSegment(pTcb, curPkt->segAdr, "Bad precedence");
		return;
	      }
	pTcb->irs = p->hdr.sequenceNumber;
	p->hdr.sequenceNumber += 1;
				/* Adjust for sequence number of syn bit. 
				*/
	pTcb->rcvNxt = p->hdr.sequenceNumber;
	pTcb->foreignSocket.port = curPkt->foreignPort;
	pTcb->foreignSocket.host = curPkt->foreignHost;

	statusCode = SendSynSegment(pTcb, SelectIss(), True, pTcb->rcvNxt);
	pTcb->state = SynRcvd;	/* Adjust state since set to SynSent in
				   SendSynSegment. */
	if (statusCode != OK)
	  {
	    DiscardSegment(pTcb, curPkt->segAdr, "Unsuccessful send of syn");
	    return;
	  }
	HandleOther(pTcb, curPkt, False);
	return;
      }

    /* 
     * Any other control or text-bearing segment
     * (not containing Syn) must have an Ack and thus would
     * be discarded by the Ack processing.  An incoming Rst
     * segment could not be valid, since it could not have
     * been sent in response to anything sent by this 
     * incarnation of the connection.  So it is unlikely to
     * get to here.  Drop the segment and return.
     */

    DiscardSegment(pTcb, curPkt->segAdr, "Should never have gotten here!");
  }




/*
 * DoSegSynSent:
 */

DoSegSynSent(pTcb, curPkt, p)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    TcpPktPtr p;		/* Ptr to start of Tcp packet in buffer. */
  {
    Boolean acceptableAck;	/* Signals whether ack numbers in segment
				   are acceptable. */

 /* 
  * Check the Ack bit. 
  */

    acceptableAck = False;
    if (p->hdr.ack)
	if (!IN(pTcb->iss + 1, p->hdr.acknowledgementNumber, 
		pTcb->sndNxt))
	  {
	    if (!p->hdr.rst)
		SendControlSegment(pTcb, pTcb->localPort,
			pTcb->foreignSocket.port,
			pTcb->foreignSocket.host,
			p->hdr.acknowledgementNumber, False, Null32, 
				True, False);
	    if (InternetDebug > 3)
	      {
		printf("iss = %u;  ack = %u;  sndNxt = %u\n",
		    pTcb->iss, p->hdr.acknowledgementNumber, pTcb->sndNxt);
	      }

	    DiscardSegment(pTcb, curPkt->segAdr, "Bad ack after syn sent");
	    return;
	  }
	else
	    if (IN(pTcb->sndUna, p->hdr.acknowledgementNumber, 
			pTcb->sndNxt))
		acceptableAck = True;

 /* 
  * Check the Rst bit.
  */

    if (p->hdr.rst)
      {
	if (acceptableAck)
	    CloseConnection(pTcb, curPkt, ABORTED,
		"destination rejected connection [RST]");
	else
	    DiscardSegment(pTcb, curPkt->segAdr, "Rst bit set.");
	return;
      }

 /* 
  * Check the security and precedence.
  */

    if (pTcb->security != curPkt->segSecurity)
      {
	if (p->hdr.ack)
	    SendControlSegment(pTcb, pTcb->localPort, pTcb->foreignSocket.port,
		    pTcb->foreignSocket.host,
		    p->hdr.acknowledgementNumber, False, Null32, 
				True, False);
	else
	    SendControlSegment(pTcb, pTcb->localPort, pTcb->foreignSocket.port,
		    pTcb->foreignSocket.host,
		    Null32, True, p->hdr.sequenceNumber + curPkt->segLen, 
		    True, False);
	DiscardSegment(pTcb, curPkt->segAdr, "Bad security");
	return;
      }
    if (p->hdr.ack)
      {
	if (curPkt->segPrc != pTcb->prc)
	  {
	    SendControlSegment(pTcb, pTcb->localPort, pTcb->foreignSocket.port,
		    pTcb->foreignSocket.host,
		    p->hdr.acknowledgementNumber, False, Null32, 
				True, False);
	    DiscardSegment(pTcb, curPkt->segAdr, "Bad precedence");
	    return;
	  }
      }
    else
	if (curPkt->segPrc > pTcb->prc)
	    if (pTcb->higherPrcAllowed)
		pTcb->prc = curPkt->segPrc;
	    else
	      {
		SendControlSegment(pTcb, pTcb->localPort,
			pTcb->foreignSocket.port,
			pTcb->foreignSocket.host,
			Null32, True, p->hdr.sequenceNumber + 
			curPkt->segLen, True, False);
		DiscardSegment(pTcb, curPkt->segAdr, "Bad precedence");
		return;
	      }

 /* 
  * Check the Syn bit.
  * This step should be reached only if the Ack is ok, or there
  * is no Ack, and the segment did not contain a Rst.
  */

    if (p->hdr.syn)
      {
	pTcb->irs = p->hdr.sequenceNumber;
	p->hdr.sequenceNumber += 1;
				/* Adjust for sequence number of syn bit. 
				*/
	pTcb->rcvNxt = p->hdr.sequenceNumber;
	if (acceptableAck)
	  {
	    pTcb->sndUna = p->hdr.acknowledgementNumber;
	    pTcb->sndWnd = p->hdr.window;
	    pTcb->sndWl1 = p->hdr.sequenceNumber;
	    pTcb->sndWl2 = p->hdr.acknowledgementNumber;
	  }
	if (IN(pTcb->iss + 1, pTcb->sndUna, pTcb->sndNxt))
				/* Syn has been acked. */
	  {
	    pTcb->state = Estab;
	    SendAckSegment(pTcb);
	    HandleOther(pTcb, curPkt, True);
	    return;
	  }
	else
	  {
	    pTcb->state = SynRcvd;/* Rely on Retransmission to send syn
				   segment out again. */
	    HandleOther(pTcb, curPkt, False);
	    return;
	  }
      }

 /* 
  * If neither Syn or Rst then drop the segment.
  */

    DiscardSegment(pTcb, curPkt->segAdr, 
    	"Couldn't process packet in synsent state.");
  }




/*
 * DoSegDefault:
 */

DoSegDefault(pTcb, curPkt, p)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    TcpPktPtr p;		/* Ptr to start of Tcp packet in buffer. */
  {
    Boolean acceptable;		/* Signals whether sequence numbers in
				   segment are acceptable. */
    Boolean discardFlag = False;/* True => discard segment at the end. */
    char *errStr;
    register int tmp;		/* Holds intermediate results */

 /* 
  * Check sequence number ############################################# STEP 1
  */

    acceptable = False;
    errStr = "Bad sequence number.";
    if (curPkt->segLen == 0)
	if (pTcb->rcvWnd == 0)
	  {
	    /* Mark as acceptable so that ACK, URG and RST are processed;
	       will later toss when we go to process the segment body. */
	    if (p->hdr.sequenceNumber == pTcb->rcvNxt)
		acceptable = True;
	  }
	else
	  {
	    if (IN(pTcb->rcvNxt, p->hdr.sequenceNumber, 
			pTcb->rcvNxt + pTcb->rcvWnd - 1))
		acceptable = True;
	  }
    else
	if (pTcb->rcvWnd > (Bit16) 0)
	  {
	    /* Acceptable if either the start or the end of the segment
	       is within the current receive window.                   */
	    tmp = pTcb->rcvNxt + pTcb->rcvWnd - 1;
	    if ((IN(pTcb->rcvNxt, 
		    p->hdr.sequenceNumber,
		    tmp)) ||
		(IN(pTcb->rcvNxt, 
		    p->hdr.sequenceNumber + curPkt->segLen - 1,
		    tmp)))
	      {
		acceptable = True;
	      }
	    else
	      {
		errStr = "none of segment in rcvWnd.";
#ifdef TIMERDEBUG
		if (InternetDebug>2)
		  {
	            printf("[%d]None of segment is in rcvWnd!\n",
			pTcb->netId);
		    printf("  segSeq: %8x  segLen: %8x  rcvNxt: %8x  rcvWnd: %8x\n",
			p->hdr.sequenceNumber,curPkt->segLen,
			pTcb->rcvNxt, pTcb->rcvWnd);
		  }
#endif TIMERDEBUG
	      }
	  }
	else
	  {
	    errStr = "rcvWnd is 0.";
	  }
    if (!acceptable)
      {
	if (!p->hdr.rst)
	    SendAckSegment(pTcb);
	if (InternetDebug > 6)
	  {
	    printf("DoSegDefault: tossing unacceptable pkt from %s:\n",
		inet_ntoa(curPkt->foreignHost));
	    printf("  pkt seqNo : %20u  len   : %20u\n",
		p->hdr.sequenceNumber,curPkt->segLen);
	    printf("  tcb rcvNxt: %20u  rcvWnd: %20u\n",
		pTcb->rcvNxt, pTcb->rcvWnd);
	  }
	DiscardSegment(pTcb, curPkt->segAdr, errStr);
	return;
      }

    if (p->hdr.sequenceNumber > pTcb->rcvNxt)
      {
	if ((curPkt->segLen > 0) &&     /* save for data later processing */
	    (TotalBufferCount < MaxBufAllocSaveQ))
	  {
	    QSegment(pTcb,curPkt,True);
	  }
	else				/* let it be retransmitted        */
	    DiscardSegment(pTcb, curPkt->segAdr, errStr);
	return;
      }

 /* 
  * Check the Rst bit ################################################# STEP 2
  */

    if (p->hdr.rst)
	switch (pTcb->state)
	  {
	    case SynRcvd: 
		  {
		    if (pTcb->passiveOpen)
		      {
			FlushQueue(pTcb, SendQueue);
			FlushQueue(pTcb, RetransQueue);
			pTcb->state = Listen;
			pTcb->active = False;
			pTcb->foreignSocket = pTcb->originalForeignSocket;
			DiscardSegment(pTcb, curPkt->segAdr, 
				"Reset a passive connection");
		      }
		    else
			CloseConnection(pTcb, curPkt, ABORTED,
			    "RST received in SynRcvd state");
		    return;
		  }
	    case Estab: 
	    case FinWait1: 
	    case FinWait2: 
	    case CloseWait: 
	    case Closing: 
	    case LastAck: 
	    case TimeWait: 
		  {
		    CloseConnection(pTcb, curPkt, ABORTED,
			"RST received in non-SynRcvd state");
		    return;
		  }
	  }

 /* 
  * Check security and precedence ##################################### STEP 3
  */

    if ((pTcb->security != curPkt->segSecurity) || 
		(curPkt->segPrc != pTcb->prc))
      {
	if (pTcb->state != SynRcvd)
	    CloseConnection(pTcb, curPkt, ABORTED,"Bad security/precedence");
	else
	    DiscardSegment(pTcb, curPkt->segAdr, "Bad security/precedence");
	SendControlSegment(pTcb, pTcb->localPort, pTcb->foreignSocket.port,
		pTcb->foreignSocket.host,
		Null32, True, p->hdr.sequenceNumber + curPkt->segLen, 
		True, False);
	return;
      }

 /* 
  * Check the Syn bit ################################################# STEP 4
  */

    if (p->hdr.syn)
      {
        SendControlSegment(pTcb, pTcb->localPort, pTcb->foreignSocket.port,
		pTcb->foreignSocket.host,
		Null32, True, p->hdr.sequenceNumber + curPkt->segLen, 
		True, False);
	CloseConnection(pTcb, curPkt, ABORTED,
	    "destination sent unexpected SYN");
	return;
      }

 /* 
  * Check the Ack field ############################################### STEP 5
  */

    if (!p->hdr.ack)
      {
	DiscardSegment(pTcb, curPkt->segAdr, "No ack in packet header!");
	return;
      }
    switch (pTcb->state)
      {
	case SynRcvd: 
	    if (IN(pTcb->sndUna, p->hdr.acknowledgementNumber, 
			pTcb->sndNxt))
	      {
		pTcb->state = Estab;
	    /* Continue processing in case Estab. */
	      }
	    else
	      {
		SendControlSegment(pTcb, pTcb->localPort,
			pTcb->foreignSocket.port,
			pTcb->foreignSocket.host,
			p->hdr.acknowledgementNumber, False, Null32, 
				True, False);
		DiscardSegment(pTcb, curPkt->segAdr, 
			"Bad ack number in state synRcvd.");
		return;
	      }
	case Estab: 
	case FinWait1: 
	case FinWait2: 
	case CloseWait: 
	case Closing:
	      {
		if( p->hdr.acknowledgementNumber == pTcb->sndUna )
		  {
		    /* old ack, do nothing */
		  }
		else if( IN( pTcb->sndUna, p->hdr.acknowledgementNumber, 
			pTcb->sndNxt) )
		  {
		    /* new acknowledgement */
		    pTcb->sndUna = p->hdr.acknowledgementNumber;
		    if (IN(pTcb->sndWl1 + 1, p->hdr.sequenceNumber, 
			pTcb->rcvNxt) ||
			((pTcb->sndWl1 == p->hdr.sequenceNumber) &&
			IN(pTcb->sndWl2, 
			p->hdr.acknowledgementNumber, pTcb->rcvNxt)))
		      {
		        pTcb->sndWnd = p->hdr.window;
		        pTcb->sndWl1 = p->hdr.sequenceNumber;
		        pTcb->sndWl2 = p->hdr.acknowledgementNumber;
		      }
		  }
		else if( IN( pTcb->sndUna - 1000000,
			p->hdr.acknowledgementNumber, pTcb->sndUna ) )
		  {
		    /* old ack, do nothing */
		  }
		else
		  {
		    /* acknowledgement for unsent data */
		    SendAckSegment(pTcb);
		    DiscardSegment(pTcb, curPkt->segAdr, 
		    	    "Bad ack number in state >= Estab");
		    return;
		  }

		if (pTcb->state == FinWait1)
		  {
		    if (pTcb->sndUna == pTcb->sndNxt)/* Fin acknowledged */
		      {
			pTcb->state = FinWait2;
		      }
		  }
		else if (pTcb->state == FinWait2)
		  {
		    if ((Empty(pTcb->q[RetransQueue])) && 
			    pTcb->waitSignal)
			NotifyUser(pTcb, END_OF_FILE);
		  }
		else if (pTcb->state == Closing)
		  {
		    if (pTcb->sndUna == pTcb->sndNxt)
			/* Fin acknowledged */
		      {
			pTcb->state = TimeWait;
			StartTimer(pTcb, TmeWaitTimeout, LenTmeWaitTimeout);
		      }
		    else
		      {
			DiscardSegment(pTcb, curPkt->segAdr, 
			      "sndUna != sndNxt in state Closing!!");
			return;
		      }
		  }
		break;
	      }
	case LastAck: 
	      {
	        if (!IN(pTcb->sndUna, p->hdr.acknowledgementNumber,
			pTcb->sndNxt))
		  {
		    SendAckSegment(pTcb);
		  }
		else
		  {
		    pTcb->sndUna = p->hdr.acknowledgementNumber;
		    if (pTcb->sndUna == pTcb->sndNxt)
				/* Fin acknowledged */
		        DeallocateTcb(pTcb, END_OF_FILE);
		  }
		DiscardSegment(pTcb, curPkt->segAdr,
		    "state LastAck - no data.");
		return;
	      }
	case TimeWait: 
	      {
		if (p->hdr.fin)
		  {
		    SendAckSegment(pTcb);
		    StartTimer(pTcb, TmeWaitTimeout, LenTmeWaitTimeout);
		  }
		DiscardSegment(pTcb, curPkt->segAdr,
		    "state TimeWait - no data.");
		return;
	      }
      }

 /* 
  * Check the Urg bit ################################################# STEP 6
  */

    if (p->hdr.urg)
	switch (pTcb->state)
	  {
	    case Estab: 
	    case FinWait1: 
	    case FinWait2: 
		  {
		    pTcb->rcvUrgentFlag = True;
		    pTcb->rcvUrgentPointer =
				p->hdr.sequenceNumber + p->hdr.urgentPointer;
		    if (pTcb->waitSignal)
			NotifyUser(pTcb, BEGIN_URGENT_DATA);
		    break;
		  }
	    default: 
	        /* This should not occur since a Fin has been
		   received from the remote side.  Ignore the Urg. */
		;
	  }

 /* 
  * Process the segment ############################################### STEP 7
  */

    switch (pTcb->state)
      {
	case Estab: 
	case FinWait1: 
	case FinWait2: 
	      {
		if (curPkt->segLen > 0)
		    QSegment(pTcb, curPkt, True);
		else
		    discardFlag = True;
		break;
	      }
	default: 
	      {
		if (InternetDebug)
		   printf("tcpseg: case should not occur - step 7\n");
	        discardFlag = True;
		break;
	      }
      }

 /* 
  * Check the Fin bit ################################################# STEP 8
  */

    if (p->hdr.fin)
      {
	if (pTcb->waitSignal)
	    NotifyUser(pTcb, END_OF_FILE);
	pTcb->rcvNxt = pTcb->rcvNxt + 1;
	SendAckSegment(pTcb);
	switch (pTcb->state)
	  {
	    case SynRcvd: 
	    case Estab: 
		  {
		    pTcb->state = CloseWait;
		    break;
		  }
	    case FinWait1: 
		if (p->hdr.acknowledgementNumber != pTcb->sndNxt)
		  {
		    pTcb->state = Closing;
		    break;
		  }
	        /* else fall through to FinWait2 case. */
	    case FinWait2: 
		  {
		    pTcb->state = TimeWait;
		    StartTimer(pTcb, TmeWaitTimeout, LenTmeWaitTimeout);
				/* Ack already stopped. */
		    break;
		  }
	    case TimeWait: 
		  {
		    StartTimer(pTcb, TmeWaitTimeout, LenTmeWaitTimeout);
		    break;
		  }
	    default: 
		break;
	  }

	if (pTcb->rcvQ != Null)
	  {
	    if (pTcb->segQBytesAvail == 0)
	      {
		ReplyToRead(END_OF_FILE, pTcb->rcvId, NULL, NULL, 0);
		pTcb->rcvQ = NULL;
	      }
	    else
	      {
		/* Should only get here if connection is closed but user
		   process has not yet taken all the output (possibly)
		   just handeded it above */
		if (InternetDebug)
		    printf("tcpseg: should only get here on close! step 8a\n");
		ServiceRcvRequest(pTcb);
	      }
	  }
      }
    if (discardFlag)
	DiscardSegment(pTcb, curPkt->segAdr,NULL);
  }
