/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Protocol-independent part of the internet server.
 */

#include "version.h"
#include "net.h"
#include "internet.h"
#include <Vdirectory.h>
#include <Vgroupids.h>
#include "inquery.h"
#include "stats.h"

/*
 * Miscellaneous globals
 */

ProcessId InternetServerPid;	/* Process id of internet server process */
int LogicalPid=INTERNET_SERVER; /* In case we want to be non-standard */

Boolean keepRunning = True;	/* For shutting down service */
File *d;			/* File for debug output */
char protNames[NumProtocols][10] = { "IP", "TCP", "ICMP" };
int StatsStartTime;		/* Time (secs) when internetserver started */
int StatsUser;			/* User id of person starting internetserver */

/*
 * Spinlock variables: for intra-team locking, measurement, tracking.
 */

SpinLockType PrintfLock = SpinLockUnlocked;
				/* For locking version of printf */
SpinLockType BufferLock = SpinLockUnlocked;
				/* For locking buffer lists */
SpinLockType InstanceLock = SpinLockUnlocked;
				/* For locking network instances */
int InstanceLockHolder = 0;
				/* Identify who holds network instance lock */

int PrintfLockMaxDelay = 0;	/* Max delay (clicks) to service printf */
int PrintfLockAvgDelay = 0;	/* Avg delay (clicks) - last 10 printf */


/*
 * Imported routines and variables.
 */

extern char *malloc();
extern InitPnet();
extern RcvPnet();
extern RcvIp();
extern HandleDirectoryCreate();
extern HandleDirectoryRead();

extern struct FuncBlock FuncTable[];
extern int NetLevelProtocol[];
extern ProcessId icmpPid;
extern Boolean StatsEnabled;
extern int StatsSendInterval;

extern int NumAllocCalls;
extern int NumDeallocCalls;

extern int InputErrorRate;
extern int OutputErrorRate;

/*
 * Forward function declarations
 */

Boolean InvalidFileid();
SystemCode QueryInstance();
PktBuf AllocBuf();


/*
 * InternetServer:
 * Main routine of the internet server.
 * Accepts CreateInstance requests for nonexistent connection instances
 * and QueryInstances.  CreateInstance results in the creation of a separate
 * connection process that will handle all further interactions for that
 * connection.  QueryInstance should normally be handled by the individual
 * connection processes; however the request is also handled here so that
 * clients can "reconnect" to a connection.
 * Also handles NQueryFile requests for connection-independent, protocol-
 * specific information.
 */

InternetServer()
  {
    Message  msg;
    IoRequest *rqMsg = (IoRequest *)msg;
    IoReply *repMsg = (IoReply *)msg;
    CreateInstanceRequest *ciReq = (CreateInstanceRequest *) msg;
    CreateInstanceReply *ciRep = (CreateInstanceReply *) msg;
    ProcessId eventPid;
    unsigned protType;

    while (keepRunning)
      {
	eventPid = Receive(msg);
        switch (rqMsg->requestcode)
	  {
            case CREATE_INSTANCE: 
	        if (ciReq->filemode & FDIRECTORY)
		    HandleDirectoryCreate(msg, eventPid);
		else
		  {
		    protType = ciReq->type;
		    repMsg->replycode = (protType>=NumProtocols) 
		        ? BAD_ARGS
			: FuncTable[protType].CreateConnection(rqMsg,eventPid);
		  }
		break;

	    case QUERY_INSTANCE:
		repMsg->replycode = QueryInstance(rqMsg);
		break;

	    case NQUERY_FILE:
		protType = ((NQueryFileRequest *)msg)->type;
		repMsg->replycode = (protType>=NumProtocols) 
		    ? BAD_ARGS
		    : FuncTable[protType].NQueryProtocol(rqMsg, eventPid);
		break;

	    case SEND_ICMP:
		if (InternetDebug)
		    printf("Main program forwarding ICMP msg (?)\n");
		Forward(msg,eventPid,icmpPid);
		repMsg->replycode = NO_REPLY;
		break;

	    case READ_INSTANCE:
		if (rqMsg->fileid == DirectoryFileId)
		    HandleDirectoryRead(msg, eventPid);
		else repMsg->replycode = MODE_NOT_SUPPORTED;
		break;

	    case RELEASE_INSTANCE:
		if (rqMsg->fileid == DirectoryFileId)
		    HandleDirectoryRelease(msg);
		else repMsg->replycode = MODE_NOT_SUPPORTED;
		break;

	    case NetTimeout:
		if (rqMsg->fileid == DirectoryFileId)
		    HandleDirectoryRelease(msg);
		else repMsg->replycode = MODE_NOT_SUPPORTED;
		break;

	    case InqueryMessage:
		HandleQueryRequest(msg, eventPid);
		break;

	    case StatsMessage | DATAGRAM_SEND_BIT:
	    case StatsMessage:
		repMsg->replycode = HandleStatsRequest(msg, eventPid);
		break;

	    case NetSuicide:
		if (StatsEnabled) SendGlobalStats();
		keepRunning = False;
		if (InternetDebug)
		  {
		    printf("  internetserver shutdown complete;");
		    printf(" China Syndrome avoided - [for now]\n");
		  }
		repMsg->replycode = POWER_FAILURE;
		break;

	    case WRITE_INSTANCE:
	    case WRITESHORT_INSTANCE:
	    case QUERY_FILE:
	    case MODIFY_FILE:
	    case LocalNetPktRcvd:
	    default:
		if (InternetDebug)
		    printf("??? unknown request '%s' received ???\n",
			ErrorString(rqMsg->requestcode));
		repMsg->replycode = REQUEST_NOT_SUPPORTED;
		break;
	  }
	if (repMsg->replycode != NO_REPLY)
	  {
	    Reply(msg, eventPid);
	  }
      }
  }



/*
 * HandleQueryRequest:
 * Process queries for internal status.
 */

HandleQueryRequest(msg, pid)
    register InqueryRequest *msg;
    ProcessId pid;
  {
    register InqueryReply *rep = (InqueryReply *) msg;    
    register IoRequest *rq = (IoRequest *) msg;
    File *f;			/* Where to write results of query */
    SystemCode error;
    int i;


    f = OpenFile(msg->fileserver, msg->fileid, FAPPEND, &error);
    if (error != OK)
      {
	if (InternetDebug > 3) 
	    printf("QueryRequest couldn't open file: %s\n",ErrorString(error));
	rep->replycode = BAD_ARGS;
	return;
      }    

    switch(msg->opcode)
      {
	case 'n':		/* Network instance general info */
	    fprintf(f,"List of active network instances:\n");
	    LOCKID(InstanceLock,InstanceLockHolder,10);
	    for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	      {
		fprintf(f,"\nnetId: %d,  %s (protocol %d)",
		    i, protNames[NetInstTable[i].prot],
		    NetInstTable[i].prot);

		fprintf(f,", rblocksize: %d,  wblocksize: %d\n", 
			    NetInstTable[i].rblocksize,
			    NetInstTable[i].wblocksize);
		fprintf(f,"inUse: %d,  handler pid: %x, owner pid: %x",
			    NetInstTable[i].inUse, 
			    NetInstTable[i].pid,
			    NetInstTable[i].ownerPid);
		fprintf(f,", tcbId = 0x%x\n", NetInstTable[i].tcbId);
	      }
	    UNLOCK(InstanceLock,InstanceLockHolder);
	    break;

	case 'f':		/* Free resources summary */
	  {
	    int numActive = 0;
	    fprintf(f,"Buffers: in use: %d  free/total = %d/%d  ",
		TotalBufferCount-FreeBufferCount,
		FreeBufferCount, TotalBufferCount);
	    fprintf(f,"total allocations = %d\n", NumAllocCalls);
	    if (((TotalBufferCount-FreeBufferCount) !=
		(NumAllocCalls-NumDeallocCalls)) ||
		(FreeBufferCount > TotalBufferCount))
		fprintf(f,"*** BUFFER COUNT MISMATCH!!! ***!\n");
	    LOCKID(InstanceLock,InstanceLockHolder,11);
	    fprintf(f,"Instances: in use: ");
	    for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	      {
		numActive++;
		fprintf(f,"%d  ", i);
	      }
	    UNLOCK(InstanceLock,InstanceLockHolder);
	    fprintf(f," free/total = %d/%d\n",
		MAX_NET_INST-numActive,	MAX_NET_INST);
	    break;
	  }

	case 'V':		/* Version identification */
	  {
	    char wsname[40];
	    QueryWorkstationConfig("name", wsname, 40);
	    fprintf(f, "This is '%s'; internetserver version %s\n",
		wsname, VERSION);
	    break;
	  }

	case 'A':		/* Attach to I/O stream */
	  {
	    File *tmp;
	    SystemCode error;
	    tmp = OpenFile(msg->fileserver, msg->fileid, FAPPEND, &error);
	    if (error != OK)
	      {
		if (InternetDebug > 3) 
		    printf("Attach: couldn't open file: %s\n",
			ErrorString(error));
		rep->replycode = BAD_ARGS;
	      }    
	    d = tmp;
	    break;
	  }

	case 'U':		/* Unattach from I/O stream */
	  {
	    if (d != stdout) Close(d);
	    d = stdout;		/* should make this /dev/null? */
	    break;
	  }

	case 'p':		/* Particulars of a net instance */
	  {
	    i = msg->parm1;
	    if ((i > 0)  && (i < MAX_NET_INST) &&
		(NetInstTable[i].inUse != 0))
	      {
		if (NetInstTable[i].prot == TcpProt)
		    PrintTcpConnectionState(f,NetInstTable[i].tcbId);
		else
		    fprintf(f,"Sorry, don't know how to print details\n");
	      }
	    else fprintf(f,"Ignoring invalid instance number '%d'\n",i);
	    break;
	  }

	case 'I':		/* Input error rate */
	  {
	    fprintf(f,"Forced input errors currently set to 1 per %u pkts\n",
		InputErrorRate);
	    if (msg->parm1 < 0) break;		/* Query only */
	    InputErrorRate = msg->parm1;
	    if (InputErrorRate == 0) InputErrorRate = MAX_SIGNED_INT;
	    fprintf(f,"  [New rate is 1 per %u]\n",InputErrorRate);
	    break;
	  }

	case 'O':		/* Output error rate */
	  {
	    fprintf(f,"Forced output errors currently set to 1 per %u pkts\n",
		OutputErrorRate);
	    if (msg->parm1 < 0) break;		/* Query only */
	    OutputErrorRate = msg->parm1;
	    if (OutputErrorRate == 0) OutputErrorRate = MAX_SIGNED_INT;
	    fprintf(f,"  [New rate is 1 per %u]\n",OutputErrorRate);
	    break;
	  }

	case 'k':		/* Kill (release) an instance */
	  {
	    if ((i = msg->parm1) != 0)
	      {
		rq->requestcode = RELEASE_INSTANCE;
		rq->fileid = i<<1;
		Send(rq, NetInstTable[i].pid);
		if (rq->requestcode != OK) 
		    fprintf(f,"\tCouldn't release (%s)\n",
			ErrorString(rq->requestcode));
	      }
	    break;
	  }

	case 'S':		/* Stats interval change/inspect */
	  {
	    if ((i = msg->parm1) > 0)
	      {
		StatsSendInterval = i;
		fprintf(f,"Setting StatsSendInterval to %d\n",i);
	      }
	    else
	      {
		fprintf(f,"Current StatsSendInterval is %d\n",
		    StatsSendInterval);
	      }
	    break;
	  }

	case 's':		/* semaphore info */
	  {
	    fprintf(f,"InstanceLockCount: %d   BufferLockCount: %d   ",
		SpinLockCount(InstanceLock), SpinLockCount(BufferLock));
	    fprintf(f,"AllocBusyWait: %d\n", AllocBusyWait);
	    fprintf(f,"PrintfLockCount: %d ",SpinLockCount(PrintfLock));
	    fprintf(f,"  PrintfLockMaxDelay: %d  Average (last 10): %d (clicks)\n",
		PrintfLockMaxDelay, PrintfLockAvgDelay);
	    break;
	  }

	case 't':		/* timeout info */
	  {
	    fprintf(f,"CurrentTime: %d,    NextTimeout: %d,    ",
		    CurrentTime, NextTimeout);
	    fprintf(f,"MaxInt: %d\n", MAX_SIGNED_INT);
	    break;
	  }

	case 'e':		/* ethernet info */
	  {
	    PrintPnetStatus(f);
	    break;
	  }

	case 'i':		/* ip reassembly info */
	  {
	    PrintIpReassemblyStatus(f);
	    break;
	  }

	case 'd':		/* set/inspect debug verbosity */
	  {
	    if (msg->parm1 == -1)
		fprintf(f,"Debug verbosity is currently %d\n",InternetDebug);
	    else if ((msg->parm1 >= 0) && (msg->parm1 <= 9))
	      {
		fprintf(f,"Setting debug verbosity to %d\n",msg->parm1);
		InternetDebug = msg->parm1;
	      }
	    break;
	  }

	default:
	    if (msg->opcode != '?')
	        fprintf(f,"Unknown opcode ->%c<- requested\n",
		    (char) msg->opcode);
	    fprintf(f," Use one of:\n");
	    fprintf(f,"\td - set/inspect verbosity of debug msgs\n");
	    fprintf(f,"\tn - general info about net instances\n");
	    fprintf(f,"\tp - details about a particular net instance\n");
	    fprintf(f,"\tf - list free/used resources\n");
	    fprintf(f,"\tk - kill (release) an instance\n");
	    fprintf(f,"\tS - Statistics requests\n");
	    fprintf(f,"\ts - semaphore/lock info\n");
	    fprintf(f,"\tt - timeout info\n");
	    fprintf(f,"\te - ethernet info\n");
	    fprintf(f,"\ti - ip reassembly info\n");
	    fprintf(f,"\tx - exit program\n");
	    fprintf(f,"\tA - attach debug output to new I/O stream\n");
	    fprintf(f,"\tU - unattach debug output (return to stdout)\n");
	    fprintf(f,"\tV - version of internetserver (name/date)\n");
	    fprintf(f,"\tI - (forced) input error rate mod/inspect\n");
	    fprintf(f,"\tO - (forced) output error rate mod/inspect\n");
	    break;
      }    

    rep->replycode = OK;
    Close(f);
  }


/*
 * InitInternetServer:
 * Initializes global data structures and starts the helper processes
 * running.
 */

InitInternetServer(localFlag, debugFlag)
    int localFlag;		/* True if internetserver should be local. */
    int debugFlag;
  {
    int TimeoutTimer(), CleanupTimer(), RcvPnet();
    ProcessId queryId, readerId;
    int i;
    SystemCode error;

    InternetServerPid = GetPid(0, LOCAL_PID);
    srand(InternetServerPid);
    StatsUser = User(InternetServerPid);
    StatsStartTime = GetTime(0);

    InternetDebug = debugFlag;
    d = stdout;			/* Default debug file */
    InputErrorRate = MAX_SIGNED_INT;
    OutputErrorRate = MAX_SIGNED_INT;

    /*
     * Initialize the instance data structures.
     */
    LOCKID(InstanceLock,InstanceLockHolder,12);
    for (i = 0; i < MAX_NET_INST; i++)
      {
	NetInstTable[i].inUse = 0;
	NetInstTable[i].next = i + 1;
      }
    NetInstTable[MAX_NET_INST - 1].next = -1;
    NetInstFree = 0;
    NetInstHead = -1;
    NetInstTable[DirectoryInstance].pid = InternetServerPid;
    NetInstTable[DirectoryInstance].rblocksize = sizeof(InternetDescriptor);
    NetInstTable[DirectoryInstance].filelastblock = MAXUNSIGNED;
    NetInstTable[DirectoryInstance].filelastbytes = MAXUNSIGNED;
    NetInstTable[DirectoryInstance].filenextblock = 0;
    NetInstTable[DirectoryInstance].inUse = 0;
    UNLOCK(InstanceLock,InstanceLockHolder);

    /* 
     * Initialize the protocol-independent modules.
     */
    InitParms();
    InitBufMgr();
    InitPnet();

    /*
     * Initialize the various connection protocols.
     */
    for (i = 0; i < NumProtocols; i++)
      {
	FuncTable[i].InitProtocol();
      }

    /*
     * Initialize helper processes.
     */
    TimerPid = Create(1, TimeoutTimer, 1000);	/* 1000 works */
    timer1Id = Create(3, CleanupTimer, 1000);	/* 1000 works */
    readerId = Create(0, RcvPnet, 3000);	/* 3000 works */
    if ((TimerPid == 0) || (timer1Id == 0) || (readerId == 0))
      {
	printf("Error in creation of an internet helper process\n");
	ExitInternetServer();
      }
    if (!Ready(TimerPid, 0) || !Ready(timer1Id, 0) || !Ready(readerId, 0))
      {
	printf("Error in readying of an internet helper process\n");
	ExitInternetServer();
      }

    /*
     * Now that initialization is complete, register the server.
     */
    if (InternetDebug)
      {
	printf("\nMain internetserver pid = %x; using logical pid %d\n",
	    InternetServerPid, LogicalPid);
      }
    if (localFlag)
	SetPid(LogicalPid, InternetServerPid, LOCAL_PID);
    else
	SetPid(LogicalPid, InternetServerPid, ANY_PID);

    if ((error = JoinGroup( VINTERNET_SERVER_GROUP, 0)) != OK)
      {
	printf("JoinGroup failed : %s\n", ErrorString(error));
	ExitInternetServer();
      }

    InternetServer();
  }




/*** GENERAL SUPPORT ROUTINES ***/

/*
 * printf - (locking version) adapted from standard system version by
 *    addition of locking and average delay constructs.
 */

/* VARARGS1 */
printf(fmt, args)
char *fmt;
{
    register int n;

    n = SpinLockCount(PrintfLock);
    AcquireGlobalSpinLock(PrintfLock);
    n = SpinLockCount(PrintfLock) - n;
    if (n > PrintfLockMaxDelay) PrintfLockMaxDelay = n;
    PrintfLockAvgDelay = (90*PrintfLockAvgDelay + 10*n) / 100;

    _doprnt(fmt, &args, d);
    if (d->type & INTERACTIVE) 
	Flush(d);  /* Flush standard output */

    ReleaseGlobalSpinLock(PrintfLock);
    return(0);
  }



/*
 * InvalidFileid:
 * Checks if the fileid received in a message is reasonable.
 */

Boolean InvalidFileid(rqMsg)
    IoRequest *rqMsg;
  {
    int indx = rqMsg->fileid >> 1;

    switch (rqMsg->requestcode)
      {
	case CREATE_INSTANCE:
	case LocalNetPktRcvd:
	case NetTimeout:
	    return(False);
	case WRITE_INSTANCE:
	case WRITESHORT_INSTANCE:
	case READ_INSTANCE:
	case QUERY_INSTANCE:
	case RELEASE_INSTANCE:
	case QUERY_FILE:
	case MODIFY_FILE:
	    return((indx < 0) ||
		   (indx > MAX_NET_INST) ||
		   (NetInstTable[indx].inUse == 0));
	default:
	    return(True);
      }
  }




/*
 * QueryInstance:
 * Returns the state of the specified network connection instance.
 */

SystemCode QueryInstance(rqMsg)
    QueryInstanceRequest *rqMsg;
  {
    QueryInstanceReply *repMsg = (QueryInstanceReply *) rqMsg;
    int indx = rqMsg->fileid >> 1;

    if (InvalidFileid(rqMsg))
	return(BAD_ARGS);
    repMsg->fileserver = NetInstTable[indx].pid;
    if (rqMsg->fileid & 1)	/* Read instance */
      {
        repMsg->blocksize = NetInstTable[indx].rblocksize;
	repMsg->filetype = READABLE + STREAM + VARIABLE_BLOCK;
	/* NOTE: lastbytes and lastblock have no meaning for a read 
	   instance. */
	repMsg->filelastbytes = MAXUNSIGNED;	/* indefinite length */
	repMsg->filelastblock = MAXUNSIGNED;
	repMsg->filenextblock = NetInstTable[indx].filenextblock;
      }
    else			/* Write instance */
      {
        repMsg->blocksize = NetInstTable[indx].wblocksize;
	repMsg->filetype = WRITEABLE + STREAM + VARIABLE_BLOCK;
	repMsg->filelastbytes = NetInstTable[indx].filelastbytes;
	repMsg->filelastblock = NetInstTable[indx].filelastblock;
	/* NOTE: nextblock has no meaning for a write instance. */
	repMsg->filenextblock = 0;		/* not meaningful */
      }
    return(OK);
  }




/*
 * AllocNetInst:
 * Allocates a network instance descriptor.
 * Returns its index or -1 to signal none left.
 */

int AllocNetInst(prot, ownerPid, pid, rblocksize, wblocksize, tcbId)
    int prot;
    ProcessId ownerPid, pid;
    int rblocksize, wblocksize;
    unsigned tcbId;
  {
    int i;

    LOCKID(InstanceLock,InstanceLockHolder,13);
    if (NetInstFree == -1)
      {
	UNLOCK(InstanceLock,InstanceLockHolder);
	return(-1);
      }
    i = NetInstFree;
    NetInstFree = NetInstTable[i].next;
    NetInstTable[i].next = NetInstHead;
    NetInstHead = i;

    NetInstTable[i].inUse = READ_INST | WRITE_INST;
    NetInstTable[i].ownerPid = ownerPid;
    NetInstTable[i].prot = prot;
    NetInstTable[i].pid = pid;
    NetInstTable[i].tcbId = tcbId;
    NetInstTable[i].rblocksize = rblocksize;
    NetInstTable[i].wblocksize = wblocksize;
    NetInstTable[i].filelastbytes = wblocksize;
    NetInstTable[i].filelastblock = 0;
    NetInstTable[i].filenextblock = 0;
    UNLOCK(InstanceLock,InstanceLockHolder);

    ActivateNetProtocol(NetLevelProtocol[prot]);
    return(i);
  }




/*
 * DeallocNetInst:
 * Deallocates a network connection.
 */

DeallocNetInst(indx)
    int indx;
  {
    int i, netProt;

    if (InternetDebug)
	printf("------ DeallocNetInst: %d ------\n", indx);

    LOCKID(InstanceLock,InstanceLockHolder,14);
    NetInstTable[indx].inUse = 0;
    if (indx == NetInstHead)
      {
	NetInstHead = NetInstTable[indx].next;
      }
    else
      {
	for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	  {
	    if (indx == NetInstTable[i].next)
	      {
		break;
	      }
	  }
	NetInstTable[i].next = NetInstTable[indx].next;
      }

    NetInstTable[indx].next = NetInstFree;
    NetInstFree = indx;

    /* Check if a network-level protocol can be deactivated. */
    netProt = NetLevelProtocol[NetInstTable[indx].prot];

    for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
      {
	if (netProt == NetLevelProtocol[NetInstTable[i].prot])
	  {
	    break;
	  }
      }
    if (i == -1)
      {
	DeactivateNetProtocol(netProt);
      }
    UNLOCK(InstanceLock,InstanceLockHolder);
  }




/*
 * ReplyToRead:
 * Replies to a user client's read request.
 */

ReplyToRead(replycode, pid, packet, bufferptr, len)
    SystemCode replycode;	/* Replycode to send to reader. */
    ProcessId pid;		/* Process id of reader. */
    PktBuf packet;		/* Packet containing data to return to
				   reader.  NULL if no data to return. */
    char *bufferptr;		/* Address of reader's buffer. */
    int len;			/* Length of data to return. */
  {
    Message msg;
    IoReply *repMsg = (IoReply *) msg;
    SystemCode status;
    int i;
    char *t, *f;

    repMsg->replycode = replycode;
    /*
     * Move data into reader's buffer if there is any.
     */
    if (packet == NULL)
	repMsg->bytecount = 0;
    else
      {
        status = OK;
	repMsg->bytecount = len;
	
	if( len > MAX_APPENDED_SEGMENT )
	  {
	    status = MoveTo(pid, bufferptr, packet->dataptr, len);
	  }
	else if ( len > IO_MSG_BUFFER )
	  {
	    ReplyWithSegment( msg, pid, packet->dataptr, bufferptr, len );
	    return;
	  }
	else if (len > 0)
	  {
	    if (DifferentByteOrder(pid))
	      {
		ByteSwapLongCopy((char *)packet->dataptr,
		    (char *)repMsg->shortbuffer, len);
	      }
	    else
	      {
		t = (char *) (repMsg->shortbuffer);
		f = (char *) (packet->dataptr);
		for (i = 0; i < len; i++)
		    *t++ = *f++;
	      }
	  }
	if (status != OK)
	  {
	    repMsg->bytecount = 0;
	    repMsg->replycode = status;
	  }
      }
    pid = Reply(msg, pid);
  }
