/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Support routines for the IP layer.
 */


#include "net.h"
#include "internet.h"


/*
 * Imported routines
 */

extern SystemCode IpRoute();
extern SystemCode WritePnet();
extern PktBuf AllocBuf();
extern ProcessId icmpPid;
extern SpinLockType *InstanceLock;
extern int InstanceLockHolder;

/*
 * Forward declarations
 */

SystemCode FragmentIpPacket();
PktBuf AddNewFragment();
short IpCrc();

struct iphdritem
  {
    IpHdrPtr iphdr;
    struct iphdritem *next;
  };



/*
 * InitIpConnection:
 * Initializes data structures for an Ip connection.
 */

InitIpConnection(pTcb)
    IpTcbPtr pTcb;
  {
    pTcb->receiveBlocked = False;
    pTcb->netId = -1;
    pTcb->instState = True;
    InitSafeQueue(&(pTcb->readerQueue), pTcb->ringBufs);
    InitQueue(&(pTcb->readyQ));
    pTcb->waitFlag = False;
    pTcb->rcvId = 0;
    pTcb->bufferPtr = Null;
  }





/*
 * IpReassemblyTimeout:
 * Handles packet reassembly timeouts.
 * Deallocates the resources being used to reassemble pkt.
 */

IpReassemblyTimeout()
  {
    PktBuf pkt, pkt1,casthdrptr;
    Message msg;
    Requestpak *req = (Requestpak *) msg;
    IpHdrPtr iphdr;
    struct iphdritem *malloc();
    struct iphdritem *firstitem;
    struct iphdritem *lastitem;
    struct iphdritem *newitem;
    struct iphdritem *searchitem;

    firstitem = Null;
    lastitem = Null;
    newitem = Null;
    searchitem = Null;

    if (ReassemblyList.head == Null)
	return;

    for (pkt = ReassemblyList.head; pkt != Null; pkt = pkt1)
      {
        pkt1 = pkt->next;
	if (IpTimeoutVal(pkt) <= CurrentTime)
	  {
	    iphdr = (IpHdrPtr) pkt->dataptr;
	    if(firstitem == Null)
	      {
		firstitem = malloc(sizeof(struct iphdritem));
		if (firstitem == Null)
		  {
		    if (InternetDebug)
			printf("No room to manage Icmp Time Exceeded\n");
  		  }
		else
		  {
		    lastitem = firstitem;
		    firstitem->iphdr = iphdr;
		    firstitem->next = Null;
		    if(InternetDebug) printf("Sending Icmp Time Exceeded\n");
            	    req->requestcode = SEND_ICMP;
            	    req->pktdest = iphdr->srcAdr;
            	    req->icmptype = IcmpTimeExceeded;
		    casthdrptr = (PktBuf) iphdr;
		    req->recdpktptr = casthdrptr;
            	    Send(req,icmpPid); 
		  }
	      }
	    else
	      {
		searchitem = firstitem;
		while ((searchitem != Null) &&
		    (!(SamePacket(iphdr,searchitem->iphdr))))
			searchitem = searchitem->next;
		
		if (searchitem == Null)
		  {
		    if(InternetDebug) printf("Sending Icmp Time Exceeded\n");

            	    req->requestcode = SEND_ICMP;
            	    req->pktdest = iphdr->srcAdr;
            	    req->icmptype = IcmpTimeExceeded;
		    casthdrptr = (PktBuf) iphdr;
		    req->recdpktptr = casthdrptr;
            	    Send(req,icmpPid); 

		    newitem = malloc(sizeof(struct iphdritem));
		    if (newitem == Null)
		      {
		        if (InternetDebug)
			    printf("No room for Icmp Time Exceeded mgmt\n");
		      }
		    else
		      {
		        newitem->iphdr = iphdr;
		        newitem->next = Null;
		        lastitem->next = newitem;
		        lastitem = lastitem->next;
		      }
		  }
	      }

	    RemovePacket(pkt);
	    FreeBuf(pkt);
	    if (InternetDebug)
	      {
		printf("IP PACKET REASSEMBLY TIMEOUT\n");
	      }
	  }
      }
    for (searchitem = firstitem; searchitem != Null; 
		                 searchitem = searchitem->next)
      {
	free(searchitem);
      }
  }




/*
 * PrintIpReassemblyStatus:
 * Prints out status of IP reassembly list.
 */

PrintIpReassemblyStatus(f)
    File *f;
  {
    PktBuf pkt;
    HoleDescrPtr ptr;

    fprintf(f,"CurrentTime: %d.    Packets on the IP reassembly list:\n",
    		CurrentTime);
    for (pkt = ReassemblyList.head; pkt != Null; pkt = pkt->next)
      {
	fprintf(f,"Timeout time: %d.    Holes in packet %x:  ", 
		IpTimeoutVal(pkt), pkt);
	Flush(f);
	for (ptr = (HoleDescrPtr) HoleHdr(pkt); 
		ptr != NULL; 
		ptr = ptr->next)
	  {
	    fprintf(f,"first: %d,  last: %d\n", ptr->first, ptr->last);
	    Flush(f);
	  }
      }
  }




/*
 * IpSendPnet:
 * Fragments the IP packet if necessary and allowed.  Finds the appropriate
 * physical network address to send the packet(s) to; and send(s)
 * it(them).
 */

SystemCode IpSendPnet(packet)
    PktBuf packet;
 {
    IpHdrPtr iph;
    PnetAddr dst;
    SystemCode status;

    iph = (IpHdrPtr) packet->dataptr;

    if( (status = IpRoute(iph->dstAdr, &dst)) != OK )
      {
	return( status );
      }

    if (iph->tl > MaxIpPacketSize)
      {
	if (InternetDebug) printf("Trying to fragment\n");
      }

    if( iph->tl <= MaxIpPacketSize ) 	    /* fragmentation not necessary */
	return( WritePnet( packet, &dst, IpNetProt ) );

    if( iph->flags_fo & MAY_FRAGMENT )      /* fragmentation allowed */
	return( FragmentIpPacket( packet, &dst ) );

    return( BAD_BYTE_COUNT );		    /* fragmentation not allowed */
  }



/*
 * WriteIp:
 * Builds an IP packet with the specified attributes and sends it
 * to the physical network.
 */

SystemCode WriteIp(ipsrc, ipdst, prot, id, tos, ttl, packet, mf, opt)
     IpAddr	ipsrc ;		/* Source address */
     IpAddr	ipdst;		/* Destination address */
     char	prot;		/* Protocol field value */
     int	id;		/* Identifier field value */
     char	tos;		/* Type of service */
     char	ttl;		/* Time to live */
     PktBuf	packet;		/* Packet in which to build the IP
				   packet.  Points to IP data. */
     char mf;			/* May fragment */
     char	*opt;		/* Ptr to IP options - currently not used */
  {
    IpHdrPtr	iph;
    SystemCode statusCode;

    /*
     * Add on room for the IP header.
     */
    packet->dataptr -= 20;
    packet->length += 20;
    /*
     * Fill in the IP header and send the packet off.
     */
    iph = (IpHdrPtr)(packet->dataptr);
    iph->ver_ihl = (CurrentInternetVersion << 4) | 5; /* 5 is hdr length */
    iph->tos = tos;
    iph->tl = packet->length;
    iph->id = id;
    iph->flags_fo = mf ? MAY_FRAGMENT : 0;
    iph->ttl = ttl;
    iph->prot = prot;
    iph->srcAdr = ipsrc;
    iph->dstAdr = ipdst;
    /* Eventually do options processing here. */
    statusCode = IpSendPnet(packet);

    return(statusCode);
  }




/*
 * FindWhichInstance:
 * Finds the IP instance referred to by the designated IP protocol id.
 * Returns -1 if none found.
 */

int FindWhichInstance(prot)
    char prot;
  {
    int i, j = -1;
    IpTcbPtr p;

    LOCKID(InstanceLock,InstanceLockHolder,31);
    for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
      {
	if (NetInstTable[i].prot == IPtype)
	  {
	    p = (IpTcbPtr) NetInstTable[i].tcbId;
	    if (!p->instState)
		continue;	/* Connection is already closed. */
	    if (p->prot == 0)
		j = i;		/* Remember promiscous connection. */
	    if (p->prot == prot)
	      {
		UNLOCK(InstanceLock,InstanceLockHolder);
		return(i);
	      }
	  }
      }
    UNLOCK(InstanceLock,InstanceLockHolder);
    return(j);
  }




/*
 * FragmentIpPacket:
 * Does packet fragmentation and sends the fragments to the physical
 * network.  Fragmentation is performed by breaking the packet into 2
 * packets: the front packet is of maximum allowable size; the second
 * packet is recursively fed to IpSendPnet again.
 */

SystemCode FragmentIpPacket(packet, dst)
    PktBuf packet;
    PnetAddr *dst;
  {
    IpHdrPtr iph, iph1;
    PktBuf pkt;
    int len, hdrLen, nfb;
    SystemCode status;
    int oldPacketLength, oldTl, oldMoreFragments, oldChecksum;
    int savePacketLength;
    char *savePacketDataptr;

    if (InternetDebug)
      {
	printf("IP packet must be fragmented.\n");
      }
    iph = (IpHdrPtr) packet->dataptr;
    /*
     * Get a packet buffer for the back fragment.
     */
    pkt = AllocBuf();
    if (pkt == NULL)
      {
	FreeBuf(pkt);
	return(NO_SERVER_RESOURCES);
      }
    pkt->dataptr += (packet->dataptr - packet->data);
				/* Leave room for physical network headers. */
    /*
     * Package up the front fragment and send it to the physical net.
     */
    hdrLen = (iph->ver_ihl & 0x0f) << 2;
    nfb = (MaxIpPacketSize - hdrLen) / 8;
				/* Number of fragment blocks in front
				   fragment. */
    len = hdrLen + nfb * 8;
    oldPacketLength = packet->length;
    oldMoreFragments = iph->flags_fo & MORE_FRAGMENTS;
    oldTl = iph->tl;
    oldChecksum = iph->checksum;
    packet->length = len;
    iph->flags_fo |= MORE_FRAGMENTS;
    iph->tl = len;
    iph->checksum = 0;
    savePacketLength = packet->length;
    savePacketDataptr = packet->dataptr;
    WritePnet(packet, dst, IpNetProt);
    packet->length = savePacketLength;
    packet->dataptr = savePacketDataptr;
    /*
     * Move header to new packet.
     */
    hdrLen = (iph->ver_ihl & 0x0f) << 2;
		    /* Recompute header length taking into account whether or
       			not to copy various options. */
    Copy(pkt->dataptr, packet->dataptr, hdrLen); 
    pkt->length = hdrLen;
    iph1 = (IpHdrPtr) pkt->dataptr;
    iph1->tl = oldTl - nfb * 8;
    iph1->flags_fo = (iph->flags_fo & FRAGMENT_OFFSET_MASK) + nfb;
    iph1->flags_fo |= oldMoreFragments;
    iph1->checksum = 0;
    /*
     * Move remaining data to new packet.
     */
    Copy((pkt->dataptr + pkt->length),
	    (packet->dataptr + packet->length), 
	    (oldPacketLength - packet->length));
    pkt->length += (oldPacketLength - packet->length);
    /*
     * Recursively handle the back fragment.
     */
    status = IpSendPnet(pkt);
    /*
     * Restore the packet to its original state for the higher layers.
     */
    packet->length = oldPacketLength;
    iph = (IpHdrPtr) packet->dataptr;
    iph->flags_fo |= oldMoreFragments;
    iph->tl = oldTl;
    iph->checksum = oldChecksum;
    return(status);
  }




/*
 * DoReassembly:
 * Does reassembly of packets.  If a packet has been completely
 * reassembled then a ptr to it is returned; otherwise NULL is returned.
 * The reassembly algorithm used is described in the "Internet Protocol
 * Implementation Guide".
 */

PktBuf DoReassembly(packet)
    PktBuf packet;
  {
    PktBuf ptr;
    IpHdrPtr iphdr = (IpHdrPtr) packet->dataptr;

    /*
     * Find out if reassembly has already been started for this IP packet.
     */
    for (ptr = ReassemblyList.head; ptr != NULL; ptr = ptr->next)
      {
        if (SamePacket(iphdr, ((IpHdrPtr) (ptr->dataptr))))
	  {
	    break;
	  }
      }
    if (ptr == NULL)
      {
	/*
	 * Allocate reassembly resources for this packet.
	 */
	ptr = AllocBuf();
	if (ptr == NULL)
	  {
	    FreeBuf(packet);	/* Discard the packet since there are not
				   enough resources to handle it. */
	    return(NULL);
	  }
	/*
	 * Initiate reassembly resources.
	 */
	IpTimeoutVal(ptr) = (unsigned) (CurrentTime + ReassemblyTimeoutLen);
	ptr->length = (iphdr->ver_ihl & 0x0f) << 2;
	Copy(ptr->dataptr, packet->dataptr, ptr->length);
        EnQueue(&ReassemblyList, ptr);
	HoleHdr(ptr) = NULL;
        MakeHole(ptr, IpDataStart(ptr), 0, MAXPBUFSIZE);
      }
    /*
     * Add the current fragment to the packet being reassembled.
     */
    ptr = AddNewFragment(packet, ptr);
    return(ptr);
  }




/*
 * MakeHole:
 * Make a hole descriptor at the indicated location in pkt.
 * All quantities are in terms of bytes.
 */

MakeHole(packet, ptr, first, last)
    PktBuf packet;
    HoleDescrPtr ptr;
    short first, last;
  {
    ptr->first = first;
    ptr->last = last;
    ptr->next = (HoleDescrPtr) HoleHdr(packet);
    HoleHdr(packet) = (unsigned) ptr;
  }




/*
 * AddNewFragment:
 * Updates the hole descriptor list for pkt. and copies the fragment
 * data into the reassembly buffer.
 * Returns true if the pkt has been completely reassembled.
 */

PktBuf AddNewFragment(fragPkt, pkt)
    PktBuf fragPkt, pkt;
  {
    IpHdrPtr iphdr = (IpHdrPtr) pkt->dataptr;
    IpHdrPtr fragHdr = (IpHdrPtr) fragPkt->dataptr;
    HoleDescrPtr ptr, ptrPrev, ptrNext;
    short fragFirst, fragLast;

    ptr = (HoleDescrPtr) HoleHdr(pkt);
    ptrPrev = NULL;
    fragFirst = (fragHdr->flags_fo & FRAGMENT_OFFSET_MASK) << 3;
    fragLast = fragFirst + fragHdr->tl - ((fragHdr->ver_ihl & 0x0f) << 2) - 1;
    /*
     * Check if the new fragment will make the reassembled packet too large.
     */
    if ((((iphdr->ver_ihl & 0x0f) << 2) + fragLast) >= MAXPBUFSIZE)
      {
	/*
	 * Deallocate reassembly resources.
	 */
	RemovePacket(pkt);
	FreeBuf(pkt);
	FreeBuf(fragPkt);
	return(NULL);
      }
    /*
     * Traverse list of hole descriptors to see where fragment lies.
     */
    while (ptr != NULL)
      {
	ptrNext = ptr->next;
	if ((ptr->first > fragLast) || (ptr->last < fragFirst))
				/* No overlap with this hole. */
	  {
	    ptr = ptrNext;
	    ptrPrev = ptr;
	    continue;
	  }
	/*
	 * Fragment overlaps the hole.  Delete the hole descriptor.
	 */
	if (ptrPrev == NULL)
	    HoleHdr(pkt) = (unsigned) ptrNext;
	else
	    ptrPrev->next = ptrNext;
	/*
	 * Add hole descriptors for any holes left in front or back of the
	 * fragment.
	 */
	if (fragFirst > ptr->first)
	  {
	    MakeHole(pkt, ptr, ptr->first, fragFirst - 1);
				/* Hole in front of the fragment. */
	  }
	if ((fragLast < ptr->last) && (fragHdr->flags_fo & MORE_FRAGMENTS))
	  {
	    MakeHole(pkt, IpDataStart(pkt) + fragLast + 1, fragLast + 1,
			ptr->last);
				/* Hole in back of the fragment. */
	  }
	ptr = ptrNext;
      }
    /*
     * Move the fragment data into the reassembly buffer and discard the
     * fragment buffer.
     */
    Copy(IpDataStart(pkt) +
	       ((fragHdr->flags_fo & FRAGMENT_OFFSET_MASK) << 3),
	     IpDataStart(fragPkt),
	     (fragHdr->tl - ((fragHdr->ver_ihl & 0x0f) << 2)));
    FreeBuf(fragPkt);
    /*
     * If reassembly is complete then return the packet.
     */
    if (HoleHdr(pkt) == NULL)
      {
	pkt->length = ((iphdr->ver_ihl & 0x0f) << 2) + fragLast + 1;
	iphdr->tl = pkt->length;
	iphdr->flags_fo &= ~MORE_FRAGMENTS;
	/* Process options here ??? */
	RemovePacket(pkt);
	return(pkt);
      }
    else
      {
	return(NULL);
      }
  }




/*
 * RemovePacket:
 * Remove the specified packet from the reassembly list.
 */

RemovePacket(pkt)
    PktBuf pkt;
  {
    PktBuf ptr;

    if (ReassemblyList.head == NULL)
	return;
    if (ReassemblyList.head == pkt)
      {
        DeQueue(&ReassemblyList);
	return;
      }
    for (ptr = ReassemblyList.head; ptr->next != NULL; ptr = ptr->next)
	if (ptr->next == pkt)
	  {
	    ptr->next = pkt->next;
	    if (ReassemblyList.tail == pkt)
	      {
	        ReassemblyList.tail = ptr;
	      }
	    return;
	  }
    return;
  }




/*
 * IpCrc:
 * Computes a checksum on the IP header
 * Returns the result of the checksum computaion.
 */

short	IpCrc(packet)
    PktBuf	packet;
  {
    register int		headerLength, crc;
    register unsigned short	i, *data;
    IpHdrPtr			iphdr = (IpHdrPtr)(packet->dataptr);

    headerLength = (iphdr->ver_ihl & 0x0f) << 1;
				 /* # short's in the header*/
    data = (unsigned short *)packet->dataptr;
    crc = 0;
    for (i=0 ; i<headerLength ;i++)
      {
#ifdef  LITTLE_ENDIAN
	unsigned short tmp;
	tmp = *data++;
	ByteSwapShortInPlace(&tmp);
	crc += tmp;
#else
        crc += *data++;		/* 2's complement add the next header word*/
#endif  LITTLE_ENDIAN
      }
    /* Add once to get first overflow */
    crc = (crc & 0xFFFF) + (crc >> 16);
    /* Add again to catch overflows caused by above */
    crc += crc >> 16;
    i = (short)crc;
    return ( ~i );
  }
