/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * IP routing module.
 * All IP routing for the internet server is done by the code in this file.
 */


#include "net.h"
#include "internet.h"

/*
 * Gateway Table:
 * A static list of (address prefix, gateway address) pairs is used to route
 * IP datagrams to an appropriate first-hop gateway based on the leading bytes
 * of the destination IP address.  The gateway table is initialized
 * from a network-specific config file.  There is currently no support
 * for multiple gateways for a given prefix; if the gateway for a
 * a particular prefix is down, no other gateway will be tried and
 * those datagrams will be lost.  Neither are "redirects" from gateways
 * honoured.  Obviously, this should be fixed, but it is in keeping with
 * the woefully inadequate specifications for IP routing!
 */

struct GatewayTableEntry
  {
    struct GatewayTableEntry *next;	/* link to chain entries together   */
    IpAddr mask;			/* IP address mask to select prefix */
    IpAddr prefix;			/* IP address prefix (0=>default)   */
    IpAddr gateway;			/* IP address of gateway (0=>local) */
  }
    *GatewayTable = NULL;		/* global ptr to list of gateways */

/*
 * 10 Mb routing table
 */
int RoutingTableTop = 0;	/* Points to front entry of routing table.
				   New entries are added just beyond this
				   pointer, at the "rear" of the table. */

typedef struct IpRoutingTableType
  {
    IpAddr ia;
    PnetAddr ha;
  } *IpRoutingTablePtr;


struct IpRoutingTableType IpRoutingTable[MAX_NET_INST] = {0};
				/* Keep as many ip routing addrs as there are
				   network instances in case each uses a
				   different one. */


/*
 * GetLocalIpHost:
 * Returns the IP address of the local host.  We do not currently support
 * more than one network interface, and therefore there is only one
 * IP address for the local host.  This implementation queries a config
 * file to get the IP address, and if that fails it fabricates one suitable
 * for use on the 3Mb or 10Mb Ethernet in the Stanford Computer Science Dept.
 * This is all a crock -- there should be some kind of gateway query protocol
 * that we can use to get our address.
 */

IpAddr GetLocalIpHost()
  {
    SystemCode error, QueryWorkstationConfig();
    int a, b, c, d;
    char ipaddress[20];
    IpAddr ipAdr;

    if((error = QueryWorkstationConfig( "ip-address", ipaddress, 20 )) == OK)
      {
	if( sscanf( ipaddress, "[%d.%d.%d.%d]", &a, &b, &c, &d ) == 4 )
	  {
	    if( InternetDebug )
		printf("Local host adr=%d.%d.%d.%d\n", a, b, c, d);
	    return( (a << 24) | (b << 16) | (c << 8) | d );
	  }
	else printf( "Warning: Malformed IP address in config file: %s\n",
			ipaddress );
      }
    else printf("Warning: Failed to fetch IP address from config file: %s.\n",
			ErrorString( error ) );

    /*
     * If unable to determine internet address from config file, fabricate one
     * THIS IS STANFORD-DEPENDENT AND SHOULD BE CHANGED FOR OTHER SITES.
     * The third byte of the internet address is set to a non-zero value
     * to reduce the odds that another workstation with the same last
     * byte in the ethernet address will select the same IP address. Since
     * most other (class a) hosts use 0 in byte 3, collisions will only
     * affect the two colliding workstations that are inventing addresses.
     */

    if (PnetType == 10)
	ipAdr = (36 << 24) | ( 8 << 16) | (LocalPnetHost10Mb.addrlow & 0xffff);
    else
	ipAdr = (36 << 24) | (40 << 16) | (1 << 8) | LocalPnetHost3Mb;

    printf("*** Attempting to continue using IP address '%s' fabricated\n",
	inet_ntoa(ipAdr));
    printf("*** from the low order bits of the ethernet address - BEWARE!!\n");
    return(ipAdr);
  }


/*
 * InitGatewayTable:
 * Reads the static list of (address prefix, gateway address) pairs from a
 * file identified in the config file for this host and builds the global
 * gateway table.
 */

InitGatewayTable()
  {
    SystemCode error, QueryWorkstationConfig();
    char line[80], filepath[100];
    char pfxstr[80], gatestr[80];
    IpAddr msk, pfx, gate;
    File *fp;
    struct GatewayTableEntry **p;
    int i, n, b[4];

    strcpy( filepath, "[sys]" );	/* gateway file name context */
    if( (error = QueryWorkstationConfig( "ip-gateways", line, 80)) != OK )
      {
	printf(
	"Warning: Failed to fetch name of IP gateway file from config: %s.\n",
		ErrorString( error ) );
      }
    else if( (fp = Open( strcat( filepath, line), FREAD, &error )) == NULL )
      {
	printf( "Warning: Can't open IP gateway file '%s': %s.\n",
		filepath, ErrorString( error ) );
      }
    else
      {
	p = &GatewayTable;
	while( fgets( line, 80, fp ) != NULL )	/* read lines of file */
	  {
	    if( (n = sscanf( line, "%s %s", pfxstr, gatestr )) < 1 ||
		pfxstr[0] == '\0' || pfxstr[0] == '#' )
		continue;	/* ignore empty or comment lines */

	    if( n == 1 ) goto badline;	/* need two strings on each line */

	    msk = pfx = gate = 0;

	    if( strcmp( pfxstr, "default" ) != 0 )
	      {
		if( (n = sscanf( pfxstr, "%d.%d.%d.%d", b, b+1, b+2, b+3 ))
		    == 0 )
		    goto badline;

		for( i=0; i<n; ++i )
		  {
		    msk |= 0xFF << (24 - (i * 8));
		    pfx |= b[i] << (24 - (i * 8));
		  }
	      }

	    if( strcmp( gatestr, "local" ) != 0 )
	      {
		if( (n = sscanf( gatestr, "%d.%d.%d.%d", b, b+1, b+2, b+3 ))
		    != 4)
		    goto badline;

		gate = (b[0]<<24) | (b[1]<<16) | (b[2]<<8) | b[3];
	      }

	    *p = (struct GatewayTableEntry *)
			malloc( sizeof( struct GatewayTableEntry ) );
	    (*p)->next = NULL;
	    (*p)->mask = msk;
	    (*p)->prefix = pfx;
	    (*p)->gateway = gate;
	    p = &( (*p)->next );
	    continue;

badline:    printf( "Warning: Bad line in IP gateway file '%s': %s",
		    filepath, line );
	  }

	Close(fp);

	if( GatewayTable != NULL )
	  {
	    if( InternetDebug )
	      {
		printf( "IP Gateway Table:\n" );
		for( p = &GatewayTable; *p != NULL; p = &((*p)->next) )
		  {
		    msk = (*p)->mask;
		    pfx = (*p)->prefix;
		    gate = (*p)->gateway;
		    printf( (msk == 0xFF000000) ? "%d" :
			    (msk == 0xFFFF0000) ? "%d.%d" :
			    (msk == 0xFFFFFF00) ? "%d.%d.%d" :
			    (msk == 0xFFFFFFFF) ? "%d.%d.%d.%d" :
						  "default",
			    pfx>>24, (pfx>>16)&0xFF, (pfx>>8)&0xFF, pfx&0xFF);
		    printf( (gate == 0) ? " -> local\n" : " -> %d.%d.%d.%d\n",
			    gate>>24, (gate>>16)&0xFF, (gate>>8)&0xFF,
			    gate&0xFF );
		  }
	      }
	    return;
	  }
	printf( "Warning: No good lines in IP gateway file '%s'.\n",
		filepath );
      }
    printf( "         No gateways will be used.\n" );
  }



/*
 * UpdateIpRoutingTable:
 * Places a new entry into the IP routing table.
 */

UpdateIpRoutingTable(ha, ia)
    PnetAddr *ha;
    IpAddr *ia;
  {
    IpRoutingTablePtr p;
    int i;

    /* Check if an entry already exists in the table. */
    p = &IpRoutingTable[RoutingTableTop];
    for (i = 0; i < MAX_NET_INST; i++)
      {
	if (--p < IpRoutingTable)
	    p = &IpRoutingTable[MAX_NET_INST - 1];

	if (p->ia == *ia) return;
      }

    /* Add new entry to the table. */
    p = &IpRoutingTable[RoutingTableTop];
    p->ia = *ia;
    p->ha = *ha;

    if (InternetDebug > 5)
        printf("Updating IpRoutingTable: host %8x at entry %d\n", 
	    p->ia, RoutingTableTop);

    if (++RoutingTableTop >= MAX_NET_INST)
	RoutingTableTop = 0;
  }



/*
 * InitNetIp:
 * Initializes the network level part of the IP protocol family.
 */

InitNetIp()
  {
    PnetTable[IpNetProt].prot = (PnetType == 10) ? IpEtherType10Mb
						 : IpEtherType3Mb;
    InitQueue(&ReassemblyList);
  }


/*
 * InitIpRouting:
 * Initializes the IP addressing and routing information.
 */

InitIpRouting()
  {
    LocalIpHost = GetLocalIpHost();
    InitGatewayTable();
  }



/*
 * IpRoute:
 * This routine finds the network address for the first hop towards the
 * specified IP address.  Returns NOT_FOUND if unable to make the mapping.
 */

SystemCode IpRoute(ipDst, netDst)
    IpAddr ipDst;
    PnetAddr *netDst;
  {
    struct GatewayTableEntry *g;
    IpRoutingTablePtr p;
    int i, try;
    Enet3Address *e3 = (Enet3Address *) netDst;

    for( g = GatewayTable; g != NULL; g = g->next )
      {					/* search the gateway table */
	if( (ipDst & g->mask) == g->prefix )
	  {
	    if( g->gateway != 0 ) ipDst = g->gateway;
	    break;
	  }
      }

    if( PnetType == 10 )		/* map to 10Mb Ethernet address */
      {
	for( try=0; try<5; ++try )
	  {
	    p = &IpRoutingTable[RoutingTableTop];
	    for (i=0; i<MAX_NET_INST; i++)
	      {
		if (--p < IpRoutingTable)
		    p = &IpRoutingTable[MAX_NET_INST - 1];

		if (p->ia == ipDst)
		  {
		    *netDst = p->ha;
		    return( OK );
		  }
	      }
	    ArpRequest(ipDst);		/* eventually updates IpRoutingTable */
	  }
	return( NOT_FOUND );
      }
    else				/* map to 3Mb Ethernet address */
      {
	*e3 = ipDst & 0xFF;
	return( OK );
      }
  }
