/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Include file for DARPA Internet protocols.
 */



/********************
 *
 * General Definitions
 *
 ********************/


typedef struct IpSocketType	/* Internet host socket definition */
  {
    Bit16 port;
    Bit32 host;
  } IpSocket;




/********************
 *
 * IP Definitions
 *
 ********************/


/*
 * Constant definitions
 */

#define MaxIpPacketSize (MAXPBUFSIZE - 14)
				/* Maximum allowed size of IP packets
				   (leaving room for 10Mb Ethernet header) */

#define IpStackSize 3000	/* 4000 works */
				/* Size of stack for Ip connection processes.
				   */

#define CurrentInternetVersion	4


/*
 * Type definitions
 */

typedef unsigned long IpAddr;	/* IP address type. */
/*
 * The VAX/Unix C compiler always aligns longs on longword boundaries
 * so type IpAddr can't be used in the Arp packet definition.  The
 * following can be used instead.  Foo.
 */
typedef char IpAddrC[4];        /* Used when not on longword boundary */


typedef struct IpTcbType	/* IP instance transmission control block. */
  {
    int netId;			/* Index of the corresponding network instance
				   descriptor record. */
    Boolean instState;		/* State of connection wrt to open/closed. */
    Boolean receiveBlocked;	/* True if connection process is receive
				   blocked (and hence can't dequeue
				   incoming network packets). */
    RingQueue readerQueue;	/* Packet queue used to queue incoming
				   packets from the network. */
    RingBufRec ringBufs[MAX_RING_BUFS];
				/* Ring buffer pool used in conjunction with
				   the readerQueue to provide synchronization
				   between competing processes. */
    char prot;			/* IP protocol id */
    Queue readyQ;		/* Queue of packets waiting to be received. */
    Boolean waitFlag;		/* Flag to signal if a read is 
				   outstanding. */
    ProcessId rcvId;		/* Process id of reader. */
    char *bufferPtr;		/* Address of user's buffer. */
  } IpTcbRec, *IpTcbPtr;

typedef struct iphdr		/* IP packet header structure.
				   See Internet handbook for defns.
				   of the various fields. */
{
    Bit8 ver_ihl;		/* Version and header length */
    Bit8 tos;			/* Type of service */
    Bit16 tl;			/* Total length */
    Bit16 id;			/* Identifier */
    Bit16 flags_fo;		/* Fragment flags and offset */
/* High-order bit not used */
#define MAY_FRAGMENT 0x4000         /* Bit 14 */
#define MORE_FRAGMENTS 0x2000       /* Bit 13 */
#define FRAGMENT_OFFSET_MASK 0x1ff  /* Bits 0-12 */
    Bit8 ttl;			/* Time to live */
    Bit8 prot;
    Bit16 checksum;
    IpAddr srcAdr;
    IpAddr dstAdr;
}
    *IpHdrPtr;

typedef struct HoleDescrStruct	/* Descriptor for an IP fragment hole.
				   These are stored in the actual
				   holes themselves. */
  {
    short first,last;
    struct HoleDescrStruct *next;
  } *HoleDescrPtr;

/*
 * Variable declarations
 */

Queue ReassemblyList;	/* Header for the list of packets
				   currently being reassembled. */

IpAddr LocalIpHost; 		/* Internet address of local host. */


/*
 * Tunable variables - see parms.c for defns.
 */

int ReassemblyTimeoutLen;


/*
 * Macro definitions
 */

#define FragmentedPacket(iphdr) \
  (iphdr->flags_fo & (MAY_FRAGMENT | MORE_FRAGMENTS | FRAGMENT_OFFSET_MASK))
				/* Returns whether the packet represents
				   a fragment or not. */

#define SamePacket(p1,p2) ((p1->srcAdr == p2->srcAdr) && \
			(p1->dstAdr == p2->dstAdr) && \
			(p1->id == p2->id) && \
			(p1->prot == p2->prot))
				/* Returns whether p1 and p2 are parts of
				   the same IP packet. */

#define IpDataStart(pkt) (((char *)pkt->dataptr) + \
			((((IpHdrPtr)pkt->dataptr)->ver_ihl & 0x0f) << 2))
				/* Start of the IP packet's data. */

#define HoleHdr(pkt) pkt->unspecified[0]
				/* Used to store the header of the
				   fragment holes list for a packet
				   being reassembled. */
#define IpTimeoutVal(pkt) pkt->unspecified[1]
				/* Stores ptr to the timeout record
				   associated with a packet being
				   reassembled. */




/********************
 *
 * TCP Definitions
 *
 ********************/


/*
 * Various constants
 */

#define MinPacketDataLength 128	/* Minimum size of Tcp data packet
				   allowed. */

/*
 * TCP Maximum segment size:
 * Making this smaller is a kludge until we fix the bug in the
 * fragment reassembly code WIN November 1983
 */

/*# define MaxPacketDataLength 500*/

#define MaxPacketDataLength (MAXPBUFSIZE - 54)
				/* Maximum size of Tcp data packet
				   allowed.  Must be between 
				   MinPacketDataLength and 
				   MAXPBUFSIZE - 54 . */

#define TcpStackSize 6000	/* 5000 works */
				/* Size of stack to use for Tcp connection
				   processes.  This size must take into 
				   account the fact that the tcb is allocated
				   on the process stack. */

#define StandardByteDataOffset 20
				/* Standard offset of the data in bytes in 
				   a TCP packet (i.e. no options field). */

#define	TCPprot	6		/* Transmission Control Protocol type id */

/*
 * Option code byte values for Tcp options.
 */
#define EndOptionList (Bit8)00
#define NopOption (Bit8)01
#define MaxSegSizeOption (Bit8)02

/*
 * Tcp states.
 */
#define Closed 0
#define Listen 1
#define SynSent 2
#define SynRcvd 3
#define Estab 4
#define FinWait1 5
#define FinWait2 6
#define CloseWait 7
#define LastAck 8
#define Closing 9
#define TimeWait 10

/*
 * Queue enumeration type.
 */
#define NumTcpQueues 4		/* Number of queues for a Tcp connection. */

#define SendQueue 0		/* Queue used to store user's data to be
				   sent. */
#define RetransQueue 1		/* Queue used to store user's data waiting
				   to be retransmitted or acknowledged. */
#define SegQueue 2		/* Queue used to store data that has been
				   sent by the remote host but not yet
				   received by the user. */
#define SaveQueue 3		/* Queue used to store packets whose 
				   sequence numbers are beyond the rcvNxt
				   value for the connection. */

/*
 * Timeout enumeration type.
 */
#define NumTcpTimeouts 4	/* Number of Tcp timeout types. */

#define RetransTimeout 0	/* Retransmission timeout. */
#define TmeWaitTimeout 1	/* Timeout for TimeWait state. */
#define AckTimeout 2		/* Time allowed before an Ack must be
				   sent. */
#define SndWndTimeout 3		/* Retransmission timeout for periodically
				   querying the remote host in 
				   too small window size situations. */

/*
 * Dynamic retransmission timeout defns.
 */
#define NumRetransTimes 8	  /* Number of ack timeout values to use in
				     determining the retrans timeout value. */
#define RetransShift 3		  /* 2 ** RetransShift = NumRetransTimes */

#define MaxRetransTimeoutIndex 10 /* Table size for backoff multipliers */

#define MaxRetransTimeoutShift 8  /* Exponential backoff maximum shift */

/*
 * TCP connection state wrt to release and closing
 */
#define TCP_CONN_RELEASED 1	/* Selects release bit */
#define TCP_CONN_CLOSED 2	/* Selects closed bit */


/*
 * Record formats
 */

typedef struct TcbType		/* Transmission control block.  This
				   record keeps track of all state
				   information for a connection. */
  {
    Boolean receiveBlocked;	/* True if connection process is receive
				   blocked (and hence can't dequeue
				   incoming network packets). */
    RingQueue readerQueue;	/* Packet queue used to queue incoming
				   packets from the network. */
    RingBufRec ringBufs[MAX_RING_BUFS];
				/* Ring buffer pool used in conjunction with
				   the readerQueue to provide synchronization
				   between competing processes. */
    int netId;			/* Index of associated network 
				   instance descriptor. */
    unsigned instState;		/* State of connection instance wrt to release
				   by its owner and state of Tcp connection.
				   bit 1: released;
				   bit 2: closed. */
    ProcessId rcvId;		/* Process id of user Receive call
				   process. */
    ProcessId wtSignalId;	/* Process id of user WaitSignal call
				   process. */
    Bit16 localPort;		/* Local port associated with this
				   connection. */
    IpSocket   foreignSocket;	/* Foreign socket associated with this
				   connection. */
    int     state;		/* Current state of Tcp finite state
				   machine. */
    Boolean passiveOpen;	/* Indicates whether original Open was
				   passive. */
    IpSocket   originalForeignSocket;
				/* Foreign socket this connection was
				   opened with.  Used to restore a
				   connection to passive Listen state. */
    Boolean active;		/* Indicates whether connection is
				   currently active. */
    int     prc;		/* Precedence of connection. */
    Boolean higherPrcAllowed;	/* Indicates whether remote host may
				   specify a higher precedence for the
				   connection. */
    int     security;		/* Security of connection. */
    Boolean waitSignal;		/* Indicates if a WaitSignal call is
				   outstanding. */
    int maxPacketDataLength;	/* Maximum allowed segment size accepted
				   by foreign host. */
    int     rcvByteCnt;		/* Size of user's receive buffer. */
    char *rcvQ;			/* Ptr to user's Receive call buffer. 
				   Null if no Receive call is outstanding.
				*/
    PktBuf lastReadBuf;		/* Ptr to last buffer the user has already
				   received.  Used for handling duplicate
				   read requests. */
    int     segQBytesAvail;	/* Number of data bytes available in
				   segment Q. */
    Queue q[NumTcpQueues];	/* Send, Retrans, Seg, and Save queues. */
    int timers[NumTcpTimeouts];
    Bit32 sndUna;		/* send unacknowledged */
    Bit32 sndNxt;		/* send next */
    Bit16 sndWnd;		/* send window */
    Bit32 sndUp;		/* send urgent pointer */
    Bit32 sndWl1;		/* segment sequence number used for last
				   window update. */
    Bit32 sndWl2;		/* segment acknowledgement number used for
				   last window update. */
    Boolean sndUrgBit;		/* Signals that data is urgent. */
    Bit32 iss;			/* initial send sequence number. */
    Bit32 rcvNxt;		/* receive next */
    Bit16 rcvWnd;		/* receive window */
    Bit16 delRcvWnd;		/* reduction in receive window size */
    Bit32   rcvUrgentPointer;	/* receive urgent pointer */
    Boolean rcvUrgentFlag;	/* tells if urgent pointer is meaningful */
    Boolean rcvUrgentReported;	/* tells if client was informed of urgency */
    Bit32 irs;			/* receive initial sequence number */
    int retransTimes[NumRetransTimes];
				/* acknowledgement time values used to
				   determine a dynamic retrans timeout value.
				   */
    int lenRetransTimeout;	/* Retransmission timout length for the
				   connection. */
    int nextRetransTime;	/* Next slot in retransTimes to store current
				   timeout value in. */
    int user;			/* User id of the process that initiated
				   the connection in the first place */
    unsigned creationTime;	/* When this connection was created -
				   secs since Jan 1 1970 */
    unsigned segmentsSent;	/* Total number of segments sent (excl
				   retransmissions) on this connection */
    unsigned bytesSent;		/* Total number of bytes sent (excl
				    retransmissions) on this connection */
    unsigned numRetransTimeouts;/* number of retransmissions  that
				   occurred on this connection */
    unsigned totAckTime;        /* Total time between when pkt was sent
				   till when it was finally acked */
    unsigned segmentsRcvd;	/* Total number of segments rcvd (excl
				   multiple arrival due to saving) */
    unsigned bytesRcvd;		/* Total number of bytes rcvd (excl
				   multiple arrival due to saving) */
    unsigned numOutOfOrderPkts;	/* number of out of order pkts that
				   have arrived on this connection */
  } *TcpTcbPtr;

typedef struct CurPktType	/* Information associated with incoming
				   segments. */
  {
    Bit32 foreignHost;		/* Foreign host which sent the 
				   segment. */
    Bit16 foreignPort;		/* Foreign port which sent the
				   segment. */
    PktBuf segAdr;		/* Ptr to segment buffer. */
    int	 segLen;		/* segment length */
    int     segPrc;		/* segment precedence */
    int     segSecurity;	/* segment security */
    int     segDataOffset;	/* segment data offset in bytes */
  } CurPktRec, *CurPktPtr;

struct TcpHeader		/* Tcp header format.  See Internet
				   handbook for defns. of fields. */
  {
    Bit16 sourcePort;
    Bit16 destinationPort;
    Bit32 sequenceNumber;
    Bit32 acknowledgementNumber;
#ifdef  LITTLE_ENDIAN
    short unsigned    fin:     1;
    short unsigned    syn:     1;
    short unsigned    rst:     1;
    short unsigned    psh:     1;
    short unsigned    ack:     1;
    short unsigned    urg:     1;
    short unsigned    reserved:     6;
    short unsigned    dataOffset:     4;
#else
    short unsigned    dataOffset:     4;
    short unsigned    reserved:     6;
    short unsigned    urg:     1;
    short unsigned    ack:     1;
    short unsigned    psh:     1;
    short unsigned    rst:     1;
    short unsigned    syn:     1;
    short unsigned    fin:     1;
#endif  LITTLE_ENDIAN
    Bit16 window;
    Bit16 checkSum;
    Bit16 urgentPointer;
  };

typedef struct TcpBufType	/* Buffer type for Tcp packets. */
  {
    struct TcpHeader    hdr;	/* Tcp packet header. */
    Bit8 data[MaxPacketDataLength];
				/* Tcp packet data. */
  } *TcpPktPtr;

/*
 * The VAX/Unix C compiler rounds structures up to a multiple of 4 bytes
 * so the following should be used instead of sizeof(struct TcpBufType)
 * when the actual data length is needed.
 */

#define TcpPkt_Size (sizeof (struct TcpHeader) + MaxPacketDataLength)


/*
 * Tuning variables - see parms.c for defns.
 */

int RcvWindowSize, RcvWindowCutoff;
int LenRetransTimeout, LenTmeWaitTimeout, LenAckTimeout, LenSndWndTimeout;
int MaxSynRetransTimeouts, MaxLenRetransTimeout, MaxRetransTimeoutClicks;
float TcpRetransBackoff[MaxRetransTimeoutIndex];
int MSL, MSL2;


/*
 * Macro declarations
 */

#define DStart(packet) ((((TcpPktPtr) (packet->dataptr)) -> hdr.dataOffset) << 2)
				/* Ptr to start of data portion of a
				   TCP segment. */
#define DLength(packet) (packet->length - DStart(packet))
				/* Length of data portion of a TCP
				   segment. */

#define TcpPkt(packet) ((TcpPktPtr) (packet->dataptr))

#define ActiveTimer(pTcb, t) (pTcb->timers[t] != MAX_SIGNED_INT)
				/* Returns whether timer t is active. */

#define IN(lo,x,hi) (((lo<=x) && (x<=hi)) || ((hi<lo) && ((lo<=x) || (x<=hi))))
				/* Sequence number range check */



/********************
 *
 * ICMP Definitions
 *
 ********************/


#define DATASIZE	8
#define	ICMPprot	1


/* General Icmp Packet structures */


struct IcmpTopHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
};


typedef struct IcmpBufType {
    struct IcmpTopHeader hdr;
} *IcmpPktPtr;


#define IcmpPkt(packet) ((IcmpPktPtr) (packet->dataptr))





/*
 * Define ICMP message types
 */

#define		IcmpEchoReply			0
#define		IcmpDestUnreachable		3
#define		IcmpSourceQuench		4
#define		IcmpRedirect			5
#define		IcmpEcho			8
#define		IcmpTimeExceeded		11
#define		IcmpParameterProblem		12
#define		IcmpTimestamp			13
#define		IcmpTimestampReply		14
#define		IcmpInfoRequest			15
#define		IcmpInfoReply			16
#define		Undefinedtype		        20




/*
 * Echo and Echo Reply definitions
 */


#define		Ecode				0


struct IcmpEchoHeader {
	Bit8 type;
	Bit8 code;
	Bit16 checkSum;
	Bit16 identifier;
	Bit16 seqnum;
};

typedef struct IcmpEchoBufType {
	struct IcmpEchoHeader hdr;
}  *IcmpEchoPktPtr;


#define IcmpEchoPkt(packet) ((IcmpEchoPktPtr) (packet->dataptr))






/* 
 * Destination Unreachable definitions 
 */


#define		DUcodeNetUnreachable		0
#define		DUcodeHostUnreachable		1
#define		DUcodeProtocolUnreachable	2
#define		DUcodePortUnreachable		3
#define		DUcodeFragNeededDFSet		4
#define		DUcodeSourceRouteFailed		5


struct IcmpDUHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit32 unused;
};

typedef struct IcmpDUBufType {
    struct IcmpDUHeader hdr;
    struct iphdr ipheader;
    char data[DATASIZE];
} *IcmpDUPktPtr;


#define IcmpDUPkt(packet) ((IcmpDUPktPtr) (packet->dataptr))
	




/*
 * Source Quench definitions
 */

#define		SQcode				0


struct IcmpSQHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit32 unused;
};

typedef struct IcmpSQBufType {
    struct IcmpSQHeader hdr;
    struct iphdr ipheader;
    char data[DATASIZE];
} *IcmpSQPktPtr;


#define IcmpSQPkt(packet) ((IcmpSQPktPtr) (packet->dataptr))
	




/*
 * Redirect definitions
 */

#define		RcodeNetwork			0
#define		RcodeHost			1
#define		RcodeTypeOfServiceNetwork	2
#define		RcodeTypeOfServiceHost		3


struct IcmpRedirectHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit32 unused;
};

typedef struct IcmpRedirectBufType {
    struct IcmpRedirectHeader hdr;
    struct iphdr ipheader;
    char data[DATASIZE];
} *IcmpRedirectPktPtr;


#define IcmpRedirectPkt(packet) ((IcmpRedirectPktPtr) (packet->dataptr))
	


/*
 * Time Exceeded definitions
 */

#define		TEcodeTimeToLive		0
#define		TEcodeFragReassembly		1


struct IcmpTEHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit32 unused;
};

typedef struct IcmpTEBufType {
    struct IcmpTEHeader hdr;
    struct iphdr ipheader;
    char data[DATASIZE];
} *IcmpTEPktPtr;


#define IcmpTEPkt(packet) ((IcmpTEPktPtr) (packet->dataptr))
	
#define TERetransLimit	2


/*
 * Parameter Problem definitions
 */

#define		PPcodePointerToError		0

struct IcmpPPHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit8 pointer;
    Bit8 unused8;
    Bit16 unused16;
};

typedef struct IcmpPPBufType{
    struct IcmpPPHeader hdr;
    struct iphdr ipheader;
    char data[DATASIZE];
} *IcmpPPPktPtr;

#define IcmpPPPkt(packet) ((IcmpPPPktPtr) (packet->dataptr))




/*
 * Timestamp and Timestamp Reply definitions
 */

#define		TScode				0


struct IcmpTSHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit16 identifier;
    Bit16 seqnum;
    Bit32 originTS;
    Bit32 receiveTS;
    Bit32 transmitTS;
};    


typedef struct IcmpTSBufType {
    struct IcmpTSHeader hdr;
} *IcmpTSPktPtr;


#define IcmpTSPkt(packet) ((IcmpTSPktPtr) (packet->dataptr))




/*
 * Information Request and Information Reply definitions
 */

#define		IRcode				0	


struct IcmpIRHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit16 identifier;
    Bit16 seqnum;
};


typedef struct IcmpIRBufType {
    struct IcmpIRHeader hdr;
} *IcmpIRPktPtr;

#define IcmpIRPkt(packet) ((IcmpIRPktPtr) (packet->dataptr))


/* mask for class A and B networks */

#define NETFIELDA	0x80FFFFFF
#define NETFIELDB	0xC000FFFF



/* Common header */

struct IcmpCommonHeader {
    Bit8 type;
    Bit8 code;
    Bit16 checkSum;
    Bit32 unused;
};

typedef struct IcmpCommonBufType {
    struct IcmpCommonHeader hdr;
    struct iphdr ipheader;
    Bit16 first16;
    Bit16 second16;
    Bit32 next32;
} *IcmpCommonPktPtr;


#define IcmpCommonPkt(packet) ((IcmpCommonPktPtr) (packet->dataptr))


typedef struct DataBufType {
    Bit16 first16;
    Bit16 second16;
    Bit32 next32;
} *DataPtr;


/* More general Icmp information */


#define		IcmpStackSize			4000


typedef struct {
    SystemCode requestcode;
    char icmptype;
    PktBuf recdpktptr;
    IpAddr pktdest;
    ProcessId pid;
} Requestpak;


typedef struct {
    SystemCode replycode;
    char icmptype;
    PktBuf recdpktptr;
    IpAddr pktdest;
    ProcessId pid;
} Replypak;



#define 	SEND_ICMP	0x0051
#define		RECEIVE_ICMP	0x0052
#define		TIMEOUT_ICMP	0x0053

#define		ICMP_PROCESS	20

#define		IcmpPktRcvd	3

int IcmpTimeoutTime;


struct listitem {
    ProcessId savePid;
    int ts;
    IpAddr host;
    struct listitem *next;
};

struct listitem *firstitem;


#define	REPLYTIMEOUT	500
