/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Routines pertaining to the physical network part of the internet
 * server.
 */

#include "net.h"
#include "internet.h"
#include "arp.h"
#include <Vnaming.h>


/*
 * Various other constants
 */

#define AllocDelayTime 50	/* Number of clock ticks to delay before
				   checking to see if a buffer has become
				   available. */


/*
 * Variable declarations
 */

File *FileId;			/* File id for the ethernet device. */
int NumPktDiscards = 0;		/* Number of pkts discarded by reader */
int InputErrorRate;		/* How often to force read errors on input */
int OutputErrorRate;		/* How often to force write errors on output */
int EnetReadCount = 0;	/* Number of calls to Read */
int EnetWriteCount = 0;	/* Number of calls to Write */


/*
 * Imported routines and tables
 */

extern PktBuf AllocBuf();
extern InputErrorRate;
extern OutputErrorRate;




/*
 * QueryPnet:
 * Queries the ethernet device.
 */

SystemCode QueryPnet(request)
    NameRequest *request;
  {
    request->requestcode = NQUERY_FILE;
    request->nameptr = "ethernet";
    request->namelength = sizeof( "ethernet" );
    request->nameindex = 0;
    request->namecontextid = DEFAULT_CONTEXT;

    Send( request, GetPid( DEVICE_SERVER, LOCAL_PID ) );
  }




/*
 * InitPnet:
 * Initialization routine for physical network part of network server.
 * Includes initialization of various network level protocols.
 */

InitPnet()
  {
    Message msg;
    QueryEnetReply *reply = (QueryEnetReply *) msg;
    SystemCode error;
    int i;

    QueryPnet(reply);	/* get physical net parameters */
    if (reply->replycode != OK)
      {
	printf("ERROR - Query of Ethernet device failed: %s\n",
	    ErrorString(reply->replycode));
	ExitInternetServer();
      }
    else
      {
	if( reply->NetworkType == ENET_TYPE_3MBIT )
	  {
	    PnetType = 3;
	    LocalPnetHost3Mb = reply->HostAddress.e3;
				/* Make physical network address of the
				   local host available. */
	    SizeOfEnetHeader = 
		sizeof(Enet3Address) + sizeof(Enet3Address) + 2;
	    *((Enet3Address *) &BroadCastAddr) = 0;
	  }
	else if (reply->NetworkType == ENET_TYPE_10MBIT)
	  {
	    PnetType = 10;
	    LocalPnetHost10Mb = reply->HostAddress.e10;
				/* Make physical network address of the
				   local host available. */
	    SizeOfEnetHeader = 
		sizeof(Enet10Address) + sizeof(Enet10Address) + 2;
	    BroadCastAddr.addrhigh = 0xffff;
	    BroadCastAddr.addrmid = 0xffff;
	    BroadCastAddr.addrlow = 0xffff;
	  }
	else
	  {
	    printf("Unknown ethernet type %d???\n", reply->NetworkType);
	    ExitInternetServer();
	  }
	if (InternetDebug) 
	    printf("Using %dMb ethernet\n", PnetType);
      }

    /*
     * The following special IP initialization must be done BEFORE the
     * Ethernet device is opened because of the perverse kernel behaviour
     * that occurs after the Ethernet is opened but before the reader
     * process is ready.  If the kernel is ever fixed, this should be moved
     * inside InitNetIp.
     */

    InitIpRouting();

    /*
     * Open the physical network device
     */
    FileId = Open( "[device]ethernet", FCREATE+FBLOCK_MODE, &error );
    if ((error != OK) || (FileId == NULL))
      {
	ProcessId pid;
	extern int LogicalPid;
	if ((pid = GetPid(LogicalPid, LOCAL_PID)) != 0)
	  {
	    printf("An internetserver (pid=0x%x, logicalPid=%d)",
		pid, LogicalPid);
	    printf(" is already running locally!!\n");
	  }
	else
	  {
	    printf("Some other process is already using ");
	    printf("'[device]ethernet' on this workstation!\n");
	  }
	printf("Error indication from Open() : %s\n", ErrorString(error));
	Suicide();
      }

    /*
     * Initialize the various network level protocols.
     */
    for (i = 0; i < NumPnetProtocols; i++)
      {
	PnetTable[i].initNetProt();
      }
  }




/*
 * WritePnet:
 * Writes a packet out on the physical network with specified dest.
 * The packet type is determined from the netProt specified.
 */

SystemCode WritePnet(pkt, dst, netProt)
    PktBuf pkt;
    PnetAddr *dst;
    int netProt;
  {
    short unsigned eType;
    int size;
#ifdef  LITTLE_ENDIAN
    extern PktBuf AllocBuf();
    static PktBuf tmpWritePnetBuf = NULL; /* Temp buffer for byte-swapping */
#endif  LITTLE_ENDIAN
    PktBuf outputBuf;       /* pkt or tmpWritePnetBuf */
    char *dataptr;          /* Pointer to enet header */
    int length;             /* Number of bytes to write to ethernet */


    /*
     * At this point, pkt->dataptr should be pointing to the beginning
     * of the IP or ARP data, but there must be sufficient space
     * in the packet buffer to put the Ethernet header before it.
     */

#ifdef  LITTLE_ENDIAN
    if (tmpWritePnetBuf == NULL)    /* Do this the first time only */
      {
	tmpWritePnetBuf = AllocBuf();
	if (tmpWritePnetBuf == NULL)
	  {
	    printf("ERROR - can't alloc tmp buffer\n");
	    ExitInternetServer();
	  }
      }
    /*
     * Copy data to temporary buffer and perform byte-swapping and
     * checksum operations on the copy, so that if this routine is
     * called more than once with the same packet (e.g. packet is
     * retransmitted because no acknowledgement is received) the
     * byte-swapping doesn't happen a second time, thus undoing the
     * results of the first byte-swap.
     */
    outputBuf = tmpWritePnetBuf;
    bcopy((char *)pkt, (char *)outputBuf, &pkt->data[0] - (char *)pkt);
    outputBuf->dataptr = &outputBuf->data[pkt->dataptr - pkt->data];
    bcopy(pkt->dataptr, outputBuf->dataptr, pkt->length);
    /* Make dataptr have the same offset in new packet as in old */
#else
    outputBuf = pkt;
#endif  LITTLE_ENDIAN

    switch(netProt)         /* Do checksumming and byte swapping */
    {
      case IpNetProt:
      {
	IpHdrPtr oldiphdr;
	IpHdrPtr newiphdr;    /* Byte-swapped version (if necessary) */
	char *ipdata;   /* Pointer to whatever follows IP header */
	int iphdrlen;   /* Length of IP header */
	extern short IpCrc();   /* IP checksum routine */
	extern Bit16 CheckSum();/* TCP checksum routine */


	oldiphdr = (IpHdrPtr)pkt->dataptr;
	newiphdr = (IpHdrPtr)outputBuf->dataptr;
	/* Add of length of IP header to get addr of data */
	iphdrlen = (oldiphdr->ver_ihl & 0x0f) << 2;
	/* ipdata is the TCP or UDP (or whatever) packet */
	ipdata = (char *)newiphdr + iphdrlen;
	/* The checksum field is set to 0 for checksum computation */
	newiphdr->checksum = 0;
#ifdef  LITTLE_ENDIAN
	SwapIP(newiphdr);  /* Have to swap before computing checksum */
#endif  LITTLE_ENDIAN
	newiphdr->checksum = IpCrc(outputBuf); /* Checksum of IP header */
#ifdef  LITTLE_ENDIAN
	ByteSwapShortInPlace(&newiphdr->checksum); /* Swap checksum */
#endif  LITTLE_ENDIAN
	switch(oldiphdr->prot)
	  {
	    case TCPprot:   /* Only TCP now supported */
#ifdef  LITTLE_ENDIAN
		SwapTCP((TcpPktPtr)ipdata);
#endif  LITTLE_ENDIAN
		((TcpPktPtr)ipdata)->hdr.checkSum =
		    CheckSum((TcpPktPtr)ipdata,
			   oldiphdr->tl - iphdrlen,
			   LocalIpHost, oldiphdr->dstAdr, TCPprot);
#ifdef  LITTLE_ENDIAN
		ByteSwapShortInPlace((char *)
		   &(((TcpPktPtr)ipdata)->hdr.checkSum));
#endif  LITTLE_ENDIAN
		break;      /* End of TCP */
	  }
	break;      /* End of IP */
      }

      case ArpNetProt:
#ifdef  LITTLE_ENDIAN
	SwapARP((ArpPktPtr)outputBuf->dataptr);
#endif  LITTLE_ENDIAN
	break;

      default:
	if (InternetDebug)
	  {
	    printf("WritePnet: tried to write bogus pkt type %d to ether\n",
		netProt);
	    abort();
	  }
        break;
    }

    if (InternetDebug)
      {
	outputBuf->dataptr -= SizeOfEnetHeader;
	outputBuf->length  += SizeOfEnetHeader;
	PacketCheck(outputBuf); /* Check if the packet has
				   overwritten its boundaries. */
	/* Restore values in case packet is retransmitted */
	outputBuf->length  -= SizeOfEnetHeader;
	outputBuf->dataptr += SizeOfEnetHeader;
      }

    /*
     * Add room for the ethernet header. 
     */
    dataptr = outputBuf->dataptr - SizeOfEnetHeader;
    length = outputBuf->length + SizeOfEnetHeader;

    /* Determine the packet type to use. */
    eType = PnetTable[netProt].prot;

    /*
     * WARNING - BIG CROCK!!!
     * Due to various incorrect implementations of IP/TCP on the Stanford
     * network we only pad out for 10Mb networks.
     */
    if (PnetType == 10)
      {
	if (length < ENET_MIN_PACKET)
				/* Pad out the packet to the minimum
				   allowble length. */
	  {
	    length = ENET_MIN_PACKET;
	  }
      }

    /*
     * Fill in the header and send the packet off.
     */
    if (PnetType == 10)
      {
	register Enet10Packet *buffer10Mb;

	buffer10Mb = (Enet10Packet *)dataptr;
	buffer10Mb->SrcHost = LocalPnetHost10Mb;
/*
 * The following line seems to generate the wrong code on the 68000
 * compiler: it copies the address itself (dst) into buffer10Mb->DestHost
 * instead of the contents (*dst).  Thus it is replaced with the three
 * lines after it.
 */
/*      buffer10Mb->DestHost = *dst;    */
	buffer10Mb->DestHost.addrhigh = dst->addrhigh;
	buffer10Mb->DestHost.addrmid = dst->addrmid;
	buffer10Mb->DestHost.addrlow = dst->addrlow;
	buffer10Mb->EtherType = eType;
      }
    else
      {
	register Enet3Packet *buffer3Mb;

	buffer3Mb = (Enet3Packet *)dataptr;
	buffer3Mb->SrcHost = *((Enet3Address *) &LocalPnetHost3Mb);
	buffer3Mb->DestHost = *((Enet3Address *) dst);
	buffer3Mb->EtherType = eType;
      }

    EnetWriteCount++;
    if (EnetWriteCount % OutputErrorRate == 0)
      {
	size = 0;
	if (InternetDebug > 6)
	    printf("enet: forcing write error %u/%u\n",
		EnetWriteCount,OutputErrorRate);
      }
    else
      {
        size = Write(FileId, dataptr, length);
      }

    if (InternetDebug && 
	    ((size != length && size != 0) || (FileId->lastexception != OK)))
      {
	printf(
	    "ERROR: Write packet to ethernet device failed: %s!\n",
	    ErrorString(FileId->lastexception));
      }

    return(FileId->lastexception);
  }




/*
 * RcvPnet:
 * Packet queueing process.
 * Alternately reads a packet from the physical layer and hands
 * it to the appropriate client.  If no packets are available then a
 * spare packet is used to read packets from the physical layer so that
 * they don't clog up the device buffer.  Contents of the spare packet are
 * thrown away until regular packet buffers become available again.
 */

RcvPnet()
  {
    unsigned short enetType;
    unsigned char *enetData;
    int size;
    PktBuf packet, sparePacket;
    int getNewPkt = 1;
    int nextReassemblyTimeout = CurrentTime + ReassemblyTimeoutLen;
    int i;

    sparePacket = AllocBuf();	
				/* There will always be a packet available
				   at this point because it is startup
				   time. */
    while (1)
      {
	if (!Empty(ReassemblyList))
	  {
	    if (nextReassemblyTimeout <= CurrentTime)
	      {
		IpReassemblyTimeout();	/* Check for timed-out IP fragments */
		nextReassemblyTimeout = CurrentTime + ReassemblyTimeoutLen;
	      }
	  }

        if (getNewPkt)
	  {
	    packet = AllocBuf();
	    if (packet == NULL)
	        packet = sparePacket;
	    getNewPkt = 0;
	  }
	
        size = Read(FileId, packet->data, MAXPBUFSIZE);
	EnetReadCount++;
	if (EnetReadCount % InputErrorRate == 0)
	  {
	    size = 0;
	    if (InternetDebug > 6)
		printf("enet: forcing read error %u/%u\n",
		    EnetReadCount,InputErrorRate);
	  }

	if (PnetType == 10)
	  {
	    enetType = ((Enet10Packet *) (packet->data))->EtherType;
	    enetData = ((Enet10Packet *) (packet->data))->data;
	  }
	else
	  {
	    enetType = ((Enet3Packet *) (packet->data))->EtherType;
	    enetData = ((Enet3Packet *) (packet->data))->data;
	  }

        if (size > 0)
	  {
	    if (packet == sparePacket)
	      {
	        sparePacket = AllocBuf();
		if (sparePacket == NULL)
		  {
		    NumPktDiscards++;
		    if (InternetDebug)
		      {
			printf("*** Tossing good pkt: no spare buffers **\n");
		      }
		    sparePacket = packet;
	            continue;
		  }
	      }
	    for (i = 0; i < NumPnetProtocols; i++)
	      {
		if (PnetTable[i].active &&
			(enetType == PnetTable[i].prot))
		  {
		    getNewPkt = 1;
		    /* Strip off the ethernet header */
		    packet->length = size - SizeOfEnetHeader;
		    packet->dataptr = (char *) enetData;
		    PnetTable[i].rcv(packet);
		    break;
		  }
	      }
	  }
        else if (InternetDebug)
	  {
	    printf("ERROR in Ethernet Read: %d\n", FileId->lastexception);
	  }
      }
  }




/*
 * PrintPnetStatus:
 * Prints various statistics about the physical network connection.
 */

PrintPnetStatus(f)
    File *f;
  {
    Message msg;
    QueryEnetReply *reply = (QueryEnetReply *) msg;
    
    QueryPnet(reply);
    if (reply->replycode != OK)
      {
	fprintf(f,"ERROR from QueryPnet: %s.\n", ErrorString(reply->replycode));
      }
    else
      {
	if( PnetType == 3 )
	  {
	    fprintf(f,"Ethernet host address: 0%o\n", reply->HostAddress.e3);
	  }
	else
	  {
	    fprintf(f,"Ethernet host address: %04x.%04x.%04x\n",
		reply->HostAddress.e10.addrhigh,
		reply->HostAddress.e10.addrmid,
		reply->HostAddress.e10.addrlow);
	  }
	fprintf(f,"Number of valid packets: %d\n", reply->NumValidPackets);
	fprintf(f,"Number of collisions: %d\n", reply->NumCollisions);
	fprintf(f,"Number of overflows: %d\n", reply->NumOverflows);
	fprintf(f,"Number of CRC errors: %d\n", reply->NumCRCErrors);
	fprintf(f,"Number of sync errors: %d\n", reply->NumSyncErrors);
	fprintf(f,"Number of timeouts: %d\n", reply->NumTimeOuts);
      }
    fprintf(f,"\n");
    fprintf(f,"Number of packets discarded by internet reader process: %d\n",
    		NumPktDiscards);
  }
