/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * V Prototype command interpreter
 */

#include "Vnamecache.h"
#include "Vio.h"
#include <Venviron.h>
#include <Vsession.h>
#include <Vteams.h>
#include <Vioprotocol.h>
#include <Vtermagent.h>
#include <Vexceptions.h>
#include "sh.h"
#include <chars.h>
#include <Vexec.h>
#include <Vauthenticate.h>
#include <Vgroupids.h>

#ifdef XEXEC
#define EXEC_LOGO "\f\r\n\nExperimental V System Executive, Version 6.1\r\n\n"
#else
#define EXEC_LOGO "\f\r\n\nV System Executive, Version 6.0\r\n\n"
#endif

#define FALSE 0
#define TRUE 1

#define DEFAULT_PROMPT		"! "		/* Initial prompt */

ProcessId SetUpPipe();
SystemCode RunProgs(), RedefineEnvironment();

extern short ExecDebug;
extern SystemCode ChangeDirectory();
extern char *getenv();

static SystemCode AuthExec(), DeAuthExec(), ResetContext();
static void LoginCommand();

/* aliases are global, held in common among all execs */
AliasRec aliases[MaxAliases];

/* data structures for the prompt feature */
char	prompt[LineLength] = DEFAULT_PROMPT;
				/* prompt format (may contain "%d" ) */

SelectionRec HostSpec;	/* Used to specify where to execute each command. */
ProcessId execPid = 0;	/* Set in ExecServer */
int SeparateExecProgram;/* nonzero iff the exec is run as a separate program */

Exec(in, out, err, shIn, mydescriptor)
    File *in, *out, *err, *shIn;
    struct ExecDescriptor *mydescriptor;
  {
    char *args[MaxArgs];		/* array of argument pointers */
    ProcessId pidList[(MaxArgs>>2)+1];	/* programs started up by RunProgs */
    char lineBuf[LineLength];
    char banner[LineLength];		/* temp. banner string */
    char **argv;			/* pointer into args array */
    Message rmsg, dmsg;
    SystemCode error;
    char *errstr;
    unsigned i, concurrent, arbHost;
    int inLen, inLen1, argc;
    char *expandPtr;
    char expandBuf[ExpandBufLength];
    PtrRec BufPtrs;
    ContextPair currentExecutionContext;
    char *currentExecutionContextName = NULL;
    HistoryRec hRec, *h = &hRec;
    register RootMessage *rtMsg = (RootMessage *) rmsg;
	/* new team's root message */
    RootMessage *debugRtMsg = (RootMessage *) dmsg;
	/* root message for new team's postmortem debugger. */
    short silent, echo;
	/* flags of special behavior for noninteractive execs */
    int lastStatus = 0;
    	/* exit value of last command */
    int oldpadmode, oldoutpadmode;

#ifdef EXECSERVER
    silent = mydescriptor->ioflags & EXEC_NOPROMPT;
    echo = mydescriptor->ioflags & EXEC_ECHO;
#else
    silent = echo = FALSE;
#endif


    if (shIn == NULL)
	shIn = in;    /* read commands from "in" unless "shIn" is non-null */

    oldpadmode = QueryPad(shIn);
    oldoutpadmode = QueryPad(out);
    ModifyPad(shIn, CR_Input+LF_Output+Echo+LineBuffer+PageOutput);
    ModifyPad(out, CR_Input+LF_Output+Echo+LineBuffer+PageOutput);
				/* Set input & output cooking to that expected
				   by this exec. */

    /* get up to date */
    Resynch(in);
    Resynch(out);
    Seek(out, 0, FILE_END);
    Resynch(err);
    Seek(err, 0, FILE_END);
    Resynch(shIn);

    if (!SeparateExecProgram) SetBreakProcess(in, 0);
    if (!silent) SetVgtBanner(out, "");
    
    InitHistory(h);
    InitBuiltInCmds();
#ifndef EXECSERVER
    ChangeDirectory("[home]");
#endif

    /* The "current execution context", unlike the regular current
     * (name lookup) context, isn't inherited from our creator (although it
     * probably should be).  Thus, we have to set it ourselves here.  
     */
    if (SeparateExecProgram)
	/* An exec that was started up as a separate program.  Crock: In this 
	 * case we assume that we want to remain on the same host as this exec.
	 */
	ResetContext("[team/local", &currentExecutionContext, 
		     &currentExecutionContextName);
    else
	/* An exec that was created by the exec server (i.e. usually in its own 
	 * view).
	 */
	ResetContext("[homex]", &currentExecutionContext, 
		     &currentExecutionContextName);

    if (!silent) fprintf(out, EXEC_LOGO);

    for (;;)
      {

errLabel:
	/* Continue here to get the next command, or 
	 * on an error in a command */

        if (!SeparateExecProgram) SetBreakProcess(in, 0);
	if (!silent) SetVgtBanner(out, "");
#ifdef EXECSERVER
	mydescriptor->status = EXEC_FREE;
#endif
	if (!silent)
	  {
	    ModifyPad(shIn, CR_Input+LF_Output+Echo+LineBuffer+PageOutput);
	    ModifyPad(out, CR_Input+LF_Output+Echo+LineBuffer+PageOutput);
				/* Set input & output cooking to that expected
				   by this exec. */
	    fprintf( out, prompt, h->nextCmdnum );
	  }

	Flush(out);

	/* Get the next command line.  Editing and historical references
	   are provided. */
	inLen = 0;
	for (;;)
	  {
	    /* Get new input (editing is provided). */
	    inLen = GetLine(lineBuf, inLen, LineLength, shIn, out);
	    if (inLen == -1)
	      {
		/* EOF has been signalled. */
		ModifyPad(shIn, oldpadmode);
		ModifyPad(out, oldoutpadmode);
		return (lastStatus);
	      }
	    if (echo) fprintf(out, "%s\n", lineBuf);
	    Flush(out);
	    if (inLen == 0)
	      {
		/* No input received - start over. */
		goto errLabel;
	      }
	    else if (lineBuf[inLen-1] == BELL)
	      {
		fprintf(err, " XXX\n");
		goto errLabel;
	      }

	    /* Check for history references.  If history is invoked, then
	       redisplay the designated line for further editing. */
	    inLen1 = DoHistory(lineBuf, h);
	    if (inLen1 == -1)	/* Bad history ref. - start over. */
	      {
		fprintf(err, "Command not part of recent history.\n");
		goto errLabel;
	      }
	    else if (inLen1 == 0)
	      {
		/* No history ref. - get out of input loop. */
		break;
	      }
	    else	      
	      {
		/* History ref. - display designated line for user to edit. */

 		fprintf( out, prompt, h->nextCmdnum );

		inLen = inLen1;
		continue;
	      }
	  }

	/* Replace aliases. */
	if (!DoAliases(lineBuf, aliases, &error))
				/* Returns 1 if should continue - i.e. NOT an
				   alias defn. */
	  {
	    if (error != OK)	/* error only has meaning on an alias defn. */
	      {
		fprintf(err, "No room for a new alias defn.\n");
	      }
	    continue;		/* Done with this cmd. */
	  }

	/* Parse the command line into separate words. */
	BufPtrs.argc = 0;
	BufPtrs.argv = args;
	BufPtrs.expandBuf = BufPtrs.expandPtr = expandBuf;
	if (ParseExecLine(lineBuf, &BufPtrs, &errstr) == -1)
	  {
	    fprintf(err, "%s\n", errstr);
	    goto errLabel;
	  }
	if ((argc = BufPtrs.argc) == 0) continue;

	/* Do environment variable substitution */
	for (argv = args; *argv != NULL; argv++)
	  {
	    char *v;

	    if (**argv == '$') 
	      {
	        v = getenv((*argv) + 1);
		if (v) *argv = v;
	      }
	  }

	argv = args;

	/* Handle built-in commands */
	if (BuiltInCmd(args, argc, h, aliases, 
		       in, out, err, mydescriptor,
		       &currentExecutionContext, &currentExecutionContextName))
	    continue;		/* Done with this cmd. */

	/* Set default values */
	concurrent = arbHost = FALSE;
	rtMsg->stdinserver = in->fileserver;
	rtMsg->stdinfile = in->fileid;
	rtMsg->stdoutserver = out->fileserver;
	rtMsg->stdoutfile = out->fileid;
	rtMsg->stderrserver = err->fileserver;
	rtMsg->stderrfile = err->fileid;
	rtMsg->rootflags = 0;

	*debugRtMsg = *rtMsg;

	DefaultSelectionRec(&HostSpec);
	/* Get the default team server pid from our current execution ctx */
	HostSpec.teamServerPid = currentExecutionContext.pid;
	/* (We don't yet use the "cid" field, but someday we should). */

	/* Check for command environment redefinitions. */
	error = RedefineEnvironment(&argc, args, &concurrent, &arbHost,
				    &HostSpec, rtMsg, out);
	if (error != OK)
	  {
	    lastStatus = error;
	    goto errLabel;
	  }

	if (!silent)
	  {
	    sprintf(banner,"%.100s (loading)", argv[0] );
	    SetVgtBanner(out, banner);
	  }
#ifdef EXECSERVER
	mydescriptor->status = EXEC_LOADING;
#endif
	error = RunProgs(argv, concurrent, arbHost, &HostSpec,
			 rtMsg, debugRtMsg, out, pidList);
	if( error != OK )
	  {
	    switch (error)
	      {
		case NOT_FOUND:
		    errstr = "File \"%s\" not found";
		    break;
		case NO_PERMISSION:
		    errstr = "Permission denied on \"%s\"";
		    break;
		case NO_MEMORY:
		    errstr = "Not enough memory for \"%s\"";
		    break;
		case BAD_STATE:
		case END_OF_FILE:
		    errstr = "File \"%s\" not runnable";
		    break;
		case SERVER_NOT_RESPONDING:
		    errstr = "Couldn't find a suitable host to execute \"%s\"";
		    break;
		default:
		    errstr = "Unable to execute \"%s\": %s";
		    break;
	      }
	    fprintf( err, errstr, lineBuf, ErrorString(error) );
	    fprintf( err, "\r\n" );
	    lastStatus = error;
	    goto errLabel;
	  }

	  /*
	   * This is when the new loaded team actually runs.
	   * We wait for it to terminate by calling Wait
	   * (which does a ReceiveSpecific on the
	   * rootPid of the team), providing it is not running
	   * concurrently.
	   */
	if ( concurrent )
	  {
	    if (!silent)
	      { ProcessId *pid = pidList;
	        fprintf( out, "pid = %x", *pid++ );
		while (*pid != 0)
		    fprintf( out, ", %x", *pid++ );
		fputc('\n', out);
		Flush( out );
	      }
	  }
	else
	  { ProcessId *pid = pidList;
	    if (!SeparateExecProgram) SetBreakProcess(in, pidList[0]);
	    		/* good enough, I think */
	    if (!silent) SetVgtBanner(out, argv[0]);
#ifdef EXECSERVER
	    mydescriptor->status = EXEC_RUNNING;
	    mydescriptor->programPid = pidList[0];	/* ??? */
#endif
	    while (*pid)
		Wait(*pid++, &lastStatus);
	    Resynch(in);
	    Resynch(out);
	    Seek(out, 0, FILE_END);
	    Resynch(err);
	    Seek(err, 0, FILE_END);

	    if (lastStatus == -1)
	      {
	        fprintf(err, "\r\nProcess 0x%x died!\r\n", *(pid - 1));
		Flush(err);
	      }
	    if (!silent)
	      {
		fprintf(out, "\r\n");
		Flush(out);
	      }
	  }
      }
  }




/*
 * InitBuiltInCmds:
 */

InitBuiltInCmds()
  {
  }




/*
 * BuiltInCmd:
 * Checks if args represents a built-in shell command.  If so, it is 
 * executed and the function returns TRUE.  Otherwise FALSE is returned.
 */

int BuiltInCmd(args, argc, h, aliases, in, out, err, mydescriptor,
	       currExCtx, currExCtxName)
    register char **args;
    int argc;
    HistoryRec *h;
    AliasRec aliases[];
    File *in, *out, *err;
    struct ExecDescriptor *mydescriptor;
    ContextPair *currExCtx;
    char *currExCtxName[];
  {
    SystemCode r;

    if( Equal( args[0], "setprompt" ) )
      {
	SetPrompt( argc, args, err );
	return(TRUE);
      }
    if( Equal( args[0], "history" ) )
      {
        PrintHistory(h, out);
        return(TRUE);
      }
    if( Equal( args[0], "alias" ) )
      {
        PrintAliases(aliases, out);
	return(TRUE);
      }
    if( Equal( args[0], "unalias" ) )
      {
	RemoveAlias(args[1], aliases);
	return(TRUE);
      }

    if( Equal( args[0], "cd" ) )
      {
	if (args[1] == NULL)
	    r = ChangeDirectory("[home]");
	else
	    r = ChangeDirectory(args[1]);
	if (r != OK)
	  {
	    fprintf(err, "%s: %s\n", (args[1] ? args[1] : "[home]"),
	    		ErrorString(r));
	  }
	return(TRUE);
      }
    if( Equal( args[0], "cx" ) )
      {
	if (args[1] == NULL)
	    r = ResetContext("[homex]", currExCtx, currExCtxName);
	else
	  { int c = args[1][0];
	    r = ResetContext(args[1], currExCtx, currExCtxName);
	    if (r != OK && c != '\0' && c != '[' && c != '/')
	      {
		static char *StdCtxPrefix = "[team/";
		char *buf =
		     (char*)malloc(strlen(StdCtxPrefix)+strlen(args[1])+1);
		sprintf(buf, "%s%s", StdCtxPrefix, args[1]);
		r = ResetContext(buf, currExCtx, currExCtxName);
		free(buf);
	      }
	  }
	if (r != OK)
	  {
	    fprintf(err, "%s: %s\n", (args[1] ? args[1] : "[homex]"),
	    		ErrorString(r));
	  }
	return(TRUE);
      }

    if( Equal( args[0], "setenv" ) )
      {
	if (args[1] == NULL)
	  {
	    fprintf(err, "Usage: setenv <variable> [<value>]\n");
	    fflush(err);
	  }
	else if (args[2] == NULL) 
	    setenv(args[1], "");
	else
	    setenv(args[1], args[2]);

	return TRUE;
      }

    if( Equal( args[0], "unsetenv" ) )
      {
	if (args[1] == NULL)
	  {
	    fprintf(err, "Usage: unsetenv <variable>\n");
	    fflush(err);
	  }
	else
	    setenv(args[1], NULL);

	return TRUE;
      }

    if( Equal( args[0], "printenv" ) )
      {
        if (args[1] == NULL)
	  {
	    EnvironmentVariable *ev;

	    ev = *PerProcess->env;
	    while (ev)
	      {
		fprintf(out, "%s=%s\n", ev->name, ev->value);
		ev = ev->next;
	      }
	  }
	else
	  {
	    char *value;
	    
	    if ( (value = getenv(args[1])) != NULL )
	        fprintf(out, "%s\n", value);
	  }
	return TRUE;
      }

    if( Equal( args[0], "define" ) )
      {
	if (args[1] == NULL || args[2] == NULL)
	  {
	    fprintf(err, "Usage: define <newname> <oldname>\n");
	    fflush(err);
	  }
	else
	  {
	    r = DefineLocalName(args[1], args[2]);
	    if (r != OK)
	      {
	        fprintf(err, "%s: %s\n", args[2], ErrorString(r));
		fflush(err);
	      }
	  }

	return TRUE;
      }

    if( Equal( args[0], "undefine" ) )
      {
	if (args[1] == NULL)
	  {
	    fprintf(err, "Usage: undefine <name>\n");
	    fflush(err);
	  }
	else
	  {
	    r = UndefineLocalName(args[1]);
	    if (r != OK)
	      {
	        fprintf(err, "%s: %s\n", args[1], ErrorString(r));
		fflush(err);
	      }
	  }

	return TRUE;
      }

    if( Equal( args[0], "printdef" ) )
      {
	if (args[1] == NULL)
	  {
	    NameCacheEntry *ce;

	    ce = *PerProcess->namecache;
	    while (ce)
	      {
		if (ce->flags & ALIAS)
		    fprintf(out, "%s=%s\n", ce->name, ce->truename);
		ce = ce->next;
	      }
	  }
	else
	  {
	    extern char *ResolveLocalName();
	    char *value;

	    if ( (value = ResolveLocalName(args[1])) != NULL )
	        fprintf(out, "%s\n", value);
	    else
	      {
	        fprintf(err, "%s: not found\n", args[1]);
		fflush(err);
	      }
	  }
	return TRUE;
      }

    /* pwd would not have to be built in, but trivial to implement here */
    if( Equal( args[0], "pwd" ) )
      {
	fprintf(out, "%s\n", PerProcess->ctxname);
	return TRUE;
      }

    if( Equal( args[0], "pwx" ) )
      {
	fprintf(out, "%s\n", *currExCtxName);
	return TRUE;
      }

    if( ( Equal( args[0], "login" ) ) || ( Equal( args[0], "su" ) ) )
      {
	LoginCommand(args, in, out, err, mydescriptor,
		     currExCtx, currExCtxName);
	return TRUE;
      }

    if( Equal( args[0], "logout" ) )
      {
        int i;

	if (SeparateExecProgram)
	  {
	    exit(0);
	  }
	else
	  {
	    if (argc == 1)
		DeAuthExec( "" );
	    else
		for (i = 1; i < argc; i++) DeAuthExec(args[i]);
	  }

	return(TRUE);
      }


    return(FALSE);
  }

static SystemCode AuthExec(name, password)
char *name, *password;
  {
    char buf[100];
    MsgStruct msg;
    
    sprintf(buf, "%s:%s", name, password);
    msg.sysCode = AUTH_EXEC;
    msg.segmentPtr = buf;
    msg.segmentSize = 100;

    Send( &msg, execPid );

    return( msg.sysCode );
  }

static SystemCode DeAuthExec(name)
char *name;
  {
    MsgStruct msg;
    
    msg.sysCode = DEAUTH_EXEC;
    msg.segmentPtr = name;
    msg.segmentSize = strlen(name) + 1;
    if( Equal( name, "-a" ) )
	msg.unspecified[0] = 0;	/* Delete all execs */
    else
	msg.unspecified[0] = 1;
    
    Send( &msg, execPid );

    return( msg.sysCode );
  }

static void LoginCommand(args, in, out, err, mydescriptor,
			       currExCtx, currExCtxName)
    char *args[];
    File *in, *out, *err;
    struct ExecDescriptor *mydescriptor;
    ContextPair *currExCtx;
    char *currExCtxName[];
  {
    int oldmode;
    int chars;
    register char *flagp, *username;
    char *cmd;
    unsigned short verboseflag = FALSE;
    unsigned short quickflag = FALSE;
    unsigned short exclusiveFlag = FALSE;
    unsigned short finishFlag = FALSE;
    unsigned short remoteFlag = FALSE;
    unsigned ioflags;
    SystemCode r;
    char password[100];
    AuthRec arec;
    File *infile;
    int status;
    SystemCode error; 
    MsgStruct msg;

    /* Be sure there's an auth server around */
    msg.sysCode = NO_OPERATION;
    Send(&msg, VAUTH_SERVER_GROUP);
    if (msg.sysCode == KERNEL_TIMEOUT)
      {
	static char *argv[] = { "authserver", 0 };
        ProcessId newpid;
	
	fprintf(err, "Starting a new authentication server...");
	ExecProgram(argv, NULL, NULL, NULL, NULL, &error);
	if (error != OK)
	  {
	    fprintf(err, "failed: %s.\n", ErrorString(error));
	    return;
	  }
	fprintf(err, "done.\n");
      }

    cmd = *args++;

    while (*args && **args == '-')
      {
	flagp = &((*args)[1]);
	while (*flagp)
	  {
	    switch (*flagp)
	      {
	        case 'v': verboseflag = TRUE;    break;
	        case 'q': quickflag = TRUE;  	 break;
	        case 'x': exclusiveFlag = TRUE;  break;
		case 'f': finishFlag = TRUE;     break;
		case 'r': remoteFlag = TRUE;	 break;
	        default:  fprintf(err, "Unknown flag %c\n", *flagp); return;
	      }
	    flagp++;
	  }
	args++;
      }
    if (exclusiveFlag && remoteFlag)
      {
	fprintf(err,
"Can't have both exclusive login (-x) and allow remote execution (-r).\n");
	return;
      }
  
    username = *args++;
    if (username == NULL || *args != NULL)
      {
	fprintf(err, "Usage: %s <flags> <name>\n", cmd);
	return;
      }
  
    fprintf(out, "Password: ");
    Flush(out);

    oldmode = QueryPad(in);
    ModifyPad(in, oldmode & (~Echo));
    fgets(password, 20, in);
    ModifyPad(in, oldmode);
    putc('\n', out);
    Flush(out);

    chars = strlen(password);
    if (password[chars - 1] == '\n') password[chars - 1] = 0;

    if ( *cmd == 's' || SeparateExecProgram )
	r = Authenticate(username, password);	/* "su" */
    else
      {
	if (User(execPid) != UNKNOWN_USER)
	  {
	    fprintf(err,
	       "User already logged in.  Logout first, or use 'su'.\n");
	    return;
	  }
	r = AuthExec(username, password);	/* "login" */
      }

    if ( r != OK )
      {
	fprintf(err, "%s failed: %s\n", cmd, ErrorString(r));
	return;
      }

    if ( *cmd == 's' ) return;	/* done if "su" */

    /* Register the host as unavailable for remote task execution
     *   with the team server.  The exclusiveFlag determines if 
     *   absolutely all remote requests should be disallowed.
     */
    if (exclusiveFlag)
	UpdateHostStatus(username, -1);
    else if (remoteFlag)
	UpdateHostStatus(username, 1);
    else
	UpdateHostStatus(username, 0);

    if (!finishFlag)
      {
	Execl(stdin, stdout, stderr, &status, &error, "migrateprog", 0);
      }

    /* Set [home] */
    if ( (r = MapUserName(username, &arec)) != OK )
      {
        fprintf(err, "Warning: can't get home directory: %s\n",
		      ErrorString(r));
	return;
      }

    if ( (r = DefineLocalName("home", arec.home)) != OK )
      {
	fprintf(err, "Warning: can't define [home] as %s: %s\n",
		      arec.home, ErrorString(r));
	DestroyAuthRec(&arec);
	return;
      }
    DestroyAuthRec(&arec);

    if ( (r = ChangeDirectory("[home]")) != OK )
      {
	fprintf(err, "Warning: can't cd to [home]: %s\n", ErrorString(r));
	return;
      }

    /* Set [homex] */
    if ( (r = DefineLocalName("homex", "[team/local")) != OK )
      {
	fprintf(err, "Warning: can't define [homex] as %s: %s\n",
		     "[team/local", ErrorString(r));
	return;
      }
    if ( (r = ResetContext("[homex]", currExCtx, 
			   currExCtxName)) != OK )
      {
	fprintf(err, "Warning: can't cx to [homex]: %s\n", ErrorString(r));
	return;
      }

    /* Now do the guy's .Vinit if desired */
    if (!quickflag)
      {
        infile = Open("[home].Vinit", FREAD, &r);
	if (r != OK)
	  {
	    if (r != NOT_FOUND)
	        fprintf(err, "Warning: can't read [home].Vinit: %s\n",
			      ErrorString(r));
	    return;
	  }
	ioflags = mydescriptor->ioflags;
	mydescriptor->ioflags |= EXEC_NOPROMPT;
	if (verboseflag)
	    mydescriptor->ioflags |= EXEC_ECHO;
	else
	    mydescriptor->ioflags &= ~EXEC_ECHO;

        Exec(in, out, err, infile, mydescriptor);
	mydescriptor->ioflags = ioflags;
	Close(infile);
      }
  }

/* SetPrompt: with no args, resets the prompt back to the DEFAULT_PROMPT.
 * With one arg, converts the arg into a print format (which may have one
 * embedded "%d" to print the line number).
 * With more than one argument, assume userforgot to enclose the arg in 
 * double quotes and say so.
 */
SetPrompt( argc, args, err )
    int  argc;
    char *args[];
    File *err;
  {
    int i,j;
    int numberedprompt = FALSE;
    int gave_number_message = FALSE;

    if( argc == 1 ) {
	strcpy( prompt, DEFAULT_PROMPT, strlen(DEFAULT_PROMPT) );
      }
    else {
	if( argc > 2 ) {
	    fprintf(err, "Multiword prompts must be enclosed in double quotes\n");
	    fprintf(err,"Example: setprompt \"HOSTNAME # > \"\n");
	  }
	for( i=j=0; args[1][i]; i++ )
	  {
	    if( args[1][i] == '#' ) {	/* subst for linenumber */
		if( numberedprompt ) {
		    if( !gave_number_message ) {
			fprintf(err,
		"Only one embedded line number allowed per prompt\n" );
			gave_number_message = TRUE;
		      }
		    prompt[j++] = args[1][i];
		  }
		else {
		    numberedprompt = TRUE;
		    prompt[j++] = '%';		/* stick "%d" into the printf*/
		    prompt[j++] = 'd';		/* format for the linenumber */
		  }
	      }
	    else if( args[1][i] == '%' ) {	/* turn "%" into "%%" */
		prompt[j++] = '%';		/* stick "%%" into the printf*/
		prompt[j++] = '%';		/* format for percent sign */
	      }
	    else
		prompt[j++] = args[1][i];
	  }
	prompt[j] = '\0';
      }
  }



/*
 * RunProgs:
 * Parses the argv array for programs to run and pipes to set up.
 * This is a recursive routine that calls itself again if a pipe is
 * encountered.
 * The routine cleans up after itself if an error occurs while trying to
 * set up sequence of programs and pipes.
 */

SystemCode RunProgs(argv, concurrent, arbHost, hostSpec, rtMsg, debugRtMsg, 
		    shOut, pidList)
    char **argv;
    int concurrent, arbHost;
    SelectionRec *hostSpec;
    register RootMessage *rtMsg, *debugRtMsg;
    File *shOut;
    ProcessId *pidList;
  {
    Message msgx, msgy;
    register RootMessage *rtMsg1 = (RootMessage *) msgx;
    register CreateInstanceReply *pipeReply = (CreateInstanceReply *) msgy;
    File f;
    ProcessId newpid;
    SystemCode error;
    int argc1 = 0;
    static int inPipe = FALSE;

    while (argv[argc1] && !Equal("|", argv[argc1])) argc1++;

    if( argv[argc1] != NULL )
      {
	inPipe = TRUE;
        error = SetUpPipe(pipeReply, shOut);
	if (error != OK) return error;
        argv[argc1++] = NULL;
	*rtMsg1 = *rtMsg;
	rtMsg1->stdoutserver = pipeReply->fileserver;
	rtMsg1->stdoutfile = pipeReply->fileid;
	rtMsg1->rootflags |= RELEASE_OUTPUT;
	newpid = LoadProgram(argv, hostSpec, rtMsg1, NULL,
			     concurrent, &error);
	if (error != OK) return(error);
	SetOwnership(rtMsg1, newpid);

	/* If the host was chosen arbitrarily, then announce its name. */
	if (arbHost)
	    AnnounceRemoteHost(hostSpec, inPipe, argv, shOut);

	/* Set the new team running. */
	Reply(rtMsg1, newpid);
	*pidList++ = newpid;

	rtMsg->stdinserver = pipeReply->fileserver;
	rtMsg->stdinfile = pipeReply->fileid + 1;
	rtMsg->rootflags |= RELEASE_INPUT;
	error = RunProgs(&argv[argc1], concurrent, arbHost, hostSpec, 
			 rtMsg, debugRtMsg, shOut, pidList);
	if (error != OK)
	  {
	    Message msg;
	    ExitTeamRequest *req = (ExitTeamRequest *) msg;
	    ExitTeamReply *rep = (ExitTeamReply *) msg;
	    ProcessId teamServerPid;

	    req->requestcode = TERMINATE_TEAM;
	    req->requestType = SpecificTeam;
	    req->rootPid = newpid;
	    req->status = 0;
	    if ((teamServerPid = hostSpec->teamServerPid) == 0)
		teamServerPid = GetPid(TEAM_SERVER, LOCAL_PID);
	    Send(req, teamServerPid);
	    if (rep->replycode != OK)
		/* try to destroy as an individual process */
		DestroyProcess( newpid );
	  }
     }
    else /* base case of the recursion */
      {
	argv[argc1] = NULL;
	newpid = LoadProgram(argv, hostSpec, rtMsg, getenv("PATH"),
			     concurrent, &error);
	if (error != OK) return(error);
	pidList[0] = newpid;
	pidList[1] = 0;	/* to terminate the list */
	SetOwnership(rtMsg, newpid);

	/* If the host was chosen arbitrarily, then announce its name. */
	if (arbHost)
	    AnnounceRemoteHost(hostSpec, inPipe, argv, shOut);
	inPipe = FALSE;

	/* Set the new team running. */
	Reply(rtMsg, newpid);
      }
    return error;
  }


/*
 * SetOwnership:
 * Make the new team the owner of any stdio instances it is supposed
 *   to release
 */
SetOwnership(rtMsg, newpid)
    register RootMessage *rtMsg;
    ProcessId newpid;
  {
    if (rtMsg->rootflags & RELEASE_INPUT)
        SetInstanceOwner(rtMsg->stdinserver, rtMsg->stdinfile, newpid);
    if (rtMsg->rootflags & RELEASE_OUTPUT)
        SetInstanceOwner(rtMsg->stdoutserver, rtMsg->stdoutfile, newpid);
    if (rtMsg->rootflags & RELEASE_ERR)
        SetInstanceOwner(rtMsg->stderrserver, rtMsg->stderrfile, newpid);
  }
    

static AnnounceRemoteHost(hostSpec, inPipe, argv, out)
    SelectionRec *hostSpec;
    int inPipe;
    char **argv;
    File *out;
  {  
    int hostNameFound;
    register int i;
    char hostName[128];
	
    hostNameFound = GetHostName(hostSpec->teamServerPid, hostName);
    fprintf(out, "Executing");
    /* If we're part of a pipe, print the args too. */
    if (inPipe)
      {
	fprintf(out, " \"%s", argv[0]);
	for (i = 1; argv[i] != NULL; ++i)
	    fprintf(out, " %s", argv[i]);
	fprintf(out, "\"");
      }
    fprintf(out, " remotely at %s.\n\n", (hostNameFound ? hostName : "???"));
    Flush(out);
    hostSpec->teamServerPid = 0;
  }


extern SystemCode GetAbsoluteName();

static SystemCode ResetContext(inName, newContext, newName)
    char inName[];
    ContextPair *newContext;
    char *newName[];
  {
    /* Sets the context "newContext" to the value indicated by the 
     * name "inName", also returning the corresponding absolute name 
     * in "newName".
     */
    SystemCode err;
    char *tempNewName;
    ContextPair tempNewContext;

    /* Get the absolute name and context id */
    tempNewName = (char *) malloc(MAX_CONTEXT_NAME);
    strncpy(tempNewName, inName, MAX_CONTEXT_NAME);
    if (tempNewName[MAX_CONTEXT_NAME-1] != '\0')
	return BAD_ARGS;  /* name too long */
    err = GetAbsoluteName(tempNewName, MAX_CONTEXT_NAME, &tempNewContext);

    if (err != OK) return err;
    if (tempNewContext.pid == 0) return INVALID_CONTEXT;  /* not a context */

    if (*newName != NULL) free(*newName);
    *newContext = tempNewContext;
    *newName = tempNewName;

    return OK;
  }
