/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 * Copyright (c) 1982 Stanford University, all rights reserved.
 *
 * V kernel simulation code
 */

#include "config.h"
#include <signal.h>
#include <wait.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vethernet.h>
#include <Vgroupids.h>
#include <Vikc.h>
#include "net.h"
#include "kernel.h"
#include "debug.h"
#include "server.h"
#include "swab.h"

/* IMPORTS */
extern geteuid(), getegid();
extern ReclaimInstances(), CheckProcess();
extern SetDebugMode();
extern int PublicServer;
extern int NetworkNum;
extern int (*NetRead)(), (*NetWrite)(), (*NetSignal)(),(*NetReceiveQLength)();


/* EXPORTS */
void		KillSession();		/* also interrupt handler */
int	 	ChildrenAreDead;	/* If SIGCHLD event has occurred */
int 		Net;			/* Fid of network device */
ProcessId 	LogicalHostPid;		/* Logical host id in pid format */
ProcessId 	MyPid;			/* Pid of server (or session) */
ProcessId 	MainServerPid;		/* Pid of main server */
GroupId		CommonGroupId;		/* Local grp: main server & sessions */
unsigned	MyVUserNumber;		/* V user number for this process */

/* LOCALS */
int 		RootServer = 1;		/* 1 for main server, 0 otherwise */
ProcessId 	TimerPid;		/* Pid for timer pseudo process */
ProcessId 	RemotePid;		/* Remote pid for current operation */

/* Signal routines */
void BadSystemCallHandler();
void DeadProcessHandler();
void TerminalStopHandler();
void PacketHandler();
void AlarmHandler();
void DeadChildHandler();

#define MAX_RETRANSMISSIONS	3

#define SigMask(s) (1<<((s)-1))

ProcessId 	Logical_id_map[MAX_LOGICAL_ID];
ProcessId	newCandidatePid;
unsigned char	*RecBufPtr = 0;		/* Where to put received segments */
unsigned 	*RecBufSizePtr;		/* Max size of received segment */
unsigned	RecBufSize;		/* Remember internally max size 
					   specified */
unsigned short	CurrentSeqNo = 0x5555;  /* Generator for sequence numbers */
					/* 5555 favorite random number */
struct SendRecord *SendQueue = NULL;	/* Incoming messages are put in this
					   queue */
struct SendRecord *ReceivePtr = NULL;	/* next (or current) message to be
					   processed (received) */
struct SendRecord *SendAlloc();		

int 		AlarmSig;		/* State of the net: REPEAT, DONE,
					   RETRANSMIT, (modified asynchronously
					   by the alarm handler) */
int 		KernelOpStatus;		/* State of kernel operation: REPEAT,
					   DONE, NACK (modified asynchronously
					   by reply */
int 		SendQueueLock = 0;	/* Used to sync with alarm handler */
unsigned 	LocalAddr,RemAddr,TotalLength;
int 		CurrentOp;		/* Current operation waiting to 
					   complete */
int 		Retrans;		/* Number of retransmissions so far */
int 		AlarmRate;		/* alarm signal rate in seconds */


InitKernel( reinit )
    int reinit;
  /* Open a connection to the network, set up the network filter,
   * and initialize the pseudo-kernel data structures.
   * If reinit is non-zero, assume the network file is already open
   * and MyPid has been set.
   */
  {
    extern ProcessId GenerateNewVPid();

    RootServer = reinit == 0;
    if( reinit == 0 )
      {
	/* Main server process kernel initialization. */

	/* Set the V user number for this process to be SYSTEM_USER. */
	MyVUserNumber = SYSTEM_USER;

	/* The following routine determines what type of network the server
	 * is talking to and sets up the network access routines
	 * accordingly.  It then calls (*NetOpen)() to get a network
	 * device.
	 */
	if( (Net = InitializeNetwork(LOCAL_KERNEL_PROCESS_PID, MyVUserNumber))
	     == -1 )
	  {
	    printf( "Cannot open network file\n" );
	    exit( 1 );
	  }

	/* 
	 * Compute the logical host id and set the main server's pid.
	 */
	LogicalHostPid = GetLogicalHost();
	if (KDebug) 
	    printf("Logical host pid = 0x%08x\n", LogicalHostPid);

	MyPid = MainServerPid = LogicalHostPid + LOCAL_KERNEL_PROCESS_PID;

	/* The local process group to which the main server, and all sessions,
	 * belong:
	 */
	CommonGroupId = LogicalHostPid + COMMON_SERVER_PID;
	CommonGroupId |= GROUP_ID_BIT;

	newCandidatePid = LogicalHostPid + (time(0) & 0x0FFF);
	if (KDebug) 
	    printf("V pids begin at 0x%08x\n", newCandidatePid);
	TimerPid = GenerateNewVPid(); /* Pid of the timer pseudo-process */

	/*
	 * Initialize the group membership tables
	 */
	InitGroupIpc();

	/* Join appropriate V static groups: */
	JoinGroup(VKERNEL_SERVER_GROUP, CommonGroupId);
	JoinGroup(VSTORAGE_SERVER_GROUP, CommonGroupId);
	JoinGroup(VCSNH_SERVER_GROUP, CommonGroupId);
	JoinGroup(VTIME_SERVER_GROUP, CommonGroupId);
      }

    /* Init storage buffers */
    InitBufMgr();

    SendQueue = ReceivePtr = NULL;

    /* Set interrupt routines */
    AlarmRate = FAST_ALARM_RATE;
    signal( PACKET_RECEIVED,PacketHandler );
    signal( SIGALRM, AlarmHandler );
    signal( SIGCHLD, DeadChildHandler );
    signal( SIGTSTP, TerminalStopHandler );
    signal( SIGSYS, BadSystemCallHandler );
    ChildrenAreDead = 0;
    if ( !RootServer )
      {
	signal( SIGINT, KillSession );    /* assure smooth death of */
	signal( SIGTERM, KillSession );   /* sessions. */
	signal( SIGPIPE, SIG_IGN );	  /* don't die on writes to pipe*/
	/* ABOVE SHOULD CALL A ROUTINE THAT OPERATES ON THE SIGNAL */
      }

    /* Enable signal */
    (*NetSignal)( Net, PACKET_RECEIVED );
    alarm( AlarmRate );
  }

ProcessId ReceiveWithSegmentPlus(msg, recbufptr, recbufsizeptr,
				 forwarder, vUserNumber)
   Message msg; unsigned char *recbufptr; unsigned *recbufsizeptr;
   ProcessId *forwarder; unsigned *vUserNumber;

 /*
  * Look for an unprocessed incoming message.
  * We may have to pause() until such a message can be found.
  * Note that a pause can be released by any packet arriving.
  *
  * Note: Unlike ReceiveWithSegment(), this routine also returns the
  * forwarder ("forwarder") and the user number of the request ("vUserNumber").
  */
  {
    register unsigned length;
    register struct SendRecord *msgptr;

    RecBufSize = *recbufsizeptr; /* Save maximum segment size */
    RecBufSizePtr = recbufsizeptr;
    RecBufPtr = recbufptr;
    *recbufsizeptr = 0; /* Set to 0, set otherwise when segment received */

    while(1)
      {
	/* Find the first message which has not yet been "Receive()d". */
	for (msgptr = SendQueue; msgptr != NULL; msgptr = msgptr->next)
	    if (msgptr->state == UNPROCESSED)
	        break; /* we found one! */
	if (msgptr != NULL)
	    break;
	pause();
      }
    SendQueueLock = 1;
    sigblock( SigMask(PACKET_RECEIVED) );

    RecBufPtr = 0; /* Flag we are not willing to receive segments now */
    ReceivePtr = msgptr->next;

    /* Copy message */
     {
	register int i;

	for( i=0; i<8; i++ ) msg[i] = msgptr->msg[i];
      } 
    msgptr->state = RECEIVED;	/* mark this msg as processed (received) */
    SendQueueLock = 0;
    sigsetmask(0);

    *forwarder = msgptr->forwarder;
    *vUserNumber = msgptr->userNumber;
    return( msgptr->srcpid );
  }

ProcessId ReceiveWithSegment( msg, recbufptr, recbufsizeptr )
   Message msg; unsigned char *recbufptr; unsigned *recbufsizeptr;
  {
    int dummy1, dummy2;
    
    return (ReceiveWithSegmentPlus(msg, recbufptr, recbufsizeptr,
				   &dummy1, &dummy2));
  }


ProcessId ReplyWithSegment( msg, pid, src, dest, bytes )
   Message msg;
   ProcessId pid;
   unsigned char *src;
   unsigned dest, bytes;

  /* Check to see if there is a send request
   *  corresponding to this reply.  If there is then
   *  reply to the remote process and remove the
   *  send request from the list of queued messages.
   *
   * If there is no corresponding send to the reply
   *  then zero is returned else the pid.
   *
   */
 {
    register struct SendRecord *ptr, *prevptr;
    register unsigned evensize;
    ProcessId destpid;
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);
    int discardRecord;

    /* Disable signals to prevent trying to queue new send
     * while we are removing old ones, this is necessary to
     * keep the queue pointers ok.
     */
    SendQueueLock = 1;
    sigblock( SigMask(PACKET_RECEIVED) );
    
    if ( KDebug && bytes != 0 ) printf( "ReplyWithSegment\n" );

    /* Find the first message that was "Receive()d" from "pid".
       NOTE: There should be only one such message. */
    for (ptr = SendQueue; ptr != NULL; prevptr = ptr, ptr = ptr->next)
	if ((ptr->srcpid == pid) && (ptr->state == RECEIVED))
	    break; /* we found one! */
    if (ptr == NULL) 
      { 
	/* no more messages, thus fail */
	destpid = 0;
	goto exit; 
      }
    else
        destpid = pid;

    /* We've found the message, so reply to it: */
    discardRecord = 0;
    if (ptr->srcpid == TimerPid) /* Don't reply to the pseudo timer process. */
	discardRecord = 1;
    else if (((IoRequest *) msg)->requestcode != DISCARD_REPLY)
      {
	if (bytes > 0)
	  {
	    /* Handle the appending of a data segment. */
	    if (bytes > MAX_APPENDED_SEGMENT) bytes = MAX_APPENDED_SEGMENT;
	    evensize = bytes + (bytes&1);
	    bcopy(src, packet->data, evensize);
	    discardRecord = 1; /* We assume that the request was idempotent. */
	  }
	else
	    evensize = 0;
	/* Transmit a reply packet */
	(*NetWrite)(Net, &kpacket, SMALL_KPACKET+evensize,
		    remoteReply, ptr->sequenceNo, ptr->dstpid, pid, 0,
		    MyVUserNumber,
		    bytes, bytes, dest, msg);
      }

    if (discardRecord)
      {
	/* Take the record out of the queue and put it in free storage. */
	if (ptr == SendQueue)
	    SendQueue = ptr->next;
	else  /* take out of list */
	    prevptr->next = ptr->next;
	SendFree(ptr);
      }
    else
      {
	/* Keep the record in case a duplicate request arrives. */
	register int i;

	ptr->state = REPLIED;
	/* Save a copy of the reply message. */
	for(i=0; i<8; i++) ptr->msg[i] = msg[i];
      }

    /* reenable signal */
exit:
    SendQueueLock = 0;
    sigsetmask(0);
	
    return(destpid);
  }
ProcessId Reply( msg, pid )
   Message msg;
   ProcessId pid;
  {
   
   return( ReplyWithSegment(msg, pid, 0, 0, 0) );
  }

ProcessId Forwarder( pid )
    ProcessId pid;
   /* Return the pid of the process that forwarded the message
    * last received from the specified process, which is the
    * sender if the message was not forwarded.
    * Return 0 if the process is not awaiting reply.
    */
 {
    register struct SendRecord *ptr;

    /* Find message in the send queue */
    for (ptr = SendQueue; ptr != NULL; ptr = ptr->next)
	if ((ptr->srcpid == pid) && (ptr->state == RECEIVED))
	    return (ptr->forwarder);
    return(0);
  }

ProcessId UserNumber( pid )
    ProcessId pid;
   /* Return the userNumber of the process that forwarded the
    * message last received from the specified process, which 
    * is the sender if the message was not forwarded.
    * Return 0 if the process is not awaiting reply.
    */
 {
    register struct SendRecord *ptr;

    /* Find message in the send queue */
    for (ptr = SendQueue; ptr != NULL; ptr = ptr->next)
	if ((ptr->srcpid == pid) && (ptr->state == RECEIVED))
	    return (ptr->userNumber);
    return(0);
  }


ProcessId Forward( msg, frompid, topid )
   Message msg;
   ProcessId frompid, topid;
  /* Forward frompid to topid with the specified message */
  {
    register struct SendRecord *ptr;
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);

    /* Disable signals to prevent trying to queue new send
     * while we are removing old ones, this is necessary to
     * keep the queue pointers ok.
     */
    SendQueueLock = 1;
    sigblock( SigMask(PACKET_RECEIVED) );

    /* Find the first message that was "Receive()d" from "frompid".
       NOTE: There should be only one such message. */
    for (ptr = SendQueue; ptr != NULL; ptr = ptr->next)
	if ((ptr->srcpid == frompid) && (ptr->state == RECEIVED))
	    break; /* we found one! */
    if (ptr == NULL) 
      { 
	/* no more messages, thus fail */
	topid = 0;
	goto exit; 
      }

    (*NetWrite)(Net, &kpacket, SMALL_KPACKET,
    		remoteForward, ptr->sequenceNo,
		topid, frompid, ptr->dstpid,
		MyVUserNumber,
		0, 0, 0, msg);

      {
	/* Keep the record in case a duplicate request arrives. */
	register int i;

	ptr->state = FORWARDED;
	ptr->forwarder = topid; /* actually, the `forwarded to' process. */
	/* Save a copy of the reply message. */
	for(i=0; i<8; i++) ptr->msg[i] = msg[i];
      }

    /* reenable signal */
exit:
    SendQueueLock = 0;
    sigsetmask(0);
	
    return( topid );
  }

ProcessId ValidPid( pid )
   ProcessId pid;
   /*
    * Return non-zero if a valid Pid else 0.
    */
  {
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);
    Message msg;
    unsigned short sequenceno;
    unsigned retranscount;

    if ((pid == MyPid) || (pid == CommonGroupId))
        return(pid);
    ++CurrentSeqNo;
    RemotePid = pid;
    retranscount = 0;

    KernelOpStatus = REPEAT;	/* only needs to be set once, any change
    				   terminates the operation */
    do
      {
	/* NOTE critical race condition if the VAX is loaded
		- we don't know when the Alarm will go off
		- the SUN kernel can reply very quickly, so
		  CurrentOp must be set before doing the write.
         */
	AlarmSig = DONE;
	CurrentOp = VALID_PID_OP;
	(*NetWrite)(Net, &kpacket, SMALL_KPACKET,
		    remoteReceiveSpecific, CurrentSeqNo, MyPid,	pid, 0,
		    MyVUserNumber,
		    0, 0, 0, msg);
	AlarmSig = REPEAT;
	while( AlarmSig == REPEAT && KernelOpStatus == REPEAT ) pause();
	CurrentOp = NULL;
	if( ++retranscount > MAX_RETRANSMISSIONS || KernelOpStatus == NACKED )
	  {
	    return( 0 );
	  }
      }
    while( KernelOpStatus != DONE );
     
    return( pid );
  }

/*
 * Find our record of a sender in SendQueue so we can generate the
 *   right Send sequence number in CopyTo or CopyFrom packets.
 *   Similar to inline code in Reply and Forward.
 */
struct SendRecord *FindSender(pid)
    ProcessId pid;
  {
    register struct SendRecord *ptr;

    /* Disable signals to prevent trying to queue new send
     * while we are removing old ones, this is necessary to
     * keep the queue pointers ok.
     */
    SendQueueLock = 1;
    sigblock( SigMask(PACKET_RECEIVED) );

    /* Find the first message that was "Receive()d" from "frompid".
       NOTE: There should be only one such message. */
    for (ptr = SendQueue; ptr != NULL; ptr = ptr->next)
	if ((ptr->srcpid == pid) && (ptr->state == RECEIVED))
	    break; /* we found one! */

    SendQueueLock = 0;
    sigsetmask(0);

    return (ptr); /* NULL iff no such record was found */
  }

MoveTo( dstpid, dstaddr, srcaddr, bytes )
   ProcessId dstpid;
   unsigned dstaddr;
   char *srcaddr;
   unsigned bytes;
  /* Issue a MoveTo packet to destination workstation and wait for
   * acknowledgement. If a response is not received 
   * in the alarm time then we retransmit the request. If after
   * max retransmissions we still have not received a reply then
   * the MoveTo fails and returns RETRY.
   */
  {
    int packets,i,size, retranscount, evensize;
    unsigned dstptr;
    char *srcptr;
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);
    register struct SendRecord *srec;
    MsgStruct msg;
    
    RemotePid = dstpid;
    RemAddr = dstaddr;
    TotalLength = bytes;
    packets = bytes / MAX_APPENDED_SEGMENT;
    srec = FindSender(dstpid);
    if (srec == NULL) return (NOT_AWAITINGREPLY);
    msg.unspecified[0] = srec->sequenceNo; /* sequence number of the Send
    					    *  that passed access rights */
    ++CurrentSeqNo;
    retranscount = 0;
    if( KDebug ) printf( "MoveTo %d bytes\n", bytes );

    KernelOpStatus = REPEAT;	/* only needs to be set once, any change
    				   terminates the operation */

    do
      {
	/* NOTE critical race condition if the VAX is loaded
		- we don't know when the Alarm will go off
		- the SUN kernel can reply very quickly, so
		  CurrentOp must be set before doing the write.
         */
	AlarmSig = DONE;
	CurrentOp = MOVE_TO_OP;

	/* Transmit data in packets */
	dstptr = dstaddr;
	srcptr = srcaddr;
	for( i=0; i < packets; ++i )
	  {
	    bcopy( srcptr, packet->data, MAX_APPENDED_SEGMENT );
	    (*NetWrite)(Net, &kpacket, MAX_APPENDED_SEGMENT+SMALL_KPACKET,
			remoteMoveToReq, CurrentSeqNo, MyPid, dstpid, 0,
			MyVUserNumber,
			MAX_APPENDED_SEGMENT, bytes, dstptr, &msg);
	    dstptr += MAX_APPENDED_SEGMENT;
	    srcptr += MAX_APPENDED_SEGMENT;
	  }
	if( (size = (bytes % MAX_APPENDED_SEGMENT)) > 0 )
	  {
	    evensize = size + (size&1);
	    bcopy( srcptr, packet->data, evensize );
	    (*NetWrite)(Net, &kpacket, evensize+SMALL_KPACKET,
	    		remoteMoveToReq, CurrentSeqNo, MyPid, dstpid, 0,
			MyVUserNumber,
			size, bytes, dstptr, &msg);
	  }
	AlarmSig = REPEAT;
	while( AlarmSig == REPEAT && KernelOpStatus == REPEAT ) pause();
	
	/* KernelOpStatus set to DONE by network or 
	   AlarmSig set to RETRANSMIT by AlarmHandler */
	CurrentOp = NULL;
	if( retranscount++ > MAX_RETRANSMISSIONS || KernelOpStatus == NACKED )
	  {
	    /* We have done all our retransmissions */
	    return( TIMEOUT );
	  }
      }
    while( KernelOpStatus != DONE );

    return( OK );
  }


MoveFrom( srcpid, src, dest, bytes )

   ProcessId srcpid;
   unsigned src,dest;
   unsigned bytes;
  /* Issue a move from request and wait for response.
   *   Alarm handler makes sure we are not waiting indefinitely
   *   for a response. If we do not get all the data in this length
   *   time the a request is sent out only for the data we have not
   *   yet received.
   */
  {
    struct SmallKernelPacket kpacket;
    register kPacket *packet = &(kpacket.kernelpacket);
    MsgStruct msg;
    register struct SendRecord *srec;

    RemotePid = srcpid;
    RemAddr = src;
    LocalAddr = dest;
    TotalLength = bytes; /* count of bytes we have not get yet */
    ++CurrentSeqNo;
    Retrans = 0;
    srec = FindSender(srcpid);
    if (srec == NULL) return (NOT_AWAITINGREPLY);
    msg.unspecified[0] = srec->sequenceNo; /* sequence number of the Send
    					    *  that passed access rights */

    if( KDebug ) printf( "MoveFrom %d bytes\n", bytes );


 
    KernelOpStatus = REPEAT;	/* only needs to be set once, any change
    				   terminates the operation */

 
   while( TotalLength != 0 )
      {
	/* NOTE critical race condition if the VAX is loaded
		- we don't know when the Alarm will go off
		- the SUN kernel can reply very quickly, so
		  CurrentOp must be set before doing the write.
         */
	AlarmSig = DONE;
	CurrentOp = MOVE_FROM_OP;

	if( bytes == TotalLength ) /* No data received since last request */
	  {
	    if( ++Retrans > 8 )
	      {
		CurrentOp = NULL;
		return( TIMEOUT );
	      }
	    (*NetWrite)(Net, &kpacket, SMALL_KPACKET,
	    		remoteMoveFromReq, CurrentSeqNo, MyPid, srcpid, 0,
			MyVUserNumber,
			TotalLength, LocalAddr, RemAddr, &msg);
	  }
	else bytes = TotalLength;

	AlarmSig = REPEAT;
	/* wait for timeout or done */
	while( TotalLength != 0 && (AlarmSig == REPEAT) 
		&& (KernelOpStatus == REPEAT) ) pause();
	CurrentOp = NULL;
	if ( KernelOpStatus == NACKED ) return( TIMEOUT );
      }
    return( OK );
  }


SetPid( logical_id, pid )
     unsigned logical_id; ProcessId pid;

  /* Special version to register this process as the storage server */
  {
    if( KDebug ) printf( "setting logical pid %d to %x\n", logical_id, pid );

    Logical_id_map[logical_id] = pid;
  }


ProcessId GetPid( logical_id )
   unsigned logical_id;

   /* Special version that just returns the "active process pid".
    */
  {
    return(MyPid);
  }


AnswerGetPid( packet, logicalid, dstpid, seqno )
  struct SmallKernelPacket *packet;
  unsigned logicalid;
  ProcessId dstpid;
  short unsigned seqno;

   /* Answer a GetPid request 
    *
    */
  {
    ProcessId retpid = Logical_id_map[logicalid];

    /* Only answer GetPid requests if running as a Public Vserver */
    if( !PublicServer || ((unsigned)logicalid > MAX_LOGICAL_ID) ||
        ((retpid != MyPid) && (retpid != CommonGroupId)) ) 
      {
        if (NDebug) printf( "Discarding remote GetPid(%x)=>%x Public=%d\n",
	  logicalid,retpid,PublicServer );
	return;
      }
    if (NDebug) printf("Answering remoteGetPid: 0x%08x\n", retpid);

    (*NetWrite)(Net, packet, SMALL_KPACKET,
    		remoteGetPidReply, seqno, retpid, dstpid, logicalid,
		MyVUserNumber,
		0, 0, 0, 0);
  }

SystemCode ServerGetPid( req, pid )
    KernelRequest *req;
    ProcessId pid;
   /* Answer a server GetPid request 
    */
  {
    /* Only answer GetPid requests if running as a Public Vserver */
    ProcessId retpid = Logical_id_map[(unsigned)req->unspecified[0]];

    if( !PublicServer ||
        ((unsigned)req->unspecified[0] > MAX_LOGICAL_ID) ||
        ((retpid != MyPid) && (retpid != CommonGroupId) &&
	 (retpid != MainServerPid)) ) 
      {
        if (NDebug) printf( "Discarding remote GetPid(%x) Public=%d\n",
	  (unsigned) req->unspecified[0], PublicServer );
	return(DISCARD_REPLY);
      }
    if (NDebug) printf( "Answering ServerGetPid\n" );
    req->pid = retpid;
    return(OK);
  }

LogicalHostCollisionCheck( packet, logicalhost, dstpid, seqno )
    struct SmallKernelPacket *packet;
    unsigned logicalhost;
    ProcessId dstpid;
    short unsigned seqno;  
  /*
   * See if the logical host number collides with ours.
   */
  {
    if ( KDebug )
	printf( "lhnCheck: my host = 0x%x\n", LogicalHostPid );

    if ( logicalhost == LogicalHostPid )
      {
        (*NetWrite)(Net, packet, SMALL_KPACKET,
		    lhnCollisionReply, seqno, MyPid, dstpid,
		    logicalhost, MyVUserNumber,
		    0, 0, 0, 0);
	if ( KDebug )
	    printf( "LogicalHostPid collision dstpid = 0x%x\n", dstpid );
      }

  } /* LogicalHostCollisionCheck */


void PacketHandler()
 /* Interrupt routine to handle incoming packets
  *  will read all the packtes currently in the
  *  buffer then return after processing them.
  */
  {
    struct KernelPacket kpacket;
    int length,localaddr,remoteaddr;
    int PacketLength = sizeof(struct KernelPacket);
    ProcessId srcpid,dstpid,forwarder;
    unsigned userNumber;
    short kernelop;
    short seqno;
    Message msg;
    int qLength;/*###*/

    if (KDebug) printf("Entered PacketHandler(): ");/*###*/
    while (1)/*###*/
      {
	qLength = (*NetReceiveQLength)(Net);/*###*/
	if (KDebug) printf("qLength = %d\n", qLength);/*###*/
	if (qLength <= 0) break;/*###*/

	/* Receive waiting packet */

	kernelop = (*NetRead)(Net, PacketLength, &kpacket, &srcpid, &dstpid,
			      &forwarder, &userNumber, &length, &localaddr,
			      &remoteaddr, &seqno, msg);
	/* If this was a group send, make the packet look as if it
	 * were forwarded from the group.
	 */
	if (dstpid & GROUP_ID_BIT)
	  {
	    if (MDebug)
		printf("Group packet: 0x%x -> 0x%x\n", srcpid, dstpid);
	    forwarder = dstpid;
	    dstpid = GroupToPid(dstpid); /* returns zero if group not found */
	  }
	/* check if we want it */
      	if( dstpid == MyPid || kernelop == remoteGetPid ||
	    kernelop == lhnCollisionCheck )
	  {
	    if (NDebug)
	        printf( "op=%d, seqno = %d, srcpid = 0x%x, user num = %d, msg[0] = 0x%x\n", 
			kernelop, seqno, srcpid, userNumber, msg[0] );

 	    switch( kernelop )
	      {
		case remoteGetPid:
		    AnswerGetPid( &kpacket, forwarder,srcpid,seqno);
		    break;
			    
	        case remoteSend:
		    HandleSend( &kpacket, msg, srcpid, dstpid, forwarder,
				userNumber, length, seqno);
		    break;
			
	        case nAck:
		    ProcessNack( &kpacket, msg, srcpid,  seqno);
		    break;
			
	        case breathOfLife:
		    ProcessBreathOfLife( &kpacket, msg, srcpid, seqno );
		    break;
			
	        case remoteMoveFromReply:
		    MoveFromAnswer(&kpacket,srcpid,length,remoteaddr, seqno);
		    break;
			
		case remoteMoveToReply:
		    if( (CurrentOp == MOVE_TO_OP) && (RemotePid == srcpid)
		      && (length == TotalLength) && (RemAddr == localaddr)
		      && seqno == CurrentSeqNo )
		      {
			KernelOpStatus = DONE;
		      }
		    break; 
	
		case lhnCollisionCheck:
		    LogicalHostCollisionCheck( &kpacket, srcpid, forwarder
					       , seqno );
		    break;
	       }
	  }
	else
	    if (NDebug && (kernelop != -1))
	        printf("PacketHandler - rejected packet: op=%d, seqno = %d, srcpid = 0x%x, dstpid = 0x%x, user num = %d, msg[0] = 0x%x\n", 
			kernelop, seqno, srcpid, dstpid, userNumber, msg[0] );

      }
  }

static struct SendRecord *ReclaimSendRecord()
    /* We try to obtain a free SendRecord by reclaiming one
     * that was being used solely for duplicate suppression purposes.
     * Note that the oldest such records will be earliest in the queue.
     * NULL is returned iff no such record can be found.
     */
  {
    register struct SendRecord *ptr, *prevptr;
    
    for (ptr = SendQueue; ptr != NULL; prevptr = ptr, ptr = ptr->next)
	if ((ptr->state == REPLIED) || (ptr->state == FORWARDED))
	    break; /* we found one! */
    if (ptr != NULL)
      {
	/* Take the record out of the queue. */
	if (ptr == SendQueue)
	    SendQueue = ptr->next;
	else  /* take out of list */
	    prevptr->next = ptr->next;
      }
    return(ptr);
  }

HandleSend( packet, msg, srcpid, dstpid, forwarder, userNumber, length, seqno)
    struct KernelPacket *packet;
    register long *msg;
    ProcessId srcpid, dstpid, forwarder;
    unsigned length, userNumber;
    short unsigned seqno;
    /* Queue up send requests for receiver to pick them out.
     *
     */
  {
    register struct SendRecord *ptr, *prevptr;

    SendQueueLock = 1;

    /* Check to see if this is a retransmission */
    ptr = SendQueue;
    while (ptr != NULL)
      {
	if (ptr->srcpid == srcpid)
	    if (ptr->sequenceNo == seqno)
		break; /* this is a true retransmission */
	    else
		/* This record is for another message sent by the same pid.
		 * We discard the old record, unless it's state is RECEIVED.
		 * (This could conceivably happen if the original message was 
		 * sent to a group, and the sender has already received a 
		 * reply from someone else, then sent another message.
		 * In this case, we need to keep the old record around so 
		 * that the Reply() doesn't get matched with the new request 
		 * by mistake.)
		 */
		 /* Note: We are assuming here that the newly arriving 
		  * message really is `newer' than the old message.  This 
		  * will always be the case if messages cannot be reordered.
		  * However, should this not be the case, then this probably
		  * won't be serious.
		  */
		if (ptr->state != RECEIVED)
		  {
		    /* Free the record from the queue. */
		    struct SendRecord *oldrecord = ptr;
		    
		    if (ptr == SendQueue)
			SendQueue = ptr = ptr->next;
		    else  /* take out of list */
			prevptr->next = ptr = ptr->next;
		    SendFree(oldrecord);
		    continue; /* Note! */
		  }
	prevptr = ptr;
	ptr = ptr->next;
      }

    if (ptr != NULL) 
      { 
	/* We have a retransmission.  Our next action depends on the state of 
	 * the request:
	 */
	switch(ptr->state)
	  {
	    case UNPROCESSED:
	    case RECEIVED:
		/* Transmit breathOfLife to prevent Send() from timing out. */
		(*NetWrite)(Net, packet, SMALL_KPACKET,
			    breathOfLife, seqno, ptr->dstpid, srcpid, 0,
			    MyVUserNumber,
			    0, 0, 0, msg);
		break;

	    case REPLIED:
		/* Retransmit the reply packet, unless the replycode
		 *  was DISCARD_REPLY.  Note that there was no 
		 *  appended segment. */
		if ( ((IoReply *)(ptr->msg))->replycode != DISCARD_REPLY )
	          {
		     (*NetWrite)(Net, packet, SMALL_KPACKET,
			    remoteReply, seqno, ptr->dstpid, srcpid, 0,
			    MyVUserNumber,
			    0, 0, 0, ptr->msg);
		  }
		break;

	    case FORWARDED:
		/* Retransmit the forward.  Note that "ptr->forwarder" is the 
		   pid of the `forwarded to' process, in this case. */
		(*NetWrite)(Net, packet, SMALL_KPACKET,
			    remoteForward, seqno,
			    ptr->forwarder, ptr->srcpid, ptr->dstpid,
			    MyVUserNumber,
			    0, 0, 0, ptr->msg);
		break;
	  }
      }
    else
      {
	/* A new request - allocate a record for it. */
	ptr = SendAlloc();
	if (ptr == NULL)
	  {
	    /* We've run out of free records.  We try to reclaim a record 
	     * that was being used solely for duplicate suppression purposes.
	     */
	    ptr = ReclaimSendRecord();
	    if (ptr == NULL)
		goto exit;
	  }
	if (SendQueue == NULL)
	    SendQueue = ptr;
	else
	    prevptr->next = ptr;
	
	ptr->state = UNPROCESSED;
	ptr->srcpid = srcpid;
	ptr->dstpid = dstpid;
	ptr->forwarder = forwarder;
	ptr->userNumber = userNumber;
	ptr->sequenceNo = seqno;
	ptr->next = NULL;
	  { /* Copy message */
	    register long *mesg = ptr->msg;
	    register short i = 9;
    
	    while( --i ) *mesg++ = *msg++;
	  }
	/* If there aren't any messages which have not been "Received",
	   then put this message as first to be "Received".
	   "Receiving" - is to attempt to handle a server request.
	 */
	if( ReceivePtr == NULL )
	  {
	    ReceivePtr = ptr;
    
	    /* Try to deliver the segment if any */
	    if (RecBufPtr != NULL)
	      {
		if( KDebug ) printf( "segsize = %d\n", length );
		if( length > RecBufSize ) length = RecBufSize;
		*RecBufSizePtr = length;
		bcopy(packet->kernelpacket.data, RecBufPtr, length+(length&1)); 
		RecBufPtr = 0; /* Flag to discard subsequent segments. */
	      }
	  }
      }

exit:
    SendQueueLock = 0;
 } /* HandleSend */


ProcessBreathOfLife( packet, msg, srcpid, seqno )
  struct KernelPacket *packet;
  register long * msg;
  ProcessId srcpid;
  short unsigned seqno;
  /* Handle an incoming breath-of-life packet. */
  {

    if( (CurrentOp == VALID_PID_OP) && (srcpid == RemotePid)
	 && (seqno == CurrentSeqNo) )
      {
	if( KDebug ) printf( "ValidPid OK\n" );
	KernelOpStatus = DONE;
      }
  }

ProcessNack( packet, msg, srcpid, seqno )
  struct KernelPacket *packet;
  register long * msg;
  ProcessId srcpid;
  short unsigned seqno;
  /* Handle a Nack 
   * This means that the process no longer exists.
   */
  {

    if( (CurrentOp != NULL) && (srcpid == RemotePid)
	 && (seqno == CurrentSeqNo) )
      {
	if( KDebug ) printf( "Nack receive from pid 0x%x\n", srcpid );
	KernelOpStatus = NACKED;
      }
  } /* ProcessNack */

MoveFromAnswer(packet,srcpid,length,localaddr, seqno)
 /* Handle a move from put data in right place
  *
  */
   struct KernelPacket *packet;
   ProcessId srcpid;
   unsigned length;
   unsigned localaddr;
   short unsigned seqno;
  {

    if( (CurrentOp == MOVE_FROM_OP) && (RemotePid == srcpid) && (length <= MAX_APPENDED_SEGMENT) &&
	(seqno == CurrentSeqNo)  && (localaddr == LocalAddr) && (length <= TotalLength) )
      {
	bcopy( packet->kernelpacket.data,LocalAddr, length + (length&1) ); 

	LocalAddr += length;
	TotalLength -= length;
	RemAddr += length;
	Retrans = 0; /* as long as still getting packets there is hope */
	if( TotalLength == 0 )
	  {
	    KernelOpStatus = DONE;
	  }
      }
  }


FlushNewMessages(vUserNumber)
    unsigned vUserNumber;
    /* Discards any unprocessed message with user number "vUserNumber"
     * that should be handled instead by a (newly forked) session with this user
     * number.
     */
  {
    register struct SendRecord *ptr, *prevptr;

    SendQueueLock = 1;
    sigblock( SigMask(PACKET_RECEIVED) );

    for (ptr = SendQueue; ptr != NULL; prevptr = ptr, ptr = ptr->next)
	if ((ptr->userNumber == vUserNumber) &&
	    (ptr->state == UNPROCESSED) &&
	    ((ptr->forwarder&GROUP_ID_BIT) != 0))
	  {
	    /* Take the record out of the queue and put it in free storage. */
	    if (ptr == SendQueue)
		SendQueue = ptr->next;
	    else  /* take out of list */
		prevptr->next = ptr->next;
	    SendFree(ptr);
	  }

    SendQueueLock = 0;
    sigsetmask(0);
  }


void AlarmHandler()
 /* 
  *  Initiate retransmissions by setting global flag to say retransmit .
  *  Queue a time-out message for the server.
  */
  {
    register struct SendRecord *ptr, *msgbuf, *prevptr;
    register IoRequest *msgptr;
    register int oldsigmask;

    /*
     * Manufacture a time-out message if not interrupting critical part
     * of the kernel.
     */ 
    if( SendQueueLock )
      {
        if( (CurrentOp != NULL) && AlarmSig == REPEAT ) AlarmSig = RETRANSMIT;
	alarm( AlarmRate );
	return;
      }
    oldsigmask = sigblock( SigMask(PACKET_RECEIVED) );
    msgbuf = SendAlloc();
    if (msgbuf == NULL)
      {
	/* We've run out of free records.  We try to reclaim a record 
	 * that was being used solely for duplicate suppression purposes.
	 */
	msgbuf = ReclaimSendRecord();
	if (msgbuf == NULL)
	    goto exit;
      }

    msgbuf->state = UNPROCESSED;
    msgbuf->srcpid = TimerPid;
    msgbuf->next = NULL;
    msgptr = (IoRequest *) msgbuf->msg;
    msgptr->requestcode = TIMEOUT;

    /* Search for the end of the SendQueue */
    for (ptr = SendQueue; ptr != NULL; prevptr = ptr, ptr = ptr->next)
	{}
    if (SendQueue == NULL)
	SendQueue = msgbuf;
    else
	prevptr->next = msgbuf;

    if( ReceivePtr == NULL ) ReceivePtr = msgbuf;

exit:
    if( (CurrentOp != NULL) && AlarmSig == REPEAT ) AlarmSig = RETRANSMIT;
    alarm( AlarmRate );
    sigsetmask(oldsigmask);
  }

void DeadChildHandler()

   /*
    * If root server then do nothing.
    * Else a session must mark that a child has died.
    */
  {
    /* All children require status check before exiting */
    ChildrenAreDead = 1;

  } /* DeadChildHandler */

void KillSession()
  /* 
   * Kill all process instances associated with this session
   * before we commit suicide.
   */
  {
    signal( PACKET_RECEIVED, SIG_IGN );
    signal( SIGALRM, SIG_IGN );
    signal( SIGCHLD, SIG_IGN );
    signal( SIGINT, SIG_IGN );
    signal( SIGTERM, SIG_IGN );

    if (KDebug) printf( "KillSession\n");
    ReclaimInstances(RI_RELEASE_ALL);    /* release all instances */
    CheckProcesses();

    exit( 0 );

  } /* KillSession */

void TerminalStopHandler()
/*
 * Handles the SIGTSTP signal.
 * Turns-on or off debugging mode.
 */
  {
    static long SavedDebug = -1;	/* all debug bits */

    if ( Debug == 0 )
      {
        SetDebugMode( SavedDebug );
      }
    else
      {
        SavedDebug = Debug;
        SetDebugMode( 0 );
      }
  } /* TerminalStopHandler */


void BadSystemCallHandler()
/*
 * Not really sure what to do other then output debugging information
 */
  {
    if ( GDebug )
      {
        printf( "Bad system call interrupt\n" );
      }
  } /* BadSystemCallHandler */


ProcessId GenerateNewVPid()
    /* Generates a new (unique) V process id. */
  {
    /* Increment the pid and make sure that its low-order 16 bits don't
     * conflict with the kernel process pid or the common local group id.
     * NOTE: We also make sure that these 16 bits, if interpreted as 
     * the \high-order/ 16 bits of a pid in the \opposite/ byte-order, would
     * not conflict with the logical host number, nor with a group id.
     * We do this because the packet filters allow for incoming packets (and 
     * thus the pids) to be in little-endian OR big-endian order, without
     * explicitly checking which order the packet is \really/ in.
     */
    unsigned newPidLow = newCandidatePid & 0xFFFF;

    do
      {
	++newPidLow;
	if ((newPidLow & ByteSwapShort(GROUP_ID_BIT >> 16)) != 0)
	    newPidLow += ByteSwapShort(GROUP_ID_BIT >> 16);
	newPidLow &= 0xFFFF;
      }
    while (newPidLow == COMMON_SERVER_PID ||
	   newPidLow == LOCAL_KERNEL_PROCESS_PID ||
	   newPidLow == ByteSwapShort((unsigned)LogicalHostPid >> 16));

    newCandidatePid = LogicalHostPid + newPidLow;
    return newCandidatePid;
  }
