/*
 * Unix V file server naming protocol 
 */
#include "config.h"
#include <Vnaming.h>
#include <naming.h>
#include <types.h>
#include <stat.h>
#include <errno.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <server.h>
#include <debug.h>
#include <Vgroupids.h>
#include <Vdirectory.h>

/* Character string name request handlers */
extern SystemCode GetContextId();
extern SystemCode GetContextName();
extern SystemCode GetFileName();

/* Context cache (tree) manipulation routines */
extern SystemCode InitContexts();
extern SystemCode InsertContextNode();
extern DeleteContextNode();
extern SystemCode ChangeContext();
extern char *FindContext();
extern FreeContextTree();			

/* Unix string name handlers */
extern SystemCode GetName();
extern char *ParseName();

/* Imports */
extern FileInstance *GetInstance();
extern char BlockBuffer[];
extern SessionDesc *Session;
extern int errno;
extern int PublicServer;
extern ProcessId MyPid;
extern GroupId CommonGroupId;
extern char *HostName;

char *WellKnownContext[ MAX_WELL_KNOWN_CONTEXTS ] = { NULL };


ContextId CurrentContext = 0;		/* Context Id of DEFAULT_CONTEXT */
char *CurrentContextName = "/";	
ContextNode *RootContextNode = NULL;	/* root of tree cache */

/* Following routines are used to manage the context name cache */
FreeContextTree( cnode )
    ContextNode *cnode;
/*
 * recursively free the context tree.
 */
  { 
    if ( cnode->left != NULL ) FreeContextTree( cnode->left );
    if ( cnode->right != NULL ) FreeContextTree( cnode->right ); 
    free( cnode );
  } /* FreeContextTree */

SystemCode InitContexts()
/*
 * Free all the nodes in the context cache (tree).
 * Redefine the well-known contexts DEFAULT_CONTEXT and PUBLIC_CONTEXT, and cd
 * to DEFAULT_CONTEXT.
 */
  {
    extern char *malloc(), *strcpy();
    ContextId cid;
    register int i;

    for (i = 0; i < MAX_WELL_KNOWN_CONTEXTS; i++)
        WellKnownContext[i] = NULL;

    WellKnownContext[DEFAULT_CONTEXT] = "/";
	
    /* PUBLIC_CONTEXT is invalid for non-public servers */
    WellKnownContext[PUBLIC_CONTEXT] = PublicServer ? PUBLIC_DIR : NULL;
        
    if (RootContextNode != NULL)
      {
        FreeContextTree(RootContextNode);
	RootContextNode = NULL;
      }

    return ChangeContext(DEFAULT_CONTEXT);
  } /* InitContexts */    

char *ParseName( NewName, OldName, IsaDirectory )
    char *NewName, *OldName;
    int IsaDirectory;
/*
 * Take OldName and parse out all the extraneous '.', '..', and '/'
 *  correctly while transferring into NewName.
 * If IsaDirectory is true, then terminate the name with a '/'
 *  otherwise it shouldn't end in a '/'.
 * Always returns its first argument.
 * NOTE: OldName must start with a '/'.
 * %%% BUG: This whole approach is wrong when symbolic links to 
 *  directories are permitted.  Have to search our way backwards
 *  up the file system as "pwd" does. 
 */
  { 
    register char *np = NewName;
    register char *cp;
    
    *np++ = '/';
    cp = OldName + 1;	/* skip over initial '/' */

    /* parse "name" in unix hierarchical fashion */
    /* Note the extremely clever algorithm */
    while( *cp != NULL )
      {
	/* first get rid of any extra '/'s */
	while( *cp == '/' ) cp++;

	/* determine if this part is '.' or '..' */
        if ( *cp == '.' ) 
	  {
	    /* starts with a '.' */
	    switch( *++cp )
	      {
		case '/': /* go nowhere */
			cp++;
		case NULL: /* same as above, but no need to increment */
			break;

	        case '.': /* is at least "..", so we have to read on */
			if ( *++cp == '/' || *cp == NULL )
			  {
			    /* go up one in the unix directory tree,
			       but don't back over the root */
			    np--;
			    while( np > NewName && *--np != '/' );
			    np++;
			    break; /* switch */
			  }
			/* else the name is "..<foobar>" */
			*np++ = '.';
			/* intentional fall thru to ".<foobar>" */
		default: 
			*np++ = '.';
			while( *cp != NULL ) /* copy name including '/' */
			    if ( (*np++ = *cp++) == '/' )
			        break;
			break;
	      } /* switch */
	  }
	else
	    while( *cp != NULL ) 
	        if ( (*np++ = *cp++) == '/' )
		    break;
      }
	
    if ( IsaDirectory )
      {
        /* always end with a '/' as a convention */
        if ( np[-1] != '/' ) *np++ = '/';
      }
    else
      {
        while ( np[-1] == '/' ) 
	    if ( --np <= NewName )
	      {
	        np++;
		break;
	      }
      }

    *np = NULL;		/* terminate */
    if ( CDebug ) printf( "Parsed name = %s\n", NewName );
    return( NewName );
 
  } /* ParseName */

SystemCode InsertContextNode( Name, NameLength, Cid )
    register char *Name;
    unsigned NameLength;
    ContextId *Cid;
/*
 * Get a context id for the Name and return its value.
 * If the user cannot access the context, then set the return
 *  code accordingly.
 * Calls ParseName to convert the name to canonical form (without
 *  ".." or "." components, repeated slashes, etc.
 */
  {
    register ContextNode *cnp,*TmpCnp;
    register ContextId cid;
    static short unsigned CidGenerator;
     
    /* should be in the correct context */
    /* try to get to requested context from this one */
    if ( chdir( Name ) != 0 )
      {
        switch ( errno )
          { 
	    case ENOTDIR: /* not a directory. */
            case ENOENT: /* directory is not found. */
		    return( NOT_FOUND );	
		    break;
            case EACCES: 
		    return( NO_PERMISSION );	/* is this right? */
        	    break;
	    default:
		    return( INTERNAL_ERROR );
		    break;
          }
      }

    if ( CDebug ) printf( "InsertContextNode: Name is \"%s\"\n", Name );

    /* at this point, we know client has correct permissions */
    CidGenerator += 0xbbb1;	/* prime */
    *Cid = CurrentContext = cid = (MyPid<<16) + CidGenerator;
    
    /* see if we already have this name in our little cache */
    /* if not insert it */
    cnp = (ContextNode *)calloc( 1, sizeof( ContextNode ) );
    
    TmpCnp = cnp;	/* save this node */
    
    /* everything is OK, insert into the tree */
    if ( RootContextNode == NULL ) /* special case: no tree */
	cnp = RootContextNode = (ContextNode *) TmpCnp;
    else
      { /* search the tree */
        cnp = RootContextNode;
	for (;;)
	  {
	    if( cnp->contextid < cid)
		if ( cnp->right != NULL )
		    cnp = cnp->right;
		else
		  {
		    cnp->right = TmpCnp;
		    cnp = TmpCnp;
		    break;
		  }
	    else if( cnp->contextid > cid )
		if ( cnp->left != NULL )
		    cnp = cnp->left;
		else
		  {
		    cnp->left = TmpCnp;
		    cnp = TmpCnp;
		    break;
		  }
	    else
	      {
		/* In the tree already.  Reuse the same node. */
		free( TmpCnp );
		if ( cnp->name != NULL ) 
		    free(cnp->name);

		break;
	      }
	  } 
      }

    cnp->contextid = cid;

    /* allocate a reasonable length of string and parse the name into it */
    cnp->name = CurrentContextName 
        = ParseName( malloc( NameLength + 2 ), Name, 1 );
    return( OK );

  } /* InsertContextNode */

DeleteContextNode( cid )
    ContextId cid;
/*
 * Since deleting a name from the context cache is such a rare
 * occurance, the actual method is to invalidate the context node
 * by setting the name to NULL.  If the inode is reallocated to
 * another directory, then the InsertContextNode routine will take
 * the appropriate action by validating the node at that time.
 */
  {
    register ContextNode *cnp;
     
    if ( CDebug ) printf( "DeleteContextNode: 0x%x\n", cid );
    if ( cid < MAX_WELL_KNOWN_CONTEXTS ) 
      {
	WellKnownContext[ cid ] = NULL;
        return;
      }

    cnp = RootContextNode;
    while ( cnp != NULL )
      {
	if ( cnp->contextid == cid ) break;
	if ( cnp->contextid < cid )
	    cnp = cnp->right;
	else
	    cnp = cnp->left;
      }

    if ( cnp != NULL )
      { /* invalidate */
        free( cnp->name );
	cnp->name = NULL;
      }

  } /* DeleteContextNode */

char *FindContext( cid )
    register ContextId cid;
/*
 * search for the specified context.
 * returns the name if found, otherwise null.
 */
  {
    register ContextNode *cnp;

    if ( CDebug ) printf( "FindContext: cid = 0x%x\n", cid );

    if ( cid == CurrentContext )
    	return( CurrentContextName );

    if ( cid < MAX_WELL_KNOWN_CONTEXTS )
        return( WellKnownContext[ cid ] );


    cnp = RootContextNode;
    while ( cnp != NULL )
      {
	if ( cnp->contextid == cid ) break;
	if ( cnp->contextid < cid )
	    cnp = cnp->right;
	else
	    cnp = cnp->left;
      }
    if ( cnp == NULL )
	return( NULL );

    return( cnp->name );	/* will be null if invalid */

  } /* FindContext */    
    

SystemCode ChangeContext( cid )
    register ContextId cid;
/*
 * Change the current working directory to the specified context
 * If not possible, then return an appropriate error code.
 */
  {
    register ContextNode *cnp;
    register char *np;
    
    if ( cid == CurrentContext )
    	return( OK );

    if ( (np = FindContext( cid )) == NULL )
        return( INVALID_CONTEXT );

    /* try to change the specified context */
    if ( chdir( np ) != 0 ) 
      {
	/* have to delete this context from the tree */
	DeleteContextNode( cid );

        switch ( errno )
          { 
		
	    case ENOTDIR: /* not a directory! */
            case ENOENT: /* directory is not found! */
		    return( INVALID_CONTEXT );	/* panic if cid = 0 */
		    break;
            case EACCES: 
		    return( NO_PERMISSION );	/* is this right? */
        	    break;
	    default:
		    return( INTERNAL_ERROR );
		    break;
          }
      }
      
    /* Valid context */
    CurrentContext = cid;
    CurrentContextName = np;
    return( OK );

  } /* ChangeContext */



SystemCode MapContextPrefix( req, pid, ctx )
    register NameRequest *req;
    ProcessId pid;
    register ContextPair *ctx;
/*
 * Parse enough of the name in the given context to determine whether it 
 *  is included in the local name space.  Caller must determine whether 
 *  to DISCARD_REPLY upon error.
 */
  {
    char component[MAX_NAME_LENGTH+1];
    register int i, compstart;
    register char c;
    register SystemCode r;

    if ( CDebug ) printf("MapContextPrefix\n");

    for (;;)
      {
	/* Parse out next name component */
	compstart = req->nameindex;
        for ( i = 0; i < MAX_NAME_LENGTH; i++ )
          {
    	    if ( (c = BlockBuffer[req->nameindex]) == '\0' ) break;
	    req->nameindex++;
	    if (c == ']' || c == '/') break;
	    component[i] = c;
           }
        component[i] = '\0';

	switch ( ctx->cid )
	  {
	    case GLOBAL_ROOT_CONTEXT:
	      ctx->pid = VCSNH_SERVER_GROUP;
  
	      /* Check if this request was directed to storage servers */
	      if (strcmp(component, "storage") == 0)
		  ctx->cid = STORAGE_SERVER_CONTEXT;
	      else if (strcmp(component, ".") == 0)
		  continue;	/* stay where we are */
	      else if (*component == '\0')
		  /* Lookup succeeds but result is a multi-manager context */
		  return MULTI_MANAGER;
	      else
		  /* Lookup fails */
		  return NOT_HERE;
	      break;

	    case STORAGE_SERVER_CONTEXT:
	      ctx->pid = VSTORAGE_SERVER_GROUP;
  
	      if (strcmp(component, HostName) == 0)  /* Our own name? */
		  ctx->cid = DEFAULT_CONTEXT;
	      else if (  PublicServer &&	     /* Wild card? */
	      		 ( strcmp(component, "any") == 0 ||
			   strcmp(component, "all") == 0 )  )
		  ctx->cid = DEFAULT_CONTEXT;
	      else if (strcmp(component, ".") == 0)
		  continue;	/* stay where we are */
	      else if (strcmp(component, "..") == 0)
		{
		  /* ".." from storage servers context gets us back up to
		   *  global root context.  At this point the other
		   *  server types have given up, so we strip off the
		   *  already-parsed prefix and forward the request
		   *  to the group.
		   */
		  req->namecontextid = GLOBAL_ROOT_CONTEXT;
	  
		  Forward(req, pid, VCSNH_SERVER_GROUP);
		  return NOT_AWAITINGREPLY; /* no longer awaiting reply */
		}
	      else if (*component == '\0')
		  /* Lookup succeeds but in a multi-manager context */
		  return MULTI_MANAGER;
	      else
		  /* We are not the server the client wants to talk to */
		  return NOT_HERE;
	      break;

	    default:
	      ctx->pid = CommonGroupId;
  
	      /* Back off and handle the rest of the name locally */
	      req->nameindex = compstart;
	      return OK;

	  } /* switch */
	 
      } /* for */
    
  } /* MapContextPrefix */



char NameBuffer[ BUFFER_SIZE*2 ];

SystemCode GetName( req, pid, segsize, FileName, ctx )
    register NameRequest *req;
    ProcessId pid;
    unsigned *segsize;
    register char **FileName;
    register ContextPair *ctx;
/*
 * Return the fully qualified (local) pathname of the file.
 *  If the name specifies a multi-manager context, return its
 *  context id in ctx.
 * We don't try to handle ".." from the root of the Unix file
 *  system.  It's not worth the effort.
 */
  {
    extern char *strcat();
    unsigned NameLength;
    char *NamePtr;
    register SystemCode r;
    char *ContextName;
    SystemCode err;

    /* check validity of name length 
       NOTE: that we allow 0 length names which means "." in unix 
     */
    NameLength = req->namelength;
    if( NameLength > BUFFER_SIZE ||  NameLength < req->nameindex ) 
        return( BAD_ARGS );

    /* Was the name sent with the packet? */
    if ( *segsize < NameLength )
      { /* no, have to retrieve from remote host */
	NamePtr = req->nameptr;

	if( (r=MoveFrom( pid, NamePtr, BlockBuffer, NameLength )) != OK)
		return( r );
	*segsize = NameLength;
      }

    BlockBuffer[ NameLength ] = 0; /* terminate name */
    if ( CDebug ) 
        printf( "GetName: \"%s\"\n", &BlockBuffer[ req->nameindex ] );
   
    /* Handle multi-manager contexts we participate in. */
    ctx->cid = req->namecontextid;
    if ( (err = MapContextPrefix(req, pid, ctx)) != OK )
	return err;

    /* Is the name unix context independent? */
    if ( (*(NamePtr = &BlockBuffer[ req->nameindex ])) == '/' )
    	*FileName = NamePtr;	/* no need for context prefix */
    else
      { 
        /* get the context name */
        if ( (ContextName = FindContext( ctx->cid )) == NULL )
    	    return( INVALID_CONTEXT );

	/* prepend it to the name specified in the request */
	if (strlen(ContextName) + strlen(NamePtr) > sizeof(NameBuffer)) 
	    return ILLEGAL_NAME;
        strcpy( NameBuffer, ContextName );	/* copy context name */
	*FileName = strcat( NameBuffer, NamePtr ); /* append file name */
      }
    
    return( OK );

  } /* GetName */


/* CHARACTER STRING NAME REQUEST HANDLERS */
SystemCode GetContextId( req, pid, segsize )
    ContextRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Find the ContextId for the specified name.
 * Once found, store in a cache that maps ContextIds to names
 * (which is the harder of the two functions).
 */
  {
    register SystemCode r;
    ContextPair ctx;
    char *NamePtr;
  
    if( Session != NULL ) Session->inactivity = 0;

    if( CDebug ) printf( "GetContextId\n" );

    r = GetName( req, pid, &segsize, &NamePtr, &ctx );

    if (r != OK)
      {
        if (r == MULTI_MANAGER)
          {
	    ((ContextReply *)req)->context = ctx;
	    return OK;	
          }
	else 
	    return( r );
      }

    r = InsertContextNode( NamePtr, strlen( NamePtr ), &ctx.cid );
    if (r != OK) return( r );
    
    /* valid context */
    if ( CDebug ) printf( "Context id is 0x%x\n", ctx.cid );
    ((ContextReply *)req)->context.cid = ctx.cid;
    ((ContextReply *)req)->context.pid = CommonGroupId;
    
    return( OK );
    
  } /* GetContextId */



char AbsNameBuffer[ BUFFER_SIZE * 2 ];

SystemCode GetContextName( req, pid )
    ContextRequest *req;
    ProcessId pid;
/*
 * Search for the name of the specified context
 */

  {
    ContextId cid;
    unsigned namelen;
    register char *np;
    register ContextNode *cnp;
    unsigned segsize;
    char *segptr;

    segsize = req->namelength, segsize;
    req->namelength = 0;

    if ( Session != NULL ) Session->inactivity = 0;

    cid =  req->context.cid;

    if ( CDebug ) printf( "GetContextName: ContextId = 0x%x\n", cid );

    switch (cid)
      {
	case GLOBAL_ROOT_CONTEXT:
	  np = "[";
	  break;
	  
	case STORAGE_SERVER_CONTEXT:
	  np = "[storage/";
	  break;
	  
	default:
	  /* Look up context id in the cache */
          if ( (np = FindContext( cid )) == NULL )
              return( INVALID_CONTEXT );

          /* Tack on prefix to make this name absolute */
          strcpy(AbsNameBuffer, "[storage/");
          strcat(AbsNameBuffer, HostName);
          strcat(AbsNameBuffer, "]");
          strcat(AbsNameBuffer, np);
          np = AbsNameBuffer;
 	  break;
      }
 
    /* enough space to write it? */
    if ( (namelen = strlen( np )) > segsize )
        return( BAD_BUFFER );
  
    if ( CDebug ) printf( "Context name is \"%s\"\n", np );

    /* where to put it */
    segptr = req->nameptr;

    /* how much we are sending */
    ((ContextReply *)req)->namelength = namelen;

    /* context to interpret it in */
    cid = ANY_CONTEXT;
    ((ContextReply *)req)->context.cid = cid;
    ((ContextReply *)req)->context.pid = CommonGroupId;

    if ( namelen <= MAX_APPENDED_SEGMENT && Session != NULL )
      { /* put it in the reply packet */
        ((ContextReply *)req)->replycode = OK; 
        ReplyWithSegment( req, pid, np, segptr, namelen );
        return( NOT_AWAITINGREPLY ); 
      }

    /* long name! */
    return( MoveTo( pid, segptr, np, namelen ) );

  } /* GetContextName */



SystemCode GetFileName( req, pid )
    ContextRequest *req;
    ProcessId pid;
/*
 * Search for the instance id in the request and return the name 
 * into the requesting processes segment.
 */

  {
    ContextId cid;
    InstanceId id;
    register FileInstance *desc;
    unsigned namelen;
    unsigned segsize;
    char *segptr;

    segsize = req->namelength;
    req->namelength = 0;

    if ( Session != NULL ) Session->inactivity = 0;

    id = req->instanceid;

    if ( CDebug ) printf( "GetFileName: InstanceId = 0x%x\n", id );

    /* Find the instance */
    if ( (desc = GetInstance( id )) == NULL )
        return( INVALID_FILE_ID );
    
    if ( desc->name == NULL )
        return( NOT_FOUND );

    if (desc->type == MMCONTEXT_INSTANCE)
        strcpy(AbsNameBuffer, desc->name);
    else
      {
	/* Tack on a prefix to make name absolute */
	strcpy(AbsNameBuffer, "[storage/");
	strcat(AbsNameBuffer, HostName);
	strcat(AbsNameBuffer, "]");
	strcat(AbsNameBuffer, desc->name);
      }

    /* enough space to write it? */
    if ( (namelen = strlen( AbsNameBuffer )) > segsize )
        return( BAD_BUFFER );
  
    if ( CDebug ) printf( "File name is \"%s\"\n", desc->name );

    /* where to put it */
    segptr = req->nameptr;

    /* how much we are sending */
    ((ContextReply *)req)->namelength = namelen;

    if ( namelen <= MAX_APPENDED_SEGMENT && Session != NULL )
      { /* put it in the reply packet */
        ((ContextReply *)req)->replycode = OK; 
        ReplyWithSegment( req, pid, AbsNameBuffer, segptr, namelen );
        return( NOT_AWAITINGREPLY ); 
      }

    /* long name! */
    return( MoveTo( pid, segptr, AbsNameBuffer, namelen ) );

  } /* GetFileName */



SystemCode GetAbsoluteName( req, pid, segsize )
    ContextRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Search for the file named in the request and return the absolute name 
 *  into the requesting process's segment.  If the file does not exist
 *  but is in our partition of the name space, make our best guess as to
 *  what the name would be if it did exist.
 */
  {
    register SystemCode r;
    ContextPair ctx;
    char *np;
    int namelen;

    if( Session != NULL ) Session->inactivity = 0;

    if( CDebug ) printf( "GetAbsoluteName\n");

    r = GetName( req, pid, &segsize, &np, &ctx );

    if (r == MULTI_MANAGER)
      {
	/* Multi-manager object */
	r = OK;
	switch (ctx.cid)
	  {
	    case GLOBAL_ROOT_CONTEXT:
	      np = "[";
	      break;
	      
	    case STORAGE_SERVER_CONTEXT:
	      np = "[storage/";
	      break;
	      
	    default:
	      return NOT_FOUND;  /* should not happen */
	  }
      }
    else if (r == OK)
      {
	/* Single-manager object in our partition of the name space */

	r = InsertContextNode( np, strlen( np ), &ctx.cid );

	strcpy(AbsNameBuffer, "[storage/");
	strcat(AbsNameBuffer, HostName);
	strcat(AbsNameBuffer, "]");
	if (r == OK)
	  {
	    /* Locally implemented single-manager context */
	    strcat(AbsNameBuffer, CurrentContextName);
	  }
        else if (r == NOT_FOUND)
	  {
	    /* Nonexistent or not a context */
	    r = OK;
	    ctx.pid = 0;
	    ParseName(AbsNameBuffer + strlen(AbsNameBuffer), np, 0);
	  }
	else return( r );

	np = AbsNameBuffer;
      }
    else return( r );

    /* enough space to write it? */
    if ( (namelen = strlen( np )) > segsize )
        return( BAD_BUFFER );
  
    if ( CDebug ) printf( "Context name is \"%s\"\n", np );

    /* how much we are sending */
    ((ContextReply *)req)->namelength = namelen;

    /* context id, if given name specifies a valid context */
    ((ContextReply *)req)->context = ctx;

    if ( namelen <= MAX_APPENDED_SEGMENT && Session != NULL )
      { /* put it in the reply packet */
        ((ContextReply *)req)->replycode = OK; 
        ReplyWithSegment( req, pid, np, req->nameptr, namelen );
        return( NOT_AWAITINGREPLY ); 
      }

    /* long name! */
    return( MoveTo( pid, req->nameptr, np, namelen ) );

  } /* GetAbsoluteName */



SystemCode QueryName( req, pid, segsize )
    ContextRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Reply if the given name is within our partition of the global
 *  name space, advancing reply->nameindex just beyond the shortest
 *  prefix that maps to a single-manager context (or to the end
 *  of the name if we were still in a multi-manager context after
 *  mapping the entire name).  Then set reply->context to the context
 *  to which the indicated prefix maps.
 */
  {
    register SystemCode r;
    ContextPair ctx;
    char *np;
#define reply ((ContextReply *)req)

    if( Session != NULL ) Session->inactivity = 0;

    if( CDebug ) printf( "QueryName\n" );

    /* GetName does all the real work */
    r = GetName( req, pid, &segsize, &np, &ctx );

    reply->context = ctx;
    if (ctx.pid == CommonGroupId)
        reply->flags = GROUP_PREFIX;	/* cooperative group implementation */
    else
        reply->flags = 0;		/* partitioned among group members */

    if (r == MULTI_MANAGER) r = OK;
    return( r );

#undef reply
  } /* QueryName */


