/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 */

/*
 * Host name lookup has its own table which is V specific.
 */
#include "config.h"
#include <Venviron.h>
#include <Vsession.h>	/* for LookupServer only */
#include "debug.h"
#include "kernel.h"
#undef NULL
#include <stdio.h>
/*
 * Host server lookup using host table file provided with server.
 * The file may contain entries of the form:

# this is a comment
<hostname> <hexpid>
   
 * Blank lines are also acceptable and there must be at least one
 * space or tab inbetween <hostname> and <hexpid>.  Note that
 * <hexpid> need only contain the logical host number for each host.
 */

#define MAX_HOST_NAME_LEN	32
FILE *HostTableFile;


MakeLowerCase( cp )
    register char *cp;
  {
    cp--;
    while( *++cp != 0 )
        if ( *cp >= 'A' && *cp <= 'Z' )
	    *cp -= 'A' - 'a';
	
  } /* MakeLowerCase */

InitHostTable()
    /*
     * Open the host file.
     */
  {
    if ( (HostTableFile = fopen( HOST_TABLE_FILE, "r" )) == 0 )
      {
        if ( LDebug )
	    perror( HOST_TABLE_FILE );
	return( 0 );
      }
  } /* InitHostTable */

ProcessId GetServerPid( host )
    char *host;
    /*
     * Look for the host in the HostTableFile.
     * If there is a match, then try to parse a hexpid
     * from the entry.  If any error occurs, return 0.
     */
  {
    static int hostTableInitialized = 0;
    extern char BlockBuffer[];
    char HostName[ MAX_HOST_NAME_LEN + 1 ];
    register char *hnp;
    register int hostnamelen;
    register char *cp;
    ProcessId serverpid;
    
    if ( !hostTableInitialized )
      {
        if ( !InitHostTable() )
	    return( 0 );		/* file not found */
	hostTableInitialized = 1;
      }
    
    if ( fseek( HostTableFile, 0, 0 ) == -1 )
        return( 0 );

    /* Make the host name palatable */
    hnp = HostName;
    strncpy( hnp, host, MAX_HOST_NAME_LEN );
    hnp[ MAX_HOST_NAME_LEN ] = NULL;
    hostnamelen = strlen( hnp );
    MakeLowerCase( hnp );

    if ( LDebug )
        printf( "GetServerPid: %s\n", hnp );
    cp = BlockBuffer;
    while( fgets( cp, 256, HostTableFile ) != NULL )
      {
        char *address, *index();

	if ( *cp == '#' || *cp == '\n' )
	    continue;	 	/* this line is a comment or blank */	
        address = index(cp,'\t');
	if (address==NULL) address = index(cp,' ');
	if (address==NULL)
	  {
	    /* malformed entry */
	    if (GDebug)
		printf("serverpid string malformed %s name is %s",
		       cp, HostName);
	    continue;
	  }

	*address = '\0';
	MakeLowerCase( cp );
	if ( strcmp( cp, hnp ) == 0 )
	  {
	    /* The name and entry matched so try to read server pid */
  	   if ( sscanf( ++address, "%x", &serverpid ) == 0 ) 
	     {
		/* Couldn't convert the entry */
	        if (GDebug)
		    printf("serverpid string invalid %s name is %s",
		     	   cp, HostName);
	        return(0);
	      }
            if (GDebug)
		    printf("serverpid string valid: %s name is %s, pid %x",
		     	   cp, HostName, serverpid);

	    /* NOTE: The following statement should really be conditionally
	     * compiled, depending upon LITTLE_ENDIAN.  However, 
	     * because we still have to look up other hosts besides our own,
	     * (for Vload) for now we just assume that the host is 
	     * little-endian.
	     */
	    return ( (serverpid & 0xfffc0000) | LITTLE_ENDIAN_HOST
			| GROUP_ID_BIT | COMMON_SERVER_PID);
	  }
      } 
      
    return( 0 );	/* look up failed */
	    
  } /* GetServerPid */


SystemCode LookupServer( req, pid, segsize )
   register LookupServerRequest *req;
   ProcessId pid;
   unsigned segsize;
  /* 
   * Get the server pid corresponding to the specified server name.
   */
  {
#define reply	((LookupServerReply*)req)
    char *name;
    SystemCode r;
    unsigned char host;
    unsigned nameLength;
    char *namePtr;

    if( SDebug ) printf( "Lookup server request from %8x\n", pid );

    nameLength = req->namelength;
    namePtr = req->hostname;
    if( nameLength > 100 ) return( ILLEGAL_REQUEST );
    name = BlockBuffer;
    if( segsize < nameLength )
      {
	if( (r=MoveFrom(pid, namePtr, name, nameLength)) != OK ) return( r );
      }
    name[nameLength] = 0; /* make a string */

    if( (pid = GetServerPid(name)) == 0 ) return( ILLEGAL_NAME );
/* RSF: Kludge - "pid" is now a local group id for the appropriate V server.  
 * However, if Vload sends a message to such a pid, it gets confused by the 
 * fact that the replier's pid (an individual pid) is different.  Since Vload 
 * is currently the only thing that uses LOOKUP_SERVER, and since this 
 * routine is going away eventually, we change "pid" to a `main server' pid
 * instead.
 */
    pid = (pid&0xFFFF0000&~GROUP_ID_BIT) | LOCAL_KERNEL_PROCESS_PID;

    reply->serverpid = pid;

    return( OK );

#undef reply
  } /* LookupServer */


ProcessId GetLogicalHost()

  /* Return the logical host number of this machine as a pid.
   */
  {
    extern char *HostName;
    static ProcessId LogicalHost = 0;
    static char hostname[ MAX_HOST_NAME_LEN + 1 ];
    
    if ( LogicalHost == 0 )
      {
        /* Have to Lookup the host */
	if ( HostName == NULL )
	  {
   	    HostName = hostname;
	    gethostname( hostname, sizeof( hostname ) );
	    hostname[ sizeof(hostname)-1 ] = NULL; /* null terminate just in case */
	    MakeLowerCase(hostname);
          }	
	/* Get server pid for this host and extract Logical Host id */
	LogicalHost = GetServerPid(HostName) & (0xFFFF0000 & ~GROUP_ID_BIT);
	if (LogicalHost == 0)
	  {
	    printf("GetLogicalHost: Unable to GetServerPid for %s\n", HostName);
	    exit(1);
	  }
      }
    return( LogicalHost );
  } /* GetLogicalHost */


/*
 * Routines for maintaining the "user correspondence" file, which is used to 
 * map from V user number to a user name on the local host.
 *
 * The file contains entries of the form:
<V user number> <local user name>
# This is a comment  
 * Blank lines are also acceptable and there must be at least one
 * space or tab between <V user number> and <local user name>.
 * <V user number> is in decimal.
 */

FILE *UserCorrespondenceFile = NULL;

static int InitCorrespondenceTable()
    /* Open the user correspondence file (for reading), if it hasn't already
     * been opened. */
  {
    if (UserCorrespondenceFile == NULL)
      {
	UserCorrespondenceFile = fopen(USER_CORRESPONDENCE_FILE, "r");
	if (UserCorrespondenceFile == NULL)
	  {
	    if (LDebug)
		perror(USER_CORRESPONDENCE_FILE);
	    return (0);
	  }
      }
    return (1);
  }


static int ReopenCorrespondenceTable(openmode)
    /* Open (or reopen) the user correspondence file in the given mode. */
    char openmode[];
  {
    if (UserCorrespondenceFile == NULL)
	UserCorrespondenceFile = fopen(USER_CORRESPONDENCE_FILE, openmode);
    else
	UserCorrespondenceFile = freopen(USER_CORRESPONDENCE_FILE, openmode,
					 UserCorrespondenceFile);
    if (UserCorrespondenceFile == NULL)
      {
	if (LDebug)
	    perror(USER_CORRESPONDENCE_FILE);
	return (0);
      }
    return (1);
  }


static CloseCorrespondenceTable()
    /* Closes the user correspondence file. */
  {
    if (UserCorrespondenceFile != NULL)
      {
	fclose(UserCorrespondenceFile);
	UserCorrespondenceFile = NULL;
      }
  }


static char *LookupCorr(vUserNumber, entryIndex, entrySize)
    unsigned vUserNumber;
    int *entryIndex, *entrySize;
    /* Looks up "vUserNumber" in the (previously opened) user correspondence
     * file.  If an entry is found, then "entryIndex" returns the position in 
     * the file of the start of the entry (origin 0), and "entrySize" returns 
     * the length of the entry (in bytes).
     * This routine returns the user name if an entry is found, otherwise NULL.
     */
  {
#define ENTRY_BUFSIZE 256
    static char entryBuffer[ENTRY_BUFSIZE];  
    register char *cp;
    int countSoFar = 0;
    int newCount = 0;

    if (fseek(UserCorrespondenceFile, 0, 0) == -1)
        return(NULL);

    cp = entryBuffer;
    while(fgets(cp, ENTRY_BUFSIZE-1, UserCorrespondenceFile) != NULL)
      {
	extern char *index();

        register char *address;
	int result;
	unsigned foundVUserNumber;

	countSoFar += newCount;
	newCount = strlen(cp);
	if (*cp == '#' || *cp == '\n')
	    continue;	 	/* this line is a comment or blank */
        address = index(cp,'\t');
	if (address==NULL) address = index(cp,' ');
	if (address==NULL)
	  { /* malformed entry */
	    if (GDebug)
		printf("user correspondence string malformed: %s\n", cp);
	    continue;
	  }

	/* Try to parse the found string into a user number. */
	*address = '\0';
        if (sscanf(cp, "%d", &foundVUserNumber) == 0)
	  {
	    if (GDebug)
		printf("V user number string invalid: %s\n", cp);
	    continue;
	  }

	/* is this the one we want? */
	if (foundVUserNumber == vUserNumber)
	  {
	    char *userName = ++address;
	    
	    while ((*address != '\0') && (*address != '\n'))
		++address;

	    *address = '\0';
	    if (LDebug)
		printf("Found user name: %s\n", userName);
	    *entryIndex = countSoFar;
	    *entrySize = newCount;
	    return (userName);
	  }
      } 
      
    if (LDebug)
	printf("No existing user correspondence for V user number %d\n",
	       vUserNumber);
    return(NULL);	/* look up failed */
  }


SystemCode AddCorr(vUserNumber, userName)
    unsigned vUserNumber;
    char userName[];
    /* Adds a new user correspondence to the end of the correspondence file. */
  {
    if (!ReopenCorrespondenceTable("a"))
	return(BAD_STATE);		/* file not found!! */

    fprintf(UserCorrespondenceFile, "%d	%s\n", vUserNumber, userName);
    CloseCorrespondenceTable();
    return(OK);
  }


SystemCode DeleteCorr(vUserNumber)
    unsigned vUserNumber;
    /* Deletes any existing user correspondence for the given V user number. */
  {
#define COPY_BUFSIZE 512
    static char copyBuf[COPY_BUFSIZE];
    int entryIndex, entrySize, bytesToRead;
    FILE *tempCorrFile = NULL;

    if (!InitCorrespondenceTable())
	return (BAD_STATE);		/* file not found!! */

    /* Find the position of the existing correspondence, if any. */
    if (LookupCorr(vUserNumber, &entryIndex, &entrySize) == NULL)
	return (NOT_FOUND);

    if (fseek(UserCorrespondenceFile, 0, 0) == -1)
        return(NULL);

    /* Open a temporary file, and copy the contents of the existing
     * correspondence file to this temporary, except for the entry that we 
     * want deleted.
     */
    tempCorrFile = fopen(TEMP_CORRESPONDENCE_FILE, "w");
    if (tempCorrFile == NULL)
      {
	if (LDebug)
	    perror(TEMP_CORRESPONDENCE_FILE);
	return (BAD_STATE);
      }
    chmod(TEMP_CORRESPONDENCE_FILE, 0644);
    bytesToRead = entryIndex;
    while (bytesToRead > 0)
      {
	int bytesRead, readSize;

	readSize = (bytesToRead < COPY_BUFSIZE) ? bytesToRead : COPY_BUFSIZE;
	bytesRead = fread(copyBuf, 1, readSize, UserCorrespondenceFile);
	if (bytesRead <= 0)
	  {
	    if (LDebug)
		perror("Error reading user correspondence file");
	    fclose(tempCorrFile);
	    return (BAD_STATE);
	  }
	if (fwrite(copyBuf, 1, bytesRead, tempCorrFile) != bytesRead)
	  {
	    if (LDebug)
		perror("Error writing temporary file (1st stage)");
	    fclose(tempCorrFile);
	    return (BAD_STATE);
	  }
	bytesToRead -= bytesRead;
      }
    /* Continue copying from where the entry ends. */
    fseek(UserCorrespondenceFile, entryIndex + entrySize, 0);
    while (1)
      {
	int bytesRead;

	bytesRead = fread(copyBuf, 1, COPY_BUFSIZE, UserCorrespondenceFile);
	if (bytesRead <= 0)
	    break;
	if (fwrite(copyBuf, 1, bytesRead, tempCorrFile) != bytesRead)
	  {
	    if (LDebug)
		perror("Error writing temporary file (2nd stage)");
	    fclose(tempCorrFile);
	    return (BAD_STATE);
	  }
      }
    fclose(tempCorrFile);

    /* Now rename the temporary file as the real one. */
    CloseCorrespondenceTable();
    fclose(tempCorrFile);
    if (rename(TEMP_CORRESPONDENCE_FILE, USER_CORRESPONDENCE_FILE) == -1)
      {
	if (LDebug)
	    perror("Error on correspondence file rename");
	return (BAD_STATE);
      }

    return (OK);
  }


char *GetUserCorrespondence(vUserNumber)
    unsigned vUserNumber;
    /* Attempts to find the local user name, corresponding to the given V 
     * user number.  The user name string is returned if found, otherwise 
     * NULL is returned.
     */
  {
    int dummy1, dummy2;

    if (LDebug)
        printf("GetUserCorrespondence: %d\n", vUserNumber);

    if (!InitCorrespondenceTable())
	return(NULL);		/* file not found */
    
    return (LookupCorr(vUserNumber, &dummy1, &dummy2));
  }
