/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * Instance routines
 */

#include "config.h"
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vtime.h>
#include <sgtty.h>
#include <signal.h>
#include <pwd.h>
#include <server.h>
#include <types.h>
#include <stat.h>
#include <debug.h>
#include <sys/dir.h>

/* EXPORT */
extern FileInstance *AllocInstance();
extern FileInstance *GetInstance();
extern unsigned ReclaimInstances();
extern FreeInstance();
extern FileInstance *FindInstance();

InstanceId	Idgenerator = 0x5555; /* Random strange number to start */

FileInstance *FileInstances = NULL;

InitInstances()
    /* 
     * Initialize the instance list so that the head of the list is
     * always a null record.  This makes inserting and deleting from the
     * head of the list very simple.
     */
  {
    FileInstances = (FileInstance *) calloc( 1, sizeof(FileInstance *) );
  } /* InitInstances */


FileInstance *AllocInstance( pid, instancetype )
    ProcessId pid;
    InstanceType instancetype;

  /* Allocate a file or process instance descriptor and
   * return a pointer to the descriptor. The instance id is set
   * to the next "generation" for this descriptor.
   * Note that calloc must be used (vs. malloc), since some of the fields
   * must be zero.
   */
  {
    extern FileInstance *FileInstances;
    register FileInstance *desc;

    while( (++Idgenerator == 0) || GetInstance(Idgenerator) );

    switch ( instancetype ) 
      {
	case FILE_INSTANCE:
	    desc = (FileInstance *) calloc(1, sizeof(FileInstance));
	    break;

	case PROCESS_INSTANCE:
	    desc = (FileInstance *) calloc(1, sizeof(ProcessInstance));
	    break;

	case DIRECTORY_INSTANCE:
	case MMCONTEXT_INSTANCE:
	    desc = (FileInstance *) calloc(1, sizeof(DirectoryInstance));
	    break;

	default:
	    /* This is an internal error */
	    if (GDebug)
		printf("  AllocateInstance called with unknown instancetype = %d\n",
		       instancetype);
	    return NULL;    /* Internal error */
	    break;
      }
    if (desc == NULL) return NULL;	/* calloc failed */

    desc->name = NULL;
    desc->type = instancetype;
    desc->owner = pid;
    desc->id = Idgenerator;
    desc->link = FileInstances->link;
    FileInstances->link = desc;

    return( desc );
  }

FileInstance *GetInstance( fid )
    register InstanceId fid;

  /* Locate the instance descriptor with the corresponding
   * instance id if any and return a pointer to this descriptor.
   * Return NULL if no such descriptor.
   */
  {
    extern FileInstance *FileInstances;
    register FileInstance *desc;

    desc = FileInstances->link;

    while( desc != NULL )
      {
	if( desc->id == fid ) 
	  {
	    switch (desc->type)
	      {
		case FILE_INSTANCE:
		    if (desc->unixfile < 0)
		      {
			/* We have to reopen the file */
			/* Make a guess that we will have to reclaim an instance
			   in order to open another unix file */
			(void)ReclaimInstances(RI_FREE_DESCRIPTOR);
			desc->unixfile = open(desc->name,
					      desc->filetype&WRITEABLE ? 2 : 0);
			if (desc->unixfile < 0)
			  {
			    /* We couldn't find it, so we pretend it doesn't
			     * exist at all.
			     */
			    if (IDebug) 
				printf("  Unable to reopen \"%s\"\n",
				       desc->name);
			    FreeInstance(desc); 
			    return NULL;
			  }
		      }
		    desc->clock = 0;
		    break;

		case DIRECTORY_INSTANCE:
		    if (desc->unixfile == NULL)
		      {
			/* We have to reopen the file */
			/* Make a guess that we will have to reclaim an instance
			   in order to open another unix file */
			(void)ReclaimInstances(RI_FREE_DESCRIPTOR);
			desc->unixfile = (int)opendir(desc->name);
			if (desc->unixfile == NULL)
			  {
			    /* We couldn't find it, so we pretend it doesn't
			     * exist at all.
			     */
			    if (IDebug) 
				printf("  Unable to reopen \"%s\"\n",
				       desc->name);
			    FreeInstance(desc); 
			    return NULL;
			  }
		      }
		    desc->clock = 0;
		    break;
	      }
	    return desc;
	  }
	desc = desc->link;
      }
    return NULL;
  } /* GetInstance */

FreeInstance( desc )
    FileInstance *desc;

  /* 
   * Free the specified instance descriptor.
   */
  {
    register FileInstance *ptr;

    ptr = FileInstances;

    while (ptr != NULL && ptr->link != desc) ptr = ptr->link;

    if (ptr == NULL)
      {
	if (GDebug) 
	    printf("  Freeing bad instance descriptor\n");
      }
    else
      {       
	ptr->link = desc->link;
	ptr = desc;
	/* DirectoryInstances don't alloc their names, the context cache does */
	if ( ptr->name != NULL && ptr->type != DIRECTORY_INSTANCE ) 
	    free( ptr->name );

	if ( ptr->type == PROCESS_INSTANCE && 
	     ((ProcessInstance*)ptr)->lastdata != NULL )
	         free( ((ProcessInstance*)ptr)->lastdata );
	free( ptr );
      }
  }


unsigned ReclaimInstances(checkMode)
    int checkMode;
 /*
  * Check that the owner of each open instance exists.
  * If it doesn't, then for instances we must close the file
  * first and for jobs we must kill the process and close the pipe.
  * Returns the number of unix file descriptors that were reclaimed.
  * "checkMode" indicates further what action should be taken - see server.h 
  * for further details.
  */
  {
    register int reclaimed = 0;
    register FileInstance *ptr;
    FileInstance *prev;

    if (IDebug) 
	CheckFiles();
    prev = FileInstances;

    while ((ptr = prev->link) != NULL)
      {
	int instanceOK;
	
	if (IDebug) printf("  Checking instance 0x%x\n", ptr);
	if (checkMode == RI_RELEASE_ALL)
	    instanceOK = 0;
	else
	  {
	    instanceOK = ValidPid(ptr->owner);
	    if (instanceOK && (checkMode == RI_CHECK_ONLY))
		/* We only wanted to know if \any/ instances are valid. */
		goto reclaimExit;
	  }
	if (!instanceOK)
	    reclaimed += ReclaimSingleInstance(ptr, &prev);
	else 
	    prev = ptr; 

	if (checkMode == RI_FREE_DESCRIPTOR && reclaimed > 0)
	    goto reclaimExit; /* All we wanted was to free a descriptor. */
      }

    if (checkMode == RI_FREE_DESCRIPTOR && reclaimed <= 0)
      {
	/* No luck.  We now try to free a unix file descriptor. */
	register FileInstance *oldest;

        ptr = FileInstances;
	oldest = NULL;
        while ((ptr = ptr->link) != NULL)
          {
	    struct stat dummy;
		
	    /* We can only free file or directory instances that have not 
	     * already been freed, and that can be recovered by simply reopening
	     * the file with the stored name.
	     */
	    if (((ptr->type == FILE_INSTANCE && ptr->unixfile >= 0) ||
		 (ptr->type == DIRECTORY_INSTANCE && ptr->unixfile != NULL)) &&
	        ptr->name != NULL &&
		stat(ptr->name, &dummy) == 0)
		    /* Find least recently used descriptor */
		    if (oldest == NULL || oldest->clock < ptr->clock)
			oldest = ptr;
	  }
	if (oldest != NULL)
	  {
	    if (IDebug)
	        printf("  freeing unix file desc for \"%s\"\n", oldest->name);
	    if (oldest->type == DIRECTORY_INSTANCE)
	      {
		closedir(oldest->unixfile);
		oldest->unixfile = NULL;
	      }
	    else /* FILE_INSTANCE */
	      {
		close(oldest->unixfile);
		oldest->unixfile = -1;
	      }
	    oldest->clock = 0;	
	    reclaimed++;
	  }
      }
reclaimExit:
    if (IDebug) printf("  reclaimed %d files\n", reclaimed);
    return reclaimed;
  } /* ReclaimInstances */


FileInstance *FindInstance( id, instancetype )
    unsigned id;
    InstanceType instancetype;
/*
 * Find the id in the instance list.  
 * Only works for instancetype == PROCESS_INSTANCE, in which case
 * id is a UNIX pid.  Returns the first descriptor matching id or NULL
 * if nothing is found.  
 */
  {
    switch ( instancetype )
      {
	case PROCESS_INSTANCE:	/* find unix process id */
	  {
	    register ProcessInstance *pdesc;

	    for ( pdesc = (ProcessInstance *)FileInstances->link
		  ; pdesc != NULL
		  ; pdesc = pdesc->link ) 
		  if ((id ==  pdesc->pid) && (pdesc->type == PROCESS_INSTANCE))
		      return ((FileInstance *) pdesc);
	    break;
	  }

	default:
	  {
	    if (GDebug)
		printf("  FindInstance called with unknown instancetype = %d\n",
		       instancetype);
	    break;
	  }
      } /* switch */

    return NULL;    /* not found */

  } /* FindInstance */

CheckFiles() 
  {
    register int f;
    struct stat st;

    printf( "#  links Mode  Device  Inode  LastAccess  \n");
    for( f = 0; f < 20; f++) 
      {
	if (fstat(f, &st) == 0)
	  {
	    printf( "%2d %5x %4x  %7x  %5x %s"
			,f,st.st_nlink,st.st_mode,st.st_dev,st.st_ino,ctime(&(st.st_atime)));
	  }
      }
  }

IncInstanceClocks()
    /* Increment the clock field for all the file instances. */
   {
     register FileInstance *ptr;
     
     ptr = FileInstances;
     
     while((ptr = ptr->link) != NULL)
       {
         if (ptr->type == FILE_INSTANCE || ptr->type == DIRECTORY_INSTANCE)
	     ptr->clock++;
       }
  }


SystemCode GetInstanceInfo(req, pid)
   register IoRequest *req;
   ProcessId pid;
    /* Return (in a segment) information about current instances.
     * This feature is intended for debugging purposes.
     */
  {
    extern char BlockBuffer[];

    register char *buffer = BlockBuffer;
    register int reclaimed = 0;
    register FileInstance *ptr;
    FileInstance *prev;
    int goodCount;
    char str[128];

    buffer[0] = '\0';
    prev = FileInstances;

    while( (ptr = prev->link) != NULL )
      {
	int instanceOK;
	
	sprintf(str, "Instance 0x%x ", ptr->id);
	strcat(buffer, str); buffer += strlen(str);
	switch (ptr->type) 
	  {
	    case FILE_INSTANCE:
	      {
		struct stat dummy;
		
		sprintf(str, "(file: %s%s (%s))",
		        ptr->name,
			stat(ptr->name, &dummy) != 0 ? " (invalid name!)" : "",
			ptr->unixfile < 0 ? "closed" : "open");
		break;
	      }
	    case PROCESS_INSTANCE:
		sprintf(str, "(process: %s (%s))", ptr->name, 
			ptr->filetype&READABLE ? "readable" :
			ptr->filetype&WRITEABLE ? "writeable" : "???"); break;
	    case DIRECTORY_INSTANCE:
		sprintf(str, "(directory: %s (%s))",
		        ptr->name, ptr->unixfile == NULL ? "closed" : "open");
		break;
	    case MMCONTEXT_INSTANCE:
		sprintf(str, "(multi-manager context: %s)", ptr->name); break;
	    default:
		sprintf(str, "(???[%d])", ptr->type); break;
	  }
	strcat(buffer, str); buffer += strlen(str);

	sprintf(str, "\n    Owner = 0x%08x", ptr->owner);
	strcat(buffer, str); buffer += strlen(str);
	instanceOK = ValidPid(ptr->owner);
	if (instanceOK)
	    sprintf(str, ", valid\n");
	else
	    sprintf(str, ", invalid!  Reclaiming\n");
	strcat(buffer, str); buffer += strlen(str);

	if (!instanceOK)
	    reclaimed += ReclaimSingleInstance(ptr, &prev);
	else 
	    prev = ptr; 

      } /* while */

    sprintf(str, "reclaimed %d instances\n", reclaimed);
    strcat(buffer, str); buffer += strlen(str);

    prev = FileInstances;
    goodCount = 0;
    while( (ptr = prev->link) != NULL )
      {
	if (ptr->id != 0)
	    ++goodCount;
	prev = ptr;
      }
    sprintf(str, "Maintaining %d valid instances\n", goodCount);
    strcat(buffer, str); buffer += strlen(str);

    *buffer++ = '\0'; /* just in case */

      {
	int bytes = buffer-BlockBuffer;
	char *ptr = req->bufferptr;
	IoReply *reply = (IoReply *)req;
    
	if (bytes <= MAX_APPENDED_SEGMENT)
	  {
	    reply->bytecount = bytes;
	    reply->replycode = ((req->bytecount < bytes) ? BAD_BUFFER : OK);
	    ReplyWithSegment(reply, pid, BlockBuffer, ptr, bytes );
	    return NOT_AWAITINGREPLY;
	  } 
	else
	  {
	    return MoveTo(pid, ptr, BlockBuffer, bytes);
	  }
      }

  }

static int ReclaimSingleInstance(ptr, prev)
    FileInstance *ptr, **prev;
    /* Attempts to reclaim the instance whose descriptor is pointed to by "ptr".
     * ("*prev" is a pointer to the previous entry in the instance descriptor
     * list.)
     * If the instance really could be reclaimed then we return 1, otherwise 0.
     */
  {
    int reclaimed = 1; /* this may get reset below */
    register ProcessInstance *pd;
    int upid;

    switch (ptr->type) 
      {
	case FILE_INSTANCE:
	    if (ptr->name != NULL) 
		free(ptr->name);
	    /* try to free the unix file desc, if the file desc
	       had been recycled already then close will fail.
	       In which case we have reclaimed nothing */
	    if (close(ptr->unixfile) == -1) reclaimed = 0;
	    (*prev)->link = ptr->link;
	    free(ptr);
	    break;
    
	case DIRECTORY_INSTANCE:
	    if (ptr->unixfile != NULL)
		closedir((DIR *)ptr->unixfile);
	    else
		reclaimed = 0;
	    ptr->unixfile = NULL;
	    (*prev)->link = ptr->link;
	    free(ptr);
	    break;
    
	case MMCONTEXT_INSTANCE:	
	    (*prev)->link = ptr->link;
	    free(ptr);
	    break;
    
	case PROCESS_INSTANCE:
	    /*
	     * Kill the process, if not already dead.
	     * We must have at least one process descriptor around
	     * to "catch" the signal that should be received.
	     */
	    pd = (ProcessInstance *) ptr;
	    if (pd->state == PD_RUNNING || pd->state == PD_STOPPED)
	      {
		if (close(pd->pipedesc) == 0)
		    pd->pipedesc = -1;	/* invalidate */
		else
		    reclaimed = 0;	/* nothing freed */
	    
		upid = pd->pid;		/* save this process id */
		pd->pid = 0;		/* don't find this guy */
		if (FindInstance(upid, PROCESS_INSTANCE) != NULL)
		  {
		    /* Let the other instance worry about the SIGCHLD.
		       The owner of this instance no longer cares */
		    (*prev)->link = ((FileInstance*)pd)->link;
		  }
		else
		  {
		    /* kill the process and leave the descriptor 
		       around to catch the signal */
		    if (pd->pid != 0) killpg(getpgrp(pd->pid), SIGKILL);
		    pd->state = PD_AWAITING_DEATH;
		    pd->id = 0;		/* invalidate instanceid */
		    pd->pid = upid;
		    (*prev) = ptr;
		  }
	      }
	    else
	      {
		if (pd->state == PD_AWAITING_DEATH)
		  {
		    if (pd->pid != 0) killpg(getpgrp(pd->pid), SIGKILL);
		    reclaimed = 0;
		    (*prev) = ptr;
		  }
		else
		  {
		    /*
		     * At this point, the process is dead and
		     * we can release the instance.  If the 
		     * pipe was still open, then we have gained
		     * one more file descriptor.  
		     */
		    if (close(pd->pipedesc) != 0)
			reclaimed = 0;	/* nothing reclaimed */
		
		    (*prev)->link = ptr->link;	/* remove from list */
		    /* 
		     * Find the sibling. if not found then this is the
		     * last reference to name, thus we may free it 
		     */
		    if (FindInstance(pd->pid, PROCESS_INSTANCE) 
			 == NULL)
			free(pd->name);
		    if (pd->lastdata != NULL)
			free(pd->lastdata);
		    free(ptr);
		  }
	      }
	    break;
    
	default:
	    if (GDebug)
		printf("  ReclaimSingleInstance - bad instancetype = %d\n",
		       ptr->type);
	    reclaimed = 0;
	    (*prev) = ptr;
	    break;
      }

    return reclaimed;
  }
