/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 *
 * UNIX-specific file access routines, etc.
 */

#include "config.h"
#include <types.h>
#include <stat.h>
#include <errno.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vnaming.h>
#include <Vdirectory.h>
#include <server.h>
#include <debug.h>

extern SystemCode GetName();
extern char *ParseName();
extern FileInstance *GetInstance();
extern SystemCode SetupInstance();
extern char *FindContext();
extern SetDebugMode();
extern char *malloc();

SystemCode SetupInstance( file, desc, Name, pid )
    int file;
    register FileInstance *desc;
    char *Name;
    ProcessId pid;

    /* Check it is a disk file or directory and not a device.
     * Setup file instance descriptor for further access.
     */
  {
    extern SessionDesc *Session;
    struct stat buf;
    register SystemCode r;
    unsigned short mode;

    fstat( file, &buf );
    mode = buf.st_mode;

    if( Session == NULL )
      {
	if( !(mode&(S_IREAD >> 6)))
	  return( NO_PERMISSION);  
      }

    if( (mode &= S_IFMT) != S_IFREG && mode != S_IFDIR )
	return( NO_PERMISSION );

        /* Set up the file instance descriptor as required */
        desc->name = ParseName( malloc( strlen( Name ) + 2 ), Name, 0 );
        /* Block size picked to be most efficient for UNIX and network */

        desc->blocksize = BLOCK_SIZE;
        desc->lastblock = buf.st_size / BLOCK_SIZE;
        desc->lastbytes = buf.st_size % BLOCK_SIZE;
        desc->nextblock = 0;
        desc->unixfile = file; /* Remember the unix file descriptor */
	desc->owner = pid;

    return( OK );

  } /* SetupInstance */


SystemCode RemoveFile(req, pid, segsize)
    CreateInstanceRequest *req;
    ProcessId pid;
    unsigned segsize;
  {
    extern SessionDesc *Session;
    extern int errno;

    register SystemCode r;
    char *NamePtr;
    ContextPair dummy;

    if (Session == NULL) return NO_PERMISSION;
 
    Session->inactivity = 0;

    if ((r = GetName(req, pid, &segsize, &NamePtr, &dummy)) != OK)
	  return r;

    if (FDebug) printf("RemoveFile: \"%s\"\n", NamePtr);

    /* We check for the special case where the file that we have been 
     * asked to remove happens to be a currently maintained file instance 
     * that is currently \closed/.  In this case we need to reopen the file 
     * before unlinking it, so that the file can still be accessed thereafter
     * via the instance.
     * Note: all we do is compare pathnames, so this test won't always work
     * (e.g. if the names are different, but refer to the same file).  
     * Perhaps we should really be checking inode numbers???
     */
      {
	extern FileInstance *FileInstances;
	register FileInstance *ptr = FileInstances;
	
	while ((ptr = ptr->link) != NULL)
	    if (ptr->type == FILE_INSTANCE &&
	        strcmp(ptr->name, NamePtr) == 0 &&
		ptr->unixfile < 0)
	      {
		/* We have to reopen the file */
		/* Make a guess that we will have to reclaim an instance
		   in order to open another unix file */
		(void)ReclaimInstances(RI_FREE_DESCRIPTOR);
		ptr->unixfile= open(ptr->name, ptr->filetype&WRITEABLE ? 2: 0);
		if (ptr->unixfile < 0)
		  {
		    if (FDebug) printf("Couldn't reopen closed file!\n");
		    return BUSY;
		  }
	      }
      }

    /* Attempt to remove the requested file. */
    if (unlink(NamePtr) == -1)
      {
	if (FDebug)
	    printf("unlink() - Unix error number %d\n", errno);
	switch (errno)
	  {
	    case ENOTDIR:
	    case ELOOP:
		r = ILLEGAL_REQUEST;
		break;
	    case EACCES:
	    case EPERM:
	    case EROFS:
	    case EFAULT:
		r = NO_PERMISSION;
		break;
	    case ENOENT:
		r = NOT_FOUND;
		break;
	    default:
		r = INTERNAL_ERROR;
		break;
	  }
      }
    else
	r = OK;

    return r;
  } /* RemoveFile */


SystemCode QueryFile(req, pid)
   IoRequest *req; ProcessId pid;

  /* Handle a query file request for the named file.
   */
  {
    register FileInstance *desc;

    if ((desc = GetInstance(req->fileid)) == NULL) return INVALID_FILE_ID;
    
    return REQUEST_NOT_SUPPORTED;
  } /* QueryFile */


SystemCode ModifyFile(req, pid)
   IoRequest *req; ProcessId pid;

  /* 
   * Handle a modify file request.
   * Not implemented at present.
   */
  {
    register FileInstance *desc;
    long debug;

    if ((desc = GetInstance(req->fileid)) == NULL) return INVALID_FILE_ID;
    
    if (FDebug) 
         printf("ModifyFile: id 0x%x, pid 0x%x\n"
	 	  , req->fileid, pid);

    return REQUEST_NOT_SUPPORTED;

  } /* ModifyFile */


SystemCode RenameFile(req, pid, segsize)
    NameRequest *req;
    ProcessId pid;
    unsigned segsize;
    /* Rename a file.  The old name and new name are concatenated (separated
     * by a '\0') and pointed to by req->nameptr.  req->namelength is the
     * length of the total string.  req->unspecified[0] is the length of the
     * first name (not including the '\0').
     */
  {
    extern SessionDesc *Session;
    extern char BlockBuffer[];
    extern int errno;

    register SystemCode r;
    char *NamePtr, *sourceName, *destName;
    ContextPair dummy;
    unsigned sourceNameLength, destNameLength;

    if (Session == NULL) return NO_PERMISSION;
 
    Session->inactivity = 0;

    sourceNameLength = req->unspecified[0] + 1;		 /* includes the '\0' */
    destNameLength = req->namelength - sourceNameLength; /* doesn't ... */

    /* Get the source name: */
    if ((r = GetName(req, pid, &segsize, &NamePtr, &dummy)) != OK)
	  return r;

    /* Save this name away, so that it won't get overwritten by the 
     * destination name.
     */
    sourceName = malloc(strlen(NamePtr) + 1);
    if (sourceName == NULL)
	return NO_SERVER_RESOURCES;
    strcpy(sourceName, NamePtr);

    /* Get the destination name.  Note that we rely on the fact that the
     * second (destination) file name was left in "BlockBuffer", as a result
     * of the first GetName().  We shift this name to the front, and then call
     * GetName() a second time.
     */
    strncpy(BlockBuffer, &BlockBuffer[sourceNameLength], destNameLength);
    BlockBuffer[destNameLength] = '\0';

    req->nameptr += sourceNameLength;	/* just in case */
    segsize -= sourceNameLength;	/* ditto */
    req->namelength = destNameLength;
    req->nameindex = 0;
    if (BlockBuffer[0] == ROOT_ESCAPE_CHAR)
      {
	req->namecontextid = GLOBAL_ROOT_CONTEXT;
	++req->nameindex;
      }
    if ((r = GetName(req, pid, &segsize, &NamePtr, &dummy)) != OK)
      {
	free(sourceName);
        return ILLEGAL_NAME;
      }
    destName = NamePtr;

    if (FDebug)
	printf("RenameFile: from \"%s\" to \"%s\"\n", sourceName, destName);

    if (rename(sourceName, destName) == -1)
      {
	if (FDebug)
	    printf("rename() - Unix error number %d\n", errno);
	switch (errno)
	  {
	    case ENOTDIR:
	    case EINVAL:
	    case EISDIR:
	    case ENOTEMPTY:
		r = ILLEGAL_REQUEST;
		break;
	    case EACCES:
	    case EPERM:
	    case EROFS:
	    case EFAULT:
		r = NO_PERMISSION;
		break;
	    case ENOENT:
		r = NOT_FOUND;
		break;
	    case EXDEV:
		r = REQUEST_NOT_SUPPORTED;
		break;
	    default:
		r = INTERNAL_ERROR;
		break;
	  }
      }
    else
	r = OK;

    free(sourceName);
    return r;
  } /* RenameFile */
