/*
 * SystemCode QueryWorkstationConfig(keyword, value, maxlength)
 *
 * This routine provides a way for a program to discover details of
 *   the configuration of the workstation it is running on.  The keyword
 *   and returned value are character strings, with maxlength the size
 *   of the area available to store the value.  Conceptually, the routine
 *   queries a database of configuration information, with the workstation's
 *   identity and the keyword as keys, and returns the associated value.
 *   If the keyword has no associated value, an error code is returned,
 *   else OK.  The possible error codes include NOT_FOUND if there is
 *   no configuration information for this workstation at all, BAD_BUFFER
 *   if the value buffer is too small (in which case the value is truncated),
 *   BAD_ARGS if there is no value for the given keyword (but there
 *   is other information available for the workstation), or possibly
 *   other codes if there are internal problems.
 *
 * The current implementation uses configuration files stored in the
 *   directory [public]config on public storage servers, with names formed
 *   from the (primary) Ethernet hardware address of the workstation.
 *
 * Tim Mann, 3/13/84:  Created.
 */

#include <Vioprotocol.h>
#include <Vnaming.h>
#include <Vethernet.h>

File *OpenConfigFile(error)
    SystemCode *error;
  {
    /* Locate and open the proper config file */
    char configFileName[32];
    Message msg;
    register NameRequest *qerequest = (NameRequest *) msg;
    register QueryEnetReply *qereply = (QueryEnetReply *) msg;

    /* Figure out which config file to read based on our 
     *   Ethernet address */
    qerequest->requestcode = NQUERY_FILE;
    qerequest->nameptr = "ethernet";
    qerequest->namelength = sizeof("ethernet");
    qerequest->nameindex = 0;
    qerequest->namecontextid = DEFAULT_CONTEXT;

    Send(qerequest, GetPid(DEVICE_SERVER, LOCAL_PID));

    if (qereply->NetworkType == ENET_TYPE_3MBIT)
      {
	sprintf(configFileName, "[sys]config/C.%02x", 
		qereply->HostAddress.e3);
      }
    else /* qereply->NetworkType == ENET_TYPE_10MBIT */
      {
	sprintf(configFileName, "[sys]config/C.%04x%04x%04x",
		qereply->HostAddress.e10.addrhigh, 
		qereply->HostAddress.e10.addrmid,
		qereply->HostAddress.e10.addrlow);
      }

    return( Open(configFileName, FREAD, error) );
  }


SystemCode QueryWorkstationConfig(keyword, value, maxlength)
    char *keyword, *value;
    int maxlength;
  {
    SystemCode error;
    register int c, found;
    register char *p;
    register short size;
    File *configFile;

    configFile = OpenConfigFile(&error);
    if (error != OK) return (error);

    /* Look for our keyword */
    found = 0;
    for (;;)  /* for each line */
      {
	/* Check if this line has our keyword */
        p = keyword;
	for (;;)
	  {
	    c = getc(configFile);

	    if (c == EOF)
	      {
	        Close(configFile);
		return (BAD_ARGS);	/* keyword not found */
	      }

	    if (*p == '\0')		/* end of desired keyword */
	      {
		if (c == ':')
		    found++;		/* keyword matches */
		break;
	      }

	    if (c == ':' || c == '\n')
		break;			/* end of keyword in file, no match */

	    if (c != *p++) break;	/* keyword doesn't match */
	  }

	if (found) break;

	while (c != '\n')
	    c = getc(configFile);	/* flush rest of line */
        
      }

    /* Read the value (or as much as we have room for) */
    p = value;
    error = BAD_BUFFER;
    while (maxlength-- > 0)
      {
	c = getc(configFile);
	if (c == EOF)
	  {
	    error = OK;		/* odd -- value ends in midline */
	    *p++ = '\0';
	    break;
	  }
	if (c == '\n')
	  {
	    /* Check for a continuation line */
	    c = getc(configFile);
	    if (c == ':')
	      {
		*p++ = '\n';	/* continuation line found */
	      }
	    else
	      {
		error = OK;	/* end of value */
		*p++ = '\0';
	        break;
	      }
	  }
	else
	    *p++ = c;
      }

    Close(configFile);

    return (error);
  }
