/*
 * namecache.c
 *
 * Name caching routines.  Currently cache uses a very simple data
 *  structure; just a linked list of everything we know.  Sorted
 *  by prefix length with longer prefixes first, so that we always
 *  find the longest prefix match.  
 * A future version could use a hash table if search time becomes a 
 *  factor.
 */

#include "Vnaming.h"
#include "Vgroupids.h"
#include "Vnamecache.h"

extern char *strsave();

/*
 * Add a new entry to the name prefix cache.  
 */
NameCacheEntry *NameCacheAdd(prefix, length, from, to, truename, flags)
    char *prefix, *truename;
    ContextPair from, to;
    unsigned short length, flags;
  {
    register NameCacheEntry *ce1, *ce2, *newce;
    register char *p;

    /* Copy entry, omitting leading and 
     *  trailing delimiters if any. */
    if (*prefix == ROOT_ESCAPE_CHAR)
      {
        prefix++;
	length--;
      }
    if (length && IsDelimiter(prefix[length-1])) length--;
    p = (char *) malloc(length+1);
    if (p == NULL) return NULL;
    strncpy(p, prefix, length);    
    p[length] = '\0';

    /* Find where new entry goes and check for duplication */
    ce1 = (NameCacheEntry *) PerProcess->namecache;
    ce2 = *PerProcess->namecache;

    while (ce2 != NULL && ce2->namelength > length)
      {
	ce1 = ce2;
        ce2 = ce2->next;
      }
    
    while (ce2 != NULL && ce2->namelength == length)
      {
	if (ContextPairEqual(ce2->from, from) &&
	    strcmp(ce2->name, p) == 0)
	  {
	    /* Duplicate entry.  Delete old version.
	     *  (Could reuse list node instead of freeing.)
	     */
	    ce1->next = ce2->next;
	    free(ce2->name);
	    free(ce2);
	    ce2 = ce2->next;
	  }
	else
	  {
	    ce1 = ce2;
	    ce2 = ce2->next;
	  }
      }

    /* Create new entry and put between ce1 and ce2.
     *  (ce1 guaranteed != NULL) */
    newce = (NameCacheEntry *) malloc(sizeof(NameCacheEntry));
    if (newce == NULL) return NULL;
    newce->name = p;
    newce->from = from;
    newce->to = to;
    newce->namelength = length;
    if (truename) 
        newce->truename = strsave(truename);
    else
	newce->truename = NULL;
    newce->flags = flags;

    newce->next = ce2;
    ce1->next = newce;
    
    return newce;
  }


/*
 * Delete a name cache entry; presumably invalid.
 */
SystemCode NameCacheDelete(cacheEntry)
    NameCacheEntry *cacheEntry;
  {
    register NameCacheEntry *ce1, *ce2;

    /* Find entry if there */

    ce1 = (NameCacheEntry *) PerProcess->namecache;
    ce2 = *PerProcess->namecache;
    while (ce2 != NULL && ce2 != cacheEntry)
      {
	ce1 = ce2;
        ce2 = ce2->next;
      }
    
    if (ce2 == NULL) 
      {
        /* Wasn't in the cache */
        return NOT_FOUND;
      }

    /* Found.  Delete it */
    ce1->next = ce2->next;
    free(ce2->name);
    free(ce2);
    ce2 = ce2->next;

    return OK;
  }


/*
 * See if any prefix of the given name matches something in the
 *  cache.  We make an assumption about component delimiters
 *  (hidden in the definition of IsDelimiter) to avoid matching a 
 *  prefix like "[storage/foo" with a name like "[storage/foobar]".
 *  Cache entries do not end with a delimiter (other than a
 *  null byte string terminator).  A prefix match is defined as
 *  all the characters in the prefix matching the corresponding
 *  characters in the given name, plus the given name containing
 *  a delimiter immediately following the match.
 */
NameCacheEntry *NameCacheLookup(name, context)
    register char *name;
    ContextPair context;
  {
    register NameCacheEntry *ce;
    register unsigned length;

    /* Strip leading '[' if any */
    if (*name == ROOT_ESCAPE_CHAR)
      {
	name++;
	context.pid = VCSNH_SERVER_GROUP;
	context.cid = GLOBAL_ROOT_CONTEXT;
      }

    /* Start search */
    ce = *PerProcess->namecache;
    length = strlen(name);
    while (ce != NULL && ce->namelength > length)
      {
	/* Can't match.  Prefix longer than name */
        ce = ce->next;
      }
    
    /* Prefix short enough */
    while (ce != NULL)
      {
	if (ContextPairEqual(ce->from, context) &&
            strncmp(ce->name, name, ce->namelength)==0 &&
	    IsDelimiter(name[ce->namelength]))
	  {
	    /* Found a match! */
	    return ce;
	  }
	else
	  {
	    ce = ce->next;
	  }
      }

    return NULL;
  }

