
@Chapter(Using the Graphics Primitives)

This chapter provides the information needed to use the routines
in the graphics library.  
The main topics discussed are:  
the data structures used to represent the graphical
objects which the routines deal with, and the use of the routines.
As a convention the routines are given names of the form 
@t[ObjectFunction(arg...)].  
That is, the type of object with an initial capital letter, followed by the
function with an initial capital letter.

@Section(General Screen Handling)

The following functions are used to do full screen operations:
@begin(description)
ScreenInit()@\Initializes the screen.
Right now it simply turns on the video enable bit, which is usually turned
on by the @c[PROM] monitor anyway, so it is not necessary.

ScreenClear()@\Clears the screen to all zeros (black).

ScreenSet()@\Sets the screen to all ones (white or green).
This is used if you are drawing black objects on a white background.

ScreenFlash()@\Inverts the screen.
All ones (white or green) become zeros (back) and vice versa.

GXProbe()@\Returns the address of the frame buffer, if found.
See the last section of this chapter for more information and an example.

GXdecode(n)@\Takes a pointer into frame buffer address space and
decodes it into a string.  Returns a pointer to the (static) string.
It is useful if you are debugging or writing your own graphics primitives.
@end(description)

@Section(Data Structures)

Presently, two types of graphical objects are dealt with in the graphics 
library:  Rasters, and Lines.  This is a brief description of the data 
structures used to represent these objects.

@subsection(Rasters)

    A raster, as referred to in subsequent discussion here, is a rectangular
region of bitmapped memory.
Rasters may also be stored in the processor or @c[MULTIBUS] memory,
as described in the next section.
Rasters in the frame buffer are 
represented as a quadruple: 
pixel height, pixel width, pixel @BC[X] address, and pixel @BC[Y] address.
@BC[X] and @BC[Y] specify the pixel address of the upper left
corner of the raster, with
height and width indicating a non-negative displacement which defines the
rectangular region relative to that pixel.  In C, the declaration is as
follows:

@example(
	struct fb_ raster
	  {
                short    height;
                short    width;
                short    x;    /* pixel-x-address of upper left corner */
                short    y;    /* pixel-y-address of upper left corner */
          };
)

Notice that the representation provides the information necessary to access
the data, but does not provide
the data itself.
The data are the values of the pixels in
the described region of the frame buffer.

    RasterOp primitives manipulate raster objects as defined above.
The routines take pointers to the structures as arguments, @b[not]
the structures themselves.
An example of this is:

@example[
       struct fb_ raster    screen = { 1024, 1024, 0, 0 };

          RasterFill( &screen );
]
and the library routine then accesses the raster through @BC[X] and @BC[Y]
fields in the structure.

@subsection(Representation of Rasters in Processor Memory)

Rasters in memory are stored as uni-dimensional arrays of short words
(16 bits wide).  The representation is
the height and width of the raster, and
the start address of this array of words.
Thus we get:

@example[
	struct mem_ raster
	  {
		short	height;
		short	width;
		short	*start;
	  };

]
If a raster in the frame buffer is @t[h] by @t[w] pixels, it 
will be stored as @t[w/16] chunks of @t[h] short words, and one more
chunk of @t[h] short
words if @t[w%16] is not zero.
Think of this as breaking the raster in the 
framebuffer up into columns of 16 pixels each that are @t[h] pixels high.
The first instance of @t[h] short words in memory (that
with the lowest address in memory) will be that corresponding to the left
hand column of the raster as it appears in the frame buffer.  
This is illustrated in Figure @ref(moverast).

@Begin(Figure)
@PressPicture( Height = "5.2 Inches", File = "MoveRast.press" )
@caption(Moving a Raster from Framebuffer to Memory)
@tag(MoveRast)
@End(Figure)

@Section(The Raster Operations)

    Presently the following RasterOps are implemented:
@t[@itemize[
        RasterFill( DestRaster );

        RasterGet( MemRaster, SourceRaster );

        RasterPut( DestRaster, MemRaster );

        RasterCopy( DestRaster, SourceRaster );

        RasterPattern( DestRaster, Pattern );
	
	RasterRoll( Raster, UpBy );
]]
Notice that for historical reasons, the @i[destination] is usually
the first argument, and the second argument is the source.
This might be reversed from what you are used to, but is an
unfortunate C language standard.

@subsection[The Fill Primitive]
    The @t[Fill] operation performs the pre-specified function (raster
operation) on the destination raster.
It is the simplest and fastest graphics primitive.

@example[Dest <- Function( Pattern, Source, Dest )]
As this implies, @t[Fill] requires that the @BC[Function],
@BC[Pattern], and @BC[Source] registers be loaded
with the appropriate constant values, 
prior to the call to @t[Fill]. 
For example, the following call:

@begin(example)
 	struct fb_ raster screen = { 1024, 1024, 0, 0 };
	GXfunction = GXclear;
	RasterFill( &screen );
@end(example)
Will clear the entire screen.

@subsection[The Get Primitive]
    The @t[Get] operation performs a transfer of a raster from the
framebuffer to processor memory, with or without updating (performing a raster
operation on) the raster  in the framebuffer as it reads it out.  
The @BC[Function] and @BC[Pattern] registers
must be loaded in accordance with the update function to be performed.  
If no update is to occur (non-modifying read), the source raster in the framebuffer
is left unchanged;
set @t[GXfunction] to @t[GXcopy] for this purpose, and the
@BC[Pattern] register need not be set.

@begin(example)
	short buffer[66000];
 	struct mem_ raster MemRaster    = { 1024, 1024, buffer };
 	struct  fb_ raster SourceRaster = { 1024, 1024, 0, 0 };
	GXfunction = GXcopy;
	RasterGet( &MemRaster, &SourceRaster );
@end(example)

    An example involving the @BC[Pattern] register might be to OR in it's contents
as we read from the Frame buffer.
However, this is rarely (if ever) used.

@example[
	GXpattern =  0xcccc;
	GXfunction = GXor;
	RasterGet( &MemRaster, &SourceRaster );
]

@subsection[The Put Primitive]

    The @t[Put] operation is the inverse of the @t[Get] operation -- It moves a
raster from memory to the framebuffer.
As usual, the @BC[Function] and @BC[Pattern] registers must be loaded
with the appropriate function to apply as the raster from memory is 
moved into the framebuffer. 
Just as with @t[Get], usually this will be @t[GXcopy].

@begin(example)
	short buffer[66000];
 	struct  fb_ raster DestRaster = { 1024, 1024, 0, 0 };
 	struct mem_ raster MemRaster  = { 1024, 1024, buffer };
	GXfunction = GXcopy;
	RasterPut( &DestRaster, &MemRaster );
@end(example)

@subsection[The Pattern Primitive]

    The @t[Pattern] operation is used to draw ``textured''
areas on the screen.
A pointer to 16 shorts is passed to define the pattern,
which is replicated as often as necessary to fill the area.
As usual, the @BC[Function] and @BC[Source] registers must be loaded
with the appropriate function to apply as the pattern from memory is 
moved into the framebuffer.
Note that the pattern is always aligned (both
vertically and horizontally) to sixteen bit boundaries.
This is in contrast to most other primitives which align
the data to any bit boundary.
Thus @t[Pattern] is used for applications like filling complex
areas with textures that repeat at 16 pixel intervals.
The following example will set the entire screen to a ``gray''
pattern of alternating ones and zeros.

@begin(example)
	short gray[] = 
	  {
	    0x5555, 0xAAAA, 0x5555, 0xAAAA,
	    0x5555, 0xAAAA, 0x5555, 0xAAAA,
	    0x5555, 0xAAAA, 0x5555, 0xAAAA,
	    0x5555, 0xAAAA, 0x5555, 0xAAAA,
	  };
 	struct  fb_ raster DestRaster = { 1024, 1024, 0, 0 };
	GXfunction = GXcopyPattern
	RasterPattern( &DestRaster, grey );
@end(example)

@subsection[The Copy Primitive]

    The @t[Copy] operation, is perhaps the most sophisticated of the
RasterOp primitives presently existing, since we may update both the source
and destination rasters if we wish.
That is, we perform a modifying read,
in addition to the write (which necessarily modifies).  
This function may be
thought of as the composition of a @t[Get] and a @t[Put], without the
intermediate transfer to processor memory.  It is for moving rasters about
within the framebuffer.
Remember, the destination is the first argument, the source the second.

@begin(example)
 	struct  fb_ raster SourceRaster = { 200, 200, 0, 0 },
			   DestRaster  = { 200, 200, 600, 400 };
	GXfunction = GXcopy;
	RasterCopy( &DestRaster, &SourceRaster );
@end(example)

@subsection(The Roll Primitive)

  The @t[Roll] primitive is a special operation used for scrolling
rectangluar areas of the framebuffer.
It is equivalent to copying the top lines of a raster into memory,
copying the raster up, and restoring memory to the bottom of the raster.
The last parameter determines the number of pixels to scroll.

@begin(example)
 	struct fb_ raster screen = { 1024, 1024, 0, 0 };
	GXfunction = GXcopy;
	RasterRoll( &screen, 4 );
@end(example)

@Section(Line Graphics Primitives)

This section describes the use of the @t{Line(x,y,dx,dy)} function.
The line primitives currently consist of three macros, defined in @t[lines.h]:
@t[@itemize[
movea(x,y)

mover(dx,dy)

drawa(x,y)

drawr(dx,dy)]]

The @t[movea(x,y)] operation establishes the current position (CP) at
@value(#xy).  
Draw operations are with respect to the current position -- that is, CP 
defines one endpoint of the line drawn by @t[drawr(dx,dy)] or
@t[drawa(x,y)].
The @t[drawa(x,y)] routine draws from the CP to the "absolute" point
@value(#xy);
the CP is then updated to @value(#xy).
The @t[drawr(dx,dy)] routine draws from the CP @BC[<Xc,Yc>] to the point
@BC[<Xc+]@t[dx]@BC[,Yc+]@t[dy]@BC[>],
and the CP is then updated to the new position.

    All the routines are implemented as macros which define the appropriate
call to a fundamental line drawing routine based on the Bresenham algorithm.
The @t[movea(x,y)] and @t[mover(x,y)] routines do no drawing,
they simply change the current
position (CP), which is the origin for the next draw command.
The @BC[Width] and @BC[Function] registers must be set up before
calls to these routines.
Usually the @BC[Width] will be set to 1, and @BC[function] will
will be set to @t[GXset] or @t[GXclear] depending on if white or
black lines are being drawn respectively.
Note that setting the @t[GXwidth] to a value greater than one
does @b[not] necessarily draw wider lines lines;
it just will cause the given number of pixels to the right
of the vector to be operated on.
This will mean a wide line for vertical vectors only.

As with the rest of these graphics primitives, no error checking or clipping is
done, so be sure the coordinates are all between 0 and 1023 inclusive.
No assumption should be made about CP at the start of the session,
altough they are most likely zero.  Thereafter, the CP wil retain the value
of the last point drawn or moved to.

@section(Text Primitives)

The following primitive is used for drawing simple text:
@begin(example)
	DrawText( left, bottom, string, length, paintFlag );
@end(example)
The first two parameters give the screen coordinates of the
lower left corner of the text.  The string is is a pointer
to the character string, and the length is the number of characters.
A fixed width font is currently used, with the high-order bit
indicating inverted video (white on black).
If the paint flag is true,
the characters are ``painted'' onto the screen using the 
@t[GXpaintInverted] and @t[GXand] functions for normal and
inverse respectively.
If the paint flag is false, the characters are copied onto
the screen, destroying whatever was there previously.

@Section(Where to Find the Code and How to Use It)

  To compile C programs using rasterops, you must include the
file @t[rasterops.h] or @t[lines.h], along with @t[framebuf.h]
to define the frame buffer symbols.
The framebuffer registers can be set by just putting them on the
left side of an assignment statement.

  Add the line @t[-lgraphics] to your @t[cc68] command line when
you are linking your program together (note: no space between 
@t[l] and @t[graphics]).
Usually this is the line in the @t[makefile] which constructs the
final load file from the individual @t[.b] files.
Before you do anything in your program, you must include the lines:
@example[
	int GXProbe(), GXBase;
	GXBase = GXProbe();
]
This determines the base address of the frame buffer.
If it is zero, no frame buffer could be found.
There is currently a problem using this routine under the Vkernel
(the operating system used by the Network Graphics project);
just set @t[GXBase] to @t[GXDefaultBase] instead, at the risk
of making your programs a bit less portable.
Note carefully the capitalization of the variable @t[GXBase].

@Begin(Table)
@Begin(Example)
@include(demo.c)
@End(Example)
@Caption(A Complete Sample Program)
@Tag(demo)
@End(Table)

The code is in @t[src/graphics/lib], along with a test
program @t[testrops], relative to the SUN home directory on
your machine.
The libraries should be installed as @t[lib/libgraphics.a], so they can
be found with the @t[-lgraphics] of the loader and @t[cc68].
The include files are under @t[include] on the SUN home directory.
On the Stanford machines Shasta, Diablo, and Navajo, the SUN home
directory is @t[/usr/sun].

A sample program is given in table @ref(demo).
This program uses both RasterOps and the Line primitives.
First it draws some random vectors in the area between 800 and 1000
in both @BC[X] and @BC[Y], which is off the screen in both
landscape and portrait mode displays.
Then it copies this pattern to random places on the main area of the screen.
This program has been linked with the following @t[makefile] lines:

@begin(example)
demo: demo.b
	cc68 -r -o demo demo.b -lgraphics
@end(example)
