/*
 * This pseudo-function compiles code into a table pointed to by 'a2',
 * to apply the logical operation 'function' (as defined in <framebuf.h>,
 * using 'd1' as the source.
 * The result is 'a3@+'.
 *
 * Input:
 * register a2 - points to an array into which code is to be compiled.
 * register d7 == mask (used both a "compile"-time and "run"-time)
 * register d6 == function to compile
 * function, mask.
 * Output:
 * register a2 - point to next free location in code array.
 * Side-effects: The right 16 bits of d0 and d1 may change.
 *
 * Characteristics of the code generated:
 * Performs: dst = (dst & ~ mask) | ((dst function src) & mask)
 * That is dst remains unchanged in bits where mask is zero.
 * Here src is 'd1'; dst is 'a3@+';
 * mask and function are given at "compile"-time.
 * d7 must also be set to mask at run-time.
 * As well as d1, d7 and a3; d0 may be used as working storage.
 * All operations are word (16-bit) operations.
 * The maximum length of the code generated is 12 bytes.
 */
#include <framebuf.h>
#include <m68000.h>
#define compileW(x) {asm(" .word 0x34FC"); asm(x);}
   /* *code++ = x, i.e. movw #"x",a2@+ */

#define InvertSrc	{ compileW(" notw d1"); }
#define HasMask (mask != 0xFFFF)
#define AndWithMask	\
	    {if (HasMask) compileW(" andw d7,d1");}

static RasterCompileDummyW()
  {
    register unsigned short mask;	/* d7 */
    register function;			/* d6 */
asm("	.globl RasterCompileW");
asm("RasterCompileW:")
    switch(function & 15)
      {
	case GXclear & 15: /* 0 */
	    if (!HasMask) compileW(" clrw a3@+")
	    else
	      {
		compileW(" movw d7,d0");
		compileW(" notw d0");
		compileW(" andw d0,a3@+");
	      }
	    break;
	case 0x1:
	    InvertSrc;
	    /* ... then fall through to case 0x4 ... */
	/* case 0x2 is just before case 0x8 */
	/* case 0x3=copyInverted is just before case 0xC */
	case 0x4:
	    compileW(" orw a3@,d1");
	    AndWithMask;
	    compileW(" eorw d1,a3@+");
	    break;
	case GXinvert & 15: /* 5 */
	    compileW(" eorw d7,a3@+");
	    break;
	case 0x9:
	    InvertSrc;
	    /* ... then fall through to ... */
	case GXxor & 15: /* 6 */
	    AndWithMask;
	    compileW(" eorw d1,a3@+");
	    break;
	/* case 0x7 is just before case 0xD */
	case GXpaintInverted & 15: /* 2 */
	    InvertSrc;
	    /* ... then fall through to ... */
	case GXand & 15: /* 8 */
	    AndWithMask;
	    compileW(" andw d1,a3@+");
	    break;
	/* case 0x9 is just before case 0x6 */
	case GXnoop & 15: /* A */
	    break;
	/* case 0xB is just before case 0xE */
	case GXcopyInverted & 15: /* 0x3 */
	    InvertSrc;
	    /* ... then fall through to ... */
	case GXcopy & 15: /* C */
	    if (HasMask)
	      {
	        compileW(" movw a3@,d0");
	        compileW(" eorw d0,d1");
	        compileW(" andw d7,d1");
	        compileW(" eorw d0,d1");
	      }
	    compileW(" movw d1,a3@+");
	    break;
	case 0x7:
	    InvertSrc;
	    /* ... fall through to ... */
	case 0xD:
	    compileW(" andw a3@,d1");
	    InvertSrc;
	    AndWithMask;
	    compileW(" eorw d1,a3@+");
	    break;
	case 0xB:
	    InvertSrc;
	    /* ... fall through to ... */
	case GXpaint & 15: /* E */
	    AndWithMask;
	    compileW(" orw d1,a3@+");
	    break;
	case GXset & 15: /* F */
	    if (HasMask) compileW(" orw d7,a3@+")
	    else compileW(" movw d7,a3@+");
	    break;
      }
    rts;
}

#undef InvertSrc
#undef HasMask
#undef AndWithMask
#define InvertSrc	{ compileW(" notl d1"); }
#define HasMask (mask != -1)
#define AndWithMask	\
	    {if (HasMask) compileW(" andl d7,d1");}
/* ditto, but using long (32-bit) operations */
static RasterCompileDummyL()
  {
    register unsigned long mask;	/* d7 */
    register function;			/* d6 */
asm("	.globl RasterCompileL");
asm("RasterCompileL:")
    switch(function & 15)
      {
	case GXclear & 15: /* 0 */
	    if (!HasMask) compileW(" clrl a3@+")
	    else
	      {
		compileW(" movl d7,d0");
		compileW(" notl d0");
		compileW(" andl d0,a3@+");
	      }
	    break;
	case 0x1:
	    InvertSrc;
	    /* ... then fall through to case 0x4 ... */
	/* case 0x2 is just before case 0x8 */
	/* case 0x3=copyInverted is just before case 0xC */
	case 0x4:
	    compileW(" orl a3@,d1");
	    AndWithMask;
	    compileW(" eorl d1,a3@+");
	    break;
	case GXinvert & 15: /* 5 */
	    compileW(" eorl d7,a3@+");
	    break;
	case 0x9:
	    InvertSrc;
	    /* ... then fall through to ... */
	case GXxor & 15: /* 6 */
	    AndWithMask;
	    compileW(" eorl d1,a3@+");
	    break;
	/* case 0x7 is just before case 0xD */
	case GXpaintInverted & 15: /* 2 */
	    InvertSrc;
	    /* ... then fall through to ... */
	case GXand & 15: /* 8 */
	    AndWithMask;
	    compileW(" andl d1,a3@+");
	    break;
	/* case 0x9 is just before case 0x6 */
	case GXnoop & 15: /* A */
	    break;
	/* case 0xB is just before case 0xE */
	case GXcopyInverted & 15: /* 0x3 */
	    InvertSrc;
	    /* ... then fall through to ... */
	case GXcopy & 15: /* C */
	    if (HasMask)
	      {
	        compileW(" movl a3@,d0");
	        compileW(" eorl d0,d1");
	        compileW(" andl d7,d1");
	        compileW(" eorl d0,d1");
	      }
	    compileW(" movl d1,a3@+");
	    break;
	case 0x7:
	    InvertSrc;
	    /* ... fall through to ... */
	case 0xD:
	    compileW(" andl a3@,d1");
	    InvertSrc;
	    AndWithMask;
	    compileW(" eorl d1,a3@+");
	    break;
	case 0xB:
	    InvertSrc;
	    /* ... fall through to ... */
	case GXpaint & 15: /* E */
	    AndWithMask;
	    compileW(" orl d1,a3@+");
	    break;
	case GXset & 15: /* F */
	    if (HasMask) compileW(" orl d7,a3@+")
	    else compileW(" movl d7,a3@+");
	    break;
      }
    rts;
}
