/*
 * SetUpEnvironment()
 * 
 * Set up the environment for a new team.  Its argument vector and pointers
 *   to environment variables and name cache are passed as parameters.
 *   This code knows what the local environment variables and name cache 
 *   data structures look like; also, the format of a team enviroment block
 *   is coded into it.
 * The block is constructed locally and moved in a single CopyTo.  The
 *   alternative would be to copy each argument (etc.) individually, which
 *   would avoid some local calculations and a call to malloc(), but would
 *   be much slower due to requiring many more kernel operations.
 * All the byte swapping is a huge pain.
 */

#include "Vio.h"
#include "Vnamecache.h"

extern char *SetTeamSize();

static char *packstr();
static char *pack2();
static char *pack4();
static char *packpair();

/* Align pointer p to the next b-byte boundary, assuming b is a power of 2. */
#define align(p, b) (  (p) + ( -(int)(p) & ((b)-1) )  )


SystemCode SetUpEnvironment(pid, args, env, cache, where)
    ProcessId pid;
    char *args[];
    EnvironmentVariable *env;
    NameCacheEntry *cache;
    char *where;
  {
    unsigned long argc, envc, cachec, x;
    register unsigned long blocksize, swap;
    register char *p;
    register char **a;
    register EnvironmentVariable *ev;
    register NameCacheEntry *ce;
    char *block;
    SystemCode error;

    /* Compute how big the block will be */
    /* First the byte count and args */
    blocksize = sizeof(blocksize) + sizeof(argc);
    argc = 0;
    a = args;
    while (*a)
      {
        blocksize += strlen(*a++) + 1;
	argc++;
      }

    /* Next, the environment.  Note we should lock the environment 
     *  here to prevent it from changing out from under us.
     */
    blocksize = align(blocksize, 4) + sizeof(envc);
    envc = 0;
    ev = env;
    while (ev) 
      {
        blocksize += strlen(ev->name) + strlen(ev->value) + 2;
	envc++;
	ev = ev->next;
      }

    /* Finally, how big the name cache will be.  Should lock it too. */
    blocksize = align(blocksize, 4) + sizeof(cachec);
    cachec = 0;
    ce = cache;
    while (ce)
      {
	blocksize = align(blocksize, 4);  /* Include padding */
	blocksize += strlen(ce->name)
		     + (ce->truename ? strlen(ce->truename) : 0)
		     + 2 + 2*sizeof(ContextPair) + sizeof(unsigned short);
	cachec++;
	ce = ce->next;
      }
    blocksize = align(blocksize, 4) + sizeof(ContextPair) + 
        (PerProcess->ctxname ? strlen(PerProcess->ctxname) : 0) + 1;
    blocksize = align(blocksize, 4);

    /* Malloc a block big enough to hold the whole thing */
    block = (char *) malloc(blocksize);
    if (block == NULL) return NO_MEMORY;

    /* Now stuff everything in it */
    p = block;
    swap = DifferentByteOrder(pid);

    /* First the byte count and arguments */
    x = blocksize;
    p = pack4(p, &x, swap);
    p = pack4(p, &argc, swap);
    a = args;
    while (*a) p = packstr(p, *a++);
    
    /* Next the environment variables */
    p = align(p, 4);
    p = pack4(p, &envc, swap);
    ev = env;
    while (ev)
      {
        p = packstr(p, ev->name);
	p = packstr(p, ev->value);
	ev = ev->next;
      }

    /* Now the name cache */
    p = align(p, 4);
    p = pack4(p, &cachec, swap);
    ce = cache;
    while (ce)
      {
	p = align(p, 4);
	p = packpair(p, &ce->from, swap);
	p = packpair(p, &ce->to, swap);
	p = pack2(p, &ce->flags, swap);
	p = packstr(p, ce->name);
	if (ce->truename) 
	    p = packstr(p, ce->truename);
	else
	    p = packstr(p, "");
	ce = ce->next;
      }

    /* And finally the working context */
    p = align(p, 4);
    p = packpair(p, &PerProcess->ctx, swap);
    if (PerProcess->ctxname)
	p = packstr(p, PerProcess->ctxname);
    else
        p = packstr(p, "");

    /* Whew!  Environment block is all constructed.  Pass it to the
     *  new team.  */
    where = align(where, 4);
    if (SetTeamSize(pid, where + blocksize) != where + blocksize)
        error = NO_MEMORY;
    else 
        error = MoveTo(pid, where, block, blocksize);

    free(block);

    return error;
  }


/* Utility routines */

/* Pack in string s at location p, and return a pointer
 * just past where the terminator ended up. */
static char *packstr(p, s) 
    char *p, *s;
  {
    while (*p++ = *s++) ;
    return p;
  }

/* Pack in a two-byte integer at location p, byte-swapping if swap != 0. */
static char *pack2(p, s, swap)
    char *p, *s;
  {
    if (swap)
      {
	*p++ = *(s+1);
	*p++ = *s;
      }
    else
      {
	*p++ = *s++;
	*p++ = *s;
      }
    return p;
  }


/* Pack in a four-byte integer at location p, byte-swapping if swap != 0. */
static char *pack4(p, s, swap)
    char *p, *s;
  {
    if (swap)
      {
	s += 3;
	*p++ = *s--;
	*p++ = *s--;
	*p++ = *s--;
	*p++ = *s;
      }
    else
      {
	*p++ = *s++;
	*p++ = *s++;
	*p++ = *s++;
	*p++ = *s;
      }
    return p;
  }


/* Pack in a pair of four-byte integers at location p, as above */
static char *packpair(p, s, swap)
    char *p, *s;
  {
    p = pack4(p, s, swap);
    return (pack4(p, s+4, swap));
  }

