/*
 * Standard exception handler routine.
 *
 * Copyright (c) 1985 Distributed Systems Group, Stanford University.
 *
 * Author:  Steve Tepper
 *
 * Modifications:
 *  (ejb)  25 Sep 85--Added routines to print parameters to abort.
 *
 * There is another version of the exception-printing routine, used by the
 *   kernel's last-ditch exception-handler, in kernel/vax/exception.c.  It's
 *   pretty stupid having both.
 */


#include <Venviron.h>
#include <Vprocess.h>
#include <Vexceptions.h>
#include <Vio.h>
#include <Vquerykernel.h>

/* Machine configuration */
static configUnknown = 1;
static MachineConfigurationReply mreply;

#define ABORT_INSTR 0xffff

#define AP 12
#define STACK_ENTRY_SIZE 4

#define MAX_NUM_STRINGS 9
#define MAX_STRING_LENGTH 128
#define MAX_NUM_ARGS 64


extern SystemCode GetNumberofParams();
extern SystemCode GetParams();
extern SystemCode GetFormatString();
extern SystemCode GetStringParams();

static unsigned char GetProcessorType()
    /* Find out what machine we're running on.  Since the exception server
     * always (?) runs on the processor that generated the exception, this
     * works OK.  Woe betide us if the exception server is ever allowed to
     * reside on another machine.
     */
  {
    if (configUnknown)
      {
	QueryKernel(0, MACHINE_CONFIG, &mreply);
	configUnknown = 0;
      }
    return mreply.processor;
  }


unsigned short *StandardExceptionHandler(req, pid, fout)
    register ExceptionRequest *req;
    ProcessId pid;
    File *fout;			/* Print out messages on this file */

    /* Standard exception handling routine.  Prints out some information
     * about the process incurring the exception and returns the pc at
     * which the exception occurred.
     */
  {
    Processor_state state;
    unsigned stackSize;

    /* Print information about the state of the process */
    Resynch( fout );
    fprintf(fout,"\r\n");

    switch( req->type )
      {
      case VecMachineChk:
	fprintf(fout, " Machine check: " );
	switch(GetProcessorType())
	  {
	    case PROC_UVAX1: switch(req->subtype)
	      {
		case MC_MemControl_I:
		    fprintf(fout, "memory controller, addr=%x",  req->accaddr);
		    break;
		case MC_BadRead_I:
		    fprintf(fout, "parity/ECC error,page addr=%x",
								 req->accaddr);
		    break;
		case MC_NoMem_I:
		    fprintf(fout, "nonexistent memory, addr=%x", req->accaddr);
		    break;
		case MC_IllOp_I:
		    fprintf(fout, "unaligned word in i/o space, addr=%x", 
								 req->accaddr);
		    break;
		case MC_PTRead_I:
		    fprintf(fout,"page table read error, addr=%x",
								 req->accaddr);
		    break;
		case MC_PTWrite_I:
		    fprintf(fout, "page table write error, addr=%x",
								 req->accaddr);
		    break;
		case MC_CPUStore_I:
		    fprintf(fout, "control store parity error");
		    break;
		case MC_Micro_I:
		    fprintf(fout, "invalid state in micromachine");
		    break;
		case MC_Q22Bus_I:
		    fprintf(fout, "can't read vector from Q22 bus");
		    break;
		case MC_ParWrite_I:
		    fprintf(fout, "stack write error, addr=%x", req->accaddr);
		    break;
		default:
		    fprintf(fout, "unknown MicroVAX I type %d", req->subtype);
	      } /* End of VecMachineChk cases for MicroVAX I */
		break;

	    case PROC_UVAX2: switch (req->subtype)
	      {
		case MC_MicroFSD_II:
		    fprintf(fout, "impossible microcode state (FSD)");
		    break;
		case MC_MicroSSD_II:
		    fprintf(fout, "impossible microcode state (SSD)");
		    break;
		case MC_UndefFPU0_II:
		    fprintf(fout, "undefined FPU error code 0");
		    break;
		case MC_UndefFPU7_II:
		    fprintf(fout, "undefined FPU error code 7");
		    break;
		case MC_UndefMMTB_II:
		    fprintf(fout, 
			"undefined memory management status (TB miss)");
		    break;
		case MC_UndefMMM0_II:
		    fprintf(fout, "undefined memory management status (M=0)");
		    break;
		case MC_PTE_P0_II:
		    fprintf(fout,
			"process PTE address in P0 space (kernel bug?)");
		    break;
		case MC_PTE_P1_II:
		    fprintf(fout,
			"process PTE address in P1 space (kernel bug?)");
		    break;
		case MC_InterruptID:
		    fprintf(fout,
			"undefined interrupt ID code");
		    break;
		case MC_BusError_II  :
		case MC_BusError_II+1:
		case MC_BusError_II+2:
		case MC_BusError_II+3:
		    fprintf("%s bus error, %s address = %x",
			(req->subtype & 2) ? "write" : "read",
			(req->subtype & 1) ? "physical" : "virtual",
			req->accaddr );
		    break;
		default:
		    fprintf(fout, "unknown MicroVAX II type %d", req->subtype);
	      } /* End of VecMachineChk cases for MicroVAX II */
		break;

	    default:
		fprintf(fout,"processor type (0x%x) isn't MicroVAX I or II:\n",
		    GetProcessorType());
		fprintf(fout,"                subtype = %x, address(?) = %x",
		    req->subtype, req->accaddr);
		break;

	  }
	break;      /* End of VecMachineChk case */

      case VecKernStackInval:
	fprintf(fout, " Kernel stack invalid" );
	break;

      case VecPwrFail:
	fprintf(fout, " Power fail" );
	break;

      case VecResPrivInstrn:
	/* Load instruction */
	req->instruction = 0;
	MoveFrom( pid, (unsigned short *) &(req->instruction),
					 (unsigned short *) req->errpc, 2 );
	if( req->instruction == ABORT_INSTR )
	  {
	    /* Look at the stack to determine whether abort was called
	     * with any parameters.  If so, feed the parameters to
	     * _doprnt (which is set up to take a variable number of
	     * arguments).
	     */

	    Unspec paramaddr;
	    char strings[ MAX_NUM_STRINGS ][ MAX_STRING_LENGTH ];
	    int nparams;
	    Unspec arg[ MAX_NUM_ARGS ];
	    fprintf( fout, " Abort called:\r\n" );

	    nparams = GetNumberofParams( pid, &paramaddr, fout );
	    if( nparams > 0 )
	      {
		if( GetParams( pid, paramaddr, arg, nparams, fout )
							 != OK ) break;
		if( GetFormatString( pid, strings[0], arg[0], fout )
							 != OK ) break;
		arg[0] = (Unspec) strings[0];
		if( GetStringParams( pid, strings, arg, nparams, fout )
							 != OK ) break;
		_doprnt( arg[0], &arg[1], fout );
	        fprintf( fout, "\r\n" );
	      }

	    fprintf( fout, " Abort instruction" );
	  }
	else
	    fprintf(fout, " Reserved/privileged instruction" );
	break;

      case VecXFC:
	fprintf(fout, " XFC instruction" );
	break;

      case VecOperandReserved:
	fprintf(fout, " Reserved operand" );
	break;

      case VecAddrModeReserved:
	fprintf(fout, " Reserved address mode" );
	break;

      case VecAccessCntlViol:
	fprintf(fout, " Access control violation");
	goto MMerr;

      case VecXlationInval:
	fprintf(fout, " Translation invalid");
MMerr:
	fprintf(fout, " on %s at virtual addr=%x",
	    (req->code & MM_Write) ? "write" : "read", req->accaddr);
	if (req->code & MM_Length)
	    fprintf(fout, ", length violation");
	if (req->code & MM_ProcessPT)
	    fprintf(fout, " while accessing process page table");
	break;

      case VecTracePending:
	fprintf(fout, " Trace pending" );
	break;

      case VecBreakPoint:
	fprintf(fout, " Breakpoint" );
	break;

      case VecCompatibility:
	fprintf(fout, " Compatibility fault type %d", req->subtype);
	break;

      case VecArithmetic:
	fprintf(fout, " Arithmetic: " );
	switch(req->subtype)
	  {
	  case A_IOverflow:
	    fprintf(fout, "integer overflow");
	    break;

	  case A_IDivBy0:
	    fprintf(fout, "integer divide by 0");
	    break;

	  case A_SubRange:
	    fprintf(fout, "subscript range");
	    break;

	  case A_FOverflow:
	    fprintf(fout, "floating overflow");
	    break;

	  case A_FDivBy0:
	    fprintf(fout, "floating divide by 0");
	    break;

	  case A_FUnderflow:
	    fprintf(fout, "floating underflow");
	    break;

	  default:
	    fprintf(fout, "type %d", req->subtype);
	    break;
	  }

	break;          /* End of arithmetic traps */

      case VecCHMK:
	fprintf(fout, " CHMK instruction, code=0x%x", req->subtype);
	break;

      case VecCHME:
	fprintf(fout, " CHME instruction, code=0x%x", req->subtype);
	break;

      case VecCHMS:
	fprintf(fout, " CHMS instruction, code=0x%x", req->subtype);
	break;

      case VecCHMU:
	fprintf(fout, " CHMU instruction, code=0x%x", req->subtype);
	break;

      case VecSBI_SILO:
	fprintf(fout, " SBI silo compare" );
	break;

      case VecCorrMemRead:
	fprintf(fout, " Corrected memory read data" );
	break;

      case VecSBI_Alert:
	fprintf(fout, " SBI alert" );
	break;

      case VecSBI_Fault:
	fprintf(fout, " SBI fault" );
	break;

      case VecBusTimeout:
	fprintf(fout, " Memory write timeout" );
	break;

      case VecTrap1:
      case VecTrap2:
      case VecTrap3:
      case VecTrap4:
      case VecTrap5:
      case VecTrap6:
      case VecTrap7:
      case VecTrap8:
      case VecTrap9:
      case VecTrapA:
      case VecTrapB:
      case VecTrapC:
      case VecTrapD:
      case VecTrapE:
      case VecTrapF:
	fprintf(fout, " Software level %d trap", (req->type - VecTrap1) + 1 );
	break;

      case VecIntervalTimer:
	fprintf(fout, " Interval timer" );
	break;

      case VecUndoneEMT:
	fprintf(fout, " Emulated instruction undone" );
	break;

      case VecDoneEMT:
	fprintf(fout, " Emulated instruction done" );
	break;

      case VecRxConsStor:
	fprintf(fout, " Cassette/floppy rcv interrupt" );
	break;

      case VecTxConsStor:
	fprintf(fout, " Cassette/floppy xmt interrupt" );
	break;

      case VecRxConsole:
	fprintf(fout, " Console rcv interrupt" );
	break;

      case VecTxConsole:
	fprintf(fout, " Console xmt interrupt" );
	break;

      case VecDeqna:
	fprintf(fout, " DEQNA interrupt" );
	break;

      case SUSPENDED:
        fprintf(fout, " Forced exception");
	break;

      default: 
	fprintf(fout, " Unknown exception" );
	break;
      }

    fprintf(fout, " in process %x\r\n",pid );
    if( req->instruction != 0 ) 
        fprintf(fout, " Instruction  " );
    fprintf(fout, "  Program counter    PSL\r\n" );
    if( req->instruction != 0 )
        fprintf(fout, "     %4x     ", req->instruction );
    fprintf(fout, "       %6x              %08x\r\n",
	req->errpc, req->status );

    ReadProcessState(pid, &state);
    if ((req->status & PSL_CURMOD) == PSL_KERNEL)
	fprintf(fout, " In kernel, called from %x\r\n", state.pc);

#ifdef  NOTDEF   /* do we need this?? */

    /* Check for stack overflow */
    if (state.perProcess != 0)
      {
	stackSize = 0;
	MoveFrom(pid, &stackSize,
	      &(((PerProcessArea *) state.perProcess)->stackSize),
	      sizeof(stackSize));
	if ((unsigned) state.USER_STACK_POINTER <= (unsigned) state.perProcess ||
	      (unsigned) state.USER_STACK_POINTER > (unsigned) state.perProcess + stackSize)
	    fprintf(fout, " Probable stack overflow\r\n");
      }

    StackDump(pid, fout);
#endif NOTDEF
    Flush( fout );
    return( (unsigned short *)req->errpc );
  }


/*
 * This part of the file contains the routines called by
 * StandardExceptionHandler when it handles an abort.  The abort routine can be
 * called with parameters.  If it is, the first parameter is to be interpreted
 * as a printf string.  Successive parameters are variables to be printed using
 * that string.  In addition to the usual format specifiers, we've added a %z,
 * which interprets the associated variable as a SystemCode and prints the error
 * message for that code (obtained from ErrorString).
 */


static SystemCode GetNumberofParams( pid, paramaddr, fout )
	ProcessId pid;
	Unspec *paramaddr;
	File *fout;
  {
    /* Return the number of parameters in the call to abort, and the address
     * of the first (topmost, lowest addressed) parameter.
     */

    Unspec Ap;
    Processor_state state;
    SystemCode error;
    int nparams;

    if( !ReadProcessState( pid, &state ) )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n" );
	fprintf( fout, " Unable to query process %x.\r\n", pid );
	return( OK );
      }

    Ap = state.regs[AP];
    *paramaddr = Ap + STACK_ENTRY_SIZE;

    /* nparams = *Ap; */
    error = MoveFrom( pid, &nparams, Ap, STACK_ENTRY_SIZE );
    if( error != OK )
      {
	fprintf( fout, "Error in StandardExceptionHandler:\r\n" );
	fprintf( fout, "MoveFrom call for number of parameters: %s",
						ErrorString( error ) );
	return( OK );
      }

    if( nparams > MAX_NUM_ARGS )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n");
	fprintf( fout, " Too many parameters to abort.  (Maximum %d.)\r\n",
							 MAX_NUM_ARGS );
	return( OK );
      }
		
    return( nparams );
  }


static SystemCode GetParams( pid, paramaddr, arg, nparams, fout )
	ProcessId pid;
	Unspec paramaddr;
	long arg[];
	int nparams;
	File *fout;
  {
    /* GetParams reads the parameters from the process pid's address space
     * (starting from paramaddr) into the exception handler's address space,
     * starting at arg.
     */
  
    SystemCode error;

    error = MoveFrom( pid, arg, paramaddr, nparams * STACK_ENTRY_SIZE );
    if( error != OK )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n" );
	fprintf( fout, "	MoveFrom call for parameters:%s\r\n",
							 ErrorString( error ) );
      }
    return( error );
  }



static SystemCode GetFormatString( pid, fmtstring, fmtaddr, fout )
	ProcessId pid;
	char *fmtstring;
	Unspec fmtaddr;
	File *fout;
  {
    /* The first parameter is a pointer to a printf style formatting string.
     * That string must be copied to our address space (starting at fmtstring).
     * The first argument, which like the others will be passed to _doprnt,
     * will then be changed to fmtstring.
     */

    SystemCode error;

    error = MoveFrom( pid, fmtstring, fmtaddr, MAX_STRING_LENGTH );
    if( error != OK )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n" );
	fprintf( fout, "	MoveFrom call for abort format string:%s\r\n",
							 ErrorString( error ) );
      }
    return( error );
  }



static SystemCode GetStringParams( pid, strings, arg, nparams, fout )
	ProcessId pid;
	char strings[ MAX_NUM_STRINGS ][ MAX_STRING_LENGTH ];
	Unspec arg[];
	int nparams;
	File *fout;
  {
    /* This routine actually serves two functions.  It interprets the %z's
     * in the format string (which starts at strings[0]) as SystemCodes,
     * gets the associated value from the argument list stored starting at
     * arg, and replaces that value with a pointer to the ErrorString for
     * that SystemCode.  It then adjusts the %z to a %s.
     *
     * Second, for each of the original %s's in the format string, it copies
     * the pointed-to string into the exception handlers address space--in
     * the strings array.  The pointer that had been in the arg list is changed
     * to point to this new copy.
     *
     * Note that since we didn't want to malloc from within the exception
     * handler (too unreliable), we allocated space for these strings on the
     * stack.  That limits us in both number and length of strings.
     */

    SystemCode error = OK;
    int stringcount =  1;
    int paramcount = 1;
    int notStringError = 1;
    char *p = strings[0];

    while( *p != '\0'  &&  p < (char *) strings + MAX_STRING_LENGTH  &&
							paramcount < nparams )
      {
	if( *p == '%' )
	  {
	    p++;
	    while( ( (*p >= '0'  &&  *p <= '9') || *p == '-' || *p == '*' )  &&
				p < (char *) strings + MAX_STRING_LENGTH )
		p++;
	    switch( *p )
	      {
		case 'z':
		  arg[ paramcount ] = (Unspec) ErrorString( arg[ paramcount ] );
		  paramcount++;
		  *p = 's';
		  break;

		case 's':
		  if( stringcount > MAX_NUM_STRINGS )
		    {
		      if( notStringError )
			{
			  fprintf( fout,
				"\r\nError in StandardExceptionHandler:\r\n" );
			  fprintf( fout,
				"Too many strings (> %d) in call to abort.\r\n",
				MAX_NUM_STRINGS );
			  notStringError = 0;
			}
		      break;
		    }

		  error = MoveFrom( pid, strings[ stringcount ],
					 arg[ paramcount ], MAX_STRING_LENGTH );
		  if( error != OK )
		    {
		      fprintf( fout,
				"\r\nError in StandardExceptionHandler:\r\n" );
		      fprintf( fout,
				"MoveFrom call for param %d of abort: %s\r\n",
				paramcount, ErrorString( error ) );
		      return( error );
		    }

		  strings[ stringcount ][ MAX_STRING_LENGTH - 1 ] = '\0';
		  arg[ paramcount++ ] = strings[ stringcount++ ];
		  break;

		case '%':
		  break;

		default:
		  paramcount++;
	      }
	  }

	p++;
      }
    return( error );
  }
