/*
 * Standard exception handler routine.
 *
 * 28 Feb 83 (mmt) - Abstracted from exceptions.c file so that other exception
 *			handlers can include it.
 *  5 March 83 (WIN) - Fixed bug in MoveFrom: byte count was too large by 1
 * 19 May 83 (mmt) - StandardExceptionHandler made to return error pc value.
 * 12 Aug 83 (TPM) - Added bus error type printing and stack overflow detect.
 * 28 Jul 84 (ejb) - Changed error messages to reflect changes in our
 *			choice of abort and breakpoint instructions.
 * 25 Sep 85 (ejb) - Added calls to routines to treat abort as having a
 *			printf string and parameters to be printed.
 */


#include <Venviron.h>
#include <Vprocess.h>
#include <Vexceptions.h>
#include <Vio.h>
#include <Vquerykernel.h>

#define MaxLookAhead 10
#define ABORT_INSTR	0x4afc
#define BKPT_INSTR	0x484f

#define MAX_NUM_STRINGS 9
#define MAX_STRING_LENGTH 128
#define MAX_NUM_ARGS 64

extern SystemCode GetNumberofParams();
extern SystemCode GetParams();
extern SystemCode GetFormatString();
extern SystemCode GetStringParams();

/* Machine configuration */
static configUnknown = 1;
static MachineConfigurationReply mreply;


unsigned short *StandardExceptionHandler(req, pid, fout)
    register ExceptionRequest *req;
    ProcessId pid;
    File *fout;			/* Print out messages on this file */

    /* Standard exception handling routine.  Prints out some information
     * about the process incurring the exception and returns the pc at
     * which the exception occurred.
     */
  {
    register unsigned short *i;
    unsigned short codeBuffer[MaxLookAhead];
    register int instCnt, index = 0;
    Processor_state state;
    unsigned stackSize;

    /* Find out whether we are running on a MC68000 or MC68010 */
    if (configUnknown)
      {
	QueryKernel(0, MACHINE_CONFIG, &mreply);
	configUnknown = 0;
      }

    /* Print information about the state of the process */
    Resynch( fout );
    fprintf(fout,"\r\n");

    i = (unsigned short *) req->errpc;

    switch( req->type )
      {
      case ZERODIV:
	fprintf(fout, " Divide by zero" );
	break;

      case CHKINST:
	fprintf(fout, " Check instruction" );
	break;

      case TRAPVINST:
	fprintf(fout, " Overflow" );
	break;

      case TRACETRAP:
	fprintf(fout, " Trace bit on" );
	break;

      case PRIVVIOL:
	fprintf(fout, " Privileged instruction" );
	if ((int)i >= TEAM_ORIGIN) /* not in kernel space */
	  {
	    /* Load instruction */
	    req->instruction = 0;
	    MoveFrom( pid,(unsigned short *)&(req->instruction)+1, i, 2);
	  }
	break;

      case ILLINST:
      case EMU1010:
      case EMU1111:
	if ((int)i >= TEAM_ORIGIN) /* not in kernel space */
	  {
	    /* Load instruction */
	    req->instruction = 0;
	    MoveFrom( pid, (unsigned short *) &(req->instruction) + 1, i, 2 );
	    if( req->instruction == ABORT_INSTR )
	      {
		/* Look at the stack to determine whether abort was called
		 * with any parameters.  If so, feed the parameters to
		 * _doprnt (which is set up to take a variable number of
		 * arguments).
		 */

		Unspec paramaddr;
		char strings[ MAX_NUM_STRINGS ][ MAX_STRING_LENGTH ];
		int nparams;
		Unspec arg[ MAX_NUM_ARGS ];
		fprintf( fout, "Abort called:\r\n" );

		nparams = GetNumberofParams( pid, &paramaddr, fout );
		if( nparams > 0 )
		  {
		    if( GetParams( pid, paramaddr, arg, nparams, fout )
								 != OK ) break;
		    if( GetFormatString( pid, strings[0], arg[0], fout )
								 != OK ) break;
		    arg[0] = (Unspec) strings[0];
		    if( GetStringParams( pid, strings, arg, nparams, fout )
								 != OK ) break;
		    _doprnt( arg[0], &arg[1], fout );
		    fprintf( fout, "\r\n" );
		  }
		fprintf( fout, " Abort instruction" );
	      }

	    else if ( req->instruction == BKPT_INSTR )
	      fprintf( fout, " Illegal instruction--Breakpoint?" );

	    else
	      fprintf(fout," Illegal instruction" );
	  }
	else
	    fprintf(fout," Illegal instruction" );
	break;

      case ADDRERROR:
      case BUSERROR:
        /* Check if error was on instruction fetch */
	if (!InstructionFetch(req->code))

	 if( mreply.processor == PROC_MC68020 )
	    /* The kernel doesn't fill in the instruction field for the
	     * Sun 3 yet, so we can't do anything with it, can we?
	     * Waiting for Lance's announcement, Eric 1-22-86.
	     */
	    req->instruction = 0;
	 else

	  {
	    /* Error was not on the instruction fetch */
	    /* find actual address of erroneous instruction */
	    if ((int)i >= TEAM_ORIGIN) /* not in kernel space */
	      {
		if ((i - MaxLookAhead*2-(TEAM_ORIGIN)) < 0)
		  instCnt = (int)(i - (TEAM_ORIGIN))>>1;
		else
		  instCnt = MaxLookAhead;
			/* get code */
		MoveFrom(pid, codeBuffer, i - instCnt,
			 (unsigned short *)0 + instCnt);
			/* find instruction */
		for( index=instCnt; (index > 0) &&
			(codeBuffer[index]!= req->instruction);
			index--)
		    ; 	/* empty */

		if (codeBuffer[index] != req->instruction)
		  i = (unsigned short *) req->errpc;
		else
		  i -= MaxLookAhead - index;
	      }
	  }

	if( req->type == BUSERROR )
	  {
	    switch (req->buserrortype)
	      {
	      case OUT_OF_RANGE:
	        fprintf(fout, " Address out of range");
	        break;
	      case SPURIOUS:
	        fprintf(fout, " Spurious bus error");
		break;
	      case SYSTEM_SPACE:
		fprintf(fout, " Access to system space");
		break;
	      case PROTECTION:
		fprintf(fout, " Memory protection violation");
		break;
	      case PAGE_INVALID:
	        fprintf(fout, " Access to invalid page");
	        break;
	      case SEG_INVALID:
	        fprintf(fout, " Access to invalid segment");
	        break;
	      case MB_TIMEOUT:
	        fprintf(fout, " Multibus access timeout");
		break;
	      case PARITY:
		fprintf(fout, " Memory parity error");
		break;
	      case BUS_TIMEOUT:
		fprintf(fout, " System bus timeout ");
		break;
	      case VMEBERR:
		fprintf(fout, " VME bus error");
		break;
	      case FPABERR:
		fprintf(fout, " FPA bus error");
		break;
	      case FPAENERR:
		fprintf(fout, " FPA enable error");
		break;
	      case WATCHDOG:
		fprintf(fout, " Watchdog or user reset");
		break;
	      default:
		fprintf(fout, " unknown exception");
		break;
	      }
	  }
	else
	  {
	    fprintf(fout, " Odd address access" );
	  }

	if (InstructionFetch(req->code))
	  {
	    fprintf(fout, " on instruction fetch from");
	    req->instruction = 0;
	  }
	else
	  {
	    if (ReadAccess(req->code))
	        fprintf(fout," on read from");
	    else
        	fprintf(fout," on write to");
	  }

	fprintf(fout, " address %x", req->accaddr );
	break;

      case TRAP0:
      case TRAP1:	case TRAP2:	case TRAP3:
      case TRAP4:	case TRAP5:	case TRAP6:
      case TRAP7:	case TRAP8:	case TRAP9:
      case TRAP10:	case TRAP11:	case TRAP12:
      case TRAP13:	case TRAP14:	case TRAP15:
	fprintf(fout, " Trap instruction");
	break;

      case SPURINT:
	fprintf(fout, " Spurious interrupt ");
	break;

      case INT1:	case INT2:	case INT3:
      case INT4:	case INT5:	case INT6:
      case INT7:
	fprintf(fout, " Unexpected interrupt");
	break;

      case SUSPENDED:
        fprintf(fout, " Forced exception");
	break;

      default: 
	fprintf(fout, " Unknown exception" );
	break;
      }

    fprintf(fout, " in process %x\r\n",pid );
    if( req->instruction != 0 ) 
        fprintf(fout, " Instruction  " );
    fprintf(fout, "  Program counter    Status register\r\n" );
    if( req->instruction != 0 )
        fprintf(fout, "     %4x     ", req->instruction );
    fprintf(fout, "       %6x              %2x\r\n", i, req->status );

    ReadProcessState(pid, &state);
    if (req->status & SUPERVISOR_STATE) 
	fprintf(fout, " Exception in kernel!!\r\n");

    /* Check for stack overflow */
    if (state.perProcess != 0)
      {
	stackSize = 0;
	MoveFrom(pid, &stackSize,
	      &(((PerProcessArea *) state.perProcess)->stackSize),
	      sizeof(stackSize));
	if ((unsigned) state.USER_STACK_POINTER <= (unsigned) state.perProcess ||
	      (unsigned) state.USER_STACK_POINTER > (unsigned) state.perProcess + stackSize)
	    fprintf(fout, " Probable stack overflow\r\n");
      }

    Flush( fout );
    return(i);
  }


int InstructionFetch(code)
    int code;
  {
    StatusWord stat;

    stat.s00.word = code;
    if (mreply.processor == PROC_MC68000)
	return (stat.s00.field.fc == FC_USER_PROGRAM ||
		stat.s00.field.fc == FC_SUPER_PROGRAM);
    else if (mreply.processor == PROC_MC68010)
	return (stat.s10.field.fc == FC_USER_PROGRAM ||
		stat.s10.field.fc == FC_SUPER_PROGRAM);
    else
	return (stat.s20.field.fc == FC_USER_PROGRAM ||
		stat.s20.field.fc == FC_SUPER_PROGRAM);
  }


int ReadAccess(code)
    int code;
  {
    StatusWord stat;

    stat.s00.word = code;
    if (mreply.processor == PROC_MC68000)
	return (stat.s00.field.rw == 1);  /* MC68000: 1 = read */
    else if (mreply.processor == PROC_MC68010)
	return (stat.s10.field.rwf == 1); /* MC68010: 1 = read */
    else
	return (stat.s20.field.rw == 1);  /* MC68020: 1 = read */

  }


/*
 * This part of the file contains the routines called by
 * StandardExceptionHandler when it handles an abort.  The abort routine can be
 * called with parameters.  If it is, the first parameter is to be interpreted
 * as a printf string.  Successive parameters are variables to be printed using
 * that string.  In addition to the usual format specifiers, we've added a %z,
 * which interprets the associated variable as a SystemCode and prints the error
 * message for that code (obtained from ErrorString).
 */

#define FP 14
#define STACK_ENTRY_SIZE 4
#define ADDQL4		0x588f	/* instruction to add 4 to stack pointer */
#define ADDQL8		0x508f	/* instruction to add 8 to stack pointer */
#define ADDL		0xdffc 	/* instruction to add contents of next long to
				    stack pointer */
#define ADDW		0xdefc 	/* instruction to add contents of next word to
				    stack pointer */
#define LEA		0x4fef	/* a faster version of ADDW */



static SystemCode GetNumberofParams( pid, paramaddr, fout )
	ProcessId pid;
	Unspec *paramaddr;
	File *fout;
  {
    /* Return the number of parameters in the call to abort, and the address
     * of the first (topmost, lowest addressed) parameter.
     */

    Unspec returnpc;
    Unspec frameptr;
    unsigned short instruction;
    long count;
    Processor_state state;
    SystemCode error;
    int nparams;

    if( !ReadProcessState( pid, &state ) )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n" );
	fprintf( fout, " Pid %x does not exist.\r\n", pid );
	return( 0 );
      }

    frameptr = state.regs[FP];
    if( frameptr == 0 )
      {
	fprintf( fout, "(stack has been corrupted, too)" );
	return( 0 );
      }

    /* The first parameter is below the frame pointer and the return address. */
    *paramaddr = frameptr + 2 * STACK_ENTRY_SIZE;

    error = MoveFrom( pid, &returnpc, frameptr + STACK_ENTRY_SIZE,
							STACK_ENTRY_SIZE );
    if( error != OK )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n");
	fprintf( fout," MoveFrom call for return address from stack:%s\r\n",
							 ErrorString( error ) );
	return( 0 );
      }

    error = MoveFrom( pid, &instruction, returnpc, sizeof( instruction ) );
    if( error != OK )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n");
	fprintf( fout," MoveFrom call for return address contents:%s\r\n",
							 ErrorString( error ) );
	return( 0 );
      }

    /* The instruction at the return address tells (at least in our compiler's
     * calling convention) how much to add to the stack pointer upon return
     * from the call.  If we divide that number by the size of each stack
     * entry, we get the number of parameters.  This same trick is used in
     * the debugger's stackdump.
     */

    switch( instruction )
      {
	case ADDQL4 :  return( 1 );
	case ADDQL8 :  return( 2 );
	case LEA    :
	case ADDW   :  error = MoveFrom( pid, &count, returnpc +
				 sizeof( instruction ), sizeof( short ) );
		       break;
	case ADDL   :  error = MoveFrom( pid, &count, returnpc +
				 sizeof( instruction ), sizeof( long ) );
		       break;
        default     :  return( 0 );
      }

    if( error != OK )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n");
	fprintf( fout, " MoveFrom call for number of parameters:%s\r\n",
							 ErrorString( error ) );
	return( 0 );
      }

    nparams = count / STACK_ENTRY_SIZE;
    if( nparams > MAX_NUM_ARGS )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n");
	fprintf( fout, " Too many parameters to abort.  (Maximum %d.)\r\n",
							 MAX_NUM_ARGS );
	return( 0 );
      }
		
    return( nparams );
  }



static SystemCode GetParams( pid, paramaddr, arg, nparams, fout )
	ProcessId pid;
	Unspec paramaddr;
	long arg[];
	int nparams;
	File *fout;
  {
    /* GetParams reads the parameters from the process pid's address space
     * (starting from paramaddr) into the exception handler's address space,
     * starting at arg.
     */
  
    SystemCode error;

    error = MoveFrom( pid, arg, paramaddr, nparams * STACK_ENTRY_SIZE );
    if( error != OK )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n" );
	fprintf( fout, "	MoveFrom call for parameters:%s\r\n",
							 ErrorString( error ) );
      }
    return( error );
  }



static SystemCode GetFormatString( pid, fmtstring, fmtaddr, fout )
	ProcessId pid;
	char *fmtstring;
	Unspec fmtaddr;
	File *fout;
  {
    /* The first parameter is a pointer to a printf style formatting string.
     * That string must be copied to our address space (starting at fmtstring).
     * The first argument, which like the others will be passed to _doprnt,
     * will then be changed to fmtstring.
     */

    SystemCode error;

    error = MoveFrom( pid, fmtstring, fmtaddr, MAX_STRING_LENGTH );
    if( error != OK )
      {
	fprintf( fout, "\r\nError in StandardExceptionHandler:\r\n" );
	fprintf( fout, "	MoveFrom call for abort format string:%s\r\n",
							 ErrorString( error ) );
      }
    return( error );
  }



static SystemCode GetStringParams( pid, strings, arg, nparams, fout )
	ProcessId pid;
	char strings[ MAX_NUM_STRINGS ][ MAX_STRING_LENGTH ];
	Unspec arg[];
	int nparams;
	File *fout;
  {
    /* This routine actually serves two functions.  It interprets the %z's
     * in the format string (which starts at strings[0]) as SystemCodes,
     * gets the associated value from the argument list stored starting at
     * arg, and replaces that value with a pointer to the ErrorString for
     * that SystemCode.  It then adjusts the %z to a %s.
     *
     * Second, for each of the original %s's in the format string, it copies
     * the pointed-to string into the exception handlers address space--in
     * the strings array.  The pointer that had been in the arg list is changed
     * to point to this new copy.
     *
     * Note that since we didn't want to malloc from within the exception
     * handler (too unreliable), we allocated space for these strings on the
     * stack.  That limits us in both number and length of strings.
     */

    SystemCode error = OK;
    int stringcount =  1;
    int paramcount = 1;
    int notStringError = 1;
    char *p = strings[0];
    while( *p != '\0'  &&  p < (char *) strings + MAX_STRING_LENGTH  &&
							paramcount < nparams )
      {
	if( *p == '%' )
	  {
	    p++;
	    while( ( (*p >= '0'  &&  *p <= '9') || *p == '-' || *p == '*' )  &&
				p < (char *) strings + MAX_STRING_LENGTH )
		p++;

	    switch( *p )
	      {
		case 'z':
		  arg[ paramcount ] = (Unspec) ErrorString( arg[ paramcount ] );
		  paramcount++;
		  *p = 's';
		  break;

		case 's':
		  if( stringcount >= MAX_NUM_STRINGS )
		    {
		      /* Remember that the format string is included in this
		       * maximum--that's why >= is the test, not >.
		       */
		      if( notStringError )
			{
			  fprintf( fout,
				"\r\nError in StandardExceptionHandler:\r\n" );
			  fprintf( fout,
				"Too many strings (> %d) in call to abort.\r\n",
				MAX_NUM_STRINGS - 1 );
			  notStringError = 0;
			}
		      break;
		    }

		  error = MoveFrom( pid, strings[ stringcount ],
					 arg[ paramcount ], MAX_STRING_LENGTH );
		  if( error != OK )
		    {
		      fprintf( fout,
				"\r\nError in StandardExceptionHandler:\r\n" );
		      fprintf( fout,
				"MoveFrom call for param %d of abort: %s\r\n",
				paramcount, ErrorString( error ) );
		      return( error );
		    }

		  strings[ stringcount ][ MAX_STRING_LENGTH - 1 ] = '\0';
		  arg[ paramcount++ ] = strings[ stringcount++ ];
		  break;

		case '%':
		  break;

		default:
		  paramcount++;
	      }
	  }

	p++;
      }
    return( error );
  }
