/*
 * Distributed V Kernel
 * Copyright (c) 1986 by Stanford University, All rights reserved.
 *
 * Kernel MSCP disk driver, William Lees
 */

/*
 * This source file contains the major entry points for RQDX device server.
 * These include the device power-up routine, and well as the subroutines
 * for manipulating device UIO objects:
 *	Create, Release, Read, Write, Modify and Query
 */

/* Common definitions */

#include "rqdx.h"		/* RQDX I/O protocol definitions */
#include "process.h"		/* Process structures, ProcessId */
#include "dm.h"			/* Device management, DeviceInstance */
#include "interrupt.h"		/* Setting interrupt vectors */
#include "Vioprotocol.h"	/* V I/O protocol defs */

/* Imports */

extern int Asm_MscpInterrupt();		/* Device interrupt routine */
extern SystemCode MscpStartInit();	/* Start cntrlr port init seq */
extern SystemCode MscpUnitBlockIO();	/* Issue block I/O to a unit */

extern	SystemCode	NotWriteable();

extern short RqdxDriverState;			/* State of controller */
extern struct rqdx_unit_info RqdxUnitInfo[3];	/* State of the units */

/* Exports */

extern SystemCode MscpCreate();		/* Routine to create a UIO object */
extern SystemCode MscpPowerup();	/* Routine to init the RQDX */

/* Forward */

SystemCode MscpRead();		/* Read a UIO object */
SystemCode MscpWrite();		/* Write a UIO object */
SystemCode MscpReadWrite();	/* Common code for reading and writing */
SystemCode MscpQuery();		/* Query a UIO object */
SystemCode MscpModify();	/* Modify a UIO object */
SystemCode MscpRelease();	/* Disown a UIO object */

/* Private */

unsigned	SavedIPL;	/* Storage for IPL */

/****************************************************************************
 * Routine name:	MscpPowerup
 *
 * Description:
 *
 *	This routine is called at kernel startup to init the controller.
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	SavedIPL	Temp storage for hardware IPL
 *
 */

SystemCode MscpPowerup()
{
	SystemCode s;

	setexvec(Asm_MscpInterrupt, VecRqdx);	/* Load interrupt vector */

	asm("	mfpr	$ipl, _SavedIPL");	/* Save old IPL */
	asm("	mtpr	$15, $ipl");		/* Allow device interrupts */

/* Start controller port initialization sequence.  The hardware interrupt
 * priority was lowered to allow controller interrupts.  The sequence will
 * complete asynchronously.  We wait until the controller becomes on-line
 * or an error occurs.
 */
	if ( ( s = MscpStartInit() ) != OK ) return(s);

	while ( (RqdxDriverState != S_IDLE) &&
		(RqdxDriverState != S_RUN) &&
		(s == OK ) ) ;

	asm("	mtpr	_SavedIPL, $ipl");	/* Restore old IPL */

	return( s );
};

/****************************************************************************
 * Routine name:	MscpCreate
 *
 * Description:
 *
 *	This routine creates a UIO object for one of the units on the
 *	RQDX controller.
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxDriverState
 *	RqdxUnitInfo
 *
 */

SystemCode MscpCreate( pd, desc )
	Process *pd;
	DeviceInstance *desc;
{
	CreateInstanceRequest *req = (CreateInstanceRequest *) &(pd->msg);

/* Checks.  Unit number must be in range, controller and unit must each
 * be on-line.  Creation mode must be FCREATE.
 */
	if ( (desc->unit < 0) || (desc->unit > 2) ) return( BAD_BUFFER );

	if ( RqdxDriverState != S_RUN ) return( DEVICE_ERROR );

	if ( RqdxUnitInfo[desc->unit].unit_online == 0 )
		return( DEVICE_ERROR );

	if ( req->filemode != FREAD && req->filemode != FMODIFY )
		return( MODE_NOT_SUPPORTED );

/* Initialize the device instance descriptor. */

	desc->type = MULTI_BLOCK + READABLE;
	if ( req->filemode == FREAD )
	    desc->type += WRITEABLE;
	desc->owner = pd->pid;

	desc->readfunc = MscpRead;
	desc->writefunc = req->filemode == FREAD ? NotWriteable : MscpWrite;
	desc->modifyfunc = MscpModify;
	desc->queryfunc = MscpQuery;
	desc->releasefunc = MscpRelease;
	

/* Fill in unit specific info */

	desc->blocksize = SECSIZE * BLOCK_FACTOR;
	desc->lastblock =
		RqdxUnitInfo[desc->unit].unit_last_block / BLOCK_FACTOR;

	return( OK );
}

/****************************************************************************
 * Routine name:	MscpRead
 *
 * Description:
 *
 *	This routine is called to read RQDX UIO objects.
 *
 * Inputs:
 *
 *	pd	Descriptor of requesting process
 *	desc	UIO
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

SystemCode MscpRead( pd, desc )
	Process *pd;
	DeviceInstance *desc;
{
	return ( MscpReadWrite( 1, pd, desc ) );
};

/****************************************************************************
 * Routine name:	MscpWrite
 *
 * Description:
 *
 *	This routine performs a write to a RQDX UIO object.
 *
 * Inputs:
 *
 *	pd	Descriptor of requesting process
 *	desc	UIO
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

SystemCode MscpWrite( pd, desc )
	Process *pd;
	DeviceInstance *desc;
{
	return ( MscpReadWrite( 0, pd, desc ) );
}

/****************************************************************************
 * Routine name:	MscpReadWrite
 *
 * Description:
 *
 *	This routine is the common code to read or write a UIO object
 *
 * Inputs:
 *
 *	read	code indicating operation, 1=read, 0=write
 *	pd	Descriptor of requesting process
 *	desc	UIO
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxUnitInfo
 *
 */

SystemCode MscpReadWrite( read, pd, desc )
	short read;
	Process *pd;
	DeviceInstance *desc;
{
	IoRequest *req = (IoRequest *) &(pd->msg);
	char * user_buffer;
	SystemCode s;

/*	if (read) 
	printx("[MscpRead: unit %d, byte count %d, 1024-byte block %d]\n",
			desc->unit, req->bytecount, req->blocknumber);
	else
	printx("[MscpWrite: unit %d, byte count %d, 1024-byte block %d]\n",
			desc->unit, req->bytecount, req->blocknumber);*/

/* Check that unit in question is online and that bytecount is valid.
 * Byte count must be even, smaller than disk buffer, and smaller than
 * disk size.
 */
	if ( RqdxUnitInfo[desc->unit].unit_online == 0 )
		return( DEVICE_ERROR );

	if ( (req->bytecount & 1) == 1 ) {
		printx("Odd bytecount is illegal\n");
		req->bytecount = 0;
		return( BAD_BYTE_COUNT);
	}

	if ( req->bytecount > RQDX_DISK_BUFFER_SIZE ) {
		printx("Bytecount too big; bigger than buffer\n");
		req->bytecount = 0;
		return ( BAD_BYTE_COUNT );
	}

	if (req->bytecount >
		( ( (RqdxUnitInfo[desc->unit].unit_last_block + 1) *
			SECSIZE * BLOCK_FACTOR) - 1) ) {
		printx("Bytecount too big; bigger than disk\n");
		req->bytecount = 0;
		return( BAD_BYTE_COUNT );
	};

	if (req->bytecount <= IO_MSG_BUFFER)
		user_buffer = (char *) req->shortbuffer;
	else
		user_buffer = req->bufferptr;

	s = MscpUnitBlockIO(pd,
		read,				/* Code to indicate op */
		desc->unit,
		req->bytecount,
		user_buffer,
		req->blocknumber * BLOCK_FACTOR);

	if (s != OK) return ( s );

	pd->state = AWAITING_INT;	/* Suspend process until completion */

	return( NO_REPLY );	/* Interrupt routine will reply */
}

/****************************************************************************
 * Routine name:	MscpRelease
 *
 * Description:
 *
 *	This routine disassociates a process from a UIO object, which
 *	is statically allocated and not disposed of.
 *
 * Inputs:
 *
 *	pd	Descriptor of reqesting process
 *	desc	UIO
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 */

SystemCode MscpRelease( pd, desc)
	Process *pd;
	DeviceInstance *desc;
{
	desc->owner = 0;

	return( OK );
}

/****************************************************************************
 * Routine name:	MscpQuery
 *
 * Description:
 *
 *	This routine is supposed to return information or statistics
 *	about the UIO object. Currently a nop.
 *
 * Inputs:
 *
 *	pd		Descriptor of requesting process
 *	desc		UIO
 *	dirIndex	
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 */

SystemCode MscpQuery( pd, desc, dirIndex )
	Process *pd;
	DeviceInstance *desc;
	unsigned short dirIndex;
{
	printx("[MscpQuery]\n");
	return( OK );
}

/****************************************************************************
 * Routine name:	MscpModify
 *
 * Description:
 *
 *	This routine allows one to modify characteristics of a UIO object.
 *	Currently a nop.
 *
 * Inputs:
 *
 *	pd		Descriptor of requesting process
 *	desc		UIO
 *	dirIndex	
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 */

SystemCode MscpModify( pd, desc, dirIndex)
	Process *pd;
	DeviceInstance *desc;
	unsigned short dirIndex;
{
	printx("[MscpModify]\n");
	return( OK );
}

