/*
 * V Kernel - Copyright (c) 1985 by Stanford University
 *
 * Kernel Qvss Serial Keyboard support routines
 *
 * I have no documentation on the serial keyboard.  How it works was
 * garnered from an adb of /vmunix on a microvax running ultrix and
 * from said sources.  The initialization here is called in qvss.c.
 *
 * We only process receive interrupts, we poll for transmit interrupts
 * because they are only done at initialization.  We have to
 * write to the keyboard to initialize it.
 */

#include "Venviron.h"
#include "cbuff.h"
#include "dm.h"
#include "interrupt.h"
#include "qvss.h"
#include "qvsskeytab.h"


/* Imports */
extern (*ConsoleBellRoutine)();
extern CircularBuffer ConsoleInBuff;
extern DeviceInstance *ConsoleInInstance;

/* Forward */
extern unsigned long QvssKeyboardPutchar();

short	QvssKeyboardDefaults[15] = {
	    QV_DOWN,		/* 0 doesn't exist */
	    QV_AUTODOWN,
	    QV_AUTODOWN,
	    QV_AUTODOWN,
	    QV_DOWN,
	    QV_UPDOWN,
	    QV_UPDOWN,
	    QV_AUTODOWN,
	    QV_AUTODOWN,
	    QV_AUTODOWN,
	    QV_AUTODOWN,
	    QV_AUTODOWN,
	    QV_AUTODOWN,
	    QV_DOWN,
	    QV_AUTODOWN
    };

short	QvssKeyboardInitList[] = {	/* reset random keyboard stuff */
	    QV_AR_ENABLE,	/* autorepeat on */
	    QV_CL_ENABLE,	/* keyclick on */
	    0x84,		/* keyclick volume! */
	    QV_KBD_ENABLE,	/* keyboard itself */
	    QV_BELL_ENABLE,	/* Bell */
	    0x84,		/* Bell volume! */
	    QV_LED_DISABLE,	/* keyboard led's */
	    LED_ALL
	};
#define QVKBDINITLENGTH	(sizeof(QvssKeyboardInitList)/sizeof(short))

Call_inthandler(IntQvssUart);


/* Routines that interact directly with the keyboard */

unsigned long QvssKeyboardGetchar()
  {
    register unsigned int key;
    register unsigned int c;
    static short Lock = 0, Shift = 0, Cntrl = 0, Last = 0;

    key = 0x0ff & Qvss->regs[QV_UBUFF+UARTAOFFS];

    /* map the character received into what it really is */
    switch (key)
      {
	case QV_KEY_LOCK:
	    Lock ^= 0xffff;
	    if (Lock)
		QvssKeyboardPutchar( QV_LED_ENABLE );
	    else
		QvssKeyboardPutchar( QV_LED_DISABLE );
	    QvssKeyboardPutchar( LED_3 );
	    return(0xffffffff);
	case QV_KEY_SHIFT:
	    Shift ^= 0xffff;
	    return(0xffffffff);
	case QV_KEY_CNTRL:
	    Cntrl ^= 0xffff;
	    return(0xffffffff);
	case QV_KEY_ALLUP:
	    Cntrl = Shift = 0;
	    return(0xffffffff);
	case QV_KEY_REPEAT:
	    c = Last;
	    break;
	default:
	    c = Shift ? QvssShiftKeyTable[key]:QvssKeyTable[key];
	    if (Cntrl && c >= ' ' && c <= '~' )
	        c &= 0x1f;
	    if (Lock && c >= 'a' && c <= 'z')	/* Caps lock */
	        c -= 'a'-'A';
	    break;
	    
      }
    
    Last = c;
    return(c);
  }

unsigned long QvssKeyboardPutchar( c )
    char c;
  {
    while (0 == (Qvss->regs[QV_USCSR+UARTAOFFS] & 0x4));
    Qvss->regs[QV_UBUFF+UARTAOFFS] = c;
  }

QvssBell()
  {
    QvssKeyboardPutchar( QV_RING_BELL );
  }

QvssKeyboardInit()
  {
    register int i;
    
    QvssKeyboardPutchar( QV_DEFAULTS );
    for (i = 1; i < 15; i++)
	QvssKeyboardPutchar( QvssKeyboardDefaults[i] | (i<<3) );
    for (i = 0; i < QVKBDINITLENGTH; i++)
	QvssKeyboardPutchar( QvssKeyboardInitList[i] );

    ConsoleBellRoutine = &QvssBell;

  }

/* INTERRUPT HANDLER */
IntQvssUart() 
  /*
   * Interrupt handler for the Qvss Uarts.
   *
   * For now we know that the only interrupts we will get are from
   * Key events from the keyboard.
   */
  {
    register unsigned c;
    register Process *pd;
    register DeviceInstance *inst;
    register char *string;

    /* lower our IPL to something reasonable */
    asm("	mtpr	$0x14, $ipl");
    
    /* Machine specific read of character */
    if (0xffffffff == (c = QvssKeyboardGetchar()))
	return;

    if ( (inst = ConsoleInInstance ) != NULL )
      {
	string = NULL;
	if (c & 0x80)
	  {
	    if (0 == (string = QvssSpecialKeyTable[c&0x7f])) /* no character */
		return;
	    c = *string;
	  }

	if (MAP_TO_RPD( pd, inst->reader) != NULL )
	  {
	    /* Someone waiting, so give it to them directly */
	    
	    ((IoReply *)&pd->msg)->replycode = OK;
	    ((IoReply *)&pd->msg)->bytecount = 1;
	    ((IoReply *)&pd->msg)->shortbuffer[0] = c;
	    inst->nextblock++;
	    inst->reader = 0;

	    Addready( pd );
	  }
	else
	  { 
	    /* no one is waiting so queue the character. */
	    CbuffPut( &ConsoleInBuff, c );
	  }
	  
	if (string != NULL) /* queue the rest of the characters */
	    while (*++string)
		CbuffPut( &ConsoleInBuff, *string );
      }
  } /* IntQvssUART */

