/*
 * V Kernel - Copyright (c) 1985 by Stanford University
 *
 * Kernel Framebuffer support routines
 *
 * The "qvss" provides access to the framebuffer and to the framebuffer's
 * machine registers.  Opening this device causes the framebuffer to be
 * mapped into your address space, and reading from or writing to this
 * device causes a read/write to the machine register that controls it.
 */

#ifndef _QVSS
#define _QVSS
#include "memory.h"

/*
 * Qvss Location (in Q-bus I/O and Memory space), size, etc.
 *   PhysQvss and QVSSMEMSWITCHES assume factory settings for I/O and
 *   memory addresses respectively.
 */
#define	Qvss		((qvss_reg_t *)(SYSSEGSTART + PHYSMEMSIZE + 0x1e80))
#define	PhysQvss	(IOMEMBASE+0x1e80)
#define NUMQVSSREGS	32
#define NUMUSERQVSSREGS	6
#define QVSSSIZE	0x40000	/* in bytes */
#define	QVSSVBASEADDR	(TEAM_LIMIT-QVSSSIZE)
#define QVSSMEMSWITCHES 0xF
#define	QVSSPBASEADDR	(QbusMemOffset+(QVSSSIZE*QVSSMEMSWITCHES))

#define NUMSYSQVPAGES	NumPages(PageTblLen(NumPages(QVSSSIZE)))

typedef struct {
    unsigned short  regs[NUMQVSSREGS];
} qvss_reg_t;

/*
 * For a given team, returns the index into the system page table
 * where the page table for teamno's page table entries for the QVSS
 * begin and end.
 */
#define TEAMSYSQVSTART(teamno) \
   (TEAMSYSPTSTART(teamno)+NumPages(PageTblLen(NumPages(QVSSVBASEADDR))))

#define TEAMSYSQVEND(teamno) \
    (TEAMSYSQVSTART(teamno)+NUMSYSQVPAGES)


/*
 * For a given team, returns the base address of the beginning of the page
 * table for the Qvss, and also the address of the first non qvss page
 * table entry.
 */
#define TEAMPTQVBASE(teamno) \
   (TEAMPTBASE(teamno)+PageTblLen(NumPages(QVSSVBASEADDR)))

#define TEAMPTQVEND(teamno) \
   (TEAMPTQVBASE(teamno)+PageTblLen(NumPages(QVSSSIZE)))

/* what follows are some defines for the various registers in the
 * Qvss device.  I suggest you read a manual for a description
 * of the various fields.  It is incredible.
 */

/* the base csr */
#define QV_CSR		0
#define		QV_BIGSCREEN	0x0001
#define		QV_ENABLEVID	0x0004
#define		QV_CURSORFUNC	0x000f
#define		QV_VIDLOOP	0x0010
#define		QV_LOOPTEST	0x0020
#define		QV_INTENABLE	0x0040
#define		QV_ONCURS	0x0080
#define		QV_MOUSEA	0x0100
#define		QV_MOUSEB	0x0200
#define		QV_MOUSEC	0x0400
#define		QV_MEM0		0x0800
#define		QV_MEM1		0x1000
#define		QV_MEM2		0x2000
#define		QV_MEM3		0x4000

/* the framebuffer stuff */
#define	QV_CURSX	1
#define		QV_CURSPOSMASK	0x03ff
#define	QV_MOUSE	2
#define		QV_MOUSEXMASK	0x00ff
#define		QV_MOUSEYMASK	0xff00
#define	QV_CRTCA	4
#define		QV_CRTREG	0x001f
#define			QV_HORTOT	0
#define			QV_HORDIS	1
#define			QV_HSYNCPOS	2
#define			QV_SYNCWIDTH	3
#define			QV_VERTTOT	4
#define			QV_VERTTOTADJ	5
#define			QV_VERTDISP	6
#define			QV_VSYNCPOS	7
#define			QV_CRTMODE	8
#define			QV_SCANLINE	9
#define			QV_CSCANSTART	10
#define			QV_CSCANEND	11
#define			QV_STARTADDRHI	12
#define			QV_STARTADDRLO	13
#define			QV_CURSADDRHI	14
#define			QV_CURSADDRLO	15
#define			QV_LPENPOSHI	16
#define			QV_LPENPOSLO	17
#define		QV_VERTBLANK	0x0020
#define		QV_LPENREGFULL	0x0040
#define		QV_UPDATESTROBE	0x0080
#define	QV_CRTCD	5

/* the interrupt controller */
#define	QV_IUARTMASK	0x01	/* the interrupt bit field masks */
#define	QV_IVSYNCMASK	0x02
#define	QV_IMOUSEMASK	0x04
#define	QV_ICSTARTMASK	0x08
#define	QV_IMOUSEAMASK	0x10
#define	QV_IMOUSEBMASK	0x20
#define	QV_IMOUSECMASK	0x40

#define	QV_IUARTNUM	0	/* the interrupt bit field numbers */
#define	QV_IVSYNCNUM	1
#define	QV_IMOUSENUM	2
#define	QV_ICSTARTNUM	3
#define	QV_IMOUSEANUM	4
#define	QV_IMOUSEBNUM	5
#define	QV_IMOUSECNUM	6

#define	QV_PRIORITIES	0x01
#define	QV_IMVECTORS	0x02
#define	QV_IMINTPOLL	0x04
#define	QV_IMREG	0x60
#define	QV_IMINTENABLE	0x80
#define	QV_ICDR		6
#define	QV_ICSR		7

/* The UARTs */
#define	QV_UISM		21
#define	QV_UMODE	16
#define	QV_USCSR	17
#define	QV_UCMD		18
#define	QV_UBUFF	19
#define	UARTBOFFS	8
#define	UARTAOFFS	0

/* bits for the Qvss keyboard.  Attached to UARTA */
#define QV_UPDOWN 	0x86		/* bits for setting lk201 modes */
#define QV_AUTODOWN 	0x82
#define QV_DOWN 	0x80
#define QV_DEFAULTS 	0xd3		/* reset (some) default settings */
#define QV_AR_ENABLE 	0xe3		/* global auto repeat enable */
#define QV_CL_ENABLE 	0x1b		/* keyclick enable */
#define QV_KBD_ENABLE 	0x8b		/* keyboard enable */
#define QV_BELL_ENABLE 	0x23		/* the bell */
#define QV_LED_ENABLE 	0x13		/* light led */
#define QV_LED_DISABLE 	0x11		/* turn off led */
#define QV_RING_BELL 	0xa7		/* ring keyboard bell */
#define LED_1 		0x81		/* led bits */
#define LED_2 		0x82
#define LED_3 		0x84
#define LED_4 		0x88
#define LED_ALL 	0x8f
#define QV_KDOWN_ERROR	0x3d		/* key down on powerup error */
#define QV_POWER_ERROR 	0x3e		/* keyboard failure on powerup test */
#define QV_OUTPUT_ERROR	0xb5		/* keystrokes lost during inhibit */
#define QV_INPUT_ERROR 	0xb6		/* garbage command to keyboard */
#define QV_LOWEST	0x56		/* lowest significant keycode */
#define QV_DIV6_START	0xad		/* start of div 6 */
#define QV_DIV5_END	0xb2		/* end of div 5 */

#endif
