/*
 * Distributed V Kernel - Copyright (c) 1981 by David Cheriton.
 * (Transliterated from Zed and Verex Kernel)
 * Copyright (c) 1982 Stanford University.
 *
 *  Kernel time services - machine-independent portion
 *  Structured as a simple workfrom processor.
 *  Major service is the timeout queue, which delays a process
 *  for pd->timeout_count clicks and then invokes the
 *  pd->timeout_func.
 */

#include "process.h"
#include "timer.h" /* (Typically) Machine-dependent header */
#include "Vquerykernel.h"  /* For setting P_Delayq */

extern Process_id Device_Server_Pid;
extern SyncQueue *RemoveQueue();

/* Static Globals containing the current time GMT in seconds and
 * clicks (clock interrupts) since January 1, 1970.
 */

TimeRecord Time;
SyncQueue Delayq;
long DelayClickCount = 0;	/* Clock ticks not handled while in kernel */

InitTimeWFProcessor()
  {
    Delayq.head = NULL;
    Delayq.type = TIMEOUT_QUEUE;
#ifndef VAX
    P_Delayq_head = &Delayq.head;
#endif
  }

SystemCode GetTime( pd )
    register Process *pd;
  /* Return the current time in seconds and store
   * the current number of clicks into *clicksptr
   * if it is non-zero.
   */
  {
    register KernelRequest *reply;

    reply = (KernelRequest *)&pd->msg;
    disable;
    reply->unspecified[1] = (unsigned) Time.clicks;
    reply->unspecified[0] = (unsigned) Time.seconds;
    enable;
    return( OK );
  }

SystemCode SetTime( pd )
    register Process *pd;

  /* Set the current time to the specified number of seconds
   * and clicks.
   */
  {
    register KernelRequest *req;

    req = (KernelRequest *)&pd->msg;

    disable;
    Time.seconds = (unsigned) req->unspecified[0];
    Time.clicks = (unsigned) req->unspecified[1];
    enable;
    return( OK );
  }

DelayProcess( pd )
    register Process *pd;

  /* Block the specified process for the specified number
   * of clicks specified in timeout_count.
   * The number of clicks remaining in the delay time is returned,
   * which is 0 unless Wakeup has been used.
   */
  {
    register Process *tmp_pd, *prev;
    register long delay_clicks;

/*%%%*/if (pd->queuePtr != NULL)
	{
	  printx("DelayProcess: process %x already on queue %x\n", 
		pd, pd->queuePtr);
	  StackDump();
#ifndef VAX
	  ;asm("trap #14");;
#else
	  ;asm("mtpr $0xf05, $0x23");;
#endif
        }
    if( (delay_clicks = pd->timeout_count) == 0 )
      {
	(*(pd->timeout_func))( pd );
	return;
      }
    
    /* There may be some clicks 'queued up' because the tick interrupt came
     * while we were in the kernel. Adjust for this by adding to our count.
     */
    delay_clicks += DelayClickCount;

    /* Insert in delay queue */
    Lockq( &Delayq );    
    tmp_pd = Delayq.head;

    if( tmp_pd == NULL ) Delayq.head = pd;
    else
      if( delay_clicks < tmp_pd->timeout_count ) 
	Delayq.head = pd;
    else
      {
        do /* Find pd's position in delay queue */
          {
            delay_clicks -= tmp_pd->timeout_count;
            prev = tmp_pd;
            tmp_pd = tmp_pd->link;
          }
        while( tmp_pd != NULL && delay_clicks >= tmp_pd->timeout_count );
        prev->link = pd;
      }
    if( (pd->link = tmp_pd) != NULL )
        tmp_pd->timeout_count -= delay_clicks;

    pd->timeout_count = delay_clicks;
    pd->queuePtr = &Delayq;
    Unlockq( &Delayq );
  }

SyncQueue *RemoveDelayQueue( pd )
    register Process *pd;
  /* Remove the specified process from Delayq, if there.
   */
  {
    if (pd->queuePtr != &Delayq) return(NULL);
    return(RemoveQueue(pd));
  }

SystemCode Wakeup( pd ) register Process *pd;

  /* Unblock the specified process providing it is blocked in the 
   *  delay queue due to a Delay() kernel operation.  The state
   *  DELAYING is used only for processes that have invoked Delay(),
   *  not for other processes on the Delayq. 
   */
  {
    register Process *wakepd;
    register KernelRequest *krequest;
    register KernelRequest *req;

    req = (KernelRequest *)&pd->msg;
    if( !MAP_TO_RPD(wakepd, req->pid) ) return( NONEXISTENT_PROCESS );

    /* Remove from delay queue */
    if( RemoveDelayQueue(wakepd) )
      {
	if( wakepd->state == DELAYING )
	  {
	    krequest = (KernelRequest *)&wakepd->msg;
	    krequest->opcode = AWOKEN;
	    /* Set remaining time before wakeup */
	    krequest->unspecified[1] = wakepd->timeout_count;
	    Addready( wakepd );
	    return( OK );
	  }
	DelayProcess( wakepd );
      }
    return( NOT_FOUND );
  }

Timer_interrupt()

  /* Called on each kernel timer interrupt.
   * Several duties are performed:
   * - Increment the current time.
   * - Check and update the delay queue.
   * Before doing anything we check the value of KernelInterrupted.
   * If non-zero (meaning we have interrupted the kernel), we
   * return immediately.
   */
  {
    extern Process *Active; /* Remove when PD is passed. */
    extern short KernelInterrupted;
    int Retransmit(), NetCheck();
    extern int NetAlarm, DelayClickCount;
    register Process *pd;

    /* Update the current time */
    if( ++Time.clicks > CLICKS_PER_SEC-1 )
      {
        ++Time.seconds;
        Time.clicks = 0;
      }
    ++Active->processorTime;
    DelayClickCount++;

    if (KernelInterrupted) return;

    /* Check the delay queue and execute the timeout function of
     * all of the processes that have timed out since the last
     * timer interrupt that didn't occur while the kernel was
     * running.
     */
    do
        if ((pd = Delayq.head) == NULL)
	    DelayClickCount = 0;
	else
	  {
	    if (DelayClickCount > pd->timeout_count)
	      {
	        DelayClickCount -= pd->timeout_count;
		pd->timeout_count = 0;
	      }
	    else
	      {
		pd->timeout_count -= DelayClickCount;
		DelayClickCount = 0;
	      }
	    while (pd != NULL && pd->timeout_count == 0)
	      {
		Delayq.head = pd->link;
/*%%%*/if (pd->queuePtr != &Delayq)
	{ printx("Timer_interrupt: Bogus process %x on Delayq\n", pd);
#ifndef VAX
	  ;asm("trap #14");;
#else
	  ;asm("mtpr $0xf05, $0x23");;
#endif
	}
		pd->queuePtr = NULL;
		(*(pd->timeout_func))( pd );
		pd = Delayq.head;
	      }
	  }
    while (DelayClickCount != 0);

    if( --NetAlarm == 0 ) 
      {
        NetCheck();
      }
  }
