/*
 * Distributed V Kernel - Copyright (c) 1982 by Stanford University
 *
 * Inter kernel communication interrupt routines
 *
 */
/* #define DEBUG 1 */

#include "process.h"
#include "naming.h"
#include "Vikc.h"
#include "ikc.h"


/* Imports */

extern unsigned FreePds;
extern Unspec Retransmit();
extern LhnRec *LhnTentative, *LhnFreeList, *Lh_bundle[];
extern Process *Alienate(), *AllocatePd();
extern SyncQueue Readyq, *RemoveQueue(), *RemoveDelayQueue();
extern TimeRecord Time;
extern Process *FindSuspendPd();
extern ProcessId KernelServerPid;
extern char *HostCacheString();

/* Kernel packet handler */

extern Process_id
    pRemoteSend(),
    pRemoteReply(),
    pRemoteForward(),
    pRemoteReceiveSpecific(),
    pBreathOfLife(),
    pRemoteGetPid(),
    pRemoteGetPidReply(),
    pRemoteCopyFromReq(),
    pRemoteCopyFromRep(),
    pRemoteCopyToReq(),
    pRemoteCopyToRep(),
    pNAck(),
    pLhnCollisionCheck(),
    pLhnCollisionReply(),
    pBadPacketType();

typedef Process_id (*PFPID)(); /* pointer to a function returning a pid */

PFPID RemoteTrapTable[16] =	/* has to agree with the remote opcodes */
  {
    pBadPacketType,
    pRemoteSend,
    pRemoteReply,
    pRemoteForward,
    pRemoteReceiveSpecific,
    pBreathOfLife,
    pRemoteGetPid,
    pRemoteGetPidReply,
    pRemoteCopyFromReq,
    pRemoteCopyFromRep,
    pRemoteCopyToReq,
    pRemoteCopyToRep,
    pNAck,
    pLhnCollisionCheck,
    pLhnCollisionReply,
    pBadPacketType
  };

    
Process_id pBadPacketType( kp )
    kPacket *kp;
  {
    printx("Bad kernel packet type %x->%x (%x) [ %s ]\n", kp->srcPid,
		kp->dstPid, kp->packetType, HostCacheString(kp->srcPid) );
    return;
  }

Process_id pRemoteSend( kp )
    register kPacket *kp;
  {
    register Process *receiver, *pd, *aa;

    /* Check if a duplicate of previous (unreplied) send. */
    if( MAP_TO_APD(pd, kp->srcPid) )
      {
	if( ( pd->seqNo == kp->sequenceNo ) )
	  {
	    /* Duplicate or Forward */
	    if( RemoveDelayQueue(pd) )
	      {
/*%%%*/		if( (pd->state != REPLIED_TO) && (pd->state != FORWARDED_TO) )
		  {
		    printx("Alien in state %d in delay queue\n", pd->state );
		    DelayProcess( pd );
		    return; /* Alien in some random state. */
		  }
		/* pd->packetType is remoteReply or remoteForward. */
		/* Check if this is a send packet resulting from the forward
		 * represented by this alien.
		 */
		if( (pd->state == FORWARDED_TO) &&
			( ((ProcessId) pd->msgq.head != kp->dstPid) ||
			  ((ProcessId) pd->msgq.tail != kp->forwarder)) )
		    goto recycle_alien;

		pd->timeout_count =
		   2 * MAX_RETRANSMISSIONS * SEND_ACKED_RETRANS_PERIOD;
		
		if ( ( pd->msg.sysCode == DISCARD_REPLY ) &&
		     ( pd->state == REPLIED_TO ) )
		  {
		    /* Supress DISCARD_REPLY reply packets */
		    DelayProcess( pd );
		    return;
		  }
	      }
	    else
	      { /* Must be AWAITING_REPLY from receiver. Get the alien to
	         * send the breathOfLife because pd may be in a queue. The
		 * alien will be destroyed by timeout queue mechanism
		 * after the packet is written.
		 */
		if( (pd=Alienate(kp)) == NULL ) return;
		pd->packetType = breathOfLife;
	      }
	    WriteKPacket( pd );
	    return;
	  }
	else if (
	    ( ( pd->seqNo < kp->sequenceNo ) &&
	      ( kp->sequenceNo - pd->seqNo > DUPLICATE_SUPPRESSION_DIST ) ) ||
	    ( ( kp->sequenceNo < pd->seqNo ) &&
	      ( pd->seqNo - kp->sequenceNo < DUPLICATE_SUPPRESSION_DIST ) ) )
	  {
	    /* A duplicate from a previous message transaction */
	    return;
	  }
	else /* A new send from this pid, copy to old alien PD */
	  {
	    if( RemoveQueue(pd) == NULL ) return; /* Busy retransmitting? */
recycle_alien:
	    Copy(&(pd->packetType), kp, WORDS_PER_PACKET * sizeof(unsigned));
	    /*
	     * Now check for and eliminate any extra aliens of the same pid
	     * in case previous send was a group send that created
	     * multiple aliens.
	     */
	    aa = pd->nextPd;
	    while( aa != NULL )
	      {
		if( aa->pid == pd->pid )
		  {
		    DestroyAlien( aa );
		    aa = pd->nextPd;
		  }
		else aa = aa->nextPd;
	      }
	  }
      }
    else /* Get an alien process descriptor. */
      {
	if( Local(kp->srcPid) ) return; /* Discard if from a local process. */
	if( (pd = Alienate(kp)) == NULL ) return;
      }
    /* Remember the original receiver and forwarder pid in msgq.head and
     * msgq.tail - this hack will be cleaned up when official alien
     * process descriptor is defined.
     */
    pd->msgq.head = (Process *) kp->dstPid;
    pd->msgq.tail = (Process *) kp->forwarder;
     /* Locate the receiver. */
    if(!MAP_TO_RPD(receiver, kp->dstPid))
      { 
        if (Local(kp->dstPid) && 
	    (PidLocalPart(kp->dstPid) == LOCAL_KERNEL_PROCESS_PID))
	  {
	    kp->dstPid = KernelServerPid;
	    MAP_TO_RPD(receiver, kp->dstPid);
	  }
	else
	  {
	    if( kp->dstPid & GROUP_ID_BIT )
	      {
		/* Hacks for CopyFromSegment */
		pd->remoteSegmentPtr = (Unspec *) kp;
		pd->segmentSize = kp->length;
		if( SendGroupMembers(pd) == 0 )
		  {/* no local group members. */
		    FreeAlien( pd );
		  }
	      }
	    else if( !Local(kp->dstPid) || (PidLhn(kp->dstPid) == 0) )
	      {  /* Wrong host- valid broadcast or sending to logical 
		    host 0. */
		FreeAlien( pd );
	      }
	    else
	      {
		/* Handle nAcks to sender since not a valid local 
		   process  */
		pd->packetType = nAck;
		pd->msg.sysCode = NONEXISTENT_PROCESS;
		/* The alien will be destroyed by timeout queue mechanism.
		 * after packet is written.
		 */
		WriteKPacket( pd );
	      }
	    return;
	  }
      }

    pd->state = AWAITING_REPLY;
    pd->remoteSegmentPtr = (Unspec *) kp; /* A Hack for CopyFromSegment */
    kp->remoteaddress = receiver->dataSegmentPtr;
    pd->segmentSize = kp->length;
    DeliverMsg( receiver, pd );
  }


Process_id pRemoteReply( kp )
    register kPacket *kp;
  {
    register MsgStruct *msg;
    register Process *sender, *replierpd, *prev;
    register unsigned *ptr1, *ptr2, i;

    /*
     * Note: we validate the incoming packet against the oldSeqNo field,
     * because that field is guaranteed to contain the original sequence
     * number of the sender's current message transaction.  (The seqNo
     * field may have been overwritten during a CopyFrom operation.)
     */
    if( !MAP_TO_RPD(sender, kp->dstPid) ) return;

    if ((sender->pdFlags & SUSPEND_FLAG) &&
	(sender->blocked_on != kp->srcPid))
      {
	/* We must be trying to reply to the suspend PD. */
	sender = FindSuspendPd(kp->dstPid);
      }
    if (sender->pdFlags & FROZEN)
      {
	/* Trying to reply to a suspended process.  Reset the sender's
	   time and drop the reply.  We know that as long as the sender
	   keeps retransmitting we will continue to receive replies. 
	   This is true even for multicast, right??? */
	if( RemoveDelayQueue(sender) == NULL ) return;
	if( sender->oldSeqNo != kp->sequenceNo ) return;
	sender->numTrans = 0;
	sender->timeout_count = SEND_ACKED_RETRANS_PERIOD;
	DelayProcess( sender );
	return;
      }

    if( RemoveDelayQueue(sender) == NULL )
      {
queue_reply:
#ifdef DEBUG
	printx("Queuing reply for %x from %x\n", sender->pid, kp->srcPid );
#endif
	Lockq( &(sender->replyq) );
	if( sender->oldSeqNo != kp->sequenceNo ) goto pRRqueueDone;

	prev = (Process *) sender->replyq.head;
	while ( prev != NULL )	/* Duplicate suppression */
	  {
	    if ( prev->pid == kp->srcPid ) goto pRRqueueDone;
	    prev = prev->link;
	  }

	if( (replierpd = AllocatePd()) == NULL ) goto pRRqueueDone;
	ptr1 = (unsigned *) &(replierpd->packetType);
	ptr2 = (unsigned *) kp;
	i = WORDS_PER_PACKET - 1;
	do { *ptr1++ = *ptr2++; } while( --i != -1 );

	if ( (replierpd->link = sender->replyq.head) == NULL )
	    sender->replyq.tail = replierpd;
	sender->replyq.head = replierpd;
	replierpd->state = REPLIED_TO;
	replierpd->queuePtr = &(sender->replyq);

pRRqueueDone:
	Unlockq( &(sender->replyq) );
	return;
      }
    if( sender->oldSeqNo != kp->sequenceNo )
      {
	DelayProcess( sender );
	return;
      }
    if( sender->state != AWAITING_REPLY ) /* Not the first reply */
      {
	DelayProcess( sender ); /* Return to the delay queue. */
	goto queue_reply;
      }
    Copy_msg( &sender->msg, &kp->msg );
    if( kp->length != 0 )
      {
	register unsigned segend;

	segend = (unsigned)sender->dataSegmentPtr + sender->dataSegmentSize;

	if( ((sender->dataSegmentPro & WRITE_ACCESS) == 0 ) ||
	    ((unsigned)kp->remoteaddress < (unsigned)sender->dataSegmentPtr)||
	    ((unsigned)kp->remoteaddress > segend) ||
	    ((unsigned)kp->remoteaddress+kp->length > segend) )
	  {
	    msg = &sender->msg;
	    msg->sysCode = BAD_REPLY_SEGMENT; /* Indicate problem to sender.*/
	  }
	else
	    ReadDataPacket( sender, kp );
      }
    ReturnPid( sender, kp->srcPid );
    Addready( sender );
  }


Process_id pRemoteReceiveSpecific( kp )
    register kPacket *kp;
  {
    register Process *alien, *pd;
    register ProcessId pid;

    /* Check if a real process or a group with local members. */
    if( MAP_TO_RPD(pd, kp->dstPid) ||
	(IsGroupId(kp->dstPid) && ((pid=GetGroupMember(kp->dstPid,0)) != 0) &&
	MAP_TO_RPD(pd, pid) ) )
	  	kp->packetType = breathOfLife;
    else
      {
	if( !Local(kp->dstPid) && !LocalGroup(kp->dstPid) )
	    return; /* Dont Nack if a remote process. */

	kp->msg.sysCode = NONEXISTENT_PROCESS;
	kp->packetType = nAck;
      }
    if( (alien = Alienate(kp)) == NULL ) return;

    WriteKPacket( alien );
  }


Process_id pBreathOfLife( kp )
    register kPacket *kp;
   /* Handle a "breathOfLife" packet - the dstPid process should either be
    * AWAITING_REPLY from this process or else RECEIVE_BLKED specifically
    * from this process.
    */
  {
    register Process *sender;

    if( (sender = MapPid(kp->dstPid)) == NULL ) return;

    if( RemoveDelayQueue(sender) == NULL ) 
      {
	return;
      }

    if( ((sender->state==AWAITING_REPLY) || (sender->state==RECEIVE_BLKED)) &&
	 (sender->oldSeqNo == kp->sequenceNo) )
      {
	sender->numTrans = 0;
	sender->pdFlags |= SEND_ACKED;
	sender->timeout_count = SEND_ACKED_RETRANS_PERIOD;
      }
    if (((sender->state == MOVEFROM_BLKED) || (sender->state == MOVETO_BLKED))
        && (sender->seqNo == kp->sequenceNo))
      {
        sender->numTrans = 1;
	sender->timeout_count = COPY_RETRANS_PERIOD;
      }
    DelayProcess( sender );
  }


Process_id pRemoteForward( kp )
    register kPacket *kp;
  {
    register Process *sender;
    register ProcessId receiverPid, forwarderPid;
    MsgStruct *msg;
    /* kp->srcPid is the forwarder pid, kp->dstPid is the sender pid
     * and kp->forwarder is the receiver pid. */

    if( !MAP_TO_RPD(sender, kp->dstPid) ) return;
    if( !RemoveDelayQueue(sender) ) return;

    if( (sender->oldSeqNo != kp->sequenceNo) ||
		(sender->state != AWAITING_REPLY) )
      {
	DelayProcess( sender ); /* Add back to delay queue. */
        return;
      }

    /* Handle a slight modification in protocol. */
    /* Old version: kp->srcPid is receiver pid, kp->dstPid is sender pid, and
     *  kp->forwarder is forwarder pid.
     * New version: kp->srcPid is forwarder pid, kp->dstPid is sender pid, and
     *  kp->forwarder is receiver pid.
     */
    if( sender->blocked_on == kp->srcPid )
      {
	/* New version */
	forwarderPid = kp->srcPid;
	receiverPid = kp->forwarder;
      }
    else
      {
	/* Old version */
	forwarderPid = kp->forwarder;
	receiverPid = kp->srcPid;
      }
    Copy_msg( &sender->msg, &kp->msg );
    /* We are finished with the kernel packet. */

   /* Check the segment access indicated by the forwarded message.
    * This is required to fit within the segment access specified
    * by the original sender, both in terms of address range and
    * Read and write access.
    */
    msg = &sender->msg;
    if(  msg->sysCode & SEGMENT_PRESENT )
      {
	register unsigned diff;
	register unsigned char segProtect;

	segProtect = (unsigned char)((msg->sysCode&SEGMENT_PRESENT)>>8);
	diff = ((unsigned)msg->segmentPtr) - (unsigned)sender->dataSegmentPtr;

	if( ((segProtect|sender->dataSegmentPro) != sender->dataSegmentPro) ||
	   ((unsigned) msg->segmentPtr < (unsigned) sender->dataSegmentPtr) ||
	   ((unsigned) msg->segmentPtr >
		(unsigned)sender->dataSegmentPtr + sender->dataSegmentSize) ||
		(msg->segmentSize > sender->dataSegmentSize - diff ) )
          {
	    msg->sysCode = BAD_FORWARD;
	    Addready( sender );
	    return;
	  }
      }
    /* Set up sender to send to process to which it was forwarded. */
    sender->blocked_on = receiverPid;
    sender->forwarder = forwarderPid;
    sender->seqNo = sender->oldSeqNo;
    KSend( sender );
  }


Process_id pRemoteCopyFromReq( kp )
    register kPacket *kp;
  {
    SystemCode r;
    register unsigned segend;
    register Process *pd, *alien;

    if( !MAP_TO_RPD(pd, kp->dstPid) )
      {
	if( !Local( kp->dstPid ) ) return;
	r = NONEXISTENT_PROCESS;
        goto send_nack;
      }

    if ((pd->pdFlags & SUSPEND_FLAG) &&
        (pd->oldSeqNo != kp->msg.unspecified[0]))
      {
	pd = FindSuspendPd(kp->dstPid);
      }
    if (pd->pdFlags & FROZEN)
      {
	/* Trying to copy from a suspended process.  
	   Defer by discarding and sending back a breath-of-life. */
	if ((alien = Alienate(kp)) == NULL) return;
	alien->packetType = breathOfLife;
	alien->segmentSize= 0;
	WriteKPacket(alien);
	return;
      }

    /* Ignore if we cant extract from delay queue. */
    if( RemoveDelayQueue(pd) == NULL ) return;

    /* Check that (a) the sequence numbers match
     *	     and  (b)   if the send was a unicast send that we are
     *			awaiting reply from the requestor.
     * Note: Does not check group membership of requestor.
     */
    if ( (pd->oldSeqNo != kp->msg.unspecified[0]) ||
	 ( !IsGroupId(pd->blocked_on) && 
           ( (pd->state != AWAITING_REPLY) ||
	     (pd->blocked_on != kp->srcPid))))
      {
	/* Assume process timed out */
	r = NOT_AWAITINGREPLY;
	goto send_nack;
      }

    segend = (unsigned) kp->remoteaddress + kp->length;
    if ( ( (pd->dataSegmentPro & READ_ACCESS) == 0 ) ||
	 ( kp->remoteaddress < pd->dataSegmentPtr ) ||
         ( segend > (unsigned) pd->dataSegmentPtr + pd->dataSegmentSize ) ||
	 ( segend < (unsigned) kp->remoteaddress ) )
      {
	r = BAD_REPLY_SEGMENT;
send_nack:
	if( pd ) DelayProcess( pd );
	if( (alien = Alienate(kp)) == NULL ) return;
	alien->msg.sysCode = r;
	alien->packetType = nAck;
	WriteKPacket( alien );
	return;
      }
    
    /* A valid CopyFrom request constitutes a send acknowledgement */
    pd->pdFlags |= SEND_ACKED;
    pd->timeout_count = SEND_ACKED_RETRANS_PERIOD;
    pd->numTrans = 1;
    
    /* Use an alien when replying to a group member. We must not send the
     * reply to the group and must not change the sender's blocked_on field.
     */
    if (IsGroupId(pd->blocked_on))
      {
	DelayProcess( pd );
	if ( (alien = Alienate( kp )) == NULL) return;
	alien->team = pd->team;
	pd = alien;
      }

    /* kp->remoteaddress and kp->length specify requested data. */
    pd->segmentPtr = kp->remoteaddress;
    pd->segmentSize = kp->length;
    pd->localaddress = (Unspec *) kp->length; /* Necessary ??? */
    pd->remoteSegmentPtr = kp->localaddress;
    pd->packetType = remoteMoveFromReply;
    pd->seqNo = kp->sequenceNo;

    WriteKPacket( pd );
  }


Process_id pRemoteCopyFromRep( kp )
    register kPacket *kp;
    /* kp->remoteaddress is the local address to deliver data */
  {
    register unsigned segend;
    register Process *pd;

    /* Locate intended recipient */
    if( !MAP_TO_RPD(pd, kp->dstPid) ) return; /* Process died. */

    /* And grab ownership of PD - should be in the delay queue. */
    if( !RemoveDelayQueue(pd) ) return; /* Might also check IKP_OUT queue. */

    /* Check that pd is expecting this data from this process. */
    if( (pd->state != MOVEFROM_BLKED) || (pd->blocked_on != kp->srcPid) )
      {
	DelayProcess( pd );
	return; /* Assume that pd timed out the send. */
      }
    segend = (unsigned) pd->dataSegmentPtr + pd->dataSegmentSize;

    /* Check that it is destined for the address we are expecting. */
    if( pd->segmentPtr != kp->remoteaddress )
      { /* Presumably, a packet was dropped. */
	/* We ignore for now. */
	if( (kp->remoteaddress < pd->dataSegmentPtr) ||
	    ((unsigned)kp->remoteaddress > segend ) ||
	    ((unsigned)kp->remoteaddress + kp->length > segend) )
	  {
	    printx("Bad CopyFromReply from %x\n", kp->srcPid );
	  }
	DelayProcess( pd );
	return;
      }
    /* Check that the data fits within the segment. */
    pd->segmentPtr = (Unspec *)((unsigned)pd->segmentPtr + kp->length);
    if( (unsigned) pd->segmentPtr > segend )
      {
	printx("Bad CopyFromReply size from %x\n", kp->srcPid );
	pd->segmentPtr = pd->dataSegmentPtr;
	DelayProcess( pd );
	return;
      }
    /* Copy the data into pd's address space. */
    ReadDataPacket( pd, kp );

    /* Now check if this is the last packet. */
    if( (unsigned)pd->segmentPtr == segend )
      {
	pd->msg.sysCode = OK;
	Addready( pd );
      }
    else
      {
	pd->timeout_count = COPY_RETRANS_PERIOD;
	DelayProcess( pd );
      }
  }


Process_id pRemoteCopyToReq( kp )
    register kPacket *kp;
    /* kp->remoteaddress is a local address,
       kp->localaddress is the total length */
  {
    register unsigned segend;
    register Process *pd, *alien;
    register SystemCode r;

    /* Check that the process exists. */
    if( !MAP_TO_RPD(pd, kp->dstPid) ) 
      {
	register Process *alien;

	if( !Local(kp->dstPid) ) return; /* Wrong host. */
	r = NONEXISTENT_PROCESS;
send_nack:
	if( pd ) DelayProcess( pd );
	kp->msg.sysCode = r;
	kp->packetType = nAck;
        if( (alien = Alienate(kp)) != NULL ) WriteKPacket( alien );
        return;
      }

    if ((pd->pdFlags & SUSPEND_FLAG) &&
        (pd->blocked_on != kp->srcPid))
      {
	pd = FindSuspendPd(kp->dstPid);
      }
    if (pd->pdFlags & FROZEN)
      {
	/* Trying to copy from a suspended process.
	   Defer by discarding and sending back a breath-of-life. */
	if ((alien = Alienate(kp)) == NULL) return;
	alien->packetType = breathOfLife;
	WriteKPacket(alien);
	return;
      }

    /* Obtain ownership of PD by dequeuing. */
    if( !RemoveDelayQueue(pd) ) return; /* Might be retransmitting. */

    /* Check (a) that the sequence numbers match
     *	 and (b) if the send was a unicast send that we are
     *		 awaiting reply from the requestor.
     * Note: Does not check group membership of requestor.
     */
    if ( (pd->oldSeqNo != kp->msg.unspecified[0]) ||
	 ( !IsGroupId(pd->blocked_on) && 
           ( (pd->state != AWAITING_REPLY) ||
	     (pd->blocked_on != kp->srcPid))))
      {
	/* Assume process timed out */
	r = NOT_AWAITINGREPLY;
	goto send_nack;
      }

    /* Check that we are expecting this data next, assuming we
     * are in the middle of a CopyTo operation. */
    /* Use pd->remoteSegmentPtr for data being received. */
    if( (pd->remoteSegmentPtr == kp->remoteaddress) &&
	(pd->pdFlags & MULTI_PACKET_REPLY) && /* Not new */
	(pd->seqNo == kp->sequenceNo) )
      {
	/* In sequence portion of current CopyTo operation. */
        pd->remoteSegmentPtr =
		(Unspec *)((unsigned)kp->remoteaddress+kp->length);
	if( (unsigned) pd->remoteSegmentPtr >
		(unsigned) (pd->dataSegmentPtr) + pd->dataSegmentSize )
	  {
	    r = BAD_ADDRESS;
	    goto send_nack;
	  }
        ReadDataPacket( pd, kp );
      }
    else
      { /* Not the data expected, or else a new CopyTo group. */

	if( !(pd->pdFlags&MULTI_PACKET_REPLY) ||
	    pd->seqNo != kp->sequenceNo ||
	    pd->segmentPtr > kp->remoteaddress )
	  {

	    segend = (unsigned)pd->dataSegmentPtr + pd->dataSegmentSize;

	    /* Assume a new CopyTo group - check if OK. */
	    /* Note that we must NOT check remoteaddress + total length
	     *  against segend here, because there is the possibility that
	     *  this is not really the first packet of the group (we
	     *  could have missed the first packet), and we don't want to
	     *  send a NAck in that case.  --TPM
	     */
	    if( ((pd->dataSegmentPro & WRITE_ACCESS) == 0) ||
	         (kp->remoteaddress < pd->dataSegmentPtr) ||
	         ((unsigned)kp->remoteaddress > segend) ||
	         ((unsigned)kp->remoteaddress+(unsigned)kp->length > segend) )
	      {
#ifdef DEBUG_COPYTO
                if( !(pd->pdFlags&MULTI_PACKET_REPLY) )
                  printx("New CopyTo group\n");
                if( pd->seqNo != kp->sequenceNo )
                  {
                    printx("Sequence numbers don't match\n");
                    printx("pd->seqNo = %d, kp->sequenceNo = %d\n",
                            pd->seqNo, kp->sequenceNo );
                  }
                if( pd->segmentPtr > kp->remoteaddress )
                  {
                    printx("Segment ptr > remote address\n");
                    printx("segment ptr = %d, remoteaddress = %d\n",
                            pd->segmentPtr, kp->remoteaddress );
                  }
                if( kp->remoteaddress < pd->dataSegmentPtr )
                  {
                    printx("kp->remoteaddress < pd->dataSegmentPtr\n");
                    printx("remoteaddress = %d, dataSegmentPtr = %d\n",
                           kp->remoteaddress, pd->dataSegmentPtr );
                  }
                if( (unsigned)kp->remoteaddress > segend )
                  {
                    printx("kp->remoteaddress > segend\n");
                    printx("remoteaddress = %d, segend = %d\n",
                           kp->remoteaddress, segend );
                  }
	   if( (unsigned)kp->remoteaddress+(unsigned)kp->length > segend) )
                  {
                    printx("kp->remoteaddress+length > segend\n");
                  printx("remoteaddress = %d, length = %d, segend = %d\n",
                           kp->remoteaddress, kp->length, segend );
                  }
#endif DEBUG_COPYTO
		printx("Bad RemoteCopyToRequest from %x: ", kp->srcPid );
		if( (pd->dataSegmentPro & WRITE_ACCESS) == 0 )
		    printx( "No write access\n" );
		else
		    printx("Bad segment specification\n" );
		r = BAD_ADDRESS;
		goto send_nack;
	      }
	    pd->seqNo = kp->sequenceNo;
	    pd->segmentSize = (unsigned) kp->localaddress;
	    pd->segmentPtr = kp->remoteaddress;
	    pd->remoteSegmentPtr =
			(Unspec *)((char *)kp->remoteaddress+kp->length);
	    pd->pdFlags |= MULTI_PACKET_REPLY;
	    ReadDataPacket( pd, kp );
	  }
	else
	  {
	    /* An out of order packet from the current group */
	    /* Cause the process to retransmit? */
	    /* Right now - we do nothing. */	    
	    /* Turn off this flag since we know process must retransmit. */
	    pd->pdFlags &= ~MULTI_PACKET_REPLY;
	    goto addDelayq;
	  }
      }
    /* Check to see if CopyTo operation is complete. */
    if( (unsigned)pd->remoteSegmentPtr == 
	   (unsigned)pd->segmentPtr + pd->segmentSize)
      {
	register Process *alien;

	pd->pdFlags &= ~MULTI_PACKET_REPLY;
	if( (alien = Alienate(kp)) == NULL ) goto addDelayq;
	alien->packetType = remoteMoveToReply;
	alien->length = pd->segmentSize;
	alien->localaddress = pd->segmentPtr;
	alien->segmentSize = 0; /* Make sure its zero. */
	WriteKPacket( alien );
     }
addDelayq:
    /* Now setup to continue timing out awaiting the reply. */
    pd->pdFlags |= SEND_ACKED;
    pd->timeout_count = SEND_ACKED_RETRANS_PERIOD;
    pd->numTrans = 1;
    DelayProcess( pd );
  }


Process_id pRemoteCopyToRep( kp )
    register kPacket *kp;
  {
    register Process *pd;

    /* kp->remoteaddress is a local address */

    if( !MAP_TO_RPD(pd, kp->dstPid) ) return;

    if( !RemoveDelayQueue(pd) ) return;

    if( (pd->state != MOVETO_BLKED) ||
	(pd->seqNo != kp->sequenceNo) ||
        (pd->blocked_on != kp->srcPid) ) 
      {
	DelayProcess( pd );
#ifdef DEBUG
	printx("Bad RemoteCopyToReply received from %x\n", kp->srcPid );
#endif
	return;
      }
    if( (pd->remoteSegmentPtr != kp->localaddress) ||
			        (pd->dataSegmentSize != kp->length) )
      {
	DelayProcess( pd );
	printx("Bad RemoteCopyToReply from %x\n", kp->srcPid );
	printx("Segment ptr and size do not match\n" );
	return;
      }
    pd->msg.sysCode = OK;
    Addready( pd );
  }


Process_id pNAck( kp )
    register kPacket *kp;
  {
    register Process *pd;

    if( !MAP_TO_RPD(pd, kp->dstPid) ) return; /* Nonexistent process. */

    if( !RemoveDelayQueue(pd) ) return; /* Not waiting for nack, so ignore. */

    if( (pd->blocked_on == kp->srcPid) && (pd->seqNo == kp->sequenceNo) &&
	(pd->state != DELAYING) )
      {
	pd->msg.sysCode = kp->msg.sysCode;
	ReturnPid( pd, 0 );
	Addready( pd );
      }
    else DelayProcess( pd );
  }


ProcessId pLhnCollisionCheck( kp )
    register kPacket *kp;
  {
    printx("Received a LhnCollisionCheck packet!\n");
  }

ProcessId pLhnCollisionReply( kp )
  {
    printx("Received a LhnCollisionReply packet!\n");
  }


Process_id pRemoteGetPid( kp )
    register kPacket *kp;
  {
    printx("Received a RemoteGetPid packet!\n");
  }


Process_id pRemoteGetPidReply( kp )
    register kPacket *kp;
  {
    printx("Received a RemoteGetPidReply packet!\n");
  }
