/*
 * V Kernel - Copyright (c) 1984 by Stanford University.
 *
 * Kernel logical-physical address map cache package.
 */


#include "Vethernet.h"
#include "process.h"


#define SHORT_HOST_HASH_SHIFT 8	/* HOST_HASH_SHIFT for shorts. */

#define HOST_HASH_MASK	 63	/* Size of hash table, minus one. */
#define HOST_CACHE_SCAVENGE_NUMBER 16
				/* Number of host cache records to scavenge
				   when none are free for a new entry. */

extern NetAddress MulticastDefault;

/* maps between hardware addresses and logical  host numbers. */	
typedef struct HostCacheEntryType
  {
    struct HostCacheEntryType *link;
    NetAddress netAddress;
    unsigned short logicalHost;
  } HostCacheEntry;

HostCacheEntry HostCacheRecords[HOST_CACHE_SIZE];
HostCacheEntry *HostCacheFree;
HostCacheEntry *HostCache[HOST_HASH_MASK+1];
unsigned long HostCacheFreeIndex;


/*
 * Address mapping cache management routines
 */

HostCacheInit()
  {
    register int i;

    for (i = 0; i <= HOST_HASH_MASK; i++)
      {
	HostCache[i] = NULL;
      }

    HostCacheFlush();
  }


HostCacheFlush()
  {
    register HostCacheEntry *e, *stop;    

    HostCacheFreeIndex = 0;
    e = HostCacheFree = &HostCacheRecords[0];
    stop = &HostCacheRecords[HOST_CACHE_SIZE];
    while (e < stop) 
      {
        e->link = e+1;
	e++;
      }
    (e-1)->link = NULL;
  }


int HostCacheScavenge()
  {
  /* Starting at the index-th hash bucket, scavenge the last entry in each
   * bucket until HOST_CACHE_SCAVENGE_NUMBER free records have been obtained,
   * or we've done a complete loop around the buckets (in which case we didn't
   * need to scavenge)
   * Returns the number of free records obtained.
   */
    register int i, index;
    register HostCacheEntry *e, *p;    

    index = HostCacheFreeIndex;
    i = 0;
    do
      {
	p = (HostCacheEntry *) &HostCache[index];
	e = p->link;
	if (e != NULL)
	  {
	    while (e->link != NULL)
	      {
		p = e;
		e = e->link;
	      }
	    p->link = NULL;
	    e->link = HostCacheFree;
	    HostCacheFree = e;
	    i++;
	  }
	if (index == HOST_HASH_MASK)
	    index = 0;
	else
	    index++;
      }
    while (i < HOST_CACHE_SCAVENGE_NUMBER && index != HostCacheFreeIndex);
    HostCacheFreeIndex = index;
    return i;
  }


HostCacheEnter(logicalhost, netaddress)
    unsigned short logicalhost;
    NetAddress netaddress;
  {
    register HostCacheEntry *entry;
    register int index;

    if (logicalhost == V_BOOT_LHN) return;
    index = (logicalhost>>SHORT_HOST_HASH_SHIFT) & HOST_HASH_MASK;
    entry = HostCache[index];
    while (entry != NULL)
      {
	if (entry->logicalHost == logicalhost) 
	  {
	    entry->netAddress = netaddress;
				/* Migration may have changed the value. */
	    return;
	  }
	entry = entry->link;
      }

    if (HostCacheFree == NULL)
	if (HostCacheScavenge() == 0) /* Free up some records */
	    Kabort("HostCacheEnter: scavenging failed");

    HostCacheFree = (entry = HostCacheFree)->link;
    entry->link = HostCache[index];
    HostCache[index] = entry;
    entry->logicalHost = logicalhost;
    entry->netAddress = netaddress;
  }


NetAddress HostCacheLookup(logicalhost)
    register unsigned short logicalhost;
    /* Return the net address of a logical host if it is in our cache */
  {
  /* Search the logical host hash table for the given logical host. If there
   * is an entry move it to the front of its bucket chain to keep it from
   * being scavenged.
   */
    register HostCacheEntry *e, *p, *head;

    if ( logicalhost == 0 ) return( MulticastDefault );

    e = head = (HostCacheEntry *) &HostCache[
    		       (logicalhost>>SHORT_HOST_HASH_SHIFT) & HOST_HASH_MASK];
    while ( e->link != NULL )
      {
	p = e;
	e = e->link;
	if ( e->logicalHost == logicalhost )
	  {
	    /* Don't move it to the front if it's already there. */
	    if ( p != head )
	      {
		p->link = e->link;
		e->link = head->link;
		head->link = e;
	      }
	    return( e->netAddress );
	  }
      }
    return( MulticastDefault );
  }

char HostCacheStringBuffer[20];
char *HostCacheString( pid )
ProcessId pid;
  {
return("This space intentionally left blank");
#ifdef undef
    NetAddress a;
    a = HostCacheLookup( pid>>16 );
    
#ifdef ENET10MEG
sprintf(HostCacheStringBuffer, "%x.%x.%x", a.addrhigh, a.addrmid, a.addrlow);
#else
sprintf(HostCacheStringBuffer, "0%o", a);
#endif
    return(HostCacheStringBuffer);
#endif
  }


HostCacheDelete( logicalhost )
    register unsigned short logicalhost;
  {
    register HostCacheEntry *e, *p;

    e = (HostCacheEntry *) &HostCache[(logicalhost>>SHORT_HOST_HASH_SHIFT) & 
							     HOST_HASH_MASK];
    while (e->link != NULL)
      {
	p = e;
	e = e->link;
	if( e->logicalHost == logicalhost )
	  {
	    p->link = e->link;
	    e->link = HostCacheFree;
	    HostCacheFree = e;
	    return;
	  }
      }
  }

