/*
 * V Kernel - Copyright (c) 1984 by Stanford University
 *
 * Kernel forced exception handling routines.
 */

#include "Vexceptions.h"
#include "process.h"
#include "naming.h"

extern SyncQueue Readyq;
extern Process Idle_process;
extern ProcessId KernelServerPid;
extern Logical_id_entry Logical_id_map[];
extern int Addready();
extern SyncQueue *RemoveQueue();
extern ExtendPd FindExtendPd();
extern Process *AllocatePd();
extern ProcessId KReceiveSpecific();

int ExecuteFinishupFcn();
ProcessId ReReceive();
ProcessId ReDelay();
ProcessId ReMoveTo();
ProcessId ReMoveFrom();
ProcessId SuspendFinishUp();
ProcessId SuspendFinishUpNoRetVal();

Process *CreateSuspendProcess();
ProcessId CleanUpSuspend();
SystemCode ForceSend();


/*
 * ForceException:
 * Causes the process designated by pid to send an exception message to the
 * exception handler if there is one.  The AWAITING_REPLY and AWAITING_INT 
 * states are handled
 * slightly differently:  A separate suspend PD is created that is used to
 * send an exception message to the exception handler.
 */

SystemCode ForceException( active )
    register Process *active;
  {
    KernelRequest *req;
    ProcessId pid;
    Process *pd, *ehPd;
    ProcessId ehPid;
    static MsgStruct exceptionMsg;

    req = (KernelRequest *) &(active->msg);
    pid = req->pid;
    if (!MAP_TO_RPD(pd, pid))
      {
	return(NONEXISTENT_PROCESS);
      }
/*    if( ((active->userNumber != pd->userNumber) &&
         (active->userNumber != 0)) || 
	(pid == KernelServerPid) )
      {
	return(NO_PERMISSION);
      }*/
    if (pd->pdFlags & SUSPEND_FLAG)
      {
	return(BUSY);
      }
    ehPid = Logical_id_map[EXCEPTION_SERVER].pid;
    if( !MAP_TO_RPD(ehPd, ehPid) )
      {
	return(NO_SERVER_RESOURCES);
      }
    SetExceptionMsg(&exceptionMsg, pd);
    req->unspecified[0] = ehPid;
    req->unspecified[1] = (unsigned) &exceptionMsg;
    return(ForceSend(active, 0));
  }


/*
 * ForceSend:
 * Causes the process designated by pid to send a message to another
 * specified process.  The AWAITING_REPLY and AWAITING_INT 
 * states are handled
 * slightly differently:  A separate suspend PD is created that is used to
 * send the message.
 * The last argument tells whether the msg is in an appended segment.  If
 * it is not, then remote ForceSends are not supported.  This is for
 * compatibility.  The last argument is 0 when called by ForceException.
 */

SystemCode ForceSend( active, msgInSegment )
    register Process *active;
    int	msgInSegment;
  {
    KernelRequest *req;
    ProcessId fromPid, toPid;
    Process *pd, *pdS, *pdE;
    int status;
    unsigned msgToSend;

    req = (KernelRequest *) &(active->msg);
    fromPid = req->pid;
    toPid = req->unspecified[0];
    msgToSend = msgInSegment ? (unsigned)req->segment : req->unspecified[1];
    if (!MAP_TO_RPD(pd, fromPid))
      {
	return(NONEXISTENT_PROCESS);
      }
/*    if( ((active->userNumber != pd->userNumber) &&
         (active->userNumber != 0)) || 
	(pid == KernelServerPid) )
      {
	return(NO_PERMISSION);
      }*/
    if (pd->pdFlags & SUSPEND_FLAG)
      {
	return(BUSY);
      }

    pd->pdFlags |= SUSPEND_FLAG;

redo:
    switch (pd->state)
      {
	case READY:
	    RemoveQueue(pd);
	    if (pd->finish_up)
	      {
		status = ExecuteFinishupFcn(pd, active);
		if (status == 0)
		  {
		    pd->pdFlags &= ~SUSPEND_FLAG;
		    return(NO_SERVER_RESOURCES);
		  }
		else if (status == 2)
		  {
		    goto redo;
		  }
		pd->finish_up = (Unspec (*)()) SuspendFinishUp;
	      }
	    else
	      {
		pd->finish_up = (Unspec (*)()) SuspendFinishUpNoRetVal;
	      }
	    DoSend(active, pd, toPid, msgToSend);
	    break;

	case RECEIVE_BLKED:
	    RemoveQueue(pd);
	    pdE = AllocatePd();
	    if (pdE == NULL)
	      {
		pd->pdFlags &= ~SUSPEND_FLAG;
		return(NO_SERVER_RESOURCES);
	      }
	    ((ExtendPd) pdE)->eType = FORCE_EXCEPTION_TYPE;
	    LinkExtendPd(pd, pdE);
	    pdE->pdFlags = pd->pdFlags;
	    pdE->blocked_on = pd->blocked_on;
	    pdE->returnMessage = pd->returnMessage;
	    pdE->dataSegmentPtr = pd->dataSegmentPtr;
	    pdE->dataSegmentSize = pd->dataSegmentSize;
	    pdE->recBufSizePtr = pd->recBufSizePtr;
	    pdE->finish_up = pd->finish_up;
	    pdE->numTrans = pd->numTrans;
	    pd->finish_up = (Unspec (*)()) ReReceive;
	    DoSend(active, pd, toPid, msgToSend);
	    break;

	case AWAITING_REPLY:
	case AWAITING_INT:
	    /* Note: we leave the PD on whatever queue it is on. */
	    pdS = CreateSuspendProcess(pd);
	    if (pdS == 0)
	      {
		pd->pdFlags &= ~SUSPEND_FLAG;
		return(NO_SERVER_RESOURCES);
	      }
	    else
	      {
		pd->priority |= STOPPED_TEAM_PRIORITY;
				/* Make it non-runnable in case it gets
				   replied to before the resumption. */
		pdS->finish_up = (Unspec (*)()) CleanUpSuspend;
		DoSend(active, pdS, toPid, msgToSend);
	      }
	    break;

	case DELAYING:
	    RemoveQueue(pd);
	    pdE = AllocatePd();
	    if (pdE == NULL)
	      {
		pd->pdFlags &= ~SUSPEND_FLAG;
		return(NO_SERVER_RESOURCES);
	      }
	    ((ExtendPd) pdE)->eType = FORCE_EXCEPTION_TYPE;
	    LinkExtendPd(pd, pdE);
	    pd->priority &= PROCESS_PRIORITY_MASK;
	    pdE->timeout_count = pd->timeout_count;
	    pd->finish_up = (Unspec (*)()) ReDelay;
	    DoSend(active, pd, toPid, msgToSend);
	    break;
	case MOVEFROM_BLKED:
	    RemoveQueue(pd);
	    pdE = AllocatePd();
	    if (pdE == NULL)
	      {
		pd->pdFlags &= ~SUSPEND_FLAG;
		return(NO_SERVER_RESOURCES);
	      }
	    ((ExtendPd) pdE)->eType = FORCE_EXCEPTION_TYPE;
	    LinkExtendPd(pd, pdE);
	    pdE->blocked_on = pd->blocked_on;
	    pdE->dataSegmentPtr = pd->dataSegmentPtr;
	    pdE->remoteSegmentPtr = pd->remoteSegmentPtr;
	    pdE->dataSegmentSize = pd->dataSegmentSize;
	    pd->priority &= PROCESS_PRIORITY_MASK;
	    pd->finish_up = (Unspec (*)()) ReMoveFrom;
	    DoSend(active, pd, toPid, msgToSend);
	    break;

	case MOVETO_BLKED:
	    RemoveQueue(pd);
	    pdE = AllocatePd();
	    if (pdE == NULL)
	      {
		pd->pdFlags &= ~SUSPEND_FLAG;
		return(NO_SERVER_RESOURCES);
	      }
	    ((ExtendPd) pdE)->eType = FORCE_EXCEPTION_TYPE;
	    LinkExtendPd(pd, pdE);
	    pdE->blocked_on = pd->blocked_on;
	    pdE->segmentPtr = pd->segmentPtr;
	    pdE->segmentSize = pd->segmentSize;
	    pdE->remoteSegmentPtr = pd->remoteSegmentPtr;
	    pd->priority &= PROCESS_PRIORITY_MASK;
	    pd->finish_up = (Unspec (*)()) ReMoveTo;
	    DoSend(active, pd, toPid, msgToSend);
	    break;

	default:
	    printx("ForceException called on a process in state %d\n",
	    		pd->state);
	    Kabort("Should never have gotten here!");
      }
    return(OK);
  }


/*
 * SetExceptionMsg:
 * Creates an exception message for the exception SUSPENDED.
 */

SetExceptionMsg(req, pd)
    register ExceptionRequest *req;
    Process *pd;
  {
    req->requestcode = EXCEPTION_REQUEST;
    req->type = SUSPENDED;
#ifdef MC68010
    req->status = pd->proc_state.frame.sr;
    req->errpc = (unsigned) pd->proc_state.frame.pc;
#else
#ifndef VAX
    req->status = pd->proc_state.sr;
    req->errpc = (unsigned) pd->proc_state.pc;
#else VAX
    req->status =(unsigned) pd->proc_state.PSL;
    req->errpc = (unsigned) pd->proc_state.PC;
#endif
#endif
    req->code = req->accaddr = req->instruction = 0;
    req->segment = (char *) TEAM_START; 
			    /* Include access to the team space */
    req->segmentsize = ((unsigned) (pd->team->team_space.size))
						    - TEAM_START;
    /* Set up data segment authorization. */
    pd->dataSegmentPro = (unsigned char) ((req->requestcode)>>8);
    pd->dataSegmentPtr = (Unspec *) req->segment;
    pd->dataSegmentSize = req->segmentsize;
    pd->remoteSegmentPtr = (Unspec *) MAXUNSIGNED;
  }


/*
 * DoSend:
 */

DoSend(active, pd, toPid, msgToSend)
    Process *active;
    Process *pd;
    ProcessId toPid;
    register Unspec *msgToSend;
  {
    Team *oldteam;

    /*
     * All of this crud is to make ForceException work remotely (when
     * msgToSend is actually a pointer in kernel space), even though
     * ForceSend will not work remotely.  We don't want to do a
     * SetAddressableTeam if active is an alien!
     */
    if (AlienProcess(active))
      {
	if (!BadUserPtr(msgToSend))
	  {
	    Kabort("Using an old Remote ForceSend, not implemented");
	  }
	Copy_msg(&pd->msg, msgToSend);
      }
    else
      {
	oldteam = GetAddressableTeam();
	SetAddressableTeam(active->team);
	Copy_msg(&pd->msg, msgToSend);
	SetAddressableTeam(oldteam);
      }
    pd->blocked_on = toPid;
    pd->forwarder = pd->pid;
    pd->returnMessage = NULL;
    KSend(pd);
  }


/* 
 * ExecuteFinishupFcn:
 * Execute the finish_up function specified by pd->finish_up.
 * Assumes that pd->finish_up is non-zero.
 * Returns a value indicating whether the operation failed (0), whether
 * the process is still READY after the finish_up fcn was executed (1), or
 * whether it is no longer READY (2).
 */

int ExecuteFinishupFcn( pd, active )
    register Process *pd, *active;
  {
    Team *oldteam;
    register Unspec (*finish_up)();
    int readyFlag;
    ExtendPd sp;
    unsigned retVal;

    readyFlag = 1;
    oldteam = GetAddressableTeam();
    SetAddressableTeam(pd->team);
    finish_up = pd->finish_up;
/*??? Do we really need this disable? */
    disable;		        /* Ensure running at kernel disable level */
    pd->finish_up = NULL;	/* This has to be in front of the call to
			           finish_up because finish_up may set it to
			           a new value. */
    retVal = (unsigned) (*(finish_up))(pd);
/*??? Do we really need this disable? */
    disable;
    if( pd->state == READY )
      {
	RemoveQueue(pd);
	sp = (ExtendPd) AllocatePd();
	if (sp == NULL)
	  {
	    return(0);
	  }
	sp->eType = FORCE_EXCEPTION_TYPE;
	LinkExtendPd(pd, sp);
	sp->v0 = retVal;
      }
    else
      {
	readyFlag = 2;
      }
    SetAddressableTeam(oldteam);
    return(readyFlag);
  }


/*
 * ReReceive: restore parameters and resume the Receive.
 * This will be simpler when there is a KReceive function - DRC.
 */

ProcessId ReReceive( active )
    register Process *active;
  {
    register Process *pdE;
    ProcessId pid;

    pdE = (Process *) FindExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->pdFlags = pdE->pdFlags;
    active->blocked_on = pdE->blocked_on;
    active->returnMessage = pdE->returnMessage;
    active->finish_up = pdE->finish_up;
    active->numTrans = pdE->numTrans;
    active->dataSegmentSize = pdE->dataSegmentSize;
    active->recBufSizePtr = pdE->recBufSizePtr;
    active->dataSegmentPtr = pdE->dataSegmentPtr;
    pid = KReceiveSpecific( active );
    DeleteExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->pdFlags &= ~SUSPEND_FLAG;
    return(pid);
  }


/*
 * ReDelay: restore parameters and resume the Delay.
 */
ProcessId ReDelay( active )
    register Process *active;
  {
    register Process *pdE;

    active->priority |= active->team->team_priority;
    pdE = (Process *) FindExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->timeout_count = pdE->timeout_count;
    active->timeout_func = (Unspec (*)()) Addready;
    active->state = DELAYING;
    DeleteExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->pdFlags &= ~SUSPEND_FLAG;
    DelayProcess( active );
    return(0);
  }

/*
 * ReMoveTo:
 * Restore parameters and reexecute MoveTo.
 */

ProcessId ReMoveTo(active)
    register Process *active;
  {
    register Process *pdE;

    active->priority |= active->team->team_priority;
    pdE = (Process *) FindExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->blocked_on = pdE->blocked_on;
    active->segmentPtr = pdE->segmentPtr;
    active->segmentSize = pdE->segmentSize;
    active->remoteSegmentPtr = pdE->remoteSegmentPtr;
    DeleteExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->pdFlags &= ~SUSPEND_FLAG;
    NonLocalCopyTo(active);
    return(0);
  }


/*
 * ReMoveFrom:
 * Restore parameters and reexecute MoveFrom.
 */

ProcessId ReMoveFrom(active)
    register Process *active;
  {
    register Process *pdE;

    active->priority |= active->team->team_priority;
    pdE = (Process *) FindExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->blocked_on = pdE->blocked_on;
    active->dataSegmentPtr = pdE->dataSegmentPtr;
    active->remoteSegmentPtr = pdE->remoteSegmentPtr;
    active->dataSegmentSize = pdE->dataSegmentSize;
    DeleteExtendPd(active, FORCE_EXCEPTION_TYPE);
    active->pdFlags &= ~SUSPEND_FLAG;
    NonLocalCopyFrom(active);
    return(0);
  }


/*
 * SuspendFinishUp:
 * Finish-up fcn for the suspend operation when it must be invoked from 
 * another finish-up fcn.  Returns the value returned from
 * the original finish-up fcn.
 */

ProcessId SuspendFinishUp(active)
    register Process *active;
  {
    ExtendPd sp;

    sp = FindExtendPd(active, FORCE_EXCEPTION_TYPE);
    DeleteExtendPd(active, FORCE_EXCEPTION_TYPE);
    Addready(active);
    active->pdFlags &= ~SUSPEND_FLAG;
    return((ProcessId) sp->v0);
  }


/*
 * SuspendFinishUpNoRetVal:
 * Finish-up fcn for the suspend operation when no return value should be
 * provided.  Basically just resets the SUSPEND_FLAG.
 */

ProcessId SuspendFinishUpNoRetVal(active)
    register Process *active;
  {
    Addready(active);
    active->pdFlags &= ~SUSPEND_FLAG;
    return;
  }


/*
 * CreateSuspendProcess:
 * Create a clone of the specified process and hook it into the regular
 * process structure as a SUSPEND_PROCESS.
 */

Process *CreateSuspendProcess(pd)
    register Process *pd;
  {
    register Process *newPd;
    register int index;

    if( (newPd = AllocatePd()) == NULL ) 
      {
        return( NULL );
      }

    /* Initialize the PD. */
    Copy(newPd, pd, sizeof(Process));
    index = pd->pid & PID_HASH_MASK;
    disable;
    newPd->nextPd = Pd_bundle[index];
    Pd_bundle[index] = newPd;
    enable;
    newPd->localPid = SUSPEND_PROCESS;
    newPd->pdFlags &= ~SUSPEND_FLAG;
    newPd->next_sender = NULL;
    newPd->msgq.head = NULL;
    newPd->replyq.head = NULL;
    newPd->queuePtr = NULL;
    return( newPd );
  }


/*
 * FindSuspendPd:
 * Return a ptr to the suspend process PD for pid, or NULL if there is none.
 */

Process *FindSuspendPd(pid)
    ProcessId pid;
  {
    register Process *pd;

    pd = Pd_bundle[pid & PID_HASH_MASK];
    while (pd != IdleProcessPtr)
      {
        if ((pd->localPid == SUSPEND_PROCESS) && (pd->pid == pid))
	  {
	    return(pd);
	  }
	pd = pd->nextPd;
      }
    return(NULL);
  }


/*
 * CleanUpSuspend:
 * Finish-up fcn for the suspend process for a process which was 
 * awaiting reply.
 */

ProcessId CleanUpSuspend(active)
    register Process *active;
  {
    register Process *pd;

    MAP_TO_RPD(pd, active->pid);
    pd->pdFlags &= ~SUSPEND_FLAG;
    pd->priority = active->priority;
    if ( pd->queuePtr == &Readyq )
      {
	RemoveQueue(pd);
	Addready(pd);
      }
    FreePd(active, active->pid);
  }
