/*
 * Distributed V Kernel
 * Copyright (c) 1982 by Stanford University, All rights reserved.
 *
 * Kernel Xylogics disk driver, Paul Roy, David Cheriton
 */

#include "Vexceptions.h"
#include "Vioprotocol.h"
#include "Vstorage.h"
#include "Vquerykernel.h"
#include "interrupt.h"
#include "dm.h"
#include "process.h"
#include "machine.h"
#include "memory.h"
#include "xyl.h"
#include "disklabel.h"

/* Imports */
extern SystemCode     NotSupported(), NotWriteable();
extern unsigned char  *LastPeripheral;
extern DeviceInstance *DeviceInstanceTable[];
#ifdef SUN2
extern void AllocateDvmaSpace();
extern void MapDvma();
extern SystemCode AllocatePage();
#else
extern void AllocateMultibusMemory();
#endif

/* Exports */
extern SystemCode XylCreate();
extern SystemCode XylRead();
extern SystemCode XylWrite();
extern SystemCode XylQuery();
extern SystemCode XylModify();
extern SystemCode XylRelease();
extern SystemCode XylPowerup();

extern SystemCode XylCheckRequest(); 

char    *dkspace_va;   /* virtual address of disk's space in multibus memory */
char    *dkspace_pa;   /* physical multibus address of disk's space */
#ifndef SUN2
char    *dkbuffer_va;  /* virtual address of data buffer */
char    *dkbuffer_pa;  /* physical multibus address of data buffer */
#endif

Iopb    *ip;      /* virtual ptr to i/o parameter block */
Iopb    *ip_pa;   /* physical ptr to i/o parameter block */


SyncQueue  XylDiskq;      /* queue of process descriptors wanting disk access */

/* 
 * Indicates the current process accessing the disk controller.
 * Zero if no requestor.
 */
ProcessId XylDiskRequestor = 0;

 
/* The following space is used to read the label off of disk and 
 *   subsequently configure the driver.
 */
char            label_space[SECSIZE];
struct dk_label *label = (struct dk_label *) label_space;


/* Need some space to hold the parameters for the different drives. */
DiskParameters dk_parameters[MAX_NUMBER_DRIVES];


/* statistic global variables */
unsigned  stats_numio;      /* total number of i/o's */
unsigned  stats_totalcyl;   /* total number of cylinders seeked */
/* last cylinder location; must keep track of
  *   this for each of the units */                               
unsigned  stats_lastcyl[MAX_NUMBER_DRIVES];     
unsigned  stats_numreads;
unsigned  stats_numwrites;
unsigned  stats_averageread;
unsigned  stats_averagewrite;


unsigned short current_unit; /* the unit(drive) that currently has an operation
                              * being performed on it. 
                              */
unsigned short controller_exists; /* flag that indicates if contr. is present */
unsigned XylCreateFlag = 0; /* Non-zero while drive is being init during
			      * CreateInstance */

/* Field in pd in which disk commands are stored. */
#define diskCmd		((short) timeoutCount)

/* Macro expansion to interrupt-invoked C call to XylInterrupt */
Call_inthandler(XylInterrupt)


SystemCode XylCreate( pd, desc )
    Process *pd;
    register DeviceInstance *desc;

  /* Create an instance for a device attached to the Xylogics disk interface.
   */
 {
    register CreateInstanceRequest *req = (CreateInstanceRequest *) &(pd->msg);
    register IoRequest *ioreq = (IoRequest *) &(pd->msg);
    register unsigned  j;
    unsigned	garbage;

    if( !controller_exists )
      return( NOT_FOUND );    /* controller does not exist */
    
    /* Since it is not possible to return NO_REPLY from this routine, we
     *   cannot enqueue the following disk operations.  Thus, there can't
     *   be another operation in progress.
     * We busy wait on all the operations and set 'XylDiskRequestor' back
     *   to zero at the end of the function.
     */
    Lockq( &XylDiskq );
    if( XylDiskRequestor )
      {
#ifdef XYL_DEBUG
	printx( "XYL CreateInstance RETRY: operation in progress\n" );
#endif
        Unlockq( &XylDiskq );
        return( RETRY );
      }
    XylDiskRequestor = pd->pid;
    Unlockq( &XylDiskq );
    XylCreateFlag = 1; /* So interrupt routine does no more than ACK ints. */

    /* Initialize the device instance descriptor. */

    desc->readfunc = XylRead;
    desc->writefunc = XylWrite;
    desc->modifyfunc = XylModify;
    desc->queryfunc = XylQuery;
    desc->releasefunc = XylRelease;

    desc->owner = pd->pid;  

    desc->type = (READABLE+WRITEABLE+MULTI_BLOCK);
    if( req->filemode == FREAD )
      {
	desc->type &= ~READABLE;
	desc->writefunc = NotWriteable;
      }
    
    /* The RESET command should be the first thing attempted on a drive
     *   so that we can see if it's out there.
     */
    ioreq->blocknumber = 0;
    ioreq->bytecount = 0;
    ioreq->fileid = desc->id;              /* kludge */
    ioreq->requestcode = RESET;  /* clear the device */
    ioreq->bufferptr = label_space;
#ifdef XYL_DEBUG
    printx("Xyl: about to reset\n" );
#endif
    XylBuildandGo( pd );   /* not XylStart cuz don't want to enqueue */

    while ( CSR(devaddr) & GBSY )
      ;                   /* GBSY will be cleared when operation completes */

    if( ip->stat1 & ERROR_MASK )
      {
#ifdef XYL_DEBUG
        printx("Error: command = 0x%x, CSR(devaddr) = 0x%x\n",
                           ip->comm, CSR(devaddr));
        printx("Status 1 = 0x%x, Status 2 = 0x%x\n", ip->stat1, ip->stat2);
#endif XYL_DEBUG
	XylCreateFlag = 0;
	XylDiskRequestor = 0;
        if( ip->stat2 == DRIVE_NOT_READY )  /* drive probably doesn't exist */
	  {
	    printx( "Error: drive %d not ready\n", desc->unit );
	    return( NOT_FOUND );
	  }
        else
          return( DEVICE_ERROR );
      }

    /* Now read the disk label off of the drive.
     */
    for (j=0; j<SECSIZE; ++j)
      *((char *) label_space + j) = 0; /* first clear the label space */

    ioreq->bufferptr = label_space;
    ioreq->bytecount = SECSIZE;
    /* blocknumber gets mulitplied by BLOCK_FACTOR in XylBuild, hence 
     *   compensate with the following. */
    ioreq->blocknumber = DISK_LABEL_SECTOR / BLOCK_FACTOR;

    ioreq->requestcode = READ;  
#ifdef XYL_DEBUG
    printx("Xyl: about to read drive %d disk label\n", desc->unit );
#endif XYL_DEBUG
    XylBuildandGo( pd );  /* not XylStart cuz don't want to enqueue */

    while ( CSR(devaddr) & GBSY )
      ;                   /* GBSY will be cleared when operation completes */

    if( ip->stat1 & ERROR_MASK )
      {
#ifdef XYL_DEBUG
        printx("Error: command = 0x%x, CSR(devaddr) = 0x%x\n",
                           ip->comm, CSR(devaddr));
        printx("Status 1 = 0x%x, Status 2 = 0x%x\n", ip->stat1, ip->stat2);
#endif XYL_DEBUG
	XylCreateFlag = 0;
	XylDiskRequestor = 0;
	return( DEVICE_ERROR );
      }
    /* set the parameters for the drive */
#ifdef XYL_DEBUG
    printx("Xyl: set drive %d parameters\n", desc->unit );
    printx("Cylinders %d; Heads %d; Sectors per track %d\n",
		label->dkl_ncyl, label->dkl_nhead, label->dkl_nsect );
#endif XYL_DEBUG
    /* The following are random but reasonable constraints. */
    if( ((label->dkl_ncyl < 10) || (label->dkl_ncyl > 2000)) ||
	((label->dkl_nhead < 2) || (label->dkl_nhead > 20)) ||
	((label->dkl_nsect < 5) || (label->dkl_nsect > 1000)) )
      {
	printx( "Xyl: bad label on drive %d\n", desc->unit );
	XylCreateFlag = 0;
	XylDiskRequestor = 0;
	return( DEVICE_ERROR );
      }
    XylInitBuildandGo( SET_DRIVE, desc->unit );

    while ( CSR(devaddr) & GBSY )
      ;               /* GBSY will be cleared when operation completes */

#ifdef XYL_DEBUG
    printx("Xyl: drive %d parameters set\n", desc->unit );
#endif
    if( ip->stat1 & ERROR_MASK )
      {
#ifdef XYL_DEBUG
        printx("Error: command = 0x%x, CSR(devaddr) = 0x%x\n",
                           ip->comm, CSR(devaddr));
        printx("Status 1 = 0x%x, Status 2 = 0x%x\n", ip->stat1, ip->stat2);
#endif XYL_DEBUG
	XylCreateFlag = 0;
	XylDiskRequestor = 0;
	return( DEVICE_ERROR );
      }
     /* reset controller by reading control register - clears interrupt bit */
    garbage = CRUR(devaddr);

    /* now set the attributes of the device that are dependent on the info
     *   obtained from the disk label.
     */

    /* adjust the blocksize and lastblock to account for the possibility
     *  that the storage server uses a different blocksize than that used
     *  on the disk.
     */
    desc->blocksize = SECSIZE * BLOCK_FACTOR;

    desc->lastblock = (label->dkl_ncyl * 
                       label->dkl_nhead *
                       label->dkl_nsect) / BLOCK_FACTOR;

    /* Since we were busy-waiting above, we never made it into the interrupt
     *   routine.  Thus see if any operations were enqueued while this
     *   function was being executed.
     */
     XylCreateFlag = 0; /* Finished with busywait initialization. */
     Lockq ( &XylDiskq );
     if( (pd=XylDiskq.head) != NULL )
       {
#ifdef XYL_DEBUG
	printx("Xyl: another requestor queued during Create operation\n" );
#endif
	if( (XylDiskq.head = pd->link) == NULL )
           XylDiskq.tail = (Process *) &(XylDiskq.head);
	pd->queuePtr = NULL;
	Unlockq( &XylDiskq );
	/* start a new op. */
	XylDiskRequestor = pd->pid;
	XylBuildandGo( pd );
      }
    else 
      {
         XylDiskRequestor = 0;
         Unlockq( &XylDiskq );
      }
#ifdef XYL_DEBUG
    printx("Xyl: Create operation complete\n" );
#endif
    return( OK );
  }



SystemCode XylPowerup()
  
  /*  Powerup and initialize the Xylogics disk Interface Board.
   */
  {
    extern int Asm_XylInterrupt();
    register int (**intvec)() = (int(**)()) INT2; /* Level 2 */
    unsigned       buffersize, garbage;
    unsigned short unit, j;

    /* Plug interrupt location */
    *intvec = Asm_XylInterrupt;

    /* initialize queue pointers */
    XylDiskq.head = NULL;
    XylDiskq.tail = (Process *) &(XylDiskq.head);
    XylDiskq.type = DISK_QUEUE;

    /* Test to see if the Xylogics controller is out there. */ 
    printx("\nProbe for Xylogics controller: ");
    if( !Probe( DEVADDR, PROBE_READ ) )
      {
        printx( "controller not found\n" );
        controller_exists = 0;
        return( NOT_FOUND );
      }
    else
      {
        printx( "controller found\n" );
        *LastPeripheral++ = PRF_DISK_XYLOGICS;
        controller_exists = 1;
      }  

#ifdef SUN2
    /* Sun-2:  Allocate DVMA space for parameter blocks and for mapping
     *  in user buffers */
    buffersize = uptopage(DISK_MAX_BYTES) + PAGE_SIZE;
    AllocateDvmaSpace( buffersize + sizeof(Iopb),
        &dkspace_pa, &dkspace_va ); 
    
    /* Map in a page of physical memory for the parameter blocks */
    AllocatePage( dkspace_va + buffersize );
#else
    /* Sun-1:  Allocate contiguous Multibus memory for parameter
     *  blocks and the kernel disk buffer */
    buffersize = (unsigned) uptopage(DISK_MAX_BYTES);
    AllocateMultibusMemory( buffersize + sizeof(Iopb),
        &dkspace_pa, &dkspace_va ); 

    /* disk buffer */
    dkbuffer_va = dkspace_va;
    dkbuffer_pa = dkspace_pa;

#endif SUN2
    /* i/o parameter block */
    ip = (Iopb *) (dkspace_va + buffersize);
    ip_pa = (Iopb *) (dkspace_pa + buffersize);

     /* reset controller by reading control register - clears interrupt bit */
    garbage = CRUR(devaddr);

    for( unit = 0; unit < MAX_NUMBER_DRIVES; unit++ )
      {
        /* bootstrap dk_parameters[unit] because XylBuild needs them */
        dk_parameters[unit].nhead = 1; /* don't want divide by zero */
        dk_parameters[unit].nsect = 1; 
      }

    /* initialize statistics */
    stats_totalcyl = 0;
    stats_numio = 0;
    for( j = 0; j < MAX_NUMBER_DRIVES; j++ )
      stats_lastcyl[j] = 0;
    
    return( OK );
  }



SystemCode XylRelease( pd, desc )
  Process *pd;
  DeviceInstance *desc;
   /*
    * Release the specified disk file instance.
    */
  {
    desc->owner = 0; /* Free the descriptor */
    return( OK );
  }



SystemCode XylQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance              *inst;
    unsigned short              dirIndex;
  {
    register QueryStorageReply *reply = (QueryStorageReply *) &(pd->msg);

    /* return the average number of cylinders seeked per I/O */

    /* Return max number of blocks per request supported. */
    reply->filler = DISK_MAX_BYTES/(SECSIZE*BLOCK_FACTOR);
    reply->NumReads = stats_numreads;
    reply->NumWrites = stats_numwrites;
    reply->AverageReadLength = stats_averageread;
    reply->AverageWriteLength = stats_averagewrite;
    reply->CylindersPerIO = (stats_numio == 0) ?  0 :
                                              stats_totalcyl / stats_numio;
    reply->BufferHitRate = 0;

    return( OK );
  }



SystemCode XylModify( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance      *inst;
    unsigned short      dirIndex;
  {
    /* clear statistics */
    stats_totalcyl = 0;
    stats_numio = 0;
    stats_numreads = 0;
    stats_numwrites = 0;
    stats_averageread = 0;
    stats_averagewrite = 0;

    return( OK );
  }



SystemCode XylRead( pd, desc )
   Process *pd;
   DeviceInstance *desc;

   /* Handle a read instance request to the Xylogics Disk interface.
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    SystemCode         r;

    if( (r = XylCheckRequest( pd )) != OK ) 
      return( r );

    /* Client waits for operation to complete */ 
    pd->state = AWAITING_INT;
    ++stats_numreads;
    stats_averageread = (stats_averageread == 0 ) ? req->bytecount :
			(stats_averageread + req->bytecount) >> 2;

    /* Issue the read operation. */
    req->requestcode = READ;
    XylStart( pd );
  
    return( NO_REPLY );  /* interrupt routine will reply */
  }



SystemCode XylWrite( pd, desc )
    Process *pd;
    DeviceInstance *desc;

   /* Handle a write instance request to the Xylogics disk interface.
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    SystemCode  r;

    if( (r = XylCheckRequest( pd )) != OK ) 
      return( r );

    ++stats_numwrites;
   /* Does this calculate the average correctly? */
    stats_averagewrite = (stats_averagewrite == 0 ) ? req->bytecount :
			(stats_averagewrite + req->bytecount) >> 2;
    pd->state = AWAITING_INT; 

    /* Issue the write operation. */
    req->requestcode = WRITE;
    XylStart( pd );

    return( NO_REPLY );  /* interrupt routine will reply */
  }



XylStart( pd )
  register Process *pd;

  {
    /* Start a new opertion if one not currently in progress.  Else enqueue
     *   the pd.
     */

    pd->link = NULL;    
    Lockq ( &XylDiskq );
    if( XylDiskRequestor )
      {
#ifdef XYL_DEBUG
        printx("XylStart: enqueuing %x for %s.\n",
		 pd->pid, (pd->msg.sysCode == READ) ? "read" : "write?" );
#endif XYL_DEBUG
        pd->queuePtr = &XylDiskq;
        XylDiskq.tail = (XylDiskq.tail->link = pd);
        Unlockq( &XylDiskq );
      }
    else
      {
         /* indicate the disk requesting process. */
	XylDiskRequestor = pd->pid;

        /* start a new opertion immediately */
        Unlockq( &XylDiskq );
        XylBuildandGo( pd );
      }
  }
 

XylBuildandGo( pd )
  Process *pd;

  {
    /* Calculate cylinder, head, and sector from physical block number 
     *   and build an I/O parameter block.  Then set the controller going.
     */

    register IoRequest *req = (IoRequest *) &(pd->msg);
    register unsigned  short unit;
    register int       nhead, nsect, blk, cylinder;
    register int       sectcnt, head, sector, cmdopt;
    register char      *local_dkbufpa, *userbuffer;
    uschar             cmd;
    unsigned  tempcyl;  /* temporary used in stats calculation */

    /* record 'current_unit' for the interrupt routine */
    current_unit = unit = 
              (DeviceInstanceTable[(req->fileid) & (MAX_DEVICES-1)])->unit;

    cmd = (uschar) req->requestcode;  

    /* multiply blocknumber by the block_factor because storage server 
     *   uses different size blocks than are used on the disk.
     */
    blk = BLOCK_FACTOR * req->blocknumber;

    nhead = dk_parameters[unit].nhead;
    nsect = dk_parameters[unit].nsect;

    if (req->bytecount <= IO_MSG_BUFFER)
      userbuffer = (char *) req->shortbuffer;
    else
      userbuffer = req->bufferptr;

#ifdef SUN2
    /* Copy page map entries from user buffer to DVMA space. */
    MapDvma( dkspace_va, userbuffer, req->bytecount );

    /* Compute the Multibus address of the user buffer's image in
     *   DVMA space */
    local_dkbufpa = dkspace_pa + (userbuffer -(char *) downtopage(userbuffer)); 

#else
    /* Sun-1: copy to multibus memory from client's memory */
    if( cmd == WRITE )
      Copy_bytes( dkbuffer_va, userbuffer, req->bytecount );

    local_dkbufpa = dkbuffer_pa;
#endif
                     
    cylinder = blk / (nhead * nsect);
    head = blk % (nhead * nsect) / nsect;
    sector = blk % nsect;
    sectcnt = ( req->bytecount % SECSIZE == 0 ) ? req->bytecount/SECSIZE :
                                                  req->bytecount/SECSIZE + 1;

    /* gather statistics - used to calculate number of cylinders seeked */
    tempcyl = (cylinder >= stats_lastcyl[unit]) ? 
                          (cylinder - stats_lastcyl[unit]) :
                          (stats_lastcyl[unit] - cylinder);
    if( (MAXUNSIGNED - stats_totalcyl) < tempcyl )
      {
        /* overflow would occur */
      printx("Xylogics driver: Resetting stats_totalcyl and stats_numio\m");
      printx("  to prevent overflow.  stats_totalcyl = %d, stats_numio = %d\n",
          stats_totalcyl, stats_numio);
	stats_numio = 0;
	stats_totalcyl = 0;
      }
    else
      {
        stats_numio++;
        stats_totalcyl += tempcyl;
      }
    stats_lastcyl[unit] = cylinder;
   
    /* fill in I/O parameter block */
    ip->comm = cmd | NORMAL;
    ip->imode = INT_MODE;
    ip->stat1 = 0;
    ip->stat2 = 0;
    ip->throt = THROT;
    ip->drive = (unit << 6) | unit;  
    ip->head = head;
    ip->sect = sector;
    ip->cylh = cylinder >> 8;
    ip->cyll = cylinder;
    ip->scnth = sectcnt >> 8;
    ip->scntl = sectcnt;
    ip->datarh = ((int) local_dkbufpa >> 12) & 0xF0;  /* 20-bit addr. rel. */
    ip->datarl = 0;
    ip->datah = (int) local_dkbufpa >> 8;
    ip->datal = (int) local_dkbufpa;
    /* insert linked iopb address below if using linked iopb's */

#ifdef XYL_DEBUG
    printx("XylBuildandGo: starting controller, cmd = 0x%x\n",cmd);
#endif
    RRlow(devaddr) = 0;
    RRhigh(devaddr) = ((int) ip_pa >> 12) & 0xF0; /* 20-bit address mode */
    ARhigh(devaddr) = (int) ip_pa >> 8;
    ARlow(devaddr) = (int) ip_pa;
    CSR(devaddr) = GBSY | CLR_INT | CLR_ERR;  
  }



XylInitBuildandGo( cmd, unit )
  uschar   cmd;
  unsigned short unit;

  {  
    /* Note that this function must be invoked immediately after reading
     *   the label off of the drive.
     */

    ip->comm = cmd | NORMAL;
    ip->imode = INT_MODE;
    ip->stat1 = 0;
    ip->stat2 = 0;
    ip->throt = THROT;
    ip->drive = (unit << 6) | unit;  
    ip->head = label->dkl_nhead - 1;
    ip->sect = label->dkl_nsect - 1;
    ip->cylh = (label->dkl_ncyl - 1) >> 8;
    ip->cyll = label->dkl_ncyl - 1;
    ip->hdofst = 0;

    /* fill in the parameter table for this drive */
    dk_parameters[unit].nhead = label->dkl_nhead;
    dk_parameters[unit].nsect = label->dkl_nsect;

    /* set the controller going */
    RRlow(devaddr) = 0;
    RRhigh(devaddr) = ((int) ip_pa >> 12) & 0xF0; /* 20-bit address mode */
    ARhigh(devaddr) = (int) ip_pa >> 8;
    ARlow(devaddr) = (int) ip_pa;
    CSR(devaddr) = GBSY | CLR_INT | CLR_ERR;  

  }



SystemCode XylCheckRequest( pd ) 
  register Process *pd;

  /* Check that the read or write request has a legitimate buffer, etc.
   */
  {
    register IoRequest  *req = (IoRequest *) &(pd->msg);
    register unsigned   count;
    register SystemCode r;

    /*  Check length */
    count = req->bytecount;
    req->bytecount = 0; /* To be left zero if a check fails */

    if( count > DISK_MAX_BYTES ) return( BAD_BYTE_COUNT );

    /* Make sure data pointer is valid.
     *  Check that on a word boundary and not in the kernel area.
     */
    if( (count > IO_MSG_BUFFER) && ( (BadUserPtr(req->bufferptr)) ||
      (pd->team->team_space.size < (req->bufferptr + count)) ||
      (((int) req->bufferptr) & 1) ) )
        return( BAD_BUFFER );

    /* Looks OK. */
    req->bytecount = count;
    return( OK );
  }


XylInterrupt()

   /* Handle an interrupt from the disk interface.
    */
 {
    register IoRequest      *req;
    register IoReply        *reply;
    register Process        *pd;
    register DeviceInstance *desc;
    register int            garbage;
    SystemCode	r;
#ifndef SUN2
    unsigned count;
    Team *oldteam;
#endif SUN2

#ifdef XYL_DEBUG
     printx("Xyl: disk interrupt\n" );
#endif
    /* clear the interrupt bit and error bits */
    CSR(devaddr) = CLR_INT | CLR_ERR;

    /* Ignore interrupt otherwise if busy-wait create activity in progress. */
    if( XylCreateFlag ) return;

    MAP_TO_RPD( pd, XylDiskRequestor ); /* Locate requestor, if any */
    if( pd && (pd->state != AWAITING_INT) )
	pd = NULL;
#ifdef XYL_DEBUG
    if( pd == NULL )
      {
	printx( "Xyl: No disk requestor.\n" );
	printx("Xyl command = 0x%x, CSR(devaddr) = 0x%x\n",
							ip->comm, CSR(devaddr));
      }
#endif
    XylDiskRequestor = 0;

    /* Check for errors. */
    if( ip->stat1 & ERROR_MASK )    
      {
        if( ip->stat2 != SUCCESS )
          {
#ifdef XYL_DEBUG
            printx("Error: command = 0x%x, CSR(devaddr) = 0x%x\n",
                        ip->comm, CSR(devaddr));
            printx("Status 1 = 0x%x, Status 2 = 0x%x\n",ip->stat1, ip->stat2);
            printx("Bus_error_register = 0x%x\n",
                       (unsigned short) GetBusErrorReg() );
#endif
          }
        else
          {
            printx("Xyl controller error: stat1 is error stat2 says success.\n");
            printx("Error: command = 0x%x, CSR(devaddr) = 0x%x\n",
                       ip->comm, CSR(devaddr));
            printx("Status 1 = 0x%x, Status 2 = 0x%x\n",ip->stat1, ip->stat2);
          }
	r = DEVICE_ERROR;
      }
    else
      {  /* A successful operation. */

#ifndef SUN2
	if( ip->comm == (READ|NORMAL) && pd ) /* Read operation
	  {
	    req  = (IoRequest *) &(pd->msg);

	    /*  Sun-1: copy to client's memory from multibus memory */
	    oldteam = GetAddressableTeam();
	    count = req->bytecount;

	    SetAddressableTeam(pd->team);
	    if( count <= IO_MSG_BUFFER )
	        Copy_bytes( req->shortbuffer, dkbuffer_va, count);
	    else
	      Copy_bytes( req->bufferptr, dkbuffer_va, count);
	    	
	    SetAddressableTeam(oldteam);
	  }
#endif /* not SUN2 */
	r = OK;
      }
    if( pd != NULL ) /* A valid requestor */
      {
	reply = (IoReply *) &(pd->msg);
	if( (reply->replycode=r) != OK ) reply->bytecount = 0;
	Addready( pd );
      }
    /* start up a new operation if one waiting */
    Lockq ( &XylDiskq );
    if( (pd=XylDiskq.head) != NULL )
      {
        if( (XylDiskq.head = pd->link) == NULL )
		XylDiskq.tail = (Process *) &(XylDiskq.head);
         pd->queuePtr = NULL;
	/* indicate the disk requesting process. */
	XylDiskRequestor = pd->pid;
         Unlockq( &XylDiskq );

#ifdef XYL_DEBUG
         printx("Starting new operation from XylInterrupt\n");
#endif
         XylBuildandGo( pd );
       }
     else 
       {
         XylDiskRequestor = 0;
         Unlockq( &XylDiskq );
       }
  }
