/*
 * V Kernel - Copyright (c) 1981 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 * Copyright (c) 1982 Stanford University.
 *
 * Serial Interface Handling Routines by Andrew Shore May-June 1982
 *	to handle the CadLinc-keytronix keyboard and demultiplex 
 *	keystrokes from mouse squeaks -- implements the `mouse-device'
 *
 * Cadlinc mouse device code debugged by Marvin Theimer and prepared for final 
 * 	installation: July 30 1982.
 *
 * Serial lines are now devices.  Tim Mann, November 4, 1982.
 *
 * Mouse device code changed to reflect the possibility of both SMI and
 * Cadlinc mice.  This effects only the serial line interrupt routine.
 * - Marvin Theimer, 2/83.
 */

#include "interrupt.h"
#include "cbuff.h"
#include "dm.h"
#include "mouse.h"
#include "Vquerykernel.h"

/* Imports */
extern SystemCode NotSupported();
extern short KernelInterrupted;
extern DeviceInstance *GetDevice();

extern short MouseType;

/* Exports */
extern SystemCode SerialCreate();
extern SystemCode SerialRead();
extern SystemCode SerialWrite();
extern SystemCode SerialRelease();

/* Private */
static CircularBuffer serialCbuff[2];

/* Routines */
Init_serial_interface( linenum ) 
    int linenum;
  /* 
   * Initialize the specified serial interface.
   */
  {
    extern int Asm_SerialInterrupt();

    lineservice( Asm_SerialInterrupt );
    linereset( linenum );
    linearmtx( linenum );
    linearmrx( linenum );

  }

/* Kernel Serial interface support routines for the Device Server */

/* Pointers to instances for serial lines */
DeviceInstance	*SerialInInstance[2];
DeviceInstance	*SerialOutInstance[2];

SystemCode SerialCreate( pd, inst )  
    Process *pd;
    DeviceInstance *inst;
  /*
   * Create instances for a serial line.  The instance returned
   *  is used for output to the line.  The instance with the next
   *  larger id is used for input; its parameters may be obtained
   *  by doing a QueryInstance.
   */
 {
    CreateInstanceRequest *req = (CreateInstanceRequest *) &(pd->msg);
    register DeviceInstance *line;
    register int linenum;

    linenum = inst->unit & 1;	/* unit numer is line number */

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    if( (line = SerialInInstance[linenum]) != NULL )
      {
	if( MapPid( line->owner ) )  return (BUSY);
	SerialInInstance[linenum] = 0;
	line->owner = 0;
      }

    if( (line = SerialOutInstance[linenum]) != NULL )
      {
	if( MapPid( line->owner ) )  return (BUSY);
	SerialOutInstance[linenum] = 0;
	line->owner = 0;
      }

    /*
     * Initialize the device instance descriptors, can assume all fields
     * are zero except for id and first instance's owner and unit number.
     */

    /* Output descriptor */
    inst->readfunc = NotSupported;
    inst->writefunc = SerialWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = SerialRelease;

    inst->type = WRITEABLE+STREAM;
    inst->blocksize = 1;
    inst->lastblock = 0;
    inst->lastbytes = inst->blocksize;
    SerialOutInstance[linenum] = inst;

    /* Input descriptor */
    inst++;
    inst->owner = pd->pid;
    inst->unit = linenum;
    inst->readfunc = SerialRead;
    inst->writefunc = NotSupported;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = SerialRelease;

    inst->type = READABLE+STREAM+VARIABLE_BLOCK;
    inst->blocksize = IO_MSG_BUFFER;
    inst->lastblock = MAXBLOCKS;
    inst->lastbytes = MAXBYTES;
    SerialInInstance[linenum] = inst;
		 /* Record the instance for interrupt routine */

    Init_serial_interface(linenum);
    CbuffFlush(&serialCbuff[linenum]);

    return( OK );
 }


SystemCode SerialRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release serial line instance
   */
  {
    IoRequest  *req = (IoRequest *) &(pd->msg);
    IoReply *reply;
    register Process *rwpd;

    if( rwpd = MapPid(inst->reader) ) 
				/* Unblock reader process */
      {
	inst->reader = 0;
	reply = (IoReply *) &(rwpd->msg);
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( rwpd );
      }

    if( rwpd = MapPid(inst->writer) ) 
				/* Unblock writer process */
      {
	inst->writer = 0;
	reply = (IoReply *) &(rwpd->msg);
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( rwpd );
      }

    inst->owner = 0;
    return( OK );
  }


SerialReturnChar(reply, inst, c)
    IoReply		*reply;
    DeviceInstance	*inst;
    char		c;
  {
    reply->replycode = OK;
    reply->shortbuffer[0] = c;
    reply->bytecount = 1;
    inst->nextblock++;
    inst->reader = 0;
  }


SystemCode SerialRead( pd, inst )
    Process *pd;
    DeviceInstance	*inst;

   /* Handle a read instance request for a serial line
    */
  {
    IoRequest *req = (IoRequest *) &(pd->msg);
    register unsigned bytecount;
    register int c, i;
    register int linenum = inst->unit;
    register IoReply *reply = (IoReply *) &(pd->msg);
    register char *sbuff = &reply->shortbuffer[0];
    Process *reader;

    bytecount = req->bytecount;
    reply->bytecount = 0;
    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading. 
     */
    if( inst->reader ) 
	if( MAP_TO_RPD(reader, inst->reader) ) 
	    return( RETRY );

    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
    if( req->blocknumber != inst->nextblock ) return( BAD_BLOCK_NO );

    disable;
    /* Try to get characters from the buffer */
    for (i = 0; i < bytecount; i++)
      {
	if ( (c = CbuffGet(&(serialCbuff[linenum]))) == CBUFF_EMPTY ) break;
	*sbuff++ = c;
      }

    if (i != 0) 
      {
	reply->replycode = OK;
	reply->bytecount = i;
	inst->nextblock++;
	return (OK);  /* One or more characters returned */
      }

    /* Wait for a character to come in */
    inst->reader = pd->pid;
    pd->state = AWAITING_INT;

    return( NO_REPLY );
  }


SerialXmitChar(reply, inst, linenum)
    IoReply		*reply;
    DeviceInstance	*inst;
    short		linenum;
  {
    inst->lastblock++;
    lineput( linenum, reply->shortbuffer[0] );
		  /* character is still in msg */
    reply->bytecount = 1;
    reply->replycode = OK;
    inst->writer = 0;
  }


SystemCode SerialWrite( pd, inst )
    Process *pd;
    DeviceInstance	*inst;

   /* Handle a write instance request to a serial line
    */
  {
    IoRequest *req = (IoRequest *) &(pd->msg);
    IoReply *reply = (IoReply *) &(pd->msg);
    register int linenum = inst->unit;
    register unsigned bytecount;

    bytecount = req->bytecount;
    reply->bytecount = 0;

    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
/*  Disabled to simplify multi-process access to the console.
    if( req->blocknumber != inst->lastblock+1 ) return( BAD_BLOCK_NO );
*/
    disable;
    if (!linereadytx(linenum))
      {
	inst->writer = pd->pid;	/* Wait until UART ready */
	pd->state = AWAITING_INT;

	return (NO_REPLY);
      }

    SerialXmitChar(reply, inst, linenum);
    return( OK );
  }

Call_inthandler(SerialInterrupt)   /* Macro expansion to interrupt-invoked
				    * Call to SerialInterrupt
				    */


SerialInterrupt()

  /*
   * Handle an interrupt from the serial device.
   *   Checks both lines to see which generated the interrupt.
   */
  {
    register Process *pd;
    register unsigned char c;
    register char c1;
    register DeviceInstance *inst;
    register int	linenum;
    static short	squeakCount = 0;	/* bytes left in squeak */
    static short	significantSqueak = 0;	/* flag to say if this squeak
                                                 * represents an `event' */

    linenum = 0;
    for (linenum = 0; linenum < 2; ++linenum)
      {
	if( linereadyrx(linenum) ) /* Receiver interrupt */
	  {
	    c = lineget( linenum );

	    if( (linenum == 0) && (MouseType == PRF_CADLINC_MOUSE) )
              {
		if((squeakCount == 0) && (0x80 <= c) && (c <= 0x87)) 
							/* squeak start */
		  {
		    squeakCount = 3;
		    if( (c & 07) != MouseButtons )
		      {
		        significantSqueak = 1;
			MouseButtons = c & 07;
		      }
		  }
		else if( (squeakCount == 1) && c )
		  {
		    c1 = (char) c;
		    MouseX += (short) c1;
		    significantSqueak = 1;
		  }
		else if(( squeakCount == 2) && c )
		  {
		    c1 = (char) c;
		    MouseY += (short) c1;
		    significantSqueak = 1;
		  }
	      }
	    if ((linenum == 0) && squeakCount)	/* Within a Cadlinc
						   mouse squeak */
	      {
		if( ((--squeakCount) == 0) && significantSqueak )
		  {
		    significantSqueak = 0;
		    MouseEvent = 1;
		  }
	      }
	    else
	      {
		/* not a mouse interrupt -- unblock serial line reader 
					    or queue character for it */
		if( MouseType == PRF_CADLINC_MOUSE && 
		    linenum == 0 && c == ('S'|0x80) )
		  {
		    lineput( 0, '`' );  /* turn off CadLinc mouse */
		  }

		if ( (inst = SerialInInstance[linenum]) != NULL )
		  {
		    if ( MAP_TO_RPD(pd, inst->reader) )
		      {
			/* Reader waiting -- give it the character */
			SerialReturnChar(&(pd->msg), inst, c);
		        Addready( pd );
		      }
		    else /* Reader not waiting -- queue the character */
			CbuffPut(&(serialCbuff[linenum]), c);
		  }		
	      }	
	  }

	if( linereadytx( linenum ) ) /* Transmitter interrupt */
	  {
	    if( (inst = SerialOutInstance[linenum]) != NULL
		&& MAP_TO_RPD(pd, inst->writer) )
	      {
		SerialXmitChar(&(pd->msg), inst, linenum );
		Addready( pd );
	      }
	    else lineresettxint( linenum );
	  }
      }
  }
