/*
 * V Kernel - Copyright (c) 1985 by Lance Berc and David Cheriton
 * Copyright (c) 1985 Stanford University, all rights reserved.
 *
 * Kernel profiling for MC68000 and MC68010
 */

/* Note: If the profile segment changes location one must change the constant
 * in the interrupt routine.
 */

#include <Venviron.h>
#include <Vexceptions.h>

#ifdef SUN2
#include <Vprofile.h>
#include "memory.h"
#include "process.h"

unsigned long ProfileKernelCount, ProfileUserCount;
unsigned long ProfileInitialized = 0;
unsigned long ProfileInProgress = 0;
SystemCode ProfileInit(), ProfileCleanup(), ProfileQuery();
int (*ProfileRomAddr)();

SystemCode HandleProfile( pd )
Process *pd;
  {
    register ProfileRequest *req = (ProfileRequest *) &pd->msg;
    register SystemCode r;

    if ( req->profileReq != PROFILE_DATA ) req->segmentSize = 0;
    switch( req->profileReq ) /* These two cases can be handled at any time */
      {
	case PROFILE_INIT: return( ProfileInit( req ) );
	case PROFILE_QUERY: return( ProfileQuery( req ) );
      }
    if ( ProfileInitialized == 0 ) return( BAD_STATE );
    r = OK;
    switch( req->profileReq )
      {
	case PROFILE_PAUSE: ProfileInProgress = 0; break;
	case PROFILE_RESTART: ProfileInProgress = 1; break;
	case PROFILE_END: r = ProfileCleanup(); break;
	case PROFILE_START:
	    ProfileInProgress = 0;
	    Zero(PROFILESEG, SEG_SIZE);
	    ProfileInProgress = 1;
	    break;
	case PROFILE_DATA:
	    if (ProfileInProgress == 1) return( BAD_STATE );
	    if ( ((unsigned)req->tablePtr < (unsigned) PROFILESEG) ||
((unsigned)req->tablePtr + req->segmentSize > (PROFILESEG+SEG_SIZE)))
		return( BAD_ARGS );
	    req->kernelCount = ProfileKernelCount;
	    req->userCount = ProfileUserCount;
      }
    return(r);
  }

SystemCode ProfileQuery( req )
register ProfileQueryRequest *req;
  {
    req->tablePtr = (Unspec *) PROFILESEG;
    req->tableSize = SEG_SIZE;
    req->fieldWidth = sizeof(unsigned short);
    req->profileInUse = 1;
    req->MustBeZero = 0;
    return( OK );
  }

SystemCode ProfileInit( req )
ProfileRequest *req;
  {
    register unsigned i;
    register int (**intvec)() = (int(**)()) INT7;
    register short psr;
    extern short sr;
    extern FreePages;
    extern ProfileClick();
        
    /* Map in some memory to the profile space in ALL of the contexts.
     * This is hardware specific.
     */
    if (ProfileInitialized == 1) return( BAD_STATE );
    if (FreePages < PGS_PER_SEG) return( NO_MEMORY );
    
    ProfileKernelCount = ProfileUserCount = 0;
    AllocateMemory( PROFILESEG, SEG_SIZE );
    ProfileInitialized = 1;
    psr = sr;
    sr |= 0x700;
    ProfileRomAddr = *intvec;
    *intvec = ProfileClick;
    sr = psr;
    return( ProfileQuery( req ) );
  }

SystemCode ProfileCleanup()
  {
    register unsigned i;
    register int (**intvec)() = (int(**)()) INT7;
    
    if (ProfileInitialized == 0) return( BAD_STATE );
    *intvec = ProfileRomAddr;
    for (i = PROFILESEG; i < PROFILESEG + SEG_SIZE; i += PAGE_SIZE)
	FreePage(i);
    ProfileInitialized = 0;
    ProfileInProgress = 0;
    return( OK );
  }
  
asm(".text");
asm(".globl	ProfileClick");
asm("ProfileClick:");
asm("	movl	ProfileRomAddr, sp@-");	/* Push rom routine address   */
asm("	moveml	#/8080, sp@-");		/* Save some registers        */
asm("	movl	ProfileInProgress, d0");/* Are we profiling now ?     */
asm("	cmpl	#0, d0");
asm("	beq	ProfileClickExit");	/* Not profiling now.         */
asm("	movl	sp@(14), d0");		/* PC when interrupted -> d0  */
asm("	cmpl	#/ffff, d0");		/* Out of 1st segment ?       */
asm("	bhi	ProfileUserClick");	/* Yes, so we can't keep it	     */
asm("	addql	#1, ProfileKernelCount");
asm("	lsrl	#2, d0");		/* Divide by 4 (granularity)	     */
asm("	lsll	#1, d0");		/* Multiply by 2 (sizeof table entry)*/
asm("	addl	#/ea0000, d0");		/* Add to table base ptr	     */
asm("	movl	d0, a0");		/* d0 points to table entry	     */
asm("	cmpw	#/ffff, a0@");		/* Maxed out ?			     */
asm("	beq	ProfileClickExit");	/* Yes; don't wrap around.	     */
asm("	addqw	#1, a0@");		/* Inc the table entry		     */
asm("	bra	ProfileClickExit");
asm("ProfileUserClick:");
asm("	addql	#1, ProfileUserCount");
asm("ProfileClickExit:");
asm("	moveml	sp@+, #/0101");		/* Restore registers          */
asm("	rts");				/* Rom address is on stack    */
#else SUN2

HandleProfile() { return (BAD_STATE); }

#endif

/* This code is left here for help in deciphering the preceeding asm code.
 * It's similar in nature, but was used with the timer interrupt instead of
 * the NMI.
 */
#ifdef undef
ProfileClick()
    /* This routine increments a counter for each four byte chunk of
     * kernel code space. It is called CLICKS_PER_SEC times per second
     * by the timer interrupt routine when profiling is turned on.
     */
    register unsigned *currentFp;
    register unsigned short *Address;
    register unsigned ptr;
    extern unsigned *a6;		/* Current frame pointer */
    extern short KernelInterrupted;	/* Set in interrupt handler */
    
    if (KernelInterrupted)
      {
	/* Find the pc value at the time of the interrupt by tracing
	 * back through the stack to the return pc value of the
	 * interrupt handler's stack frame.
	 */
	currentFp = (unsigned *) a6;	 	/* current stack frame */
	currentFp = (unsigned *) (*currentFp); 	/* independant timer routine */
	currentFp = (unsigned *) (*currentFp);	/* dependant interrupt handler*/
        ptr = ((MC68010ExceptionStackFrame *) (currentFp+6))->pc;
	ptr >>= 2;				/* 4 byte granularity */
	ptr <<= 1;				/* Data entries are 2 bytes */
	Address = (unsigned short *) (PROFILESEG + ptr);
	if (Address < (unsigned short *) PROFILESEG+SEG_SIZE)
	  {
	    if (*Address < MAX_PROFILE_COUNT) (*Address)++;
	  }
/*      else
	    Kabort("Profile address out of range.");
if (*Address > 1) printx("> 1 %x %d\n", Address, *Address);
*/
        ProfileKernelCount++;
      }
    else ProfileUserCount++;
  }
#endif
