/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann
 *
 * Memory mapping routines--machine dependent
 */

#include "process.h"
#include "memory.h"
#include "interrupt.h"
#include "Vexceptions.h"
#include "Vquerykernel.h"
#include <b.out.h>

extern MachineConfigurationReply MachineConfig;

PageMapEntry OnboardFreePageList = NOPAGE;   /* Free list of pages */
PageMapEntry MultibusFreePageList = NOPAGE;
int OnboardFreePages = 0;
int MultibusFreePages = 0;
struct bhdr KernelBheader = {0},
	    TeamBheader = {0};		/* Force to data segment */
Page *FreeKernelSpace;		/* First unused page in kernel space */
Page *FreeKernMBSpace;		/* First unused page in kernel Multibus
			         *   access segment */
short MultibusHack = 0;
Team *AddressableTeam;

int Asm_MultibusParityError(), MultibusParityError();

Init_memory_mapping()
  /*
   * Setup the memory mapping as required.
   * We assume the memory mapping has been initialized by the Sun PROM
   *   monitor in the standard way:
   * Addresses 0 to MBMEM (1 megabyte) are mapped to the Sun onboard or P2
   *   bus memory, if that much memory exists.  If there is more than one
   *   megabyte of P2 bus memory, the additional memory is not mapped.
   * Addresses MBMEM to TEAM_LIMIT are mapped to ordinary Multibus memory,
   *   if that much exists.
   * Addresses TEAM_LIMIT to MBIO are mapped to devices in the Multibus 
   *   memory address space.  A frame buffer is usually located at
   *   TEAM_LIMIT.
   * Addresses MBIO to (slightly below) CONTEXT_SIZE are mapped to the
   *   Multibus I/O address space.
   *
   * We assume the loader program has placed the kernel at KERNEL_START and
   *   the first team at TEAM_START, with the team's b.out header just
   *   below TEAM_START, and the kernel's b.out header just below that.
   *
   * This routine maps memory as follows:
   * The kernel is assigned addresses between 0 and TEAM_START.  Any unused
   *   memory in this area is added to the free list.
   * Addresses from TEAM_START to TEAM_LIMIT are available to teams.  The first
   *   team is initially given just enough mapped memory in this area
   *   to hold its code and data.  All other memory is added to the free
   *   list.
   * The area from V_SUN1_FRAMEBUFFER to CONTEXT_SIZE is left mapped as the
   *   Sun monitor has it, to permit access to devices in the Multibus memory 
   *   and I/O space by all teams, except for the segment at KERNMB, which is
   *   reserved for kernel access to multibus memory.
   *
   * If MultibusHack is turned on, things are a bit different.  We leave 
   *   Multibus memory space from 00000 to 80000 mapped to virtual addresses 
   *   100000 through 180000 in all address spaces, to allow access to special
   *   Multibus devices or shared memory.  Teams may only grow up to 100000.
   *   We look for remappable Multibus memory between 80000 and the start
   *   of the frame buffer.
   *
   * The free list is actually maintained as two separate lists, one for
   *   onboard and the other for Multibus memory, since Multibus memory is
   *   much slower.  Pages are allocated from the onboard list until it
   *   is empty, then from the Multibus list.
   */
  {
    int onboardPages, multibusPages;
    char *ttop, *ktop, *cptr;
    int ctx, i;
    PageMapEntry freepg, oldpg;
    Page *page;
    Segment *seg;

    /* Plug Multibus parity error interrupt location (level 4) */
    *( (int (**)()) INT4 ) = Asm_MultibusParityError;

    /* Copy the b.out headers into kernel data space */
    TeamBheader = *((struct bhdr *) (TEAM_START - sizeof(struct bhdr)));
    KernelBheader = *((struct bhdr *) (TEAM_START - 2*sizeof(struct bhdr)));

    ttop = (char *) TEAM_START + TeamBheader.tsize + TeamBheader.dsize
				     + TeamBheader.bsize + INIT_STACK;
    ktop = (char *) KERNEL_START + KernelBheader.tsize + KernelBheader.dsize 
				     + KernelBheader.bsize;

    /* Zero the bss area in case the loader didn't do it */
    cptr = (char *) TEAM_START + TeamBheader.tsize + TeamBheader.dsize;
    for (i = TeamBheader.bsize; i; i--) *cptr++ = '\0';

    cptr = (char *) KERNEL_START + KernelBheader.tsize + KernelBheader.dsize;
    for (i = KernelBheader.bsize; i; i-- ) *cptr++ = '\0';

    FreeKernelSpace = uptopage(ktop);
    FreeKernMBSpace = (Page *) KERNMB;

    FindConfiguration();  /* we need this information right away */

    GetMemorySize(&onboardPages, &multibusPages);
    MachineConfig.fastMemory = onboardPages * PAGESIZE;
    MachineConfig.slowMemory = multibusPages * PAGESIZE;
    MachineConfig.memory = MachineConfig.fastMemory + 
	MachineConfig.slowMemory;

    printx("Onboard memory: 0x%x bytes.  Multibus memory: 0x%x bytes.\r\n",
	MachineConfig.fastMemory, MachineConfig.slowMemory);

    for (ctx = 0; ctx < NUMCTXS; ctx++) /* Protect kernel from users */
      {
	setcontext(ctx);
	for (seg = 0; seg < (Segment *) TEAM_START; seg++)
	  {
	    segmap(seg) = RWX___ | (segmap(seg) & (PHYSSEGS-1));
	  }
	/* Reserve kernel Multibus access segment, but don't bother
	 *  unmapping pages from it */
	segmap(KERNMB) = RWX___ | (segmap(KERNMB) & (PHYSSEGS-1));
      }

    /* Unmap segments from contexts other than 0 */
    /* Leave frame buffer and Multibus I/O in all contexts */
    for (ctx = 1; ctx < NUMCTXS; ctx++)
      {
	setcontext(ctx);
	if (MultibusHack)
	  {
	    for (seg = (Segment *) TEAM_START; 
		        seg < (Segment *) MBMEM; seg++)
	        segmap(seg) = UNUSED;
	    for (seg = (Segment *) MBHACK_END; 
		        seg < (Segment *) TEAM_LIMIT; seg++)
	        segmap(seg) = UNUSED;
	  }
	else
	  {
	    for (seg = (Segment *) TEAM_START; 
		        seg < (Segment *) TEAM_LIMIT; seg++)
	        segmap(seg) = UNUSED;
	  }   
      }
    setcontext(0);

    /* Construct free list of pages */

    /* First, unmap unused kernel memory from kernel segments */
    for (page = uptopage(ktop); page < ((Page *) TEAM_START); page++)
	FreePage(page);

    /* Next, free unused onboard memory (up to MBMEM) */
    for ( page = uptopage(ttop); 
	  page < (Page *) min(onboardPages*PAGESIZE, MBMEM); 
	  page++ )
	FreePage(page);

    /* Unmap nonexistent pages from onboard area */
    for ( ; page < (Page *) MBMEM; page++)
	pgmap(page) = NOPAGE;

    /* Now free any Multibus memory found */
    if (MultibusHack)
      {
        for ( page = (Page *) MBHACK_END;
	      page < (Page *) (MBHACK_END + multibusPages*PAGESIZE); page++)
	    FreePage(page);
      }
    else
      {
        for ( ; page < (Page *) (MBMEM + multibusPages*PAGESIZE); page++)
	    FreePage(page);
      }


    /* Unmap nonexistent pages from Multibus area, and
     *   free unused segments.  Leave frame buffer and Multibus I/O. */
    if (MultibusHack)
      {
        for ( ; page < (Page *) MBMEM; page++)
	    pgmap(page) = NOPAGE;
        for (seg = (Segment *) uptoseg(ttop);
	     seg < (Segment *) MBMEM; seg++)
            segmap(seg) = UNUSED;
        for (seg = (Segment *) MBHACK_END;
	     seg < (Segment *) V_SUN1_FRAMEBUFFER; seg++)
            segmap(seg) = UNUSED;
      }
    else
      {
        for ( ; page < (Page *) TEAM_LIMIT; page++)
	    pgmap(page) = NOPAGE;
        for (seg = (Segment *) uptoseg(ttop);
	     seg < (Segment *) TEAM_LIMIT; seg++)
	    segmap(seg) = UNUSED;
      }

    /* Add the rest of onboard memory to the free list, if
     *  there was more than 1 megabyte */
    oldpg = pgmap(TESTPAGE);
    for (freepg = MBMEM/PAGESIZE; freepg < onboardPages; freepg++)
      {
	pgmap(TESTPAGE) = freepg | ONBOARD;
	FreePage(TESTPAGE);
      }
    pgmap(TESTPAGE) = oldpg;

  }

SystemCode SetTeamSize( active )
    Process *active;
  /*
   * Set the memory size for a team, mapping or unmapping pages as needed.
   */
  {
    KernelRequest *req = (KernelRequest *) &(active->msg);
    register Process *pd;
    char *KSetTeamSize();

    if( !(pd = MapPid(req->pid)) ) return( NONEXISTENT_PROCESS );

    req->unspecified[0] = (unsigned) KSetTeamSize(pd->team,req->unspecified[0]);
    return( OK );
  }

char *KSetTeamSize(td, size)
    register Team *td;
    char *size;
  /*
   * Called by SetTeamSize to do the actual work.
   */
  {
    register Page *firstUnused, *newFirstUnused;
    int oldcxt, fail;

    /* Range check on size */
    if (size < (char *) TEAM_START) size = (char *) TEAM_START;
    if (MultibusHack)
	if (size >= (char *) MBMEM) size = (char *) MBMEM;
    else
	if (size >= (char *) TEAM_LIMIT) size = (char *) TEAM_LIMIT;

    oldcxt = getcontext();	/* Temporarily switch contexts */
    SetContextReg(td->team_space.context);

    firstUnused = uptopage(td->team_space.size);
    newFirstUnused = uptopage(size);

    fail = 0;
    if (firstUnused <= newFirstUnused)
      {
	/* More memory needed */
	for (; firstUnused < newFirstUnused; firstUnused++)
	  {
	    if (fail = AllocatePage(firstUnused)) break;
	  }
      }
    else
      {
	/* Less memory needed */
	for (firstUnused--; firstUnused >= newFirstUnused; firstUnused--)
	  {
	    FreePage(firstUnused);
	    if (firstUnused == uptoseg(firstUnused))
	      {
		segmap(firstUnused) = UNUSED;  /* Release the phys seg */
	      }
	  }
      }

    setcontext(oldcxt);
    if (fail)
      {
	return (td->team_space.size = ((char *) firstUnused));
      }
    else
      {
	return (td->team_space.size = size);
      }

  }



FreePage(pgaddr)
    Page *pgaddr;
  /*
   * Add a page to the free list, unmapping it from the current context
   */
  {
    PageMapEntry freepg;

    freepg = pgmap(pgaddr);	 /* Get map entry pointing to this page */
    if ( (freepg & PAGETYPE) == ONBOARD )
      {
	/* Point page to old list top */
	pgaddr->nextpage = OnboardFreePageList;
	OnboardFreePageList = freepg;
	OnboardFreePages++;
      }
    else
      {
	pgaddr->nextpage = MultibusFreePageList;
	MultibusFreePageList = freepg;
        MultibusFreePages++;
      }
    pgmap(pgaddr) = NOPAGE;	 /* Unmap page */
  }


AllocatePage(pgaddr)
    Page *pgaddr;
  /*
   * Map a page into the current context at the given address, if one
   *  can be found.  Return 0 for success, 1 for failure.
   */

  {
    PageMapEntry newpg;
    int segno;

    /* Find a page, if available */
    if ( (newpg = OnboardFreePageList) == NOPAGE &&
	 (newpg = MultibusFreePageList) == NOPAGE )
      {
	return(1);
      }

    /* Make sure there is a physical segment mapped to this logical seg */
    if ((segmap(pgaddr) & SEGMAP) == UNUSED)
      {
	/* Find a physical segment with no page mapped into this
	 *  slot.  This page slot is guaranteed to be the first in the
	 *  segment, and if it is empty, the segment is guaranteed
	 *  to be unused. This works since we always fill segments from
	 *  low addresses to high, and always unmap empty segments.
	 */
	for (segno = 0; segno < PHYSSEGS; segno++)
	  {
	    segmap(pgaddr) = RWXRWX | segno;
	    if (pgmap(pgaddr) == NOPAGE) break;
	  }
	if (segno == PHYSSEGS)	  /* No segs available */
	  {
	    segmap(pgaddr) = UNUSED;
	    return (1);
	  }
      }

    /* Map in the page */
    pgmap(pgaddr) = newpg;

    /* Remove it from the free list */
    if ( (newpg & PAGETYPE) == ONBOARD )
      {
	OnboardFreePageList = pgaddr->nextpage;
	OnboardFreePages--;
      }
    else
      {
	MultibusFreePageList = pgaddr->nextpage;
	MultibusFreePages--;
      }

    return (0);
  }

/* Allocate physical memory. This routine is meant to be called by the kernel
 * to allocate memory for such uses as process decriptor space, ethernet
 * buffer space, and profiling data space. Is does not check to see if the
 * memory has already been allocated.
 */
SystemCode AllocateMemory( start, length )
char *start;
unsigned length;
  {
    register char *ptr, *end;
    
    end = start + length;
    
    for ( ptr = start; ptr < end; ptr += PAGESIZE )
	if ( AllocatePage( ptr ) != OK )
	  {
	    for ( end = start; end < ptr; end += PAGESIZE )
		FreePage( end );
	    return( NO_MEMORY );
	  }
    return( OK );    
  }

/* Very rudimentary. All of the memory BETTER BE ALLOCATED. */
FreeMemory( start, length )
char *start;
unsigned length;
  {
    register char *ptr, *end;
    
    end = start + length;
    
    for ( ptr = start; ptr < end; ptr += PAGESIZE ) FreePage( ptr );
    return( OK );    
  }


void ReclaimMemory(td)
    Team *td;
  /*
   * Reclaim the memory allocated to a defunct team.
   */
  {
    KSetTeamSize(td, NULL);
  }


InterTeamCopy(dest, dteam, src, steam, bytes)
    Team *dteam, *steam;
    char *dest, *src;
    long unsigned bytes;
  /*
   * This routine is used by MoveFrom and MoveTo to copy bytes between
   *   different teams.
   */

  {
    short oldcxt;
    SegMapEntry oldxfs, xfs;
    int sbytes;

    oldcxt = getcontext();		/* Save old context */
    SetContextReg(dteam->team_space.context);
    oldxfs = segmap(XFERSEG);

    while (bytes > 0)
      {
	/* Set up to move all the desired bytes from one source segment */
	sbytes = ((char *) uptoseg(src+1)) - src;
	sbytes = min(sbytes, bytes);

	/* Map segment containing source bytes into xfer segment */
	SetContextReg(steam->team_space.context);
	xfs = segmap(src);
	SetContextReg(dteam->team_space.context);
	segmap(XFERSEG) = xfs;

	/* Move the bytes */
	Copy(dest, XFERSEG + ((unsigned) src) % SEGSIZ, sbytes);
	bytes -= sbytes;
	src += sbytes;
	dest += sbytes;

      }

    segmap(XFERSEG) = oldxfs;
    setcontext(oldcxt);

  }



AllocateMultibusMemory(size, multibusAddress, mappedAddress)
    unsigned size;
    unsigned *multibusAddress;
    Page **mappedAddress;
  /*
   * Allocate 'size' bytes of contiguous Multibus memory and map it into
   *   kernel space.  Returns the Multibus address of the block in
   *   'multibusAddress', and the address within kernel space in
   *   'mappedAddress'.
   * This routine may only be called during kernel initialization, after
   *   Init_memory_mapping().
   */
  {
    Page *p;
    PageMapEntry newpg;
    Page *limit;
    int *initptr;

    if (MultibusHack)
      {
	/*
	 * This part of the MBHACK is provided for the convenience of the
	 *   multiprocessor gateway machine.  If there is nothing on the 
	 *   Multibus free list, but someone has called this routine, we try 
	 *   taking some pages out of the beginning of the region from Multibus
	 *   addresses 0x00000-0x80000 (the shared-memory region).
	 */
	if (MultibusFreePageList == NOPAGE)
	  {
	    /* p is where the memory is coming from */
	    p = ((Page *) MBMEM) + (FreeKernMBSpace - (Page *) KERNMB);
	    limit = uptopage( ((char *)p) + size );
    
	    /* Set up return values */
	    *mappedAddress = FreeKernMBSpace;
	    *multibusAddress = (unsigned) p - MBMEM;
    
	    /* Move the pages to their new home in kernel space */
	    for (; p < limit; p++)
	      {
		if (!Probe(p, PROBE_INIT)) Kabort("No Multibus memory");
		for (initptr = (int *)p; 
		     initptr < ((int *)p) + (PAGESIZE/sizeof(int));
		     initptr++)
		  {
		    *initptr = -1;  /* not 0!  See GetMemorySize() */
		  }
		newpg = pgmap(p);
		pgmap(p) = NOPAGE;
		pgmap(FreeKernMBSpace) = newpg;
		FreeKernMBSpace++;
	      }	    
	    return;
	  }
      }

    /*
     * Memory was stacked on Multibus free list, so taking pages off
     *   one by one gives contiguous memory with steadily decreasing
     *   addresses.  Thus we fill the requested block from back to front.
     */
    p = FreeKernMBSpace + (size - 1)/PAGESIZE;	/* Start of last page
    						 * in block */
    if (p >= ((Page *) KERNMB) + PGPERSEG)
	 Kabort("Not enough kernel space");
    for (; p >= FreeKernMBSpace; p--)
      {
        newpg = MultibusFreePageList;	/* Find a multibus page */
	if (newpg == NOPAGE) Kabort("No Multibus memory");
	pgmap(p) = newpg;		/* Map it in */
	MultibusFreePages--;
	MultibusFreePageList = p->nextpage;
      }

    /* Set up return values */
    *mappedAddress = FreeKernMBSpace;
    FreeKernMBSpace += (size + PAGESIZE - 1)/PAGESIZE;
    *multibusAddress = (newpg & 0x0FFF) << 11;
  }

	
GetMemorySize(onboardPages, multibusPages)
    int *onboardPages, *multibusPages;
  /*
   * Probe to determine size of on-board and Multibus memory on 
   *  a Sun. Returns the amount of each type of memory, in pages.
   *  Multibus memory is assumed to start at Multibus address 00000,
   *  unless MultibusHack is on, in which case it starts at 
   *  (MBHACK_END - MBMEM).
   */
  {
    PageMapEntry oldPage;
    register int opages, mpages, *p;

    /* Save old contents of page map entry we are using */
    oldPage = pgmap(TESTPAGE);

    /* Test onboard pages until one doesn't work */
    opages = TEAM_START/PAGESIZE;  /* Don't test kernel memory */
    while (opages < MAX_ONBOARD_PAGES)
      {
	pgmap(TESTPAGE) = ONBOARD | opages;
	if (Probe(TESTPAGE, PROBE_PRESERVE))
	    opages++;	/* more memory here */
	else
	    break;	/* no more */
      }
	
    /* Test Multibus pages until an error occurs,
     *  writing into all locations of each page found
     *  to initialize it to correct parity.
     */
    if (MultibusHack)
	mpages = (MBHACK_END - MBMEM)/PAGESIZE;
    else
        mpages = 0;

    while (mpages < MAX_MULTIBUS_PAGES)
      {
	pgmap(TESTPAGE) = MULTMEM | mpages;
	if (Probe(TESTPAGE, PROBE_INIT))
	    mpages++;	/* more memory here */
	else
	    break;	/* no more */

	for (p = (int *) TESTPAGE; p < (int *) (TESTPAGE + PAGESIZE); p++)
	    *p = -1;  /* Must not be *p = 0 because that would generate 
		       * a clrw instruction, which reads before writing! */
      }

    *onboardPages = opages;

    if (MultibusHack)
      {
        *multibusPages = mpages - (MBHACK_END - MBMEM)/PAGESIZE;
      }
    else
      {
        *multibusPages = mpages;
      }

    pgmap(TESTPAGE) = oldPage;
  }


/*
 * Interrupt handler for Multibus parity error interrupts.
 */
MultibusParityError()
  {
    Kabort("Multibus parity error");
  }

Call_inthandler(MultibusParityError);



Probe(address, how)
    short *address;
    int how;
  /*
   * Look for memory at address.
   * If how == PROBE_INIT,
   *   write before reading to initialize, then look for writeable memory.
   * If how == PROBE_PRESERVE,
   *   look for writeable memory, but preserve contents of location.
   * If how == PROBE_READ,
   *   just try reading and see whether it causes a bus error.
   */
  {
    register short *Zero = (short *) 0;
    register short z, x;
    int BusErrorOnProbe(), (*oldBusErrorVector)();

    /* Set bus error vector */
    oldBusErrorVector = *( (int (**)()) BUSERROR );
    *( (int (**)()) BUSERROR ) = BusErrorOnProbe;

    z = *Zero;  /* Save old contents of address 0 */
		
    if (how == PROBE_INIT) 
	*address = 0x5c5c;	/* Initialize the test location */

    x = ~*address;		/* Read from it */

    if (how == PROBE_READ)
      {
	/* Success if we get here with no bus error */
	*( (int (**)()) BUSERROR ) = oldBusErrorVector;
	return (1);
      }
    else
      {
	*address = x;		/* Write back the complement */
	if (*address == x && *Zero == z) 
	  {
	    /* Success */
	    *address = ~x;
	    *( (int (**)()) BUSERROR ) = oldBusErrorVector;
	    return (1);
          }
    	else
	  {
	    /* Failure */
	    asm("ProbeFailed:");
	    *( (int (**)()) BUSERROR ) = oldBusErrorVector;
	    *Zero = z;
            return (0);
          }
      }

    asm("	.globl BusErrorOnProbe");
    asm("BusErrorOnProbe:");
#ifdef MC68010
    asm("	addl	#58, sp");
#else !MC68010
    asm("	addl	#14, sp");
#endif MC68010
    asm("	jmp	ProbeFailed");

  }


/* Memory protection types.  Maps from one of 16 protection
 *  codes, defined in memory.h, into individual bits */
char permission[16] =
  {
    0,			/*  0 */
    SX,			/*  1 */
    SR,			/*  2 */
    SR|SX,		/*  3 */
    SR|SW,		/*  4 */
    SR|SW|SX,		/*  5 */
    SR|UR,		/*  6 */
    SR|SW|UR,		/*  7 */
    SR|UR|SW,		/*  8 */
    SR|SW|UR|UW,	/*  9 */
    SR|SW|UR|UX,	/* 10 */
    SR|SW|UR|UW|UX,	/* 11 */
    SR|SX|UR|UX,	/* 12 */
    SR|SW|SX|UR|UX,	/* 13 */
    SR|SW|SX|UX,	/* 14 */
    SR|SW|SX|UR|UW|UX	/* 15 */
  };


/*
 * Another routine to tell you whether you can read a given location in
 *   memory.  This one, however, is intended specifically for virtual
 *   addresses.  It doesn't actually try to read from the location, just
 *   look at the memory mapping hardware, which had better be set
 *   up properly.
 *
 * May need major fiddling if we introduce virtual memory.
 */

KernelCanRead(address, length)
    unsigned    address, length;
  {
    register unsigned	testadr;

    /* Test one byte in each page from address to address+length-1 */
    /*   If we can read one byte in the page, we can read anything.*/
    /* Since the Sun-1 stores the protection bits in the segment   */
    /*   map, we could speed this up by only checking protection   */
    /*   for one page in each segment.  However, we should still   */
    /*   (I think) check the page-type on every page, unless we're */
    /*   very certain of our logic.				   */
    for (testadr = address; (unsigned)(testadr-address) < length;
			     /* ^ Does this work for all cases? */
	 testadr = (testadr+PAGESIZE) & (-PAGESIZE) /*gag*/)
      {
	/*
	 * The kernel is actually allowed to read from addresses above
	 *   CONTEXT_SIZE, but that's where all the funny hardware lives
	 *   and it shouldn't normally be trying to read from there, so we
	 *   regard it as an error
	 */
	if ( testadr >= CONTEXT_SIZE ||
	     ! ( SR & permission[PermissionField(segmap(testadr))] ) ||
	     NONEXIS == (pgmap(testadr) & PAGETYPE) )
	    return 0;
      }
    return 1;
  }

int DiagnoseBusError(req)
    ExceptionRequest *req;
  /*
   * Diagnose the cause of a bus error.  Used by exception handler.
   * This code still works for either MC68000 or MC68010 because
   *   the only bits of "code" we look at are the function code,
   *   which is in the same place in the word for both.
   */
  {
    SegMapEntry smentry, attempt;
    MC68000StatusWord stat;

    if (req->accaddr > ADDR_LIMIT)
      {
	/* Garbage address -- these processors have 
	 * only 24 bit addressing */
	return (OUT_OF_RANGE);
      }

    if (req->accaddr > CONTEXT_SIZE)
      {
	/* Address outside mapped region */
	if (req->status & SUPERVISOR_STATE)
	    return (SPURIOUS);
	else
	    return (SYSTEM_SPACE);
      }

    /* Get segment map entry for this address */
    smentry = segmap(req->accaddr) & SEGMAP;

    if (smentry == UNUSED) return (SEG_INVALID);

    /* Determine type of access */
    stat.word = req->code;
    switch (stat.field.fc)
      {
      case FC_SUPER_PROGRAM:
	attempt = SX;		/* Supervisor instruction */
	break;

      case FC_SUPER_DATA:
	if (stat.field.rw == 1)
	    attempt = SR;	/* Supervisor read */
	else
	    attempt = SW;	/* Supervisor write */
	break;

      case FC_USER_PROGRAM:
	attempt = UX;		/* User instruction */
	break;

      case FC_USER_DATA:
	if (stat.field.rw == 1)
	    attempt = UR;	/* User read */
	else
	    attempt = UW;	/* User write */
	break;

      default:
	attempt = SX;		/* Should never happen */
	break;
      }

    if ((attempt & permission[PermissionField(smentry)]) == 0)
        return (PROTECTION);

    /* Check page type */
    switch (pgmap(req->accaddr) & PAGETYPE)
      {
      case ONBOARD:
        return (PARITY);

      case NONEXIS:
        return (PAGE_INVALID);

      case MULTMEM:
      case MULTIO:
        return (MB_TIMEOUT);
      }
  }


/*
 * Get memory statistics for QueryKernel operation
 */
GetMemoryStats(reply)
    register MemoryStatisticsReply *reply;
  {
    reply->unusedFastMemory = OnboardFreePages * PAGESIZE;
    reply->unusedSlowMemory = MultibusFreePages * PAGESIZE;
  }

ClearModifiedPages( pd )
  {
    return( REQUEST_NOT_SUPPORTED );
  }

/*
 * Routines to set/get the currently addressable team space.
 * Very simple on the Sun-1.  There are 16 sets of segment map
 *   registers, and we allow at most 16 teams, so we just need to
 *   reset the context register.
 */
/* GetAddressableTeam() is now a macro */

SetAddressableTeam(newteam)
    register Team *newteam;
  {
    AddressableTeam = newteam;
    SetContextReg(newteam->team_space.context);    
  }
